/* ---------------------------------------------------------------------------*

   Name:     uupoll

   Author:   Klaus Dahlenburg <kdburg@incoahe.hanse.de>

   Status:   Public domain

   Copyright: none

   Funktion: The main intention behind this program was to get a full
             replacement of the uupoll supplied by NeXT when using an
             UUCP or a file structure that is different from that hardwired
             config in NeXT's uupoll. The lack of source made it impossible
             to modify the supplied uupoll.
  
   Call:     uupoll [-n] [-x] [-g[A | 0-9,A-Z,a-z]] site ...
             
                    -n    just place a poll file but do not call uucico;
                          This option can be given only once.
                    -x    meaningful only for sites not affected by the -n
                          option. It prevents the creation of a poll file; 
                          the default is to place one. In case the poll fails 
                          there will be no attempt to poll those sites on 
                          the next general (unspecific) poll. If using 
                          autopoll the site will be called at the next + 1
                          run of autopoll.
                    -g    any grade may be given to meet the criteria for
                          a successful poll. The default being specified
                          in conf.h (A).
                          This option may be given individually for each
                          site to call.
                    site  the name of the site to be called. As many sites
                          as necessary may be specified separated by at least
                          one blank.
             Note: any site will be called with the options currently in
                   effect. The working order is left to right. Example:
                   uupoll -gQ site1 site2 -gZ site3 -n site4
                   site1 and site2 will be called immediate with grade Q
                   site3 will be called immediate with grade Z. Site4 will
                   have a poll file created with grade Z.

   Environment: NeXT 2.1

   Called Programs: sort, uniq, uucico (or autopoll), uuname

   Compile:     no special options are needed

   Comments:    - should run setuid UUCP or whatever userid is necessary to
                  write to the spool directory with the proper ownership of
                  the files and to run uucico.
                - No alias expansion is done on the given names.
*/

#include "conf.h"

#if !defined(lint)
static char rcsid[] = "$Id: uupoll.c,v 2.1 1993/05/16 21:48:15 kdburg Rel $";
#endif /* not lint */

/* $Log: uupoll.c,v $
 * Revision 2.1  1993/05/16  21:48:15  kdburg
 * changed exit() to _exit() in case the exec fails within child
 *
 * Revision 2.1  1993/05/16  21:48:15  kdburg
 * changed exit() to _exit() in case the exec fails within child
 *
 * Revision 2.0  1993/05/16  14:11:04  kdburg
 * initial revision
 * */

#define CAT 16
#define SEVERE 8
#define WARNING 4
#define OK 0

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/file.h>
#include <sys/param.h>
#include <pwd.h>
#include <sys/time.h>

int maxtab = 0;                            /* high-water-mark for site tab */

char This_Site[MAXHOSTNAMELEN+1] = "";    /* our name */
char *Msg_Log;                            /* pointer to msglog filename */
char *Poll_Dir;                           /* pointer to poll program */
char *grade;                              /* pointer to the grade */
char *Poll_Pgm_Name;                      /* programname of pollpgm */
char *called_as;                          /* called by this name */
char Sort[] = "uuname | sort | uniq";     /* how to obtain site names */
char System[MAXHOSTNAMELEN+1] = "";       /* intermediate holds site name */
char workf[300];

struct Sites {
       char name[MAXHOSTNAMELEN+1];    /* name of site as supplied by uuname */
       char grade[1];                  /* as passed or default */
       int flag;                       /* TRUE this site should be polled */
       int asap;                       /* 1 without -n; 2 with -x option */
};
struct Sites Sitetab[SITE_MAX];
struct timeval tp;
struct timezone tzp;
struct passwd *pwd;

/* define the prototypes
 * */

extern int gethostname(char *name, int namelen);
extern int system(char *cmd);
extern int fork();
extern int execlp(char *name, char *arg0, ...);
extern void *malloc(size_t byteSize);
extern int Check_Args(int argc, char *argv[]);
extern int getuid();
extern int Housekeeping(char *argv[]);
extern int Call_Site();
extern int open(char *path, int flags, int mode);
extern int close(int fd);
#ifdef __STRICT_ANSI__
extern FILE *popen(char *command, char *type);
extern int pclose(FILE *stream);
extern void _exit(int status);
#endif  /* __STRICT_ANSI__ */
#ifdef __STRICT_BSD__
extern int fprintf(FILE *stream, const char *format, ...);
extern int fclose(FILE *stream);
extern char *strerror(int errnum);
extern int fflush(FILE *stream);
extern void exit(int status);
#endif /* __STRICT_BSD__ */

/* --------------------------------------------------------------------------*/
/*                             Main                                          */
/* --------------------------------------------------------------------------*/

int main(int argc, char *argv[])
{
 int k = 0;
 int Maxrc = OK;            /* holds the max. error code encountered */

 Maxrc = Housekeeping(argv);            /* set up runtime vars */

/* If any errors shown up so far they are of such a nature that it is 
 * very questionable to continue; so we better bail out in this case. 
 */
  if (Maxrc <= WARNING) {
     k = Check_Args(argc, argv);
     Maxrc = Maxrc >= k ? Maxrc:k;
     if (Maxrc <= WARNING) {      /* rc > WARNING here will have us to skip too */
        k = Call_Site();
        Maxrc = Maxrc >= k ? Maxrc:k;
     }
  }
  gettimeofday(&tp, &tzp);
  fprintf(stderr,"%s: (I) ended with rc = %i on %s\n",
                 called_as,Maxrc,ctime(&tp.tv_sec));
  fclose(stderr);
  exit (Maxrc);
}

/* --------------------------------------------------------------------------*/
/*                             Functions                                     */
/* --------------------------------------------------------------------------*/

int Housekeeping(char *argv[]) {
 FILE *infile;
 int i,s,k,n = 0;
 int Rc = OK;

/* separate the program name by which we're called from the path
 * */

 strcpy(workf,argv[0]);
 k = strlen(*argv)+1;
 for(i=k+1;i>=0 && workf[--i] != '/';)              /* get last sep */
                      ;
 if ((called_as = (char *)malloc (k - i)) == NULL) {
    fprintf(stderr,"%s: (C) malloc failed (called_as). Reason: %i (%s)\n",
                    argv[0],errno,strerror(errno));
    return (CAT);
 }
 for(i++,s=0;(called_as[s++] = workf[i++]);)         /* name without path */
                      ;

/* if defined set up the name of the message log file otherwise stderr will be used
 * */

#ifdef ULOG_FILE
   if ((Msg_Log = (char *)malloc (sizeof(ULOG_FILE))) == NULL) {
      fprintf(stderr,"%s: (C) malloc failed (Msg_Log). Reason: %s\n",
                     called_as,strerror(errno));
      return (CAT);
   }
   strcpy(Msg_Log,ULOG_FILE);
   if ((freopen(Msg_Log,"a",stderr)) == NULL) {
      fprintf(stdout,"%s: (C) Could not open msglog: %s\n",called_as,Msg_Log);
      return (Rc >= CAT ? Rc:CAT);
   }
#endif  /* ULOG_FILE */

/* put out the started message including the time and the userid.
 * */

 pwd = getpwuid(getuid());
 gettimeofday(&tp, &tzp);
 fprintf(stderr,"\n%s: (I) started by `%s' on %s",
               called_as,(pwd==NULL) ? "???":pwd->pw_name,ctime(&tp.tv_sec));

/* set up the program to call which actually does the poll
 * */

#ifdef AUTO_POLL
   if ((Poll_Dir = (char *)malloc (sizeof(AUTO_DIR))) == NULL) {
      fprintf(stderr,"%s: (C) malloc failed (Poll_Dir). Reason: %i (%s)\n",
                      called_as,errno,strerror(errno));
      return (CAT);
   }
   strcpy(Poll_Dir,AUTO_DIR);             /* autopoll lives here  */
#else  /* ! AUTO_POLL */
   if ((Poll_Dir = (char *)malloc (sizeof(CICO_DIR))) == NULL) {
      fprintf(stderr,"%s: (C) malloc failed (Poll_Dir). Reason: %i (%s)\n",
                      called_as,errno,strerror(errno));
      return (CAT);
   }
   strcpy(Poll_Dir,CICO_DIR);             /* uucico lives here  */
#endif /* AUTO_POLL */

 k = strlen(Poll_Dir)+1;
 for(i=k;i>=0 && Poll_Dir[--i] != '/';)             /* get last sep */
                      ;
 if ((k - i) == 2) {
    fprintf(stdout,"%s: (E) the program to call is missing: %s\n",
                    called_as,Poll_Dir);
    Rc = Rc >= SEVERE ? Rc:SEVERE;
 }
 else {
    if ((Poll_Pgm_Name = (char *)malloc (k - i)) == NULL) {
       fprintf(stderr,"%s: (C) malloc failed (Poll_Pgm_Name). Reason: %i (%s)\n",
                      called_as,errno,strerror(errno));
       return (CAT);
    }
    for(i++,s=0;(Poll_Pgm_Name[s++] = Poll_Dir[i++]);)  /* name without path */
                        ;
 }

/* set up the default grade 
 * */

 if ((grade = (char *)malloc (sizeof("A"))) == NULL) {
    fprintf(stderr,"%s: (C) malloc failed (grade). Reason: %i (%s)\n",
                    called_as,errno,strerror(errno));
    return (CAT);
 }
 strcpy(grade,"A");
#ifdef DEF_GRADE
   if (strlen(DEF_GRADE) > 1) {
      fprintf(stderr,"%s: (W) grade %s invalid; default `%s' used\n",
                      called_as,DEF_GRADE,grade);
      Rc = Rc >= WARNING ? Rc:WARNING;
   }
   else
      strcpy(grade,DEF_GRADE);             /* autopoll lives here  */
#endif /* DEF_GRADE */

/* obtain our sitename
 * */

 if ((gethostname(This_Site,MAXHOSTNAMELEN+1)) != 0) {
    fprintf(stderr,"%s: (W) hostname could not be obtained. Reason-code: %i (%s)\n",
                    called_as,errno,strerror(errno));
    Rc = Rc >= WARNING ? Rc:WARNING;
 }

/* Obtain all sitenames known by this site via uuname. Get them sorted and
 * discard any duplicates
 * */

  i = 0;
  if ((infile=popen(Sort,"r")) != NULL) {
     while(fgets(Sitetab[i].name,MAXHOSTNAMELEN+1,infile) != NULL) {
          if (i > SITE_MAX) {            /* let'm run so that we can give */
             i++;                        /* the user some guidance */
             continue;                   /* we'll tell the user later on */
          }
          n = strlen(Sitetab[i].name)-1; /* offset: next to last char */
          Sitetab[i].name[n] = '\0';     /* strip trailing newline */
          Sitetab[i].flag = 0;           /* TRUE: poll this site */
          Sitetab[i].asap = 0;           /* TRUE: when -n is absent */
          *Sitetab[i].grade = *grade;    /* that's the default grade */
          maxtab = i++;                  /* set high-water-mark */
     }
     pclose(infile);
  }

/* in case the internal table overflowed we'll now give notice and tell
 * the user by which amount the table has to be increased to hold all site-
 * names
 */
  if (i > SITE_MAX) {
     fprintf(stderr,"%s: (E) number of sites > internal tab\n",called_as);
     fprintf(stderr,"%s: (E) increase SITE_MAX to >= %d and recompile\n",
                    called_as,i);
     Rc = Rc >= SEVERE ? Rc:SEVERE;
     }

/* check for any failures that may have occured during command execution
 * and therefore we're unable to obtain the site names at all. 
 */
  if (maxtab == 0) {
     fprintf(stderr,"%s: (E) could not obtain sitenames.\n",called_as);
     Rc = Rc >= SEVERE ? Rc:SEVERE;
  }
 return (Rc);
}


int Check_Args(int argc, char *argv[]) {
  int i,s,k,n = 0;
  int Rc = OK;
  int One_Site = 0;          /* TRUE: found at least one valid site to call */
  int poll_file = 1;         /* FALSE: after -x option given                  */
  int def_flag = 0;          /* TRUE: when option -n was encountered */

   /* --------------------------------------------------------------*/
   /*               check the arguments passed to us                */
   /*                                                               */
   /* These are: -n  -> place a POLL file but do not start uucico   */
   /*            -x  -> do not place a poll file (immed. poll only) */
   /*            -g? -> specify a grade with the POLL file. The ?   */
   /*                   can be by 0-9, A-Z, a-z                     */
   /*            site   name of the site to call. There many be as  */
   /*                   many as necessary separated by at least one */
   /*                   blank                                       */
   /* Note: all options will stay in effect as long as they are'nt  */
   /*       changed by a new setting. The options -n and -x can't   */
   /*       be negated once given; that means place all sites       */
   /*       that should be immediately polled to the left of the    */
   /*       -n option; the same applies to the -x option which must */
   /*       be left of the -n option to come into effect!           */
   /*       The working order is left to right!                     */
   /* --------------------------------------------------------------*/

 for (i = 1, s = 0; i < argc; i++) {
   k = strlen(argv[i]);
   switch (*argv[i]) {

      /*      ---->     handle the options         */

      case '-':
           n = 1;
           switch (*(argv[i]+n)) {
              case 'n':
                   if (k > 2) {
                      fprintf(stderr,"%s: (E) invalid specification %s\n",
                                     called_as,argv[i]);
                      Rc = Rc >= SEVERE ? Rc:SEVERE;
                      break;
                   }
                   def_flag = 1;
                   break;
              case 'x':
                   if (k > 2) {
                      fprintf(stderr,"%s: (E) invalid specification %s\n",
                                     called_as,argv[i]);
                      Rc = Rc >= SEVERE ? Rc:SEVERE;
                      break;
                   }
                   if (def_flag) {
                      fprintf(stderr,"%s: (W) -x after -n has no effect\n",
                                     called_as);
                      Rc = Rc >= WARNING ? Rc:WARNING;
                      }
                   else {
                      poll_file = 0;
                   }
                   break;
               case 'g':
                    if (k > 3) {
                       fprintf(stderr,"%s: (E) invalid specification %s\n",
                                       called_as,argv[i]);
                      Rc = Rc >= SEVERE ? Rc:SEVERE;
                      break;
                    }
                    if (*(argv[i]+n+1) == '\0') {
                       fprintf(stderr,"%s: (E) missing grade\n",called_as);
                       Rc = Rc >= SEVERE ? Rc:SEVERE;
                       break;
                    }
                    if (isalnum(*(argv[i]+n+1)) == 0) {
                       fprintf(stderr,"%s: (E) invalid grade %s\n",
                                      called_as,argv[i]);
                       Rc = Rc >= SEVERE ? Rc:SEVERE;
                       break;
                    }
                    *grade = *(argv[i]+n+1);
                    break;
               default:
                    fprintf(stderr,"%s: (W) missing/unknown option `-%s' ignored\n",
                                    called_as,argv[i]+n);
                     Rc = Rc >= WARNING ? Rc:WARNING;
                    break;
           }  /* end of switch (*(argv[i]+n)) */
           break;

  /*      ---->     handle the sitenames         */

     default:
        if (strcmp(argv[i],This_Site) == 0) {
           fprintf(stderr,"%s: (W) ignoring to call *ourself* %s\n",
                          called_as,argv[i]);
           Rc = Rc >= WARNING ? Rc:WARNING;
           break;
        }
        strcpy(System,argv[i]);
        for(s=0;s<=maxtab;s++) {
            if ((n=strcmp(Sitetab[s].name,System)) >= 0) {
                break;
            }
        }
        if (n != 0) {
           fprintf(stderr,"%s: (W) unknown site (ignored): %s\n",
                          called_as,System);
           Rc = Rc >= WARNING ? Rc:WARNING;
           break;
        }

  /*      ---->     if there are no errors we arrive here to save the data  */

        *Sitetab[s].grade = *grade;
        One_Site = Sitetab[s].flag = 1; /* indicate to poll this site */
        if (def_flag)
           Sitetab[s].asap = 0;         /* poll on next schedule */
        else {
           Sitetab[s].asap = 1;         /* poll immediately */
           if (! poll_file)
              Sitetab[s].asap++;        /* and do not place a poll file */
        }
        s++;
        break;
   }        /* end of switch (*argv[i]) */ 
 }          /* end of for(...) */

/* now let's check what we've gotten so far. If no valid data has been */
/* entered we will indicate to skip further processing                */

 if (! One_Site) { 
    fprintf(stderr,"%s: (E) found no site to call\n",called_as);
    Rc = Rc >= SEVERE ? Rc:SEVERE;
 }

return (Rc);
}

int Call_Site() {

/* For all sites that carry the -n flag we will place                   */
/* a poll file into the apropriate directory. For all others there will */
/* be an immediate call to uucico (or autopoll)                         */
/* Those sites that have been named on the command have the flag byte   */
/* set to one.                                                          */

  int fdpoll;                /* fildes for the poll file */
  int mode = 00647;          /* mode for poll file */
  int i = 0;
  int Rc = OK;
  int pid = 0;               /* process-id after fork() */

 for(i=0;(i<=maxtab);i++) {
    if (Sitetab[i].flag == 0)          /* should we trigger this one ?   */
        continue;                      /* nope */

/* processing done for delayed polls only                                */

    if (Sitetab[i].asap <= 1) {       /* do not place a poll file for sites */
                                      /* that will be polled immediate and  */
                                      /* carry the -x option                */
#ifdef HAVE_SPOOLDIR_TAYLOR
       sprintf(workf,"%s/%s/C./C.%sPOLL",
              SPOOL_DIR,Sitetab[i].name,Sitetab[i].grade);
#endif
#ifdef HAVE_SPOOLDIR_HDB
       sprintf(workf,"%s/%s/C.%s%sPOLL",
               SPOOL_DIR,Sitetab[i].name,Sitetab[i].name,Sitetab[i].grade);
#endif
#ifdef HAVE_SPOOLDIR_BSD
       sprintf(workf,"%s/C./C.%s%sPOLL",
               SPOOL_DIR,Sitetab[i].name,Sitetab[i].grade);
#endif

       fflush(stderr);
       if ((fdpoll=open(workf,O_CREAT,mode)) <= 0) {
          fprintf(stderr,"%s: (E) couldn't place poll file for system: %s. Reason: %s\n",
                          called_as,Sitetab[i].name,strerror(errno));
          Rc = Rc >= SEVERE ? Rc:SEVERE;
       }
       else {
          fprintf(stderr,"%s: (I) site %s will be called %s\n",
                         called_as,Sitetab[i].name,
                        Sitetab[i].asap == 0 ? "upon next poll":"immediately");
          if (close(fdpoll) != 0) {
             fprintf(stderr,"%s: (W) close failed for poll file; system: %s. Reason: %s\n",
                          called_as,Sitetab[i].name,strerror(errno));
             Rc = Rc >= WARNING ? Rc:WARNING;
          }
       }
    }

/* processing done for immediate  polls only
 * there is no wait for the completion of the called program that actually
 * calls the site
 * */

    fflush(stderr);
    if (Rc <= WARNING) {
       if (Sitetab[i].asap >= 1) {
          switch (pid = fork()) {
             case -1:
                 fprintf(stderr,"%s: (C) could not fork() Reason-code: %i (%s)\n",
                                 called_as,errno,strerror(errno)); 
                   return (Rc >= CAT ? Rc:CAT);
             case 0:
                execlp(Poll_Dir, Poll_Pgm_Name, "-r1", "-s", Sitetab[i].name,0);
                fprintf(stderr,"%s: (C) could not start %s. Reason-code: %i (%s)\n",
                                called_as,Poll_Pgm_Name,errno,strerror(errno));
                _exit (CAT);            /* child: bail out */
             default: 
                fflush(stderr);
                fprintf(stderr,"%s: (I) %s [%d] started; site: %s\n",
                                called_as,Poll_Pgm_Name,pid,Sitetab[i].name);
           }     /* switch (pid = fork()) */
       }      /* if (Sitetab ...)  */
    }      /* if (Rc ...) */
 }      /* for(i=0;(i<= ...))  */
 return (Rc);
}
