/* ---------------------------------------------------------------------------*

   Name:     autopoll

   Author:   Klaus Dahlenburg <kdburg@incoahe.hanse.de>

   Status:   Public domain

   Copyright: none

   Function: Autopoll may be called just as uucico is called. The difference
             is that autopoll will call uucico and if the return code is
             not zero a check is made on the status files to see which site's
             call failed. Those sites will be called again at that time found
             in the status file plus any imposed wait. The next call will be
             scheduled via an at job which in turn is handled by cron. 
             Atrun depends on the scheduling granularity of cron so the
             actual times may be later than planned.  
             Autopoll will check the options -f and -s (-S) as well as the name
             of the site passed. All other options will be passed unchecked.
             The -f and -S options will indicate to autopoll that any wait
             to call a site should be ignored; if the call fails the next
             call to those sites will be at the current time plus 120 secs.
             When the time found plus any wait evaluates to a time that 
             passed already the next call will be the current time plus 60
             secs. The name of the site if given must be a valid one and not
             the host itself otherwise it will be ignored.
  
   Call:     autopoll [ options ]

                      all option that apply to uucico may be given and
                      will be passed verbatim. See man uucico(8).
             
   Environment: NeXT 2.1, Taylor UUCP
   
   I/O:         stdin: unused.
                stdout: used only when ALOG_DIR is defined and the file
                        can't be opened. It will be a single message to tell
                        just that and the run is aborted.
                stderr: all messages go here.
                        If ALOG_DIR is defined (see conf.h) all messages will
                        be appended to a file autopoll.msglog in that 
                        directory; the file will be created automatically if
                        necessary; a redirection is then no longer possible. 
                Needs access to .Status files (see Comments later on).

   Called Programs: sort, uniq, uucico, uuname, at

   Compile:     no special options are needed. Compiled with gcc 2.3.3 -O2.
                Compile with the supplied cc might produce erroneous code
                for the check options switch case 's' code: the break inside
                the first if (..) {... break} is ignored.

   Comments:    - should run setuid UUCP or whatever userid is necessary to
                  access (RDONLY) the .Status files and to run the programs
                  named under "Called Programs" above.
                - No alias expansion is done on the given names for the
                  check against uuname's output..
                - Invalid arguments will yield in an exit code of 1 as do
                  "normal" failures. It may therefore happen that a site
                  is called at the intervals with the same invalid arguments.
                - "Wrong time to call" is not handled properly and may 
                  call the site at the intervals until the time ban is lifted.
                - human action is necessary as we can't distinguish between
                  "normal" failures and "errors" such as wrong password,
                  number to dial etc. The logs should be checked periodically.
                - if CICO_DIR points to a non existent program the run may
                  end with signal 10: Bus Error.
                - is has been observed that uucico will time out with "Dial
                  failed" when called via autopoll; setting chat-timeout to
                  value of 40 cured that problem.
*/

#include "conf.h"

#if !defined(lint)
static char rcsid[] = "$Id: autopoll.c,v 2.2 1993/05/17 20:47:05 kdburg Rel $";
#endif /* not lint */

/* $Log: autopoll.c,v $
 * Revision 2.2  1993/05/17  20:47:05  kdburg
 * execution of at cmd also ok always said failed...
 *
 * Revision 2.2  1993/05/17  20:47:05  kdburg
 * execution of at cmd also ok always said failed...
 *
 * Revision 2.1  1993/05/16  21:49:13  kdburg
 * changed exit() to _exit() in case the exec fails within child
 *
 * Revision 2.0  1993/05/16  14:12:05  kdburg
 * initial revision
 * */

#define CAT 16
#define SEVERE 8
#define WARNING 4
#define OK 0

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/wait.h>

int maxtab = 0;                           /* high-water-mark for site tab */
int Single_Site_Tab = 0;                  /* entry into site tab for a site  */
                                          /* passed via -s or -S option  */

char *Msg_Log;                            /* pointer to msglog filename */
char *Poll_Pgm;                           /* here lives autopoll */
char *Cico;                               /* here lives cico */
char *Uucico;                             /* isolated name of cico */
char *called_as;                          /* called by this name */
char *grade;                              /* pointer to the grade */
char This_Site[MAXHOSTNAMELEN+1] = "";    /* our name */
char Sort[] = "uuname | sort | uniq";     /* how to obtain site names */
char System[MAXHOSTNAMELEN+1] = "";       /* intermediate holds site name */
char workf[300];

struct Sites {
       char name[MAXHOSTNAMELEN+1];    /* name of site as supplied by uuname */
       char grade[1];                  /* as passed or default */
       int flag;                       /* TRUE this site should be polled */
       int asap;                       /* 1 without -n; 2 with -x option */
};
struct Sites Sitetab[SITE_MAX];
struct timeval tp;
struct timezone tzp;
struct passwd *pwd;

/* define the prototypes
 * */

extern int gethostname(char *name, int namelen);
extern int system(char *cmd);
extern int fork();
extern int Call_Cico(int argc, char *argv[], union wait *W_Stat); 
extern int Chk_Status(int argc, char *argv[],
                      char*site,
                      int force,
                      union wait *W_Stat);
extern int Check_Site(char *site);
extern int unlink(char *path);
extern void *malloc(size_t byteSize);
extern int execve(char *name, char *argv[], char *envp[]);
extern int execlp(char *name, char *arg0, ...);
extern int chmod(char *path, int mode);
extern int getuid();
extern int getpid();
extern int Housekeeping(char *argv[]);
extern void free(void *ptr);
#ifdef __STRICT_ANSI__
extern FILE *popen(char *command, char *type);
extern int pclose(FILE *stream);
extern void _exit(int status);
#endif  /* __STRICT_ANSI__ */
#ifdef __STRICT_BSD__
extern int fprintf(FILE *stream, const char *format, ...);
extern int fclose(FILE *stream);
extern char *strerror(int errnum);
extern int fflush(FILE *stream);
extern void exit(int status);
extern int fscanf(FILE *stream, const char *format, ...);
extern int sscanf(char *s, const char *format, ...);
#endif /* __STRICT_BSD__ */

/* --------------------------------------------------------------------------*/
/*                             Main                                          */
/* --------------------------------------------------------------------------*/

int main(int argc, char *argv[])
{
 union wait *W_Stat;
 int Maxrc = OK;                          /* Max err-code encountered so far */
 int k = 0;

 Maxrc = Housekeeping(argv);

/* If any errors poped up so far they are of such a nature that it is very
 * questionable to continue; so we better bail out in this case. 
 */
  if (Maxrc <= WARNING) {
     if ((W_Stat = (union wait *)malloc ((unsigned) (1) *sizeof(union wait))) == NULL) {
        fprintf(stderr,"%s: (C) malloc failed (W_Stat). Reason: %i (%s)\n",
                           called_as,errno,strerror(errno));
        Maxrc = CAT;
     }
     else {
        k = Call_Cico(argc, argv, W_Stat);
        Maxrc = Maxrc >= k ? Maxrc:k;
        free(W_Stat);
     }
  }
  gettimeofday(&tp, &tzp);
  fprintf(stderr,"%s: (I) ended with rc = %i on %s\n",
                 called_as,Maxrc,ctime(&tp.tv_sec));
  fclose(stderr);
  exit (Maxrc);
}

/* --------------------------------------------------------------------------*/
/*                           Functions                                       */
/* --------------------------------------------------------------------------*/

int Housekeeping(char *argv[]) { 

 FILE *infile;
 int i,n,s,k = 0;
 int Rc = OK;

/* 
 *  get our name sans path
 * */

  strcpy(workf,argv[0]);
  k = strlen(*argv)+1;
  for(i=k;i>=0 && workf[--i] != '/';)                 /* get last sep */
                      ;
  if ((called_as = (char *)malloc (k - i)) == NULL) {
     fprintf(stderr,"%s: (C) malloc failed (called_as). Reason: %i (%s)\n",
                     argv[0],errno,strerror(errno));
     return (CAT);
  }
  for(i++,s=0;(called_as[s++] = workf[i++]);)         /* name without path */
                      ;
 
/* if defined set up the name of the message log file otherwise stderr will be used
 * */

#ifdef ALOG_FILE
   if ((Msg_Log = (char *)malloc (sizeof(ALOG_FILE))) == NULL) {
      fprintf(stderr,"%s: (C) malloc failed (Msg_Log). Reason: %i (%s)\n",
                     called_as,errno,strerror(errno));
      return (Rc >= CAT ? Rc:CAT);
   }
   strcpy(Msg_Log,ALOG_FILE);
   if ((freopen(Msg_Log,"a",stderr)) == NULL) {
      fprintf(stdout,"%s: (C) Could not open msglog: %s\n",called_as,Msg_Log);
      return (Rc >= CAT ? Rc:CAT);
   }
#endif  /* ALOG_FILE */

/* put out the started message including the time and the userid.
 * */

  pwd = getpwuid(getuid());
  gettimeofday(&tp, &tzp);
  fprintf(stderr,"\n%s: (I) started by `%s' on %s",
                 called_as,(pwd==NULL) ? "???":pwd->pw_name,ctime(&tp.tv_sec));

/* set up the default grade 
 * */

  if ((grade = (char *)malloc (sizeof("A"))) == NULL) {
     fprintf(stderr,"%s: (C) malloc failed (grade). Reason: %i (%s)\n",
                     called_as,errno,strerror(errno));
      return (Rc >= CAT ? Rc:CAT);
  }
  strcpy(grade,"A");
#ifdef DEF_GRADE
   if (strlen(DEF_GRADE) > 1) {
      fprintf(stderr,"%s: (W) grade %s invalid; default `%s' used\n",
                      called_as,DEF_GRADE,grade);
      Rc = Rc >= WARNING ? Rc:WARNING;
   }
   else
      strcpy(grade,DEF_GRADE);
#endif /* DEF_GRADE */

/* get the program to actually call the site. This is normally UUCICO.
 * */

  if ((Cico = (char *)malloc (sizeof(CICO_DIR))) == NULL) {
     fprintf(stderr,"%s: (C) malloc failed (Cico). Reason: %i (%s)\n",
                     called_as,errno,strerror(errno));
     return (Rc >= CAT ? Rc:CAT);
  }
  strcpy(Cico,CICO_DIR);                      /* uucico lives here  */
  k = strlen(Cico)+1;
  for(i=strlen(Cico)+1;i>=0 && Cico[--i] != '/';)   /* get last sep */
                      ;
  if ((k - i) == 2) {
     fprintf(stderr,"%s: (E) invalid specification CICO_DIR: %s\n",
                    called_as,CICO_DIR);
    Rc = Rc >= SEVERE ? Rc:SEVERE;
  }
  else {
    if ((Uucico = (char *)malloc (k - i)) == NULL) {
       fprintf(stderr,"%s: (C) malloc failed (Uucico). Reason: %i (%s)\n",
                      called_as,errno,strerror(errno));
      return (Rc >= CAT ? Rc:CAT);
    }
    for(i++,s=0;(Uucico[s++] = Cico[i++]);)   /* name without path */
                       ;
  }

/* get the path to ourself.
 * */

  if ((Poll_Pgm = (char *)malloc (sizeof(AUTO_DIR))) == NULL) {
     fprintf(stderr,"%s: (C) malloc failed (Poll_Pgm). Reason: %i (%s)\n",
                     called_as,errno,strerror(errno));
     return (Rc >= CAT ? Rc:CAT);
  }
  strcpy(Poll_Pgm,AUTO_DIR);                        /* we live here  */
  k = strlen(Poll_Pgm)+1;
  for(i=strlen(Poll_Pgm)+1;i>=0 && Poll_Pgm[--i] != '/';)   /* get last sep */
                      ;
  if ((k - i) == 2) {
     fprintf(stdout,"%s: (E) invalid specification of AUTO_DIR: %s\n",
                    called_as,AUTO_DIR);
    Rc = Rc >= SEVERE ? Rc:SEVERE;
  }

/* obtain our sitename
 * */

  if ((gethostname(This_Site,MAXHOSTNAMELEN+1)) != 0) {
     fprintf(stderr,"%s: (W) hostname could not be obtained\n",called_as);
     Rc = (Rc >= WARNING) ? Rc:WARNING;
  }

/* 
 *  obtain all active  sitenames
 * */

  i = 0;
  if ((infile=popen(Sort,"r")) != NULL) {
     while(fgets(Sitetab[i].name,MAXHOSTNAMELEN+1,infile) != NULL) {
          if (i > SITE_MAX) {            /* let'm run so that we can give */
             i++;                        /* the user some guidance */
             continue;                   /* we'll tell the user later on */
          }
          n = strlen(Sitetab[i].name)-1; /* offset: next to last char */
          Sitetab[i].name[n] = '\0';     /* strip trailing newline */
          Sitetab[i].flag = 0;           /* TRUE: poll this site */
          Sitetab[i].asap = 0;           /* TRUE: when -n is absent */
          strcpy(Sitetab[i].grade,grade);  /* set the default grade */
          maxtab = i++;                  /* set high-water-mark */
     }
     pclose(infile);
  }

/* in case the internal table overflows we'll now give notice and tell
 * the user by which amount the table has to be increased to hold all site-
 * names
 */
  if (i > SITE_MAX) {
     fprintf(stderr,"%s: (E) number of sites > internal tab\n",called_as);
     fprintf(stderr,"%s: (E) increase SITE_MAX to >= %d and recompile\n",
                    called_as,i);
     Rc = Rc >= SEVERE ? Rc:SEVERE;
     }

/* check for any failures that may have occured during command execution
 * and therefore we're unable to obtain the site names at all. 
 */
  if (maxtab == 0) {
     fprintf(stderr,"%s: (E) could not obtain sitenames.\n",called_as);
     Rc = Rc >= SEVERE ? Rc:SEVERE;
  }
 return (Rc);
}

int Call_Cico(int argc, char *argv[], union wait *W_Stat) {

/*  Start uucico and wait for completion. In case the return code is '0'  
 *  we're finished; otherwise we'll have to check the status files for any 
 *  non successful call (retry time > 0). 
 *  Any such site will be called again at the time plus any wait
 */

 int W_Ret = 0;
 char Single_Site[MAXHOSTNAMELEN+1] = "";
 int j,k,i,x,n = 0;
 int pid = 0;
 int force = 0;
 int Rc = OK;
 for (i = 1; i < argc; i++) {
     k = strlen(argv[i]);
     switch (*argv[i]) {

       /*      ---->     handle the options         */

        case '-':
             n = 1;
             switch (*(argv[i]+n)) {
                 case 'S':
                         force = 1;
                 case 's':
                      if (k > 2) {
                         strncpy(Single_Site,argv[i]+2,k-2);
                         x = Check_Site(Single_Site);
                         Rc = (x == OK) ? Rc:x;
                         break;
                      }
                      j = i;
                      ++j;
                      if (argv[j] == 0) {
                         fprintf(stderr,"%s: (E) System to call is missing\n",
                                         called_as);
                         Rc = Rc >= SEVERE ? Rc:SEVERE;
                      }
                      else {
                         strcpy(Single_Site,argv[j]);
                         i++;        /* advance argv counter */
                         x = Check_Site(Single_Site);
                         Rc = (x == OK) ? Rc:x;
                      }
                      break;
                 case 'f':
                      force = 1;
                      break;
                 default:
                      break;
             }
         default:
              break;
     }
 }
 if (Rc > WARNING) {
     return (Rc);
 }

 fflush(stderr);
 switch(pid = fork()) { 
    case -1:
         fprintf(stderr,"%s: (C) could not fork(). Reason-code: %i (%s)\n",
                         called_as,errno,strerror(errno)); 
         return (CAT);
    case 0:
         if ((argv[0] = (char *)malloc ((unsigned)strlen(Uucico)+1)) == NULL) {
            fprintf(stderr,"%s: (C) malloc failed (argv[0]). Reason: %i (%s)\n",
                           called_as,errno,strerror(errno));
            _exit (CAT);
         }
         strcpy(argv[0],Uucico);            /* change name to be uucico */
         execve(Cico, argv, NULL);
         fprintf(stderr,"%s: (C) could not start %s. Reason-code: %i (%s)\n",
                         called_as,Uucico,errno,strerror(errno));
         _exit (CAT);            /* child: bail out */
    default: 
         fprintf(stderr,"%s: (I) starting %s [%d]\n\n",called_as,Uucico,pid);
         fflush(stderr);          /* maybe we come behind uucico's output */
                                  /* if any; it's a race condition        */
         W_Ret = wait(W_Stat);
         if (W_Stat->w_termsig == 0) {
            if (W_Stat->w_retcode == 0) {
                fprintf(stderr,"%s: (I) %s [%d] ended normally\n",
                                called_as,Uucico,pid);
                return (OK);
            }
            if (W_Stat->w_retcode != CAT) {
               fprintf(stderr,"%s: (W) Check %s's log for any errors too!\n",
                               called_as,Uucico);
               fprintf(stderr,"\n%s: (I) %s [%d] ended with rc = %i\n",
                               called_as,Uucico,pid,W_Stat->w_retcode);
               return (Chk_Status(argc, argv, Single_Site, force, W_Stat));
             }
             else
             return (CAT);    /* we where unable to exec */
          }
          else {
             fprintf(stderr,"\n%s: (E) %s [%d] terminated by signal %i\n",
                            called_as,Uucico,pid,W_Stat->w_termsig);
            return (SEVERE);
          }
 }      /* switch (pid = fork()) */
 return (OK);   /* silence the compiler */
}

int Chk_Status(int argc, char *argv[],
               char *Single_Site,
               int force,
               union wait *W_Stat) {

/* 
 * For all sites found in Site_Tab their status files will be checked.
 * The table scan will be bypassed for a call to a specific site.
 * If the call failed the wait period is > 0. We will schedule an at-job
 * to be run at the time found + the delta. In case we find an old entry
 * where the time + delta is lower than the current time we'll advance
 * the current time by 60 secs. and use that value instead.
 * In case we are invoked to call a specific site and either the -f option or
 * the site was given as -S... indicating to disregard any wait, we'll
 * use the time found in the status file increased by 120 secs.
*/

 int W_Ret = 0;
 FILE *infile, *outfile;
 long secs = 0;
 long add = 0;
 char mon[3];
 int day, hh, mm, ss;
 char oname[24];
 char jname[20];
 char tstr[20];
 char ctag[] = "  ";
 char at_cmd[] = "at";
#ifdef AT_OPTIONS
 char at_opt[] = AT_OPTIONS;
#else
 char at_opt[] = "-mc";
#endif
 int pid = 0;
 int our_pid = 0;           /* holds the pid of this process */
 int i = 0;
 int ecnt = 0;
 int Rc = WARNING;         /* uucico got rc = 1 otherwise we were not here */

/*
 * Note
 *     We have to increase the sum of time and wait by at least one minute.
 *     That is because this time denotes the earliest point *after* which 
 *     we may call again.
 *     When a site is called at the wrong time the follwing actions are
 *     taken: wait = 0  && ! force --> no further action (indicator: see log)
 *            wait = 0  && force   --> (W) message generated; no further action
 *            wait > 0  && ! force --> normal scheduling at time + wait
 *            wait > 0  && force   --> normal scheduling at time+120 secs
 *     We can't depend on the string "Wrong time to call" because the .Status
 *     file may not be updated due to the -c switch. This may lead to a
 *     situation where the site will be called over and over again while it's
 *     still the wrong time. (No we don't want to go fishing for a message in
 *     the uucp LOG!)
 * */

 for (i = Single_Site_Tab; i <= maxtab; i++) {
   sprintf(workf,"%s%s",STATUS_DIR,Sitetab[i].name);
   if ((infile=fopen(workf,"r")) != NULL) {
      fscanf(infile,"%*d %*d %ld %ld",&secs,&add);
      if (add != 0 || force) {
         if (force) {
            if (add != 0)
               add = secs + 120;                    /* shorten the wait */
            else { 
               fprintf(stderr,"%s (W) May be wrong time to call: %s\n",
                               called_as,Sitetab[i].name);
               Rc = Rc >= WARNING ? Rc:WARNING;
               fclose(infile);
               continue;
            }  /* add != 0 */
         }
         else   /* ! force */
            add += secs + 60;                    /* take the full wait */
	 gettimeofday(&tp, &tzp);
	 if (tp.tv_sec >= add) {
	    add = tp.tv_sec + 60;
	 }  /* force */
         our_pid = getpid(); 
         sscanf(ctime(&add),"%*s %s %d %d:%d:%d",mon,&day,&hh,&mm,&ss);
         sprintf(oname,"/tmp/at.%d.%02d%02d%02d",our_pid,hh,mm,ss);
         sprintf(jname,"at.%d.%02d%02d%02d",our_pid,hh,mm,ss);
/* 
 * if we can't open the workfile to be passed to at we'll abandon this site and set
 * the rc accordingly
 * */
         if ((outfile=fopen(oname,"w")) == NULL) {
            fprintf(stderr,"%s: (E) could not open workfile %s. No scheduling for: %s\n",
                            called_as,oname,Sitetab[i].name);
            Rc = Rc >= SEVERE ? Rc:SEVERE;
            fclose(infile);
            continue;
         }
/* 
 * ok, continue by writing tthe command to be executed to the file and then call
 * the at command
 * */
         fprintf(outfile,"%s -r1 -s%s\n",Poll_Pgm,Sitetab[i].name);
         sprintf(tstr,"%02d%02d",hh,mm);
         sprintf(ctag,"%d",day);
         fclose(outfile);
         if ((chmod(oname,00644)) != 0) {
               fprintf(stderr,"%s: (W) chmod to %s failed. Reason_code: %i (%s)\n",
                               called_as,oname,errno,strerror(errno));
               Rc = Rc >= WARNING ? Rc:WARNING;
         }
         switch (pid = fork()) {
           case -1:
                 fprintf(stderr,"%s: (C) could not fork(). Reason-code: %i (%s)\n",
                                called_as,errno,strerror(errno)); 
                 fclose(infile);
                 return (Rc >= CAT ? Rc:CAT);
           case 0:
                 if (*at_opt == '\0')
                    execlp(at_cmd, at_cmd, tstr, mon, ctag, oname, 0);
                 else
                    execlp(at_cmd, at_cmd, at_opt, tstr, mon, ctag, oname, 0);

                 fprintf(stderr,"%s: (C) could not start AT-cmd. Reason-code: %i (%s)\n",
                                 called_as,errno,strerror(errno));
                 _exit (CAT);            /* child: bail out */
           default: 
                 fprintf(stderr,"%s: (I) at [%d] started. Job name: %s\n",
                                called_as,pid,jname);
                 W_Ret = wait(W_Stat);
                 unlink(oname);
                 if (W_Stat->w_termsig == 0) {
                    if (W_Stat->w_retcode != 0)
                     {
                       if (W_Stat->w_retcode != CAT)
                        {
                          fprintf(stderr,"%s: (E) at-cmd failed for some reason\n",
                                         called_as);
                          Rc = Rc >= SEVERE ? Rc:SEVERE;
                        }
                       else
                        {
                         Rc = Rc >= CAT ? Rc:CAT;
                        }

                       fprintf(stderr,"%s: (I) at [%d] ended with rc = %i\n",
                                     called_as,pid,W_Stat->w_retcode);
                       /* bail out in case wait returned > SEVERE */
                       if (Rc == CAT)
                        {
                          fclose(infile);
                          return (Rc);
                        }
                     }
                    else
                     {
                      fprintf(stderr,"%s: (I) at-cmd [%d] ended normally\n",
                                         called_as,pid);
                     } 
                 }
                 else {
                    fprintf(stderr,"%s: (E) at [%d] terminated by signal %i\n",
                                    called_as,pid,W_Stat->w_termsig);
                    Rc = Rc >= SEVERE ? Rc:SEVERE;
                 }
         }   /* switch (pid = fork()) */
      }    /* if (add == 0 ... */
      fclose(infile);
   }     /* if ((infile=fopen .... */
   else {
      fprintf(stderr,"%s: (W) no access to status file for: %s\n",
                     called_as,Sitetab[i].name);
      Rc = Rc >= WARNING ? Rc:WARNING;
      if (*Single_Site != '\0')
         ecnt = maxtab+1;
      else
        ecnt++;
   }
 if (*Single_Site != '\0')
    break;                          /* this invocation was specific */       
 }           /* for (i = 0; ...  */

 if (ecnt == maxtab+1) {
    fprintf(stderr,"%s: (E) no access to status files; no scheduling done\n",
                    called_as);
    Rc = Rc >= SEVERE ? Rc:SEVERE;
 }
return (Rc);
}

int Check_Site(char *Single_Site) {
  
/*
 * check the site passed via -s or -S option to be a valid one and
 * not to be our hostname.
 * */

  int i,j = 0;
    if (strcmp(Single_Site,This_Site) == 0) {
        fprintf(stderr,"%s: (E) won't call *ourself* %s\n",
                        called_as,Single_Site);
        return(SEVERE);
    }
    for(i=0;i<=maxtab;i++) {
       if ((j=strcmp(Sitetab[i].name,Single_Site)) >= 0) {
          break;
       }
    }
    if (j != 0) {
        fprintf(stderr,"%s: (E) unknown site: %s\n",
                        called_as,Single_Site);
        return(SEVERE);
    }
  Single_Site_Tab = i;
  return(OK);
}
