########################## -*- Mode: Perl -*- ##########################
##
## File             : Database.pm
##
## Description      : handling of WAIS databases
##
#
# Copyright (C) 1996 Ulrich Pfeifer, Norbert Goevert
#
# This file is part of SFgate.
#
# SFgate is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# SFgate is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SFgate; see the file COPYING.  If not, write to
# the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
#
##
## Author           : Norbert Goevert
## Created On       : Thu Feb 15 14:37:11 1996
##
## Last Modified By : Norbert Goevert
## Last Modified On : Thu Jan 16 15:24:23 1997
##
## $State: Exp $
##
## $Id: Database.pm,v 5.1.1.3 1997/04/04 17:30:02 goevert Exp goevert $
##
## $Log: Database.pm,v $
## Revision 5.1.1.3  1997/04/04 17:30:02  goevert
## patch11: fix for undefined hash references
##
## Revision 5.1.1.2  1997/02/17 12:56:11  goevert
## patch10: new sub get_name
##
## Revision 5.1.1.1  1996/12/23 12:49:53  goevert
## patch6: handling for WAIT databases
##
## Revision 5.1  1996/11/05 16:55:17  goevert
## *** empty log message ***
##
## Revision 5.0.1.5  1996/11/04 13:09:22  goevert
## patch21: cons instead of MakeMaker
##
## Revision 5.0.1.4  1996/07/03 13:27:40  goevert
## patch19: database specification bug fixed
##
## Revision 5.0.1.3  1996/06/04 16:16:07  goevert
## patch17: fixed bug with non-standard port numbers
##
## Revision 5.0.1.2  1996/05/13 11:28:25  goevert
## patch1:
##
########################################################################


use strict;


package SFgate::Databases::Database;


sub new
{
    my $class = shift;
    my $self  = {};
    bless $self, $class;
    
    $self->initialize(@_);

    return $self;
}


sub initialize
{
    my $self = shift;

    my($database_id,
       $server,
       $port,
       $name,
       $path,
       $description,
       $attributes,
       $converter,
       $diagnostic) = @_;
    ## local variables
    local($_);
    my(%inverted_attributes, $value, $type, $types, %types);

    $self->{'database_id'} = $database_id;
    $self->{'server'}      = $server;
    $self->{'port'}        = $port;
    $self->{'name'}        = $name;
    $self->{'path'}        = $path;
    $self->{'description'} = $description;
    if (ref($attributes) eq 'HASH' && keys %$attributes) {
        # invert attributes hash and get attribute types
        foreach (keys %$attributes) {
            $value = $$attributes{$_};
            ($_, $types) = split(':', $_, 2);
            foreach $type (split(',', $types)) {
                push(@{$types{$_}}, $type);
            }
            my $attribute;
            foreach $attribute (split(',', $value)) {
                push(@{$inverted_attributes{$attribute}}, $_);
            }
        }
        $self->{'attributes'}  = \%inverted_attributes;
        $self->{'types'}       = \%types;
    }
    # copy %$converter hash
    if (ref($attributes) eq 'HASH' && keys %$converter) {
        foreach (keys %$converter) {
            $self->{'converter'}->{$_} = $converter->{$_};
        }
    }
    $self->{'diagnostics'} = [];
    $self->add_diagnostic($diagnostic) if $diagnostic;
}


sub display
{
    my $self = shift;

    ## local variables
    local($_);
    my(@converter, @attributes);

    print $self->{'database_id'} . ': ' . $self->get_database . "\n";
    print 'query: ' . $self->{'query'} . "\n" if $self->{'query'};
    
    @attributes = keys(%{$self->{'attributes'}});
    if (@attributes) {
        print " Attributes:\n";
        foreach (keys %{$self->{'attributes'}}) {
            print "  $_ => ", join(', ', @{$self->{'attributes'}->{$_}}), "\n";
        }
    }

    @converter = keys(%{$self->{'converter'}});
    if (@converter) {
        print " Converter:\n";
        foreach (@converter) {
            print "  $_ => $self->{'converter'}->{$_}\n";
        }
    }

    if ($self->{'diagnostic'}) {
        print " Diagnostic:\n  $self->{'diagnostic'}\n";
    }
}


sub match_attribute
{
    my $self = shift;
    my($attribute, $form, $lattice) = @_;
    ## local variables
    local($_);
    my(@lattice_attributes, $success, $predecessor, @successors, $top);
    ## return values
    my(@attributes);

    require SFgate::Attributes::Form;
    require SFgate::Attributes::Lattice;

    # get lattice attribut(s) for form attribute
    @lattice_attributes = $form->get_lattice_attributes($attribute);
    if (!@lattice_attributes
        && $lattice->is_attribute($attribute)) {
        push(@lattice_attributes, $attribute);
    }
    
    foreach (@lattice_attributes) {

        $success = 0;
        
        # equality
        if (defined($self->{'attributes'}->{$_})) {
            if (@{$self->{'attributes'}->{$_}} > 1
                || $self->{'attributes'}->{$_}->[0] ne 'global') {
                # 'global' is not the only one
                @attributes = (@attributes, @{$self->{'attributes'}->{$_}});
                next;
            }
        }

        # specialization
        @successors = $lattice->get_successors($_);
        foreach $attribute (@successors) {
            if (defined($self->{'attributes'}->{$attribute})) {
                @attributes = (@attributes, @{$self->{'attributes'}->{$attribute}});
                $success = 1;
            }
        }
        next if $success;
            
        # generalization
        $predecessor = $lattice->get_predecessor($_);
        $top = $lattice->get_top;
        while ($predecessor ne $top) {
            if (defined($self->{'attributes'}->{$predecessor})) {
                @attributes = (@attributes, @{$self->{'attributes'}->{$predecessor}});
                $success = 1;
                last;
            }
            $predecessor = $lattice->get_predecessor($predecessor);
        }
        next if $success;

        # no specialization or generalization
        # and equality with `global'
        push(@attributes, 'global')
            if $self->{'attributes'}->{$_}->[0] eq 'global';
    }

    my(@result_attributes, %double);
    foreach (@attributes) {
       push(@result_attributes, $_) unless $double{$_};
       $double{$_} = 1;
    }
    
    return \@result_attributes;
}


sub get_server
{
    my $self = shift;

    return $self->{'server'};
}


sub get_name
{
    my $self = shift;

    return $self->{'name'};
}


sub get_database
{
    my $self = shift;

    my($db) = $self->{'name'};
    $db = $self->{'path'} . '/' . $db if $self->{'path'};
    
    return "$self->{'server'}:$self->{'port'}/$db"
        if $self->{'port'};

    return "$self->{'server'}/$db";
}


sub get_waisquery
{
    my $self = shift;

    return () if @{$self->{'diagnostics'}};
    return () if $self->{'server'} eq 'wait';
    
    my($db) = $self->{'name'};
    $db = $self->{'path'} . '/' . $db if $self->{'path'};

    if ($self->{'server'} eq 'local') {
        return ({'query'    => $self->{'query'},
                 'database' => $db,
                 'host'     => 'localhost',
                 'tag'      => $self->{'database_id'}});
    }
    else {
        return ({'query'    => $self->{'query'},
                 'database' => $db,
                 'port'     => $self->{'port'},
                 'host'     => $self->{'server'},
                 'tag'      => $self->{'database_id'}});
    }
}


sub get_waitquery
{
    my $self = shift;

    return () if @{$self->{'diagnostics'}};
    return unless $self->{'server'} eq 'wait';
    
    my($db) = $self->{'name'};
    $db = $self->{'path'} . '/' . $db if $self->{'path'};

    return ({'query'    => $self->{'query'},
             'database' => $db,
             'host'     => 'localhost',
             'tag'      => $self->{'database_id'}});
}


sub get_documentrequest
{
    my $self = shift;

    my($docid, $type) = @_;
    ## local variables
    my($local_id);
    
    my $db = $self->{'name'};
    $db = $self->{'path'} . '/' . $db if $self->{'path'};

    if ($self->{'server'} eq 'local' || $self->{'server'} eq 'wait') {
        return ('database' => $db,
                'docid'    => $docid,
                'host'     => 'localhost',
                'type'     => $type);
    }
    else {
        my $database = $self->{'name'};
        $database = $self->{'path'} . '/' . $database if $self->{'path'};
        return ('database' => $db,
                'docid'    => $docid,
                'host'     => $self->{'server'},
                'port'     => $self->{'port'},
                'type'     => $type);
    }
}


sub get_converter
{
    my $self = shift;

    my($converter) = @_;

    if ($self->{'converter'}->{$converter}) {
        return $self->{'converter'}->{$converter};
    }

    return $converter;
}


sub get_database_file
{
        my $self = shift;
        
        return $self->{'path'} . '/' . $self->{'name'};
    }


sub get_description
{
    my $self = shift;

    return $self->{'description'} if $self->{'description'};
    return $self->{'name'}        if $self->{'name'};
    return $self->{'database_id'};
}


sub set_query
{
    my $self = shift;

    my($query) = @_;

    if ($query) {
        $self->{'query'} = $query unless $self->get_query || $self->get_diagnostics;
    }
    else {
        $self->add_diagnostic("Empty query");
    }
}


sub get_query
{
    my $self = shift;

    return $self->{'query'};
}


sub set_conditions
{
    my $self = shift;

    my($conditions) = @_;

    $self->{'conditions'} = $conditions;

    my $condition;
    foreach $condition (@$conditions) {
        $self->{'encoded_conditions'} .= '||' . join('|', @$condition);
    }
#    $self->{'encoded_conditions'} =~ s/\|{2}//;
}


sub get_conditions
{
    my $self = shift;

    return $self->{'conditions'};
}


sub get_encoded_conditions
{
    my $self = shift;

    return $self->{'encoded_conditions'};
}


sub add_diagnostic
{
    my $self = shift;

    my($diagnostic) = @_;
    
    push(@{$self->{'diagnostics'}}, $diagnostic);
}


sub get_diagnostics
{
    my $self = shift;

    return @{$self->{'diagnostics'}};
}


sub test_type
{
    my $self = shift;

    my($field, $type) = @_;
    ## local variables
    local($_);

    foreach (@{$self->{'types'}->{$field}}) {
        return 1 if $type eq $_;
    }

    return 0;
}


1;
