/* etl_msgm.c - Daemon Message printing, sizes, etc.
 *
 * Copyright (C) 1994, 1995 Free Software Foundation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.ai.mit.edu.
 * 
 * Description:
 * 
 *   Common routines for managing daemon messages, sizes, etc.
 * Created from sources in et_daem.c which was found to be common with
 * gtalkd source code.
 *
 * $Log: etl_msgm.c,v $
 * Revision 1.8  1995/03/25  04:15:06  zappo
 * Updated copyright
 *
 * Revision 1.7  1995/03/03  02:48:40  zappo
 * Added extionsion printing, swap id_num before printing, and second
 * half of messages are not printed unless verbosity >=  2
 *
 * Revision 1.6  1995/02/28  02:58:16  zappo
 * Made sure extension in gtalk message type variable was printed
 *
 * Revision 1.5  1995/02/11  14:12:02  zappo
 * Added new message name "try here" record.
 *
 * Revision 1.4  1995/01/29  14:25:34  zappo
 * Fixed some -Wall warnings
 *
 * Revision 1.3  1995/01/14  17:43:27  zappo
 * Added printing of special message extensions.
 * 
 * History:
 * zappo   9/14/94         Created
 *
 * Tokens: ::Header:: etalklib.h
 */
#include "etalklib.h"
#include "otalk.h"
#include "talk.h"
#include "gtalk.h"
#include "etl_union.h"

char *msg_types[] = {
  "Leave Invite",
  "Look up",
  "Delete",
  "Announce",
  "Reply Query",
};
char *msg_responses[] = {
  "Success",
  "Not Here",
  "Failed",
  "Machine Unknown",
  "Permision Denied",
  "Unknown Request",
  "Bad Version",
  "Bad Address",
  "Bad Control Address",
  "No Caller",
  "Try Here",
};

union ctl_msg      Control;
union ctl_response Response;
char               ControlExtension[256];


/*
 * Function: control_size, response_size
 *
 * These functions return the size of each portion of the
 * previously declaired unions.  This allows a send based
 * on message size which, in turn, is based on daemon type.
 * 
 * Parameters:  io   - device we are sending to.  need to know type.
 *              type - Type of
 * History:
 * eml 3/1/94
 * zappo   9/16/94         Added parameter type
 */
int control_size(type)
     enum DaemonType type;
{
  switch(type) {
  case GTALKD: return sizeof(Control.gtalk);
  case NTALKD: return sizeof(Control.talk);
  case OTALKD: return sizeof(Control.otalk);
  default: return 0;
  }
}
int response_size(type)
     enum DaemonType type;
{
  switch(type) {
  case GTALKD: return sizeof(Response.gtalk);
  case NTALKD: return sizeof(Response.talk);
  case OTALKD: return sizeof(Response.otalk);
  default: return 0;
  }
}


/*
 * Function: ETM_control_print
 *
 * the following debugging command will print the contents of the 
 * control message.
 * 
 * Parameters:  type - Type of
 *
 * History:
 * eml 3/1/94
 * zappo   9/16/94         changed parameter io to type
 */
void ETM_control_print(type)
     enum DaemonType type;
{
  printf("Control message is:\n");
  switch(type) {
  case GTALKD:
    printf("Control.talk.vers     : %d\n", Control.gtalk.vers);
    if(Control.talk.type > REPLY_QUERY)
      printf("Control.talk.type     : Unknown %d\n", Control.gtalk.type);
    else
      printf("Control.talk.type     : %s\n", msg_types[Control.gtalk.type]);
    printf("Extension length      : %d\n", Control.gtalk.extended);
    if(verbose >= 2)
      {
	printf("Control.talk.addr     : "); 
	print_swapped_sockaddr(&Control.gtalk.addr);
	printf("\nControl.talk.ctl_addr : ");
	print_swapped_sockaddr(&Control.gtalk.ctl_addr);
	printf("\nControl.talk.pid      : %ld\n", htonl(Control.gtalk.pid));
	printf("Control.talk.id_num   : %ld\n", htonl(Control.gtalk.id_num));
	printf("Control.talk.l_name   : %s\n", Control.gtalk.l_name);
	printf("Control.talk.r_name   : %s\n", Control.gtalk.r_name);
	printf("Control.talk.r_tty    : %s\n", Control.gtalk.r_tty);
	if(Control.gtalk.extended)
	  printf("Control.talk.extension: %s\n", ControlExtension);
      }
    break;
  case NTALKD:
    printf("Control.talk.vers     : %d\n", Control.talk.vers);
    if(Control.talk.type > REPLY_QUERY)
      printf("Control.talk.type     : Unknown %d\n", Control.talk.type);
    else
      printf("Control.talk.type     : %s\n", msg_types[Control.talk.type]);
    if(verbose >= 2)
      {
	printf("Control.talk.addr     : "); 
	print_swapped_sockaddr(&Control.talk.addr);
	printf("\nControl.talk.ctl_addr : ");
	print_swapped_sockaddr(&Control.talk.ctl_addr);
	printf("\nControl.talk.pid      : %ld\n", htonl(Control.talk.pid));
	printf("Control.talk.id_num   : %ld\n", htonl(Control.talk.id_num));
	printf("Control.talk.l_name   : %s\n", Control.talk.l_name);
	printf("Control.talk.r_name   : %s\n", Control.talk.r_name);
	printf("Control.talk.r_tty    : %s\n", Control.talk.r_tty);
      }
    break;
  case OTALKD:
    if(Control.otalk.type > ANNOUNCE)
      printf("Control.otalk.type     : Unknown %d\n", Control.otalk.type);
    else
      printf("Control.otalk.type     : %s\n", msg_types[Control.otalk.type]);
    if(verbose >= 2)
      {
	printf("Control.otalk.addr     : ");
	print_swapped_sockaddr(&Control.otalk.addr);
	printf("\nControl.otalk.ctl_addr : ");
	print_swapped_sockaddr(&Control.otalk.ctl_addr);
	printf("\nControl.otalk.pid      : %ld\n", htonl(Control.otalk.pid));
	printf("Control.otalk.id_num   : %ld\n", htonl(Control.otalk.id_num));
	printf("Control.otalk.l_name   : %s\n", Control.otalk.l_name);
	printf("Control.otalk.r_name   : %s\n", Control.otalk.r_name);
	printf("Control.otalk.r_tty    : %s\n", Control.otalk.r_tty);
      }
    break;
  default:
    printf("print_control: Input device host has daemon type %d.\n",
	   type);
  }
}

/*
 * Function: ETM_response_print
 *
 * Print out the contents of the last response message.
 * 
 * Parameters:  type - Type of daemon.
 *
 * History:
 * eml 4/15/94
 * zappo   9/16/94         Added parameter type
 */
int ETM_response_print(type)
     enum DaemonType type;
{
  printf("Response message is:\n");
  switch(type)
    {
    case GTALKD:
      printf("Response.talk.vers    : %d\n", Response.talk.vers);
      if(Response.talk.type > BADCTLADDR)
	printf("Response.talk.type    : Unknown %d\n", Response.talk.type);
      else
	printf("Response.talk.type    : %s\n", msg_types[Response.talk.type]);
      printf("Response.talk.answer  : %s\n", msg_responses[Response.talk.answer]);
      if(verbose >= 2)
	{
	  printf("Response.talk.id_num  : %ld\n", ntohl(Response.talk.id_num));
	  printf("Response.talk.addr    : ");
	  print_swapped_sockaddr(&Response.talk.addr);
	  printf("\n");
	  printf("Response.gtalk.r_name : %s\n", Response.gtalk.r_name);
	}
      break;
    case NTALKD:
      printf("Response.talk.vers    : %d\n", Response.talk.vers);
      if(Response.talk.type > BADCTLADDR)
	printf("Response.talk.type    : Unknown %d\n", Response.talk.type);
      else
	printf("Response.talk.type    : %s\n", msg_types[Response.talk.type]);
      printf("Response.talk.answer  : %s\n", msg_responses[Response.talk.answer]);
      if(verbose >= 2)
	{
	  printf("Response.talk.id_num  : %ld\n", ntohl(Response.talk.id_num));
	  printf("Response.talk.addr    : ");
	  print_swapped_sockaddr(&Response.talk.addr);
	  printf("\n");
	}
      break;
    case OTALKD:
      printf("Response.otalk.type   : %s\n", msg_types[Response.otalk.type]);
      if(Response.otalk.type > BADCTLADDR)
	printf("Response.otalk.answer : Unknown %d\n", Response.otalk.answer);
      else
	printf("Response.otalk.answer : %s\n", msg_responses[Response.otalk.answer]);
      if(verbose >= 2)
	{
	  printf("Response.otalk.id_num : %ld\n", ntohl(Response.otalk.id_num));
	  printf("Response.otalk.addr   : ");
	  print_swapped_sockaddr(&Response.otalk.addr);
	  printf("\n");
	}
      break;
    default:
      printf("Response_Display: IO device has wrong host daemon type %d.\n",
	     type);
    }
  return Success;
}


/*
 * Function: DMN_check_compile
 *
 * Checks the compiled sizes of talk daemon messages against those
 * stored in config.h
 *
 * Returns:     int  - status
 * Parameters:  None
 *
 * History:
 * eml     Aug 17, 1994    
 */
int DMN_check_compile()
{
#define SOC sizeof(CTL_MSG_OLD)
#define SOR sizeof(CTL_RESPONSE_OLD)
#define NOC sizeof(CTL_MSG)
#define NOR sizeof(CTL_RESPONSE)
#define GOC sizeof(CTL_MSG_GNU)
#define GOR sizeof(CTL_RESPONSE_GNU)

  if((SOC != OTALK_CM_SIZE) || (SOR != OTALK_CMR_SIZE) ||
     (NOC != NTALK_CM_SIZE) || (NOR != NTALK_CMR_SIZE) ||
     (GOC != GTALK_CM_SIZE) || (GOR != GTALK_CMR_SIZE))
    {
      printf("\nERROR in compile sizes!\n\n");
      printf("Old control %d, Old response %d\n", SOC, SOR);
      printf("New control %d, New response %d\n", NOC, NOR);
      printf("GNU control %d, GNU response %d\n", GOC, GOR);
      printf("\nSee config.h for desired sizes.\n");
      return Fail;
    }
  else
    {
      return Success;
    }
}
