/*---------------------------------------------------------------------------
QueryManager.m -- Copyright (c) 1991 Rex Pruess

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or send
   electronic mail to the the author.

QueryManager fires up new Query windows.  A timer is set to limit the
amount of time we wait for the Query window to come to life.  If the
"fields" data have not been read in X amount of time, then we tell the
user & quit waiting for the data.

Rex Pruess <Rex-Pruess@uiowa.edu>

$Header: /rpruess/apps/Ph/query.subproj/RCS/QueryManager.m,v 2.1 91/12/10 16:31:56 rpruess Exp $
-----------------------------------------------------------------------------
$Log:	QueryManager.m,v $
Revision 2.1  91/12/10  16:31:56  rpruess
Moved the timer code from QueryManager to Query.  Each Query object is
ow responsible for determining when it has received the server "fields"
data.  This code was added prior to the initial Ph Version 2.0 release.

Revision 2.0  91/11/19  08:22:28  rpruess
Revision 2.0 is the initial production release of Ph.

-----------------------------------------------------------------------------*/

#define WINDOWOFFSET 20.0	/* How many pixels to offset each new window */
#define MAXOFFSET WINDOWOFFSET * 10.0

/* Standard C header files */
#include <stdio.h>
#include <strings.h>

/* Objective-C & Appkit header files */
#import <objc/List.h>
#import <appkit/Application.h>
#import <appkit/Control.h>
#import <appkit/defaults.h>
#import <appkit/Panel.h>

/* Application class header files */
#import "QueryManager.h"
#import "Query.h"
#import "../PhShare.h"
#import "../qiServers.h"

@implementation QueryManager

/*---------------------------------------------------------------------------
Initialization
-----------------------------------------------------------------------------*/
- init
{
   self = [super init];

   [NXApp loadNibSection:"QueryManager.nib" owner:self withNames:NO];

   qiManager = nil;
   servers = nil;

   offset = 0.0;

   queryList = [[List alloc] init];

   return self;
}

/*---------------------------------------------------------------------------
There are some object we must talk to later.
-----------------------------------------------------------------------------*/
- initIDs:aServers qiManager:aQiManager
{
   servers = aServers;
   qiManager = aQiManager;
   return self;
}

/*---------------------------------------------------------------------------
Bring the server's query window to life.
-----------------------------------------------------------------------------*/
- (BOOL) startQuery:(const char *)aServer
{
   int             i;
   id              theQuery;

   /*** Loop through the queryList to see if we have the Query object. */

   for (i = 0; i < [queryList count]; i++) {

      theQuery = [queryList objectAt:i];

      if (strcmp ([theQuery getServerName], aServer) == 0) {

	 if ([[theQuery qiObject] hasQiFields] == NO) {
	    strcpy (errMsg, "%s is initializing itself.  Please wait.");

	    NXRunAlertPanel (NULL, errMsg, NULL, NULL, NULL, aServer);
	    return NO;
	 }

	 [theQuery showWindow:self];
	 return YES;
      }

   }

   /*** Allocate the new Query object and set mandatory variables. */

   theQuery = [[Query alloc] init];
   [theQuery setQueryVars:qiManager domain:[servers getDomainName:aServer]];
   [theQuery setQueryWindow:[servers getSiteName:aServer] offset:offset];

   /*** Adjust the offset for the next Query window a few pixels from the
        this window.  Otherwise, the user might not see the new window. */

   if (offset < MAXOFFSET)
      offset += WINDOWOFFSET;
   else
      offset = WINDOWOFFSET / 2.0;

   /*** Open the Query session.  If it fails, free the Query object. */

   if ([theQuery openSession:aServer] == NO) {
      [theQuery free];
      return NO;
   }

   [queryList addObjectIfAbsent:theQuery];

   return YES;
}

/*---------------------------------------------------------------------------
Execution of this method will result in the clearing of the fields in the
window which is currently the key window.
-----------------------------------------------------------------------------*/
- clearQueryFields:sender
{
   [queryList makeObjectsPerform:@selector (clearQueryFields)];
   return self;
}

/*---------------------------------------------------------------------------
Execution of this method will result in the clearing of the view in the
window which is currently the key window.
-----------------------------------------------------------------------------*/
- clearQueryView:sender
{
   [queryList makeObjectsPerform:@selector (clearQueryView)];
   return self;
}

/*---------------------------------------------------------------------------
Return the Query ID for the specified server.
-----------------------------------------------------------------------------*/
- getQueryId:(const char *)aServer
{
   int             i;
   id              theQuery;
   
   for (i = 0; i < [queryList count]; i++) {

      theQuery = [queryList objectAt:i];

      if (strcmp ([theQuery getServerName], aServer) == 0)
	 return theQuery;

   }

   return nil;
}

@end
