#ifndef lint
static char *RCSid = "$Header: proxyarpd.c,v 1.7 89/02/23 14:25:11 he Locked $";
#endif

/*
 * $Log:	proxyarpd.c,v $
 * Revision 1.7  89/02/23  14:25:11  he
 * Improved robustness in code fetching ether address.
 * Implementet retry of getting ether address of host that
 * did not respond during configuration setup.
 * 
 * Revision 1.6  88/12/05  11:43:39  he
 * Error report from Mark Prior <mrp@sirius.ua.oz.au>. Problem with
 * referencing a u_long via a "constructed" pointer on the Sparc.
 * (Probably an alignment restriction.) Fixed by replacing dereference
 * with bcopy.
 * 
 * Revision 1.5  88/09/17  20:23:59  he
 * Fixed an error where an #ifdef should have been an #if instead.
 * 
 * Revision 1.4  88/09/17  18:57:18  he
 * Support for SunOS 4.0 added (from David Robinson, NASA JPL).
 * Changed from using u_long to struct sockaddr_in for Sun-4
 * portability (also from David Robinson, NASA JPL).
 * 
 * Revision 1.3  88/08/04  18:49:16  he
 * Removed ntohl's and htonl's (in the hope that we only
 * store and handle network byte order longs).
 * Inserted check for gateway ethernet address validity
 * in do_analyze.
 * 
 * Revision 1.2  88/07/25  18:39:53  he
 * Made it pass lint as well as passing Saber-C checks.
 * 
 * Revision 1.1  88/07/25  13:56:21  he
 * Initial revision
 * 
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/nit.h>

#if OS_REV >= 4
#include <net/nit_if.h>
#include <net/nit_pf.h>
#include <net/packetfilt.h>
#include <sys/stropts.h>
#endif

#include <net/if.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <signal.h>
#include <sys/file.h>
#include <errno.h>

#define strdup(s)	strcpy(malloc((unsigned) strlen(s)+1),s)

#ifndef CONFFILE
#define CONFFILE	"/etc/proxytab"
#endif

extern char *malloc();
extern int errno;


char *cmdname;


main(argc,argv)
     int argc;
     char **argv;
{
  char buffer[10000];
  char *ifname;
  int n;
  int if_fd;
  void process_arp();

  cmdname = strdup(argv[0]);
  
  if( argc != 2 ){
    (void) fprintf(stderr, "Usage: %s ifname\n", cmdname);
    exit(1);
  }

  if(strncmp(argv[1],"ie0",3) &&
     strncmp(argv[1],"ie1",3) &&
     strncmp(argv[1],"le0",3) &&
     strncmp(argv[1],"ec0",3) &&
     strncmp(argv[1],"ec1",3)){
    (void) fprintf(stderr, "%s: ifname must be one of ie0,ie1,le0,ec0,ec1\n",
		   cmdname);
    exit(1);
  }
     
  ifname = strdup(argv[1]);

  if_fd = arp_open(ifname);
  if(if_fd < 0){
    perror("arp_open");
    exit(1);
  }

  if(! configure(CONFFILE, ifname, if_fd)){
    (void) fprintf(stderr, "%s: unable to configure\n", cmdname);
    exit(2);
  }

  /* loop forever */
  while( (n = read(if_fd, buffer, sizeof(buffer))) != -1 )
    process_arp(buffer, n, if_fd);

  perror("if read");
  exit(4);
}


int 
arp_open(ifname)
     char *ifname;
{
  int if_fd;
#if OS_REV < 4
  struct sockaddr_nit snit;
  struct nit_ioc nioc;

  if_fd = socket( AF_NIT, SOCK_RAW, NITPROTO_RAW );
  if( if_fd < 0 ) return(-1);

  snit.snit_family = AF_NIT;
  (void) strncpy( snit.snit_ifname, ifname, sizeof(snit.snit_ifname));

  if( bind(if_fd, (struct sockaddr *)&snit, sizeof(snit)) ) return(-1);

  bzero((char*)&nioc, sizeof(nioc));
  nioc.nioc_bufspace = NITBUFSIZ;
  nioc.nioc_chunksize = NITBUFSIZ;
  nioc.nioc_typetomatch = ETHERTYPE_ARP;
  nioc.nioc_snaplen = 32767;
  nioc.nioc_flags = NF_TIMEOUT;
  nioc.nioc_timeout.tv_usec = 200;
  if (ioctl(if_fd, SIOCSNIT, &nioc) != 0) return(-1);
  
  return(if_fd);
#else
  struct strioctl si;
  struct ifreq ifr;
  struct ether_header eh;
  struct packetfilt pf;
  register u_short *fwp;

  /*
   * Open NIT device
   */
  if ((if_fd = open ("/dev/nit", O_RDWR)) == -1)
    return (-1);

  /*
   * Arrange to get message-discard from the stream
   */
  if (ioctl (if_fd, I_SRDOPT, (char *)RMSGD) < 0){
    perror ("ioctl(RMSGD)");
    return (-1);
  }

  /*
   * Push on protocol filter
   */
  if (ioctl (if_fd, I_PUSH, "pf") < 0){
    perror ("ioctl(I_PUSH)");
    return (-1); 
  }
  /*
   * Filter only ARP requests
   */
  fwp= &pf.Pf_Filter[0];
  *fwp++ = ENF_PUSHWORD + 
           ((u_int)&eh.ether_type - (u_int)&eh.ether_dhost) /
	     sizeof (u_short);
  *fwp++ = ENF_PUSHLIT|ENF_EQ;
  *fwp++ = htons(ETHERTYPE_ARP);
  pf.Pf_FilterLen = fwp - &pf.Pf_Filter[0];
  pf.Pf_Priority = 5;
  si.ic_cmd = NIOCSETF;
  si.ic_timout = INFTIM;
  si.ic_len = sizeof (pf);
  si.ic_dp = (char *)&pf;
  if (ioctl (if_fd, I_STR, (char *)&si) < 0){
    perror ("ioctl(NIOCSETF)");
    return (-1);
  }

  /*
   * Bind NIT stream
   */
  (void )strncpy (ifr.ifr_name, ifname, sizeof (ifr.ifr_name) - 1);
  ifr.ifr_name[sizeof (ifr.ifr_name) - 1] = '\0';
  si.ic_cmd = NIOCBIND;
  si.ic_timout = INFTIM;
  si.ic_len = sizeof (ifr);
  si.ic_dp = (char *)&ifr;
  if (ioctl (if_fd, I_STR, (char *)&si) < 0){
    perror ("ioctl(NIOCBIND)");
    return (-1);
  }

  return (if_fd);
#endif
}

  
void
process_arp(buffer, blen, if_fd)
     char *buffer;
     int blen;
     int if_fd;
{
  void			do_analyze();
#if OS_REV >= 4
  do_analyze(buffer, blen, if_fd);
#else
  struct nit_hdr	*hdr;

  while( blen > 0){
    hdr = (struct nit_hdr*) buffer;
    buffer += sizeof(struct nit_hdr);
    blen -= sizeof(struct nit_hdr);

    if( hdr->nh_state == NIT_CATCH ){
      do_analyze(buffer, hdr->nh_wirelen, if_fd);
      buffer += hdr->nh_wirelen;
      blen -= hdr->nh_wirelen;
    }
  }
#endif
}


struct in_addr		my_inet_addr;
struct ether_addr	my_ether_addr;
struct in_addr		my_netmask;
struct in_addr		my_network;


int
get_my_addrs(ifname, if_fd, ping_fd)
     char *ifname;
     int if_fd, ping_fd;
{
  char buf[BUFSIZ];
  char lbuf[80];
  struct ifreq *ifr, ifreq;
  struct ifconf ifc;
  struct sockaddr_in *sin;
  int n;
  char *sprintf();

  ifc.ifc_len = sizeof(buf);
  ifc.ifc_buf = buf;
  if( ioctl(ping_fd, SIOCGIFCONF, &ifc) < 0 ){
    perror("get if config");
    return(0);
  }

  n = ifc.ifc_len / sizeof(struct ifreq);
  for( ifr = ifc.ifc_req; n; n--, ifr++ ){

    if( ifr->ifr_addr.sa_family != AF_INET ) continue;

    if( strcmp(ifname, ifr->ifr_name) != 0 ) continue;

    ifreq = *ifr;
    sin = (struct sockaddr_in *)&ifreq.ifr_addr;

    if( ioctl(ping_fd, SIOCGIFADDR, (char *)&ifreq) < 0){
      (void) sprintf(lbuf, "%s: get if addr", ifname);
      perror(lbuf);
      return(0);
    }

    my_inet_addr = sin->sin_addr;

    if( ioctl(ping_fd, SIOCGIFNETMASK, &ifreq) < 0 ){
      (void) sprintf(lbuf, "%s: get if netmask", ifname);
      perror(lbuf);
      return(0);
    }

    my_netmask = sin->sin_addr;
    my_network.s_addr = my_inet_addr.s_addr & my_netmask.s_addr;
    
    if( ioctl(if_fd, SIOCGIFADDR, &ifreq) < 0 ){
      (void) sprintf(lbuf, "%s: get if etheraddr", ifname);
      perror(lbuf);
      return(0);
    }

    my_ether_addr = *((struct ether_addr *) &ifreq.ifr_addr.sa_data[0]);
  }
  return(1);
}
  
struct gateway {
  unsigned long		network;
  unsigned long 	inet_addr;
  struct ether_addr 	ether_addr;
  int			eaddr_valid;
  int			failed_queries;
  struct gateway 	*next;
};

#define IGNORE_QUERIES	20

struct gateway	*gateways;
int ping_fd;			/* socket to ping other host(s) to */
				/* force an ARP entry in local cache */


int
configure(fname, ifname, if_fd)
     char *fname;
     char *ifname;
     int if_fd;
{
  FILE *fp;
  char read_if[80], read_net[80], read_gateway[80];
  char buf[BUFSIZ];
  struct gateway	*gp;
  struct netent		*np;
  struct hostent	*hp;
  struct ether_addr 	*eap, *ip_to_ether();
  u_long		addr;
  int			ret;

  /* Open dgram (udp) socket, so we can get if config. */
  /* Use this one later for pinging for an ether address... */
  ping_fd = socket(AF_INET, SOCK_DGRAM, PF_UNSPEC);
  if(ping_fd == -1){
    perror("ping socket");
    return(0);
  }

  if( ! get_my_addrs(ifname, if_fd, ping_fd) ) return(0);

  /* Read configuration file. */
  if( (fp = fopen(fname,"r")) == NULL ){
    perror(fname);
    return(0);
  }

  while(fgets(buf,BUFSIZ,fp) != NULL ){

    if( buf[0] == '#' ) continue; /* comment */

    /* 
     * Format of line is:
     * interface	leads_to_net	gateway_name_or_addr
     * Example:
     * le0		128.39.47.0	loke.idt.unit.no
     * le0		transmnett	telesun.tele.unit.no
     * (where transmnett = 128.39.44.0)
     *
     * Finding gateway_name's address can be time consuming on a host
     * with SunOS YP hostname lookup where the hostname is the same
     * for all the addresses in the /etc/hosts file, since we
     * (potentially) have to scan the whole YP database linearly.
     */

    ret = sscanf(buf, "%s%s%s", read_if, read_net, read_gateway);
    if( ret == EOF ){
      (void) fprintf(stderr, "%s: can't parse\n", buf);
      continue;
    }

    if( strcmp(read_if, ifname) ) continue; /* line not for us */
    
    gp = (struct gateway*) malloc(sizeof(struct gateway));
    if( gp == NULL ){
      (void) fprintf(stderr, "out of memory\n");
      return(0);
    }

    (void) bzero((char*)gp, sizeof(struct gateway));
    gp->network = inet_addr(read_net);
    if( gp->network == -1 ){
      np = getnetbyname(read_net);
      if( np == NULL ){
	(void) fprintf(stderr,"%s: no such network?\n",read_net);
	continue;
      }
      if( np->n_addrtype != AF_INET) continue;
      gp->network = np->n_net;
    }
    
    gp->inet_addr = inet_addr(read_gateway);
    if( gp->inet_addr == -1 ){
#ifdef h_addr			/* we have a 4.3-style resolver */
				/* with this, lookup is fast for names also. */
      char **hap;

      hp = gethostbyname(read_gateway);
      if( hp == NULL ){
	(void) fprintf(stderr, "%s: no such host?\n", read_gateway);
	continue;
      }

      if( hp->h_addrtype != AF_INET) continue;
      if( hp->h_length != 4){
	(void) fprintf(stderr, "%s: addrlen != 4?\n", read_gateway);
	continue;
      }

      hap = hp->h_addr_list;
      for( ; *hap; hap++ ){
	addr = *( (u_long *)*hap );
	if( (addr & my_netmask.s_addr) != my_network.s_addr ) continue;
	gp->inet_addr = addr;
	break;
      }
      if( gp->inet_addr == 0 ){
	(void) fprintf(stderr,
		       "could not find %s's address on this network (%s)\n",
		       read_gateway, inet_ntoa(my_network));
	continue;
      }
#else
      /* Have to look through all entries (until match), since a */
      /* gateway must have more than one entry in /etc/hosts. (You may */
      /* avoid this by giving the host different names for the */
      /* differrent interfaces...) This takes quite a while. We try to */
      /* lookup directly first. */

      hp = gethostbyname(read_gateway);
      if( hp == NULL ){
	(void) fprintf(stderr, "%s: no such host?\n", read_gateway);
	continue;
      }

      if( hp->h_addrtype != AF_INET) continue;
      if( hp->h_length != 4){
	(void) fprintf(stderr, "%s: addrlen != 4?\n",read_gateway);
	continue;
      }

      addr = *( (u_long *)hp->h_addr );
      if( addr & my_netmask.s_addr != my_network.s_addr ){
				/* have to do linear search */
				/* code stolen from proxyd by Barry Shein */
	sethostent(1);
	while((hp = gethostent()) != NULL){

	  if((hp->h_addrtype != AF_INET) ||
	     (my_network.s_addr !=
	      ( *((u_long *)hp->h_addr)) & my_netmask.s_addr))
	    continue;

	  /* On this net and an internet host */
	  /* Check name(s) */
	  if( strcmp(read_gateway, hp->h_name) != 0){
	    register char **ap = hp->h_aliases;

	    /* Check aliases also */
	    for(; *ap; ap++)
	      if(strcmp(*ap, read_gateway) == 0) break;
	    
	    if(*ap == NULL) continue;
	  }
	  
	  /* Found it */
	  addr = *((u_long *)hp->h_addr);
	  break;
	}
	if(hp == NULL){
	  (void) fprintf(stderr,
			 "%s: can't find address on this network (%s)\n",
			 read_gateway,
			 inet_ntoa( my_network ));
	  continue;
	}
	endhostent();
      }
      gp->inet_addr = addr;
#endif /* have 4.3-based resolver? */
    }

    /* Now we have the gateway address on this net. */
    /* Try getting his ethernet address also. */

    eap = ip_to_ether(gp->inet_addr);
    if( eap != NULL ){
      gp->eaddr_valid = 1;
      gp->ether_addr = *eap;
    }

    /* Put at head of list */
    gp->next = gateways;
    gateways = gp;
  }
  (void) fclose(fp);

  (void) detach();
  return(1);
}


u_short discard_port = 0;


struct ether_addr *
ip_to_ether(inet_addr)
     u_long inet_addr;
{
  struct servent *sp;
  static struct arpreq arpreq;
  struct sockaddr_in *sin;
  struct sockaddr_in to;
  int i;

  if( inet_addr == my_inet_addr.s_addr ) return(&my_ether_addr);

  if( discard_port == 0 ){
    sp = getservbyname("discard", "udp");
    if( sp == NULL ){
      perror("discard/udp");
      return(NULL);
    }
    discard_port = (u_short) sp->s_port;
  }

  bzero( (char *) &arpreq, sizeof( arpreq ) );
  sin = (struct sockaddr_in*) &arpreq.arp_pa;

  sin->sin_family = AF_INET;
  sin->sin_addr.s_addr = inet_addr;
  arpreq.arp_ha.sa_family = AF_UNSPEC;
  
  i = 4;			/* Don't know if this is reasonable, */
				/* does the local host send multiple */
				/* ARP's to resolve the address? */
				/* Should then perhaps not try to send */
				/* multiple packets, just wait 2-3 */
				/* rounds before giving up. */
  while( i-- ){
    if( ioctl(ping_fd, SIOCGARP, (caddr_t)&arpreq) < 0 ){
      if( errno == ENXIO ){
	to.sin_family = AF_INET;
	to.sin_port = discard_port;
	to.sin_addr.s_addr = inet_addr;
	if( sendto(ping_fd, "a", 1, 0,
		   (struct sockaddr*)&to, sizeof(to)) < 0 ) return(NULL);
	if( i < 3 ) usleep(100);
      }
      else
	return(NULL);
    }
    /* Did we get a completed ARP entry? */
    if( arpreq.arp_flags & ATF_COM ) break;
  }
				/* Too many retries? */
  if( i <= 0 ) return(NULL);
				/* Special sanity check for zero */
				/* ethernet address. 6 is magic */
				/* constant (length of ether address). */ 
  for( i = 0; arpreq.arp_ha.sa_data[i] == 0 && i < 6; i++);
  if( i == 6 ) return(NULL);
				/* All ok, return hardware address */
  return( (struct ether_addr *) &arpreq.arp_ha.sa_data[0]);
}


void
do_analyze(buffer, blen, if_fd)
     char *buffer;
     int blen, if_fd;
{
  struct ether_header 	*e_head;
  struct ether_arp	*arpp;
  u_long		addr;
  char 			localbuf[BUFSIZ];
  struct gateway	*gp, *find_gateway_to();
  struct ether_addr 	*eap, *ip_to_ether();
  int			n;
  
  /* First check that packet is sufficiently long */
  if(sizeof(struct ether_header)
     + sizeof(struct ether_arp) > blen ) return;

  e_head = (struct ether_header *) buffer;
  buffer += sizeof(struct ether_header);
  
				/* sanity check */
  if( e_head->ether_type != ETHERTYPE_ARP ) return;

  arpp = (struct ether_arp *) buffer;
  /* handle ARP packet */
  
				/* more sanity checks */
  if( arpp->arp_hrd != ARPHRD_ETHER ) return;
  if( arpp->arp_pro != ETHERTYPE_IP ) return;
  if( arpp->arp_hln != 6) return;
  if( arpp->arp_pln != 4) return;
  if( arpp->arp_op != ARPOP_REQUEST ) return;
  
  /*
   * Original version dumped core below
   * addr = *((u_long*) &arpp->arp_tpa[0]);
   *
   * Bug reported by: Mark Prior <mrp@sirius.ua.oz.au>
   *
   * Comment:
   *
   * This is an effect of the Sparc and its fairly strict rules
   * for alignment. Accessing/copying as bytes works OK. Just
   * avoid using direct dereference for all versions.
   *
   */

#if OS_REV >= 4
  bcopy( (char*)arpp->arp_tpa, (char*)&addr, 4);
#else
  bcopy( (char*)arpp->arp_xtpa, (char*)&addr, 4);
#endif
				/* should perhaps check to see if we */
				/* have an ARP request for local part */
				/* of -1 or 0, and report that */
				/* (mis-behaving hosts). For now we */
				/* let that be. */ 

  if( addr & my_netmask.s_addr == my_network.s_addr ) return;
				/* directly attached */

  gp = find_gateway_to(addr & my_netmask.s_addr);
  if( gp == NULL ) return;	/* no gateway known */

				/* Could probably have a time-out, so */
				/* that we eventually will be able to */
				/* find the gateway's ethernet address */
				/* if it is down when this program is */
				/* started from another host. Probably */
				/* an argument for running this */
				/* program on the gateway (if it is a */
				/* Sun, that is...) */

  if( ! gp->eaddr_valid ) {	/* invalid ether address? */
				/* If we have received enough failed */
				/* queries... */

    /* Actually, we should check the time since last query, and retry */
    /* getting the ether address after a reasonable time. */

    if( gp->failed_queries++ > IGNORE_QUERIES ) {
				/* retry getting ether address */
      eap = ip_to_ether(gp->inet_addr);
      if( eap != NULL ){	/* success? */
	gp->eaddr_valid = 1;	/* yes */
	gp->ether_addr = *eap;	/* copy ether address */
      }else{			/* no success */
	gp->failed_queries = 0;	/* start another round */
	return;			/* and just return */
      }
    }else
      return;			/* ignore query by just returning */
  }

  n = build_arp_reply( gp, arpp, e_head, localbuf );
  (void) arp_write( if_fd, localbuf, n );
}


struct gateway *
find_gateway_to(network)
     u_long network;
{
  struct gateway *gp;

  for( gp = gateways; gp && gp->network != network; gp = gp->next );
  return(gp);
}


int
build_arp_reply( gp, q_arpp, q_e_head, buf )
     struct gateway *gp;
     struct ether_arp *q_arpp;
     struct ether_header *q_e_head;
     char *buf;
{
  struct ether_header *a_e_head;
  struct ether_arp *a_arpp;
  int len;
  
  a_e_head = (struct ether_header*) buf;
  a_arpp = (struct ether_arp*) (buf + sizeof(struct ether_header));
  len = sizeof(struct ether_header) + sizeof(struct ether_arp);
  bzero( buf, len );

  a_e_head->ether_dhost = q_e_head->ether_shost;
  a_e_head->ether_shost = my_ether_addr;
  a_e_head->ether_type = ETHERTYPE_ARP;
  
  a_arpp->arp_hrd = ARPHRD_ETHER;
  a_arpp->arp_pro = ETHERTYPE_IP;
  a_arpp->arp_hln = 6;
  a_arpp->arp_pln = 4;
  a_arpp->arp_op = ARPOP_REPLY;
				/* Here we tell the necessary "lie": */
#if OS_REV >= 4
  bcopy((char*) &gp->ether_addr,      (char*) &a_arpp->arp_sha,
	sizeof(struct ether_addr));
  bcopy((char*) &q_arpp->arp_tpa[0], (char*) &a_arpp->arp_spa[0], 4);

  bcopy((char*) &q_arpp->arp_sha, (char*) &a_arpp->arp_tha,
	sizeof(struct ether_addr));
  bcopy((char*) &q_arpp->arp_spa[0], (char*) &a_arpp->arp_tpa[0], 4);
#else
  bcopy((char*) &gp->ether_addr,      (char*) &a_arpp->arp_xsha[0],
	sizeof(struct ether_addr));
  bcopy((char*) &q_arpp->arp_xtpa[0], (char*) &a_arpp->arp_xspa[0], 4);

  bcopy((char*) &q_arpp->arp_xsha[0], (char*) &a_arpp->arp_xtha[0],
	sizeof(struct ether_addr));
  bcopy((char*) &q_arpp->arp_xspa[0], (char*) &a_arpp->arp_xtpa[0], 4);
#endif
  return(len);
}
	

int
arp_write(fd, buf, len)
     int fd;
     char *buf;
     int len;
{
  struct sockaddr sa;
  int offset = sizeof(sa.sa_data);
  int result;

  sa.sa_family = AF_UNSPEC;
  bcopy(buf, sa.sa_data, offset);

#if OS_REV >=4
  {
    struct strbuf ctl, data;
    ctl.maxlen = ctl.len = sizeof(sa);
    ctl.buf = (char *)&sa;
    data.maxlen = data.len = len - offset;
    data.buf = buf + offset;
    if ((result = putmsg(fd, &ctl, &data, 0)) < 0)
      return (-1);
    /*
     * Flush buffers
     */
    (void)ioctl (fd, I_FLUSH, FLUSHW);
    return(result+offset);
   }
#else
  result = sendto(fd, buf+offset, len-offset,
		  0, &sa, sizeof(sa));
  return (result+offset);
#endif
}


detach()
{
  int fd;

  (void) signal(SIGTTOU, SIG_IGN);
  (void) signal(SIGTTIN, SIG_IGN);
  (void) signal(SIGTSTP, SIG_IGN);

#ifndef DEBUG  
  if (fork())
    exit(0);			/* parent */
  
  (void) setpgrp(0, getpid());		/* change process group */
  if ((fd = open("/dev/tty", O_RDWR)) >= 0) {
    (void) ioctl(fd, TIOCNOTTY, 0);	/* lose controlling terminal */
    (void) close(fd);
  }
#endif
}
  
