/*
** pkvm.c - Partial Kernel "Virtual" Memory access function emulation.
**
** Copyright (c) 1997 Peter Eriksson <pen@lysator.liu.se>
**
** This program is free software; you can redistribute it and/or
** modify it as you wish - as long as you don't claim that you wrote
** it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#include "config.h"

#if defined(NEED_LIBKVM) && !defined(HAVE_KVM_OPEN)

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <nlist.h>

#include "pidentd.h"
#include "pkvm.h"

#ifndef _PATH_UNIX
#define _PATH_UNIX "/vmunix"
#endif
#ifndef _PATH_KMEM
#define _PATH_KMEM "/dev/kmem"
#endif
#ifndef _PATH_SWAP
#define _PATH_SWAP "/dev/swap"
#endif


kvm_t *
kvm_open(char *namelist,
	  char *corefile,
	  char *swapfile,
	  int flag,
	  char *errstr)
{
    kvm_t *kd;

    
    if (!namelist)
	namelist = _PATH_UNIX;
    if (!corefile)
	corefile = _PATH_KMEM;
    if (!swapfile)
	swapfile = _PATH_SWAP;
    
    kd = (kvm_t *) malloc(sizeof(kvm_t));
    if (kd == NULL)
    {
	if (errstr)
	    perror(errstr);
	return NULL;
    }
    
    kd->namelist = s_strdup(namelist);
    
    if ((kd->fd = s_open(corefile, flag)) < 0)
    {
	if (errstr)
	    perror(errstr);
	s_free(kd->namelist);
	s_free(kd);
	return NULL;
    }
    
    return kd;
}


int
kvm_close(kvm_t *kd)
{
    int code;
    
    code = close(kd->fd);
    s_free(kd->namelist);
    s_free(kd);
    
    return code;
}


/*
** Extract offsets to the symbols in the 'nl' list. Returns 0 if all found,
** or else the number of variables that was not found.
*/
int
kvm_nlist(kvm_t *kd, struct nlist *nl)
{
    int code;
    int i;
    
    code = nlist(kd->namelist, nl);
    if (code != 0)
	return code;
    
    /*
    ** Verify that we got all the needed variables. Needed because some
    ** implementations of nlist() returns 0 although it didn't find all
    ** variables.
    */
    if (code == 0)
    {
	for (i = 0; nl[i].n_name && nl[i].n_name[0]; i++)
	    if( nl[i].n_value == 0)
		code++;
    }
    
    return code;
}


/*
** Get a piece of the kernel memory
*/
int
kvm_read(kvm_t *kd,
	 unsigned long addr,
	 void *buf,
	 int len)
{
    errno = 0;
    
    if (lseek(kd->fd, addr, 0) != addr || errno != 0)
    {
	if (debug)
	    fprintf(stderr,
		    "kvm_read(%d,%08lx,..,%d): lseek failed (errno=%d)\n",
		    kd->fd, addr, len, errno);
	return -1;
    }
    
    if (s_read(kd->fd, (char *) buf, len) != len || errno != 0)
    {
	if (debug)
	    fprintf(stderr,
		    "kvm_read(%d,%08lx,..,%d): read failed (errno=%d)\n",
		    kd->fd, addr, len, errno);
	return -1;
    }
	
    return len;
}


#else
/* Just a dummy function */
int kvm_dummy(void)
{
    return -1;
}
#endif
