/*
 * nfslib.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/errno.h>
#include <string.h>
#include <rpc/types.h>
#include <nfs/nfs.h>
#include <rpc/rpc.h>
#include <rpcsvc/mount.h>
#include "xkernel.h"
#include "userupi.h"
#include "ip.h"
#include "dns.h"
#include "udp.h"
#include "sun_rpc/sun_rpc.h"
#include <ctype.h>
#include "nfs/xnfs.h"
#include "nfslib.h"


Protl NFS, DNS, IP;

static void nfs_demux();

SUNRPCaddr CLIENTE, SERVER;
char *CLIENTNAME, *SERVERNAME;


int null()
{
}

int uid = 0;
int gid = 0;

/******************************
/*	nfs_null
/******************************/

int nfs_null()
{
}

/******************************
/*	nfs_init
/******************************/

int
nfs_init (auid, agid)
     int auid, agid;
{
   uid = auid;
   gid = agid;
   IP = xgetprotlbyname("ip");
   NFS = xgetprotlbyname("nfs");
   DNS = xgetprotlbyname("dns");

}


/******************************
/*	nfs_mount
/******************************/

int
nfs_mount (nfs_server, mount_dir)
     char *nfs_server, *mount_dir;
{
  struct minfo Mount;
  struct resolve_result res;
  IPaddr myipaddr;


  /* get nfs server ip address */
  strcpy (res.name, nfs_server);
  xcontrolprotl (DNS, RESOLVE, &res, sizeof (res));

  Mount.host = res.addr[0];
  strcpy (Mount.fs_pathname, mount_dir);

  xcontrolprotl (IP, GETMYADDR, &myipaddr, sizeof (myipaddr));
  sprintf (res.name, "%d.%d.%d.%d", myipaddr.host.a, myipaddr.host.b, 
                                    myipaddr.host.c, myipaddr.host.d);
  xcontrolprotl (DNS, RRESOLVE, &res, sizeof (res));

  strcpy (Mount.machname, res.name);
  Mount.uid = uid;
  Mount.gid = gid;

  if (xcontrolprotl (NFS, NFS_MOUNT, &Mount, sizeof (struct minfo)))
    {
      printf ("mount failed\n");
      return (-1);
    }
  return (0);
}


/******************************
/*	nfs_readdir
/******************************/

int
nfs_readdir (dirname, offset, nbytes, darr)
     char *dirname;
     int offset, nbytes;
     struct dirents *darr;
{
  int i;
  struct rddir rddarg;
  struct entrys *tentry;
/*
 *  Set the structure values to be passed to control op in nfs
 *  fields with value -1 are ignored
 */
  strcpy (rddarg.name, dirname);
  rddarg.offs = offset;
  rddarg.nbytes = nbytes;

  if (xcontrolprotl (NFS, NFS_READDIR, &rddarg, sizeof (struct rddir)))
    {
      printf ("readdir failed\n");
      return (-1);
    }


  /* compute and allocate array for entries */
  i = rddarg.nbytes; /* set it when leaving doreaddir */
  darr->entries = (struct entrys *) malloc(sizeof(struct entrys) * i);
 
  darr->nentries = i;
  tentry = darr->entries;

  for (i = 0; i < darr->nentries; i++) {
     tentry->fileid = rddarg.direct[i].fileid;
     strcpy(tentry->name, rddarg.direct[i].name);
     tentry+=1;
  }
  return (0);
}


/******************************
/*	nfs_mkdir
/******************************/

int
nfs_mkdir (dirname, mode)
     char *dirname;
     long mode;
{
  struct create crea;
/*
 *  Set the structure values to be passed to control op in nfs
 *  fields with value -1 are ignored
 */
  strcpy (crea.name, dirname);
  crea.mode = mode;
  crea.uid = uid;
  crea.gid = gid;

  if (xcontrolprotl (NFS, NFS_MKDIR, &crea, sizeof (struct create)))
    {
      printf ("mkdir failed\n");
      return (-1);
    }
  return (0);
}


/******************************
/*	nfs_rmdir
/******************************/

int
nfs_rmdir (dirname)
     char *dirname;
{
  if (xcontrolprotl (NFS, NFS_RMDIR, dirname, strlen (dirname)+1))
    {
      printf ("rmdir failed\n");
      return (-1);
    }
  return (0);
}


/******************************
/*	nfs_setattr
/******************************/

int
nfs_setattr (filename, crea)
     char *filename;
     struct create *crea;
{
/*
 *  Set the structure values to be passed to control op in nfs
 *  fields with value -1 are ignored
 */
  struct create bufcrea;

  strcpy(bufcrea.name, filename);
  bufcrea.mode = crea->mode;
  bufcrea.uid = crea->uid;
  bufcrea.gid = crea->gid;
  bufcrea.size = crea->size;
  bufcrea.atime.tv_sec = crea->atime.tv_sec;
  bufcrea.atime.tv_usec = crea->atime.tv_usec;
  bufcrea.mtime.tv_sec = crea->mtime.tv_sec;
  bufcrea.mtime.tv_usec = crea->mtime.tv_usec;

  if (xcontrolprotl (NFS, NFS_SETATTR, &bufcrea, sizeof (struct create)))
    {
      printf ("setattr failed\n");
      return (-1);
    }
  return (0);
}


/******************************
/*	nfs_remove
/******************************/

int
nfs_remove(filename)
     char *filename;
{
  if (xcontrolprotl (NFS, NFS_REMOVE, filename, strlen (filename)+1))
    {
      printf ("remove file failed\n");
      return (-1);
    }
  return (0);
}


/******************************
/*	nfs_rename
/******************************/

int
nfs_rename(oldname, newname)
     char *oldname, *newname;
{
  struct xlinkargs rename;

  strcpy(rename.from, oldname);
  strcpy(rename.to, newname);

  if (xcontrolprotl (NFS, NFS_RENAME, &rename, sizeof (struct xlinkargs)))
    {
      printf ("rename file failed\n");
      return (-1);
    }
  return (0);
}


/******************************
/*	nfs_link
/******************************/

int
nfs_link(file1, file2)
     char *file1, *file2;
{
  struct xlinkargs mlink;

  strcpy(mlink.from, file1);
  strcpy(mlink.to, file2);

  if (xcontrolprotl (NFS, NFS_LINK, &mlink, sizeof (struct xlinkargs)))
    {
      printf ("link files failed\n");
      return (-1);
    }
  return (0);
}


/******************************
/*	nfs_symlink
/******************************/

int
nfs_symlink(file1, file2)
     char *file1, *file2;
{
  struct xsymlinkargs mlink;

  strcpy(mlink.from, file1);
  strcpy(mlink.to, file2);

  if (xcontrolprotl (NFS, NFS_SYMLINK, &mlink, sizeof (struct xsymlinkargs)))
    {
      printf ("symlink files failed\n");
      return (-1);
    }
  return (0);
}


/******************************
/*	nfs_rlink
/******************************/

int
nfs_rlink(filename)
     char *filename;
{
  if (xcontrolprotl (NFS, NFS_READLINK, filename, strlen (filename)+1))
    {
      printf ("remove file failed\n");
      return (-1);
    }
  return (0);
}


/******************************
/*	nfs_getattr
/******************************/

int
nfs_getattr (filename, crea)
     char *filename;
     struct create *crea;
{
  struct create bufcrea;

  strcpy (bufcrea.name, filename);
  bufcrea.uid = uid;
  bufcrea.gid = gid;

  if (xcontrolprotl (NFS, NFS_GETATTR, &bufcrea, sizeof (struct create)))
    {
      printf ("getattr failed\n");
      return (-1);
    }
  crea->mode = bufcrea.mode;
  crea->uid = bufcrea.uid;
  crea->gid = bufcrea.gid;
  crea->size = bufcrea.size;
  crea->atime.tv_sec = bufcrea.atime.tv_sec;
  crea->atime.tv_usec = bufcrea.atime.tv_usec;
  crea->mtime.tv_sec = bufcrea.mtime.tv_sec;
  crea->mtime.tv_usec = bufcrea.mtime.tv_usec;
  return (0);
}


/******************************
/*	nfs_create
/******************************/

NFS_FILE *
nfs_create (filename,  mode)
     char *filename;
     int mode;
{
  NFS_FILE *stream;
  NFS_FD file;
  Part part[4];
  NFS_FD fd;
  struct nfssattr stat;


  fd = (NFS_FD) malloc(sizeof(NFS_FD_struct));
  if(fd == NULL) {
     return NULL;
  }

  fd->proto = xcreateprotl (nfs_demux, null, null);
  if(fd->proto == ERR_PROTL) {
     free(fd);
     return NULL;
  }
  stat.sa_mode = mode;
  stat.sa_uid = uid;
  stat.sa_gid = gid;

  part[0].address = (char *) NFS_CREATE;
  part[0].length = sizeof (int);
  part[1].address = filename;
  part[1].length = strlen (filename) + 1;
  part[2].address = (char *) &stat;
  part[2].length = sizeof(struct nfssattr);
  part[3].address = (char *) NULL;
  part[3].length = 0;

  fd->file = (Sessn) xopen (fd->proto, NFS, part);
  if(fd->file == ERR_SESSN) {
     free(fd);
     return NULL;
  }
 
  file = fd;
  if (file == NULL)
      return NULL;

  stream = (NFS_FILE *) malloc (sizeof (NFS_FILE));
  if (stream == NULL)
      return NULL;
  stream->cnt = 0;
  stream->ptr = stream->buf;
  stream->file = file;

  return stream;
}

/******************************
/*	nfs_fopen
/******************************/

NFS_FILE *
nfs_fopen (filename)
     char *filename;
{
  NFS_FILE *stream;
  NFS_FD file;

  file = nfs_open (filename,NFS_OPEN);

  if (file == NULL)
      return NULL;

  stream = (NFS_FILE *) malloc (sizeof (NFS_FILE));
  if (stream == NULL)
      return NULL;
  stream->cnt = 0;
  stream->ptr = stream->buf;
  stream->file = file;

  return stream;
}


/******************************
/*	nfs_fread
/******************************/

int
nfs_fread (data, size, count, fp)
     char *data;
     int size;
     int count;
     NFS_FILE *fp;
{
  unsigned long n;
  int l;
  int cnt;

  l = 0;
  n = (unsigned long) count *(unsigned long) size;
again:
  if ((cnt = fp->cnt) > 0)
    {
      cnt = (cnt < n) ? cnt : n;
      bcopy (fp->ptr, data, cnt);
      fp->cnt -= cnt;
      fp->ptr += cnt;
      l += cnt;
      data += cnt;
      n -= cnt;
    }
  /* n == how much more */
  if (n > 0)
    {
      if (n < BUFFER_SIZE)
	{
	  /* read in BUFFER_SIZE bytes into fp->buf and do it again */
	  fp->ptr = fp->buf;
	  if ((cnt = nfs_read_data (fp->file, fp->buf, BUFFER_SIZE)) <= 0)
	    {
	      goto ret;
	    }
	  fp->cnt = cnt;
	  goto again;
	}
      else
	{			/* read in n bytes into data */
	  l += nfs_read_data (fp->file, data, n);
	}
    }
ret:
  return ((l > 0) ? ((unsigned long) l / (unsigned long) size) : 0);
}


/******************************
/*	nfs_read_data
/******************************/

int
nfs_read_data (fd, buf, bufsize)
     NFS_FD fd;
     char *buf;
     int bufsize;
{
  int n;
  int rc;

  n = 0;
  while(bufsize > NFS_MAXDATA) {
     rc = xcontrolsessn (fd->file, NFS_READ, buf, NFS_MAXDATA);
     if(rc == -1) {
        return -1;
     }
     n += rc;
     buf += NFS_MAXDATA;
     bufsize -= NFS_MAXDATA;
  }
  rc = xcontrolsessn (fd->file, NFS_READ, buf, bufsize);
  if(rc == -1) {
     return -1;
  }
  return n + rc;
}


/******************************
/*	nfs_write
/******************************/

int
nfs_write (fd, buf, bufsize)
     NFS_FD fd;
     char *buf;
     int bufsize;
{
  int n;
  int rc;
  
  n = 0;
  while(bufsize > NFS_MAXDATA) {
     rc = xcontrolsessn (fd->file, NFS_WRITE, buf, NFS_MAXDATA);
     if(rc == -1) {
        return -1;
     }
     n += rc;
     buf += NFS_MAXDATA;
     bufsize -= NFS_MAXDATA;
  }
  rc = xcontrolsessn (fd->file, NFS_WRITE, buf, bufsize);
  if(rc == -1) {
     return -1;
  }
  return n + rc;
}


/******************************
/*	nfs_fstat
/******************************/

int
nfs_fstat(fd, buf)
NFS_FD fd;
struct stat *buf;
{
   struct nfsfattr stat;
   
   (void) xcontrolsessn(fd->file, NFS_GETATTR, &stat, sizeof(stat));
   buf->st_dev = stat.na_fsid;
   buf->st_ino = stat.na_nodeid;
   buf->st_mode = stat.na_mode;
   buf->st_uid = stat.na_uid;
   buf->st_nlink = stat.na_nlink;
   buf->st_gid = stat.na_gid;
   buf->st_rdev = stat.na_rdev;
   buf->st_size = stat.na_size;
   buf->st_atime = stat.na_atime.tv_sec;
   buf->st_mtime = stat.na_mtime.tv_sec;
   buf->st_ctime = stat.na_ctime.tv_sec;

   return 0;
}



/******************************
/*	nfs_fseek
/******************************/

int 
nfs_fseek (fp, offset, origin)
     NFS_FILE *fp;
     long offset;
     int origin;
{
  long pos;
  long count;
  int rv;

  count = fp->cnt;
  if ((count == 0) || (origin == SEEK_END))
    {
      return nfs_lseek (fp->file, offset, origin);
    }

  pos = offset;
  if (origin == SEEK_SET)
    {
      long realpos;
      if ((realpos = nfs_tell (fp->file)) < 0)
	{
	  return -1;
	}
      pos += count - realpos;
    }
  else
    offset -= count;

  if ((pos <= count) && (pos >= (fp->buf - fp->ptr)))
    {
      fp->ptr += pos;
      fp->cnt -= pos;
      return 0;
    }
  fp->ptr = fp->buf;
  fp->cnt = 0;
  return nfs_lseek (fp->file, offset, origin);
}


/******************************
/*	nfs_fclose
/******************************/

void
nfs_fclose (file)
     NFS_FILE *file;
{
  if (file != NULL) {
    nfs_close (file->file);
    free (file);
  }
}


/******************************
/*	nfs_open
/******************************/

NFS_FD
nfs_open (filename,  mode)
     char *filename;
     int mode;
{
  Part part[3];
  NFS_FD fd;

  fd = (NFS_FD) malloc(sizeof(NFS_FD_struct));
  if(fd == NULL) {
     return NULL;
  }

  fd->proto = xcreateprotl (nfs_demux, null, null);
  if(fd->proto == ERR_PROTL) {
     free(fd);
     return NULL;
  }

  part[0].address = (char *) mode;
  part[0].length = sizeof (int);
  part[1].address = filename;
  part[1].length = strlen (filename) + 1;
  part[2].address = (char *) NULL;
  part[2].length = 0;

  fd->file = (Sessn) xopen (fd->proto, NFS, part);
  if(fd->file == ERR_SESSN) {
     free(fd);
     return NULL;
  }
 
  return fd;
}

/******************************
/*	nfs_filbuf
/******************************/

int
nfs_filbuf (file)
     NFS_FILE *file;
{
  int len;

  len = nfs_read_data (file->file, file->buf, BUFFER_SIZE);

  if (len <= 0)
    {
      printf ("EOF encountered by nfs_filbuf");
      return EOF;
    }
  file->cnt = len;
  file->ptr = file->buf;
  return nfs_getch (file);
}

/******************************
/*	nfs_getch
/******************************/

int
nfs_getch (file)
     NFS_FILE *file;
{
  int i;

  if (--file->cnt >= 0)
    {
      i = *file->ptr++;
      return i;
    }
  else
    {
      return nfs_filbuf (file);
    }
}

int
nfs_ungetch (ch, file)
     NFS_FILE *file;
     int ch;
{
  ++(file->cnt);
  return *--(file->ptr) = ch;
}

/******************************
/*	nfs_fgets
/******************************/

char *
nfs_fgets(data, limit, fp)
char *data;
int limit;
NFS_FILE *fp;
{
   char *p = data;
   int c;

   while((--limit > 0) && ((c = nfs_getch(fp)) != EOF))
      if((*p++ = c) == '\n')
	  break;
   *p = '\0';
   return((c == EOF && p == data) ? NULL : data);
}

static char _numstr[] = "0123456789ABCDEF";

#define skip()  while(isspace(c)) { if ((c=nfs_getch(file))<1) goto done; }
#define TEN_MUL(X)      ((((X) << 2) + (X)) << 1)
#define mklower(c) (isupper(c) ? tolower(c) : (c))
#define mkupper(c) (islower(c) ? toupper(c) : (c))

/******************************
/*	nfs_fscanf
/******************************/

static int
nfs_fscanf (file, fmt, args)
     NFS_FILE *file;
     char *fmt;
     char *args;
{
  long n;
  int c, width, lval, cnt = 0;
  int store, neg, base, wide1, endnull, rngflag, c2;
  unsigned char *p;
  unsigned char delim[128], digits[17], *q;
  char *strchr (), *strcpy ();

  if (!*fmt)
    return (0);

  printf ("fmt = %s", fmt);
  c = nfs_getch (file);
  while (c > 0)
    {
      printf ("looping");
      store = FALSE;
      if (*fmt == '%')
	{
	  n = 0;
	  width = -1;
	  wide1 = 1;
	  base = 10;
	  lval = FALSE;
	  store = TRUE;
	  endnull = TRUE;
	  neg = -1;

	  printf ("decoding it");
	  strcpy (delim, "\011\012\013\014\015 ");
	  strcpy (digits, _numstr);	/* "01234567890ABCDEF" */

	  if (fmt[1] == '*')
	    {
	      endnull = store = FALSE;
	      ++fmt;
	    }

	  while (isdigit (*++fmt))	/* width digit(s) */
	    {
	      if (width == -1)
		width = 0;
	      wide1 = width = TEN_MUL (width) + (*fmt - '0');
	    }
	  --fmt;
	fmtnxt:
	  ++fmt;
	  printf ("switching on %d", *fmt);
	  switch (mklower (*fmt))	/* tolower() is a MACRO! */
	    {
	    case '*':
	      endnull = store = FALSE;
	      goto fmtnxt;
	    case 'l':		/* long data */
	      lval = TRUE;
	    case 'h':		/* short data (for compatibility) */
	      goto fmtnxt;
	    case 'i':		/* any-base numeric */
	      base = 0;
	      goto numfmt;
	    case 'b':		/* unsigned binary */
	      base = 2;
	      goto numfmt;
	    case 'o':		/* unsigned octal */
	      base = 8;
	      goto numfmt;
	    case 'x':		/* unsigned hexadecimal */
	      base = 16;
	      goto numfmt;

	    case 'd':		/* SIGNED decimal */
	      printf ("SIGNED decimal");
	      neg = FALSE;
	      /* FALL-THRU */

	    case 'u':		/* unsigned decimal */
	    numfmt:skip ();

	      if (isupper (*fmt))
		lval = TRUE;

	      if (!base)
		{
		  printf ("!base is TRUE");
		  base = 10;
		  neg = FALSE;
		  if (c == '%')
		    {
		      base = 2;
		      goto skip1;
		    }
		  else if (c == '0')
		    {
		      c = nfs_getch (file);
		      if (c < 1)
			goto savnum;
		      if ((c != 'x')
			  && (c != 'X'))
			{
			  base = 8;
			  digits[8] = '\0';
			  goto zeroin;
			}
		      base = 16;
		      goto skip1;
		    }
		}

	      if ((neg == FALSE) && (base == 10)
		  && ((neg = (c == '-')) || (c == '+')))
		{
		skip1:
		  c = nfs_getch (file);
		  if (c < 1)
		    goto done;
		}

	      digits[base] = '\0';
	      p = ((unsigned char *)
		   strchr (digits, mkupper (c)));

	      if ((!c || !p) && width)
		goto done;

	      while (p && width-- && c)
		{
		  n = (n * base) + (p - digits);
		  c = nfs_getch (file);
		zeroin:
		  p = ((unsigned char *)
		       strchr (digits, mkupper (c)));
		}
	    savnum:
	      if (store)
		{
		  p = ((unsigned char *) *args);
		  if (neg == TRUE)
		    n = -n;
		  if (lval)
		    *((long *) p) = n;
		  else
		    *((int *) p) = n;
		  ++cnt;
		}
	      break;

	    case 'c':		/* character data */
	      width = wide1;
	      endnull = FALSE;
	      delim[0] = '\0';
	      goto strproc;

	    case '[':		/* string w/ delimiter set */

	      /* get delimiters */
	      p = delim;

	      if (*++fmt == '^')
		fmt++;
	      else
		lval = TRUE;

	      rngflag = 2;
	      if ((*fmt == ']') || (*fmt == '-'))
		{
		  *p++ = *fmt++;
		  rngflag = FALSE;
		}

	      while (*fmt != ']')
		{
		  if (*fmt == '\0')
		    goto done;
		  switch (rngflag)
		    {
		    case TRUE:
		      c2 = *(p - 2);
		      if (c2 <= *fmt)
			{
			  p -= 2;
			  while (c2 < *fmt)
			    *p++ = c2++;
			  rngflag = 2;
			  break;
			}
		      /* fall thru intentional */

		    case FALSE:
		      rngflag = (*fmt == '-');
		      break;

		    case 2:
		      rngflag = FALSE;
		    }

		  *p++ = *fmt++;
		}

	      *p = '\0';
	      goto strproc;

	    case 's':		/* string data */
	      skip ();
	    strproc:
	      /* process string */
	      p = ((unsigned char *) *args);

	      /* if the 1st char fails, match fails */
	      if (width)
		{
		  q = ((unsigned char *)
		       strchr (delim, c));
		  if ((c < 1)
		      || (lval ? !q : (int) q))
		    {
		      if (endnull)
			*p = '\0';
		      goto done;
		    }
		}

	      for (;;)		/* FOREVER */
		{
		  if (store)
		    *p++ = c;
		  if (((c = nfs_getch (file)) < 1) ||
		      (--width == 0))
		    break;

		  q = ((unsigned char *)
		       strchr (delim, c));
		  if (lval ? !q : (int) q)
		    break;
		}

	      if (store)
		{
		  if (endnull)
		    *p = '\0';
		  ++cnt;
		}
	      break;

	    case '\0':		/* early EOS */
	      --fmt;
	      /* FALL THRU */

	    default:
	      printf ("Default HIT!");
	      goto cmatch;
	    }
	}
      else if (isspace (*fmt))	/* skip whitespace */
	{
	  skip ();
	}
      else
	{			/* normal match char */
	cmatch:
	  if (c != *fmt)
	    break;
	  c = nfs_getch (file);
	}

      if (store)
	args++;

      if (!*++fmt)
	break;
    }

done:				/* end of scan */
  printf ("we're done");
  if ((c < 0) && (cnt == 0))
    {
      printf ("scanf EOF'ed");
      return (EOF);
    }
  nfs_ungetch (c, file);
  return (cnt);
}


/******************************
/*	nfs_tell
/******************************/

int
nfs_tell (fd)
     NFS_FD fd;
{
  return xcontrolsessn (fd->file, NFS_TELL, NULL, 0);
}

/******************************
/*	nfs_lseek
/******************************/

off_t
nfs_lseek (fd, offset, whence)
     NFS_FD fd;
     off_t offset;
     int whence;
{
  return xcontrolsessn (fd->file, NFS_SEEK, &offset, whence);
}


/******************************
/*	nfs_close
/******************************/

nfs_close (fd)
     NFS_FD fd;
{
  xclose (fd->file);
  xdestroyprotl(fd->proto);
  free(fd);
  return 0;
}


static void
nfs_demux (s, buf, len)
     Sessn s;
     char *buf;
     int len;
{
  buf[len] = 0;
}



