/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */


#include "../exim.h"
#include "appendfile.h"



/* Options specific to the appendfile transport. They must be in alphabetic
order (note that "_" comes before the lower case letters). */

optionlist appendfile_transport_options[] = {
  { "*expand_group",     opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, expand_gid)) },
  { "*expand_user",      opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, expand_uid)) },
  { "bsmtp",             opt_local_smtp,
      (void *)(offsetof(appendfile_transport_options_block, local_smtp)) },
  { "check_group",       opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, check_group)) },
  { "create_directory",  opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, create_directory)) },
  { "create_file",       opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, create_file_string)) },
  { "delivery_date_add", opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, delivery_date_add)) },
  { "directory",         opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, dirname)) },
  { "directory_mode",    opt_octint,
      (void *)(offsetof(appendfile_transport_options_block, dirmode)) },
  { "file",              opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, filename)) },
  { "file_must_exist",   opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, file_must_exist)) },
  { "from_hack",         opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, from_hack)) },
  { "group",             opt_expand_gid,
      (void *)(offsetof(appendfile_transport_options_block, gid)) },
  { "lock_interval",     opt_time,
      (void *)(offsetof(appendfile_transport_options_block, lock_interval)) },
  { "lock_retries",      opt_int,
      (void *)(offsetof(appendfile_transport_options_block, lock_retries)) },
  { "lockfile_mode",     opt_octint,
      (void *)(offsetof(appendfile_transport_options_block, lockfile_mode)) },
  { "lockfile_timeout",  opt_time,
      (void *)(offsetof(appendfile_transport_options_block, lockfile_timeout)) },
  { "mode",              opt_octint,
      (void *)(offsetof(appendfile_transport_options_block, mode)) },
  { "notify_comsat",     opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, notify_comsat)) },
  { "prefix",            opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, prefix)) },
  { "require_lockfile",  opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, require_lockfile)) },
  { "return_path_add",   opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, return_path_add)) },
  { "suffix",            opt_stringptr,
      (void *)(offsetof(appendfile_transport_options_block, suffix)) },
  { "use_lockfile",      opt_bool,
      (void *)(offsetof(appendfile_transport_options_block, use_lockfile)) },
  { "user",              opt_expand_uid,
      (void *)(offsetof(appendfile_transport_options_block, uid)) },
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int appendfile_transport_options_count =
  sizeof(appendfile_transport_options)/sizeof(optionlist);

/* Default private options block for the appendfile transport. */

appendfile_transport_options_block appendfile_transport_option_defaults = {
  NULL,           /* file name */
  NULL,           /* dir name */
  "From ${return_path} ${tod_bsdinbox}\n", /* prefix */
  "\n",           /* suffix */
  NULL,           /* expand_uid */
  NULL,           /* expand_gid */
  "anywhere",     /* create_file_string (string value for create_file) */
  local_smtp_off, /* local_smtp */
  -1,             /* uid */
  -1,             /* gid */
  0600,           /* mode */
  0700,           /* dirmode */
  0600,           /* lockfile_mode */
  30*60,          /* lockfile_timeout */
  10,             /* lock_retries */
   3,             /* lock_interval */
  create_anywhere,/* create_file */
  FALSE,          /* check_group */
  TRUE,           /* create_directory */
  FALSE,          /* notify_comsat */
  TRUE,           /* require_lockfile */
  TRUE,           /* use_lockfile */
  TRUE,           /* from_hack */
  TRUE,           /* return_path_add */
  TRUE,           /* delivery_date_add */
  FALSE           /* file_must_exist */
};



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up. */

void appendfile_transport_init(transport_instance *tblock)
{
appendfile_transport_options_block *ob =
  (appendfile_transport_options_block *)(tblock->options_block);

/* Only one of a file name or directory name must be given. */

if (ob->filename != NULL && ob->dirname != NULL)
  log_write(LOG_PANIC_DIE, "Exim configuration error for %s transport:\n  "
  "only one of \"file\" or \"directory\" can be specified", tblock->name);

/* If a file name was specified, it must be an absolute path. */

if (ob->filename != NULL && ob->filename[0] != '/')
  log_write(LOG_PANIC_DIE, "Exim configuration error for %s transport:\n  "
  "the file option must specify an absolute path", tblock->name);

/* If a directory name was specified, it must be an absolute path.*/

if (ob->dirname != NULL && ob->dirname[0] != '/')
  log_write(LOG_PANIC_DIE, "Exim configuration error for %s transport:\n  "
  "the directory option must specify an absolute path", tblock->name);

/* If a fixed uid field is set, then a gid field must also be set. */

if (ob->uid >= 0 && ob->gid < 0)
  log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
    "user set without group for the %s transport", tblock->name);
    
/* If "create_file" is set, check that a valid option is given, and set the
integer variable. */

if (ob->create_file_string != NULL)
  {
  int value; 
  if (strcmp(ob->create_file_string, "anywhere") == 0) value = create_anywhere;
  else if (strcmp(ob->create_file_string, "belowhome") == 0) value = 
    create_belowhome;
  else if (strcmp(ob->create_file_string, "inhome") == 0) 
    value = create_inhome;
  else
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
      "invalid value given for \"file_create\" for the %s transport: %s",
        tblock->name, ob->create_file_string);       
  ob->create_file = value;       
  }  

/* Copy the uid, gid, and local_smtp options into the generic slots */

tblock->uid = ob->uid;
tblock->gid = ob->gid;
tblock->expand_uid = ob->expand_uid;
tblock->expand_gid = ob->expand_gid;
tblock->local_smtp = ob->local_smtp;
}



/*************************************************
*                  Notify comsat                 *
*************************************************/

/* The comsat daemon is the thing that provides asynchronous notification of
the arrival of local messages, if requested by the user by "biff y". It is a
BSD thing that uses a TCP/IP protocol for communication. A message consisting
of the text "user@offset" must be sent, where offset is the place in the
mailbox where new mail starts. There is no scope for telling it which file to
look at, which makes it a less than useful if mail is being delivered into a
non-standard place such as the user's home directory. */

static void notify_comsat(char *user, int offset)
{
int fd;
struct sockaddr_in sa;
struct hostent *hp;
struct servent *sp;
char *msg;
char buffer[256];

DEBUG(2) debug_printf("notify_comsat called\n");

if ((hp = gethostbyname("localhost")) == NULL)
  {
  DEBUG(2) debug_printf("\"localhost\" unknown\n");
  return;
  }

if (hp->h_addrtype != AF_INET)
  {
  DEBUG(2) debug_printf("local host not TCP/IP\n");
  return;
  }

if ((sp = getservbyname("biff", "udp")) == NULL)
  {
  DEBUG(2) debug_printf("biff/udp is an unknown service");
  return;
  }

sa.sin_port = sp->s_port;
sa.sin_family = hp->h_addrtype;
memcpy(&sa.sin_addr, hp->h_addr, hp->h_length);

if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
  {
  DEBUG(2) debug_printf("failed to create comsat socket: %s\n",
    strerror(errno));
  return;
  }

sprintf(buffer, "%s@%d\n", user, offset);
if (sendto(fd, buffer, (int)strlen(buffer) + 1, 0, (struct sockaddr *)(&sa),
    sizeof(sa)) < 0)
  {
  DEBUG(2) debug_printf("send to comsat failed: %s\n", strerror(errno));
  }

close(fd);
}



/*************************************************
*       Write block and check for errors         *
*************************************************/

/* On failing to write the given number of bytes, but having written
some, try to write the rest in order to force an over-quota error to
appear. Otherwise no error is set. */

static BOOL write_block(int fd, char *buffer, int count)
{
int rc;
if ((rc = write(fd, buffer, count)) != count &&
    (rc < 0 || write(fd, buffer+rc, count-rc) != count-rc)) return FALSE;
return TRUE;
}



/*************************************************
*             Write formatted string             *
*************************************************/

static BOOL write_string(int fd, char *format, ...)
{
va_list ap;
va_start(ap, format);
vsprintf(big_buffer, format, ap);
va_end(ap);
return write_block(fd, big_buffer, strlen(big_buffer));
}



/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for interface details.

Appendfile delivery is tricky and has led to various security problems in other
mailers. The logic used here is therefore laid out in some detail. When this
function is called, we are running in a subprocess which has had its gid and
uid set to the appropriate values. Therefore, we cannot write directly to the
smail logs. Any errors must be handled by setting appropriate return codes.
Note that the default setting for addr->transport_return is DEFER, so it need
not be set unless some other value is required.

(1)  If the local_part field starts with '/' then this is a delivery to a file
     after .forward expansion. Otherwise, there must be a configured file name
     or directory name.

The following items apply in the case when a file name (as opposed to a
directory name) is given:

(2f) If the file name to be expanded contains a reference to $local_part,
     check that the user name doesn't contain a / because that would cause
     a reference to another directory.

(3f) Expand the file name.

(4f) If the file name is /dev/null, return success (optimization).

(5f) Stat the file.

(6f) If the file already exists:

     If it's not a regular file, complain to local administrator and defer
     delivery with a local error code that causes subsequent delivery to
     be prevented until manually enabled.

     Check owner, group and permissions. If the owner is OK and the required
     permissions are *less* than the existing ones, chmod the file. Otherwise,
     complain and defer.

     Save the inode number.

     Open with O_WRONLY + O_APPEND, thus failing if the file has vanished.

     If open fails because the file does not exist, go to (7); on any other
     failure except EWOULDBLOCK, complain & defer. For EWOULDBLOCK (NFS
     failure), just defer.

     Check the inode number hasn't changed - I realize this isn't perfect (an
     inode can be reused) but it's cheap and will catch some of the races.

     Check it's still a regular file.

     Check that the owner and permissions haven't changed.

(7f) If file does not exist initially:

     Open with O_WRONLY + O_EXCL + O_CREAT with configured mode.

     If open fails because the file already exists, go to (6). To avoid looping
     for ever in a situation where the file is continuously being created and
     deleted, go round the (6)->(7)->(6) loop no more than 10 times. If this
     count expires, complain and defer with a code that freezes delivery
     attempts.

     If open fails for any other reason, defer for subsequent delivery.

(8f) We now have the file checked and open for writing. Lock it - if locking
     fails, close and defer temporarily.

(9f) Save access time (for subsequent restoration), size of file (for comsat and
     for re-setting if delivery fails in the middle - e.g. for quota exceeded).

The following items apply in the case when a directory name is given:

(2d) Create a new file in the directory using a temporary name, by opening for
     writing and with CREAT.

(3d) Generate a unique name for the file from the time and the inode number,
     and rename file file.

The following items apply in all cases:

(10) We now have the file open for writing, and locked if it was given as a
     file name. Write the message and flush the file.

     If there is a quota error on writing, defer the address. Timeout logic
     will determine for how long retries are attempted. We restore the mailbox
     to its original length. There doesn't seem to be a uniform error code
     for quota excession (it even differs between SunOS4 and SunOS5) so a
     system-dependent macro called ERRNO_QUOTA is used for it, and the value
     gets put into errno_quota.

     For any other error (most commonly disc full), do the same.

(11) Restore the atime; notify_comsat if required; close the file (which
     unlocks it if it was locked).

This transport never yields FAIL. */


void appendfile_transport_entry(
  transport_instance *tblock,      /* data for this instantiation */
  address_item *addr)              /* address we are working on */
{
appendfile_transport_options_block *ob =
  (appendfile_transport_options_block *)(tblock->options_block);
struct stat statbuf;
char *filename;
char *lockname;
char *hitchname;
char tempname[256];
time_t now = time(NULL);
struct utimbuf times;
BOOL smtp_dots = FALSE;
BOOL return_path_add = ob->return_path_add;
BOOL delivery_date_add = ob->delivery_date_add;
BOOL isdirectory = FALSE;
int uid = getuid();
int gid = getgid();
int saved_size;
int hd = -1;
int i, len, fd, yield;

/* See if this is the address_file transport, used to deliver to files
specified via .forward. */

if (addr->local_part[0] == '/') filename = addr->local_part;

/* Handle a non-address file delivery. One of the file or directory options is
mandatory; it will have already been checked to be an absolute path. */

else
  {
  char *fdname = ob->filename;
  if (fdname == NULL)
    {
    fdname = ob->dirname;
    isdirectory = TRUE;
    }

  if (fdname == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Mandatory file or directory option "
      "missing from %s transport", tblock->name);
    return;
    }

  filename = expand_string(fdname);

  if (filename == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (file or directory "
      "name for %s transport) failed", fdname, tblock->name);
    return;
    }

  /* If the file name contained a reference to $local_part, check that it
  doesn't attempt to change directory. */

  if (strchr(deliver_localpart, '/') != NULL &&
     (strstr(fdname, "${local_part}") != NULL ||
      strstr(fdname, "$local_part") != NULL))
    {
    addr->message = "appendfile: user name contains /";
    addr->errno = ERRNO_USERSLASH;
    addr->special_action = SPECIAL_FREEZE;
    return;
    }
  }

DEBUG(9) debug_printf("appendfile: mode=%o notify_comsat=%d "
  "%s=%s prefix=%s suffix=%s\n",
  ob->mode, ob->notify_comsat,
  isdirectory? "directory" : "file",
  filename,
  (ob->prefix == NULL)? "null":ob->prefix,
  (ob->suffix==NULL)? "null":ob->suffix);
  

/* If the -N option is set, can't do any more. */

if (dont_deliver)
  {
  debug_printf("*** delivery by %s transport bypassed by -N option\n",
    tblock->name);
  addr->transport_return = OK;
  return;
  }       

/* Handle the case of a file name. If the file name is /dev/null, we can save
ourselves some effort and just give a success return right away. */

if (!isdirectory)
  {
  char *lock_fail_type = "";
    
  if (strcmp(filename, "/dev/null") == 0)
    {
    addr->transport_return = OK;
    return;
    }

  /* Stat the file, and act on existence or non-existence. This is in a loop
  to handle the case of a file's being created or deleted as we watch. */

  for (i = 0; i < 10; i++)
    {
    if (lstat(filename, &statbuf) != 0)
      {
      /* Let's hope that failure to stat (other than non-existence) is a
      rare event. */

      if (errno != ENOENT)
        {
        addr->errno = errno;
        addr->message = string_sprintf("attempting to stat mailbox %s",
          filename);
        addr->special_action = SPECIAL_FREEZE;
        return;
        }

      /* File does not exist. If it is required to pre-exist this state is an
      error. */

      if (ob->file_must_exist)
        {
        addr->errno = errno;
        addr->message = string_sprintf("mailbox %s does not exist, "
          "but file_must_exist is set", filename);
        addr->special_action = SPECIAL_FREEZE;
        return;
        }

      /* If file creation is permitted in certain directories only, check that
      this is actually the case. Current checks are for in or below the
      home directory. */

      if (ob->create_file != create_anywhere)
        {
        BOOL OK = FALSE;
        if (deliver_home != NULL)
          {
          int len = (int)strlen(deliver_home);
	  char *file = filename;
          while (file[0] == '/' && file[1] == '/') file++;
          if (strncmp(file, deliver_home, len) == 0 && file[len] == '/' &&
            (ob->create_file == create_belowhome ||
              strchr(file+len+2, '/') == NULL)) OK = TRUE;
          }

        if (!OK)
          {
          addr->errno = errno;
          addr->message = string_sprintf("mailbox %s does not exist, "
            "but creation outside the home directory is not permitted",
            filename);
          addr->special_action = SPECIAL_FREEZE;
          return;
          }
        }

      /* Attempt to create and open the file. If open fails because of
      pre-existence, go round the loop again. For any other error,
      defer the address. */

      fd = open(filename, O_WRONLY | O_APPEND | O_EXCL | O_CREAT, ob->mode);
      if (fd < 0)
        {
        if (errno == EEXIST) continue;
        addr->errno = errno;
        addr->message = string_sprintf("while creating mailbox %s",
          filename);
        return;
        }

      /* We have successfully created and opened the file. Ensure that the group
      and the mode are correct, and then get out of the loop. */

      chown(filename, uid, gid);
      chmod(filename, ob->mode);
      break;
      }

    /* The file already exists. Test its type, ownership, and permissions, and
    save the inode for checking later. */

    else
      {
      int mode = statbuf.st_mode;
      int inode = statbuf.st_ino;

      if ((mode & S_IFMT) != S_IFREG)
        {
        addr->errno = ERRNO_NOTREGULAR;
        addr->message = string_sprintf("mailbox %s is not a regular file",
          filename);
        addr->special_action = SPECIAL_FREEZE;
        return;
        }

      /* Group is checked only if check_group is set. */

      if (statbuf.st_uid != uid || (ob->check_group && statbuf.st_gid != gid))
        {
        addr->errno = ERRNO_BADUGID;
        addr->message = string_sprintf("mailbox %s has wrong uid or gid",
          filename);
        addr->special_action = SPECIAL_FREEZE;
        return;
        }

      /* If the mode is not what it would be for a newly created file, reduce the
      permissions if possible. If the newly created permissions are greater than
      the existing permissions, don't change things. */

      if ((mode = (mode & 07777)) != ob->mode)
        {
        int diffs = mode ^ ob->mode;
        if ((diffs & mode) == diffs)
          {
          DEBUG(2) debug_printf("chmod %o %s\n", ob->mode, filename);
          if (chmod(filename, ob->mode) < 0)
            {
            addr->errno = errno;
            addr->message = string_sprintf("attempting to chmod mailbox %s",
              filename);
            addr->special_action = SPECIAL_FREEZE;
            return;
            }
          }
        else
          {
          addr->errno = ERRNO_BADMODE;
          addr->message = string_sprintf("mailbox %s has wrong mode", filename);
          addr->special_action = SPECIAL_FREEZE;
          return;
          }
        }

      /* We are happy with the existing file. Open it, and then do further
      tests to ensure that it is the same file that we were just looking at.
      If the file does not now exist, restart this loop. For an NFS error,
      just defer; other opening errors are more serious. */

      fd = open(filename, O_WRONLY | O_APPEND, ob->mode);
      if (fd < 0)
        {
        if (errno == ENOENT) continue;
        if (errno == EWOULDBLOCK)
          {
          addr->errno = errno;
          return;
          }
        addr->errno = errno;
        addr->message = string_sprintf("while opening mailbox %s", filename);
        addr->special_action = SPECIAL_FREEZE;
        return;
        }

      /* This fstat really shouldn't fail, as we have an open file! There's a
      dilemma here. We use fstat in order to be sure we are peering at the file
      we have got open. However, that won't tell us if the file was reached
      via a symbolic link. We checked this above, but there is a race exposure
      if the link was created between the previous lstat and the open. However,
      it would have to be created with the same inode in order to pass the
      check below. */

      if (fstat(fd, &statbuf) < 0)
        {
        addr->errno = errno;
        addr->message = string_sprintf("attempting to stat open mailbox %s",
          filename);
        addr->special_action = SPECIAL_FREEZE;
        close(fd);
        return;
        }

      /* Check the inode; this is isn't a perfect check, but gives some
      confidence. */

      if (inode != statbuf.st_ino)
        {
        addr->errno = ERRNO_INODECHANGED;
        addr->message = string_sprintf("opened mailbox %s inode number changed "
          "from %d to %d", filename, inode, statbuf.st_ino);
        addr->special_action = SPECIAL_FREEZE;
        close(fd);
        return;
        }

      /* Check it's still a regular file and the uid, gid, and permissions have
      not changed. */

      if ((statbuf.st_mode & S_IFMT) != S_IFREG)
        {
        addr->errno = ERRNO_NOTREGULAR;
        addr->message =
          string_sprintf("opened mailbox %s is not a regular file", filename);
        addr->special_action = SPECIAL_FREEZE;
        close(fd);
        return;
        }

      if (statbuf.st_uid != uid || (ob->check_group && statbuf.st_gid != gid))
        {
        addr->errno = ERRNO_BADUGID;
        addr->message = string_sprintf("opened mailbox %s has wrong uid or gid",
          filename);
        addr->special_action = SPECIAL_FREEZE;
        close(fd);
        return;
        }

      if ((statbuf.st_mode & 07777) != ob->mode)
        {
        addr->errno = ERRNO_BADMODE;
        addr->message = string_sprintf("opened mailbox %s has wrong mode %o "
          "(%o expected", filename, statbuf.st_mode & 07777, ob->mode);
        addr->special_action = SPECIAL_FREEZE;
        close(fd);
        return;
        }

      /* The file seems to be OK. Break the loop. */

      break;
      }
    }


  /* If i >= 10, we have gone round the exist/non-exist loop 10 times. Something
  is terribly wrong... */

  if (i >= 10)
    {
    addr->errno = ERRNO_EXISTRACE;
    addr->message = string_sprintf("mailbox %s: existence unclear", filename);
    addr->special_action = SPECIAL_FREEZE;
    return;
    }


  /* We now have an open file, and must lock it. The locking of mailbox files
  is worse than the naming of cats ("a difficult matter" - T.S. Eliot).
  Research in other programs that lock mailboxes shows that there is no
  universally standard method. Having mailboxes NFS-mounted on the system that
  is delivering mail is clearly a bad thing, but people do run like this,
  and so the code must do its best to cope.

  Two different locking approaches are taken. Unless no_use_lockfile is set,
  we attempt to build a lock file in a way that will work over NFS, and we
  also use fcntl. Failures to lock cause retries after a sleep, but only for
  a certain number of tries.

  The logic for creating the lock file is:

  . The name of the lock file is <mailbox-name>.lock

  . First, create a "hitching post" name by adding the primary host name,
    current time and pid to the lock file name. This should be unique.

  . Create the hitching post file using WRONLY + CREAT + EXCL.

  . If that fails EACCES, we assume it means that the user is unable to create
    files in the mail spool directory. Some installations might operate in this
    manner, so there is a configuration option to allow this state not to be an
    error - we proceed to lock using fcntl only.

  . Otherwise, an error causes a deferment of the address.

  . Close the hitching post file.

  . Hard link it to the lock file name, ignoring failure.

  . Stat the hitching post file.

  . Unlink it.

  . Now examine the stat data. The number of links to the file must be exactly
    2 for the locking to have succeeded. This method allows for the lock file to
    be created by some other process right up to the moment of the attempt to
    hard link it.

  . System crashes may cause lock files to get left lying around, and some means
    of flushing them is required. The approach of writing a pid (used by smail
    and by elm) into the file isn't useful when NFS may be in use. Pine uses a
    timeout, which seems a better approach. Since any program that writes to a
    mailbox using a lock file should complete its task very quickly, Pine removes
    lock files that are older than 5 minutes. We allow the value to be
    configurable on the director. */

  /* Build the lock file name once and for all if it's needed */

  if (ob->use_lockfile)
    {
    len = (int)strlen(filename);
    lockname = store_malloc(len + 8);
    sprintf(lockname, "%s.lock", filename);
    hitchname = store_malloc(len + 32 + (int)strlen(primary_hostname));
    sprintf(hitchname, "%s.%s.%08x.%08x", lockname, primary_hostname,
      now, getpid());

    DEBUG(9) debug_printf("lock name: %s\nhitch name: %s\n", lockname,
      hitchname);
    }

  /* Locking retry loop */

  for (i = 0; i < ob->lock_retries; sleep(ob->lock_interval), i++)
    {
    flock_t lock_data;

    /* Try to build a lock file if so configured */

    if (ob->use_lockfile)
      {
      lock_fail_type = "lock file"; 
      hd = open(hitchname, O_WRONLY | O_CREAT | O_EXCL, ob->lockfile_mode);
      if (hd < 0 && (errno != EACCES || ob->require_lockfile))
        {
        addr->errno = errno;
        addr->message = "creating lock file hitching post";
        return;
        }

      /* If a hitching post exists, attempt to hitch it to the lock file; get
      its subsequent state, and then get rid of its name. If the hitch was
      not successful, try again, having unlinked the lock file if it is too
      old. */

      if (hd > 0)
        {
        link(hitchname, lockname);
        fstat(hd, &statbuf);
        close(hd);
        unlink(hitchname);
        if (statbuf.st_nlink != 2)
          {
          if (ob->lockfile_timeout > 0 && stat(lockname, &statbuf) == 0 &&
              now - statbuf.st_ctime > ob->lockfile_timeout)
            {
            DEBUG(2) debug_printf("unlinking timed-out lock file\n");
            unlink(lockname);
            }
          continue;
          }
        }
      }

    /* Either a lock file has been successfully created, or there was an EACCES
    error but a lock file is not required. Now try to lock using fcntl. If we
    fail, get rid of any lock file we created. */

    lock_fail_type = "fcntl";
    lock_data.l_type = F_WRLCK;
    lock_data.l_whence = lock_data.l_start = lock_data.l_len = 0;
    if (fcntl(fd, F_SETLK, &lock_data) >= 0) break;
    if (hd > 0) unlink(lockname);
    }

  /* Check for too many tries at locking */

  if (i >= ob->lock_retries)
    {
    addr->errno = ERRNO_LOCKFAILED;
    addr->message = string_sprintf("failed to lock mailbox %s (%s)", filename,
      lock_fail_type);
    close(fd);
    return;
    }

  DEBUG(9) debug_printf("mailbox %s is locked\n", filename);

  /* Save access time (for subsequent restoration), modification time (for
  restoration if updating fails), size of file (for comsat and for re-setting if
  delivery fails in the middle - e.g. for quota exceeded). */

  if (fstat(fd, &statbuf) < 0)
    {
    addr->errno = errno;
    addr->message = string_sprintf("while fstatting opened mailbox %s",
      filename);
    close(fd);
    return;
    }

  times.actime = statbuf.st_atime;
  times.modtime = statbuf.st_mtime;
  saved_size = statbuf.st_size;
  }

/* Handle the case of creating a unique file in a given directory. A temporary
name is used to create the file. Later, when it is written, the name is changed
to a unique one. There is no need to lock the file. An attempt is made to
create the directory if it does not exist. */

else
  {
  sprintf(tempname, "%s/temp.%d", filename, getpid());
  fd = open(tempname, O_WRONLY|O_CREAT, ob->mode);
  if (fd < 0 &&                                    /* failed to open, and */
      (errno != ENOENT ||                          /* either not not exists */
       !ob->create_directory ||                    /* or not allowed to make */
       mkdir(filename, ob->dirmode) < 0 ||         /* or failed to create dir */
       (fd = open(tempname, O_WRONLY|O_CREAT, ob->mode)) < 0)) /* then open */
    {
    addr->errno = errno;
    addr->message = string_sprintf("while creating file %s", tempname);
    return;
    }
  chown(tempname, uid, gid);
  chmod(tempname, ob->mode);
  }


/* At last we can write the message to the file, preceded by any configured
prefix line, and followed by any configured suffix line. If there are any
writing errors, we must defer. */

yield = OK;
errno = 0;

/* If the local_smtp option is not unset, we need to write SMTP prefix
information. The various different values for batching are handled outside; if
there is more than one address available here, all must be included. Force
SMTP dot-handling. */

if (ob->local_smtp != local_smtp_off)
  {
  smtp_dots = TRUE;
  return_path_add = delivery_date_add = FALSE;
  if (!write_string(fd, "MAIL FROM: <%s>\n",
    (addr->errors_address != NULL)? addr->errors_address : sender_address))
      yield = DEFER;
  else
    {
    address_item *a;
    for (a = addr; a != NULL; a = a->next)
      {
      if ((a->local_part[0] == ',' || a->local_part[0] == ':')?
        !write_string(fd,
          "RCPT TO: <@%s%s>\n", a->domain, a->local_part) :
        !write_string(fd,
          "RCPT TO: <%s@%s>\n", a->local_part, a->domain))
        { yield = DEFER; break; }
      }
    if (yield == OK && !write_string(fd, "DATA\n")) yield = DEFER;
    }
  }

/* Now any other configured prefix. */

if (yield == OK && ob->prefix != NULL)
  {
  char *prefix = expand_string(ob->prefix);
  if (prefix == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (prefix for %s "
      "transport) failed", ob->prefix, tblock->name);
    return;
    }
  if (!write_string(fd, "%s", prefix)) yield = DEFER;
  }

/* The options to write_message request a return-path header if configured, the
unix "from" hack if configured, no CRs added before LFs, and no SMTP dot
handling except when local_smtp is set. Pass the errors_address from the
address if present. This is used to create return-path if requested (forced
off for local_smtp). */

if (yield == OK &&
  !transport_write_message(fd, return_path_add, delivery_date_add,
    ob->from_hack, FALSE, smtp_dots, addr->errors_address, 0)) yield = DEFER;

/* Now a configured suffix. */

if (yield == OK && ob->suffix != NULL)
  {
  char *suffix = expand_string(ob->suffix);
  if (suffix == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (suffix for %s "
      "transport) failed", ob->suffix, tblock->name);
    return;
    }
  if (!write_string(fd, "%s", suffix)) yield = DEFER;
  }

/* If local_smtp, write the terminating dot. */

if (yield == OK && ob->local_smtp != local_smtp_off &&
  !write_block(fd, ".\n", 2)) yield = DEFER;

/* Force out the remaining data to check for any errors */

if (yield == OK && fsync(fd) < 0) yield = DEFER;

/* Handle error while writing the file. Control should come here directly after
the error, with the reason in errno. */

if (yield != OK)
  {
  /* Save the error number. It will ultimately cause a strerror() call to
  generate some text. */

  addr->errno = errno;

  /* Handle quota excession. Set more_errno to the time since the mailbox
  was last read, and add an explanatory phrase for the error message, since
  some systems don't have special quota-excession errors. */

  if (errno == errno_quota)
    {
    addr->more_errno = time(NULL) - times.actime;
    #ifndef EDQUOT
    addr->message = string_sprintf("probably user's quota exceeded while "
      "writing to mailbox %s", filename);
    #endif
    DEBUG(9) debug_printf("Quota exceeded for %s: time since mailbox "
      "read = %s\n", filename, readconf_printtime(addr->more_errno));
    }

  /* For other errors, a general-purpose explanation */

  else addr->message = string_sprintf("error while writing to mailbox %s",
    filename);

  /* For a write to a directory, remove the file. */

  if (isdirectory) unlink(tempname);

  /* For a file, reset the file size to what it was before we started, leaving
  the last modification time unchanged, so it will get reset also. All systems
  investigated so far have ftruncate(), whereas not all have the F_FREESP
  fcntl() call (BSDI & FreeBSD do not). */

  else ftruncate(fd, saved_size);
  }

/* Handle successful writing - we want the modification time to be now.
Remove the default backstop error number. For a directory, now is the time
to rename the file with a unique name, constructed from the time and the file's
inode in base 62 in the form tttttt-iiiiii. As soon as such a name appears
it may get used by another process. */

else
  {
  times.modtime = now;
  addr->errno = 0;

  if (isdirectory)
    {
    char newname[256];
    char *p = newname + (int)strlen(tempname);

    if (fstat(fd, &statbuf) < 0)
      {
      addr->errno = errno;
      addr->message = string_sprintf("while fstatting opened message file %s",
        tempname);
      close(fd);
      return;
      }

    /* Build the new name. It starts with 'q' for smail compatibility. */

    strcpy(newname, tempname);
    while (p[-1] != '/') p--;
    *p++ = 'q';
    strcpy(p, string_base62(time(NULL)));
    p += (int)strlen(p);
    sprintf(p, "-%s", string_base62(statbuf.st_ino));

    if (rename(tempname, newname) < 0)
      {
      addr->errno = errno;
      addr->message = string_sprintf("while renaming message file %s as %s",
        tempname, newname);
      unlink(tempname);
      close(fd);
      return;
      }
    }
  }


/* For a file, restore the last access time (atime), and set the modification
time as required - changed if write succeeded, unchanged if not. */

if (!isdirectory) utime(filename, &times);

/* Notify comsat if configured to do so. It only makes sense if the configured
file is the one that the comsat daemon knows about. */

if (ob->notify_comsat && yield == OK)
  notify_comsat(deliver_localpart, saved_size);

/* Remove the lock file on the mailbox, if present, then close the file, which
will release the fcntl lock, if present. Set the final return code, and we are
done. */

if (hd > 0) unlink(lockname);
close(fd);

DEBUG(2) debug_printf("appendfile yields %d with errno=%d more_errno=%d\n",
  yield, addr->errno, addr->more_errno);

addr->transport_return = yield;
}

/* End of transport/appendfile.c */
