/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* General functions concerned with transportation, and generic options for all
transports. */


#include "exim.h"



/* Generic options for transports, all of which live inside transport_instance
data blocks. There are in fact a couple of options which have generic entries
in the structure, but are set only by certain transports. These are uid and
gid, and they are handled by having private options with the same values,
and copying them into the generic slots at the end of initialization. This
means they can only be set by some transports, but the values are available at
the top level. */

optionlist optionlist_transports[] = {
  { "driver",    opt_stringptr,
                 (void *)offsetof(transport_instance, driver_name) },
};

int optionlist_transports_size =
  sizeof(optionlist_transports)/sizeof(optionlist);


/*************************************************
*             Initialize transport list           *
*************************************************/

/* Read the transports configuration file, and set up a chain of transport
instances according to its contents. Each transport has generic options and may
also have its own private options. This function is only ever called when
transports == NULL. We use generic code in readconf to do most of the work. */

void transport_init(void)
{
transport_instance *t;

readconf_driver_init("transport",
  (driver_instance **)(&transports),     /* chain anchor */
  (driver_info *)transports_available,   /* available drivers */
  sizeof(transport_info),                /* size of info block */
  &transport_defaults,                   /* default values for generic options */
  sizeof(transport_instance),            /* size of instance block */
  optionlist_transports,                 /* generic options */
  optionlist_transports_size);

/* Now scan the configured transports and ... ? */

for (t = transports; t != NULL; t = t->next)
  {
  /* No checks currently done */
  }
}



/*************************************************
*             Timeout handler                    *
*************************************************/

/* This is called by the any transport that wants to apply timeouts
to its activities. It then sets transport_chunk_timeout before calling
transport_write_message() to apply timeouts to individual message chunks. */

void transport_timeout_handler(int sig)
{
transport_sigalrm_seen = TRUE;
}



/*************************************************
*             Write block of data                *
*************************************************/

/* Subroutine called by write_chunk actually to write a data block. It
applies a timeout if transport_chunk_timeout is greater than zero. Also,
on some systems at least, if a quota is exceeded *during* the write, the yield
is the number of bytes written and errno is not set. Therefore we always try to
write a second time to output the remainder of the data after a non-negative
return from write() (except after a timeout). This time, no bytes should get
written, and a proper error should get put into errno. */

static BOOL write_block(int fd, char *block, int len)
{
int i, rc;
for (i = 0; i < 2; i++)
  {
  if (transport_chunk_timeout > 0) alarm(transport_chunk_timeout);
  rc = write(fd, block, len);
  if (transport_chunk_timeout > 0) alarm(0);
  if (rc == len) return TRUE;
  if (rc < 0) return FALSE;  
  if (transport_sigalrm_seen) 
    {
    errno = ETIMEDOUT; 
    return FALSE;
    }  
  len -= rc;
  block += rc;
  }
return rc == len;
}




/*************************************************
*              Write character chunk             *
*************************************************/

/* Subroutine used by transport_write_message to scan character chunks for
newlines and act appropriately. The object is to minimise the number of writes.
In practice, either from_hack or (use_crlf & smtp_dots) is likely to be set,
but not both. However, the code works with any combination.

If the from hack or smtp_dots options are requested, we must do the necessary
at the start of the chunk if it is the start of the data section, or if the
previous last char was '\n'. This is handled by a static flag.

If a transport wants data transfers to be timed, it sets a non-zero value
in transport_chunk_timeout and sets up transport_timeout_handler() as the
SIGALRM handler. A non-zero transport_chunk_timeout causes a timer to be set
for each chunk of data written from here via write_block() above. If time runs
out, then a write() fails and provokes an error return. The caller can then
inspect transport_sigalrm_seen to check for a timeout. */

static BOOL was_nl;     /* NL status flag */

static BOOL write_chunk(int fd, char *chunk, int len, BOOL from_hack,
  BOOL use_crlf, BOOL smtp_dots)
{
char *start = chunk;
char *end = chunk + len;
char *outptr = deliver_out_buffer;
char *ptr;

/* If none of the processing options is set, we can just write out the
block in one go without further ado. */

if (!(from_hack || use_crlf || smtp_dots)) return write_block(fd, chunk, len);

/* Otherwise, scan the buffer for newlines while copying into the output
buffer. This is to ensure we write the main data output in big chunks. Assume
len > 0. First handle the case when the preceding char was NL. */

if (was_nl)
  {
  if (from_hack && len >= 5 && strncmp(start, "From ", 5) == 0)
    *outptr++ = '>';
  else if (smtp_dots && *start == '.')
    *outptr++ = '.';
  was_nl = FALSE;
  }

/* Now process the characters in the chunk. */

for (ptr = start; ptr < end; ptr++)
  {
  register int ch;

  /* If there isn't room for another 4 characters, flush the buffer. */

  if ((len = outptr - deliver_out_buffer) > DELIVER_BUFFER_SIZE - 5)
    {
    if (!write_block(fd, deliver_out_buffer, len)) return FALSE;
    outptr = deliver_out_buffer;
    }

  if ((ch = *ptr) == '\n')
    {
    int left = end - ptr - 1;  /* count of chars left after NL */

    /* Insert CR before NL if required */

    if (use_crlf) *outptr++ = '\r';

    /* Now the NL */

    *outptr++ = '\n';

    /* If it happens that \n is the last character in the chunk, set the
    flag for next time (and for checking the final char). */

    if (left <= 0) was_nl = TRUE;

    /* The "from hack" inserts ">" before any line starting with "From ". It
    is a case-sensitive test. */

    else if (from_hack && left >= 5 && strncmp(ptr+1, "From ", 5) == 0)
      *outptr++ = '>';

    /* The smtp dot convention inserts a dot before any line starting with a
    dot. (We don't need to test left >= 1 because we know it isn't <= 0.) */

    else if (smtp_dots && ptr[1] == '.') *outptr++ = '.';
    }

  else *outptr++ = ch;
  }

/* Write out any remaining data in the buffer before returning. */

return (len = outptr - deliver_out_buffer) <= 0 ||
  write(fd, deliver_out_buffer, len);
}




/*************************************************
*                Write the message               *
*************************************************/

/* This function writes the message to the given file descriptor. The headers
are in the in-store data structure, and the rest of the message is in the open
file descriptor deliver_datafile. Make sure we start it at the beginning. There
are three options:

. If add_return_path is TRUE, a "return-path:" header is added to the message,
  containing the sender's address, unless it is null, or the user_null_sender 
  flag is set (indicating an error message).

. If from_hack is true, the "from hack" is done to lines starting with the text
  "From " (what a shoddy specification). The from hack is not applied to
  headers or the prefix or suffix.

. If use_crlf is true, newlines are turned into CRLF (SMTP output).

. If smtp_dots is true, lines starting with . get an extra . added.

The yield is TRUE if all went well, and FALSE if not. Exit *immediately* after
any writing or reading error, leaving the code in errno intact. Error exits
can include timeouts for certain transports, which are requested by setting
transport_chunk_timeout non-zero. */


BOOL transport_write_message(int fd, BOOL add_return_path,
  BOOL add_delivery_date, BOOL from_hack, BOOL use_crlf, BOOL smtp_dots,
  char *errors_to, int size_limit)
{
int written = 0;
int len;
header_line *h;

/* Add return-path: if requested, and there is one. */

if (add_return_path && (errors_to != NULL || 
    (sender_address[0] != 0 && !user_null_sender)))
  {
  char buffer[SENDER_ADDRESS_MAXLENGTH + 20];
  sprintf(buffer, "Return-path: <%s>\n",
    (errors_to != NULL)? errors_to : sender_address);
  len = strlen(buffer);
  if (!write_chunk(fd, buffer, len, FALSE, use_crlf, FALSE)) return FALSE;
  }

/* Add delivery-date: if requested. */

if (add_delivery_date)
  {
  char buffer[100];
  sprintf(buffer, "Delivery-date: %s\n", tod_stamp(tod_full));
  len = strlen(buffer);
  if (!write_chunk(fd, buffer, len, FALSE, use_crlf, FALSE)) return FALSE;
  }

/* Then the message's headers. Don't write any that are flagged as "old"; that
means they were rewritten, or are a record of envelope rewriting, or were
removed (e.g. Bcc). */

for (h = header_list; h != NULL; h = h->next)
  {
  if (h->type == htype_old) continue;
  if (!write_chunk(fd, h->text, h->slen, FALSE, use_crlf, smtp_dots))
    return FALSE;
  }

/* Separate headers from body with a blank line; set the static flag for
the chunk writer. Note: use_crlf won't be in use when writing to a file, so we
are only every trying to write one byte to a file. If this fails because of
quote excession, a proper errno should result. */

if (use_crlf? (write(fd, "\r\n", 2) != 2) : (write(fd, "\n", 1) != 1))
  return FALSE;
was_nl = TRUE;

/* Ensure the body is positioned at the start of its file, then write it,
applying the size limit if required. */

lseek(deliver_datafile, 0, SEEK_SET);
while ((len = read(deliver_datafile,deliver_in_buffer,DELIVER_BUFFER_SIZE)) > 0)
  {
  if (!write_chunk(fd, deliver_in_buffer, len, from_hack, use_crlf, smtp_dots))
    return FALSE;
  if (size_limit > 0)
    {
    written += len;
    if (written > size_limit) break;
    }
  }

/* A read error on the body will have left len == -1 and errno set. */

if (len != 0) return FALSE;

/* If the last character of the body was not '\n', manufacture one. Note:
use_crlf won't be in use when writing to a file, so we are only every trying to
write one byte to a file. If this fails because of quote excession, a proper
errno should result. */

if (!was_nl)
  {
  if (use_crlf? (write(fd, "\r\n", 2) != 2) : (write(fd, "\n", 1) != 1))
    return FALSE;
  }

return TRUE;
}




/*************************************************
*            Update waiting database             *
*************************************************/

/* This is called when an address is deferred by remote transports that are
capable of sending more than one message over one connection. A database is
maintained for each transport, keeping track of which messages are waiting for
which hosts. The transport can then consult this when eventually a successful
delivery happens, and if it finds that another message is waiting for the same
host, it can fire up a new process to deal with it using the same connection.

The database records are keyed by host name. They can get full if there are
lots of messages waiting, and so there is a continuation mechanism for them.

Each record contains a list of message ids, packed end to end without any
zeros. Each one is MESSAGE_ID_LENGTH bytes long. The count field says how many
in this record, and the sequence field says if there are any other records for
this host. If the sequence field is 0, there are none. If it is 1, then another
record with the name <hostname>:0 exists; if it is 2, then two other records
with sequence numbers 0 and 1 exist, and so on.

Currently, an exhaustive search of all continuation records has to be done to
determine whether to add a message id to a given record. This shouldn't be
too bad except in extreme cases. I can't figure out a *simple* way of doing
better.

Old records should eventually get swept up by the exim_tidydb utility. */


void transport_update_waiting(host_item *hostlist, char *tpname)
{
char buffer[256];
char *prevname = "";
host_item *host;
EXIM_DB *dbm_file;

/* Open the database for this transport */

sprintf(buffer, "wait-%s", tpname);
dbm_file = db_open(buffer, O_RDWR|O_CREAT);
if (dbm_file == NULL)
  {
  log_write(LOG_MAIN, "Failed to open %s database", buffer);
  return;
  }

/* Scan the list of hosts for which this message is waiting, and ensure
that the message id is in each host record. */

for (host = hostlist; host!= NULL; host = host->next)
  {
  BOOL already = FALSE;
  db_wait *host_record;
  char *s;
  int i, host_length;

  /*Skip if this is the same host as we just processed; otherwise remember
  the name for next time. */

  if (strcmp(prevname, host->name) == 0) continue;
  prevname = host->name;

  /* Look up the host record; if there isn't one, make an empty one. */

  host_record = db_read(dbm_file, host->name);
  if (host_record == NULL)
    {
    host_record = store_malloc(sizeof(db_wait) + MESSAGE_ID_LENGTH);
    host_record->count = host_record->sequence = 0;
    }

  /* Compute the current length */

  host_length = host_record->count * MESSAGE_ID_LENGTH;

  /* Search the record to see if the current message is already in it. */

  for (s = host_record->text; s < host_record->text + host_length;
       s += MESSAGE_ID_LENGTH)
    {
    if (strncmp(s, message_id, MESSAGE_ID_LENGTH) == 0)
      { already = TRUE; break; }
    }

  /* If we haven't found this message in the main record, search any
  continuation records that exist. */

  for (i = host_record->sequence - 1; i >= 0 && !already; i--)
    {
    db_wait *cont;
    sprintf(buffer, "%s:%d", host->name, i);
    cont = db_read(dbm_file, buffer);
    if (cont != NULL)
      {
      int clen = cont->count * MESSAGE_ID_LENGTH;
      for (s = cont->text; s < cont->text + clen; s += MESSAGE_ID_LENGTH)
        {
        if (strncmp(s, message_id, MESSAGE_ID_LENGTH) == 0)
          { already = TRUE; break; }
        }
      store_free(cont);
      }
    }

  /* If this message is already in a record, no need to update. */

  if (already) continue;


  /* If this record is full, write it out with a new name constructed
  from the sequence number, increase the sequence number, and empty
  the record. */

  if (host_record->count >= WAIT_NAME_MAX)
    {
    sprintf(buffer, "%s:%d", host->name, host_record->sequence);
    db_write(dbm_file, buffer, host_record, sizeof(db_wait) + host_length);
    host_record->sequence++;
    host_record->count = 0;
    host_length = 0;
    }

  /* If this record is not full, increase the size of the record to
  allow for one new message id. */

  else
    {
    db_wait *newr =
      store_malloc(sizeof(db_wait) + host_length + MESSAGE_ID_LENGTH);
    memcpy(newr, host_record, sizeof(db_wait) + host_length);
    host_record = newr;
    }

  /* Now add the new name on the end */

  memcpy(host_record->text + host_length, message_id, MESSAGE_ID_LENGTH);
  host_record->count++;
  host_length += MESSAGE_ID_LENGTH;

  /* Update the database */

  db_write(dbm_file, host->name, host_record, sizeof(db_wait) + host_length);
  }

/* All now done */

db_close(dbm_file);
}




/*************************************************
*         Test for waiting messages              *
*************************************************/

/* This function is called by a remote transport which uses the previous
function to remember which messages are waiting for which remote hosts. It's
called after a successful delivery and it's job is to check whether there is
another message waiting for the same host, and if so, spawn a new exim process
to deliver it. However, it doesn't do this if the current continue sequence
is greater than the maximum. */

BOOL transport_check_waiting(char *transport_name, char *hostname, int fd_in,
   int fd_out, int batch_max)
{
db_wait *host_record;
int host_length, path_len, pid, status;
EXIM_DB *dbm_file;
char new_message_id[MESSAGE_ID_LENGTH + 1];
char buffer[256];

DEBUG(4) debug_printf("transport_check_waiting entered\n");

/* Do nothing if we have hit the batch maximum. */

if (batch_max > 0 && continue_sequence >= batch_max)
  {
  DEBUG(4) debug_printf("batch max reached: returning\n");
  return FALSE;
  }

/* Open the waiting information database. */

sprintf(buffer, "wait-%s", transport_name);
dbm_file = db_open(buffer, O_RDWR|O_CREAT);
if (dbm_file == NULL)
  {
  log_write(LOG_MAIN, "Failed to open %s database", buffer);
  return FALSE;
  }

/* See if there is a record for this host; if not, there's nothing to do. */

host_record = db_read(dbm_file, hostname);
if (host_record == NULL)
  {
  db_close(dbm_file);
  DEBUG(4) debug_printf("no messages waiting for %s\n", hostname);
  return FALSE;
  }

/* Scan the message ids in the record from the end towards the beginning,
until one is found for which a spool file actually exists. If the record gets
emptied, delete it and continue with any continuation records that may exist.
*/

host_length = host_record->count * MESSAGE_ID_LENGTH;

/* Loop to handle continuation host records in the database */

for (;;)
  {
  BOOL found = FALSE;

  sprintf(buffer, "%s/input/", spool_directory);
  path_len = (int)strlen(buffer);

  for (host_length -= MESSAGE_ID_LENGTH; host_length >= 0;
       host_length -= MESSAGE_ID_LENGTH)
    {
    struct stat statbuf;
    strncpy(new_message_id, host_record->text + host_length,
      MESSAGE_ID_LENGTH);
    new_message_id[MESSAGE_ID_LENGTH] = 0;
    sprintf(buffer + path_len, "%s-D", new_message_id);

    /* The listed message may be the one we are currently processing. If
    so, we want to remove it from the list without doing anything else.
    If not, do a stat to see if it is an existing message. If it is, break
    the loop to handle it. No need to bother about locks; as this is all
    "hint" processing, it won't matter if it doesn't exist by the time exim
    actually tries to deliver it. */

    if (strcmp(new_message_id, message_id) != 0 &&
        stat(buffer, &statbuf) == 0)
      {
      found = TRUE;
      break;
      }
    }

  /* If we have removed all the message ids from the record delete the record.
  If there is a continuation record, fetch it and remove it from the file,
  as it will be rewritten as the main record. Repeat in the case of an
  empty continuation. */

  while (host_length <= 0)
    {
    int i;
    db_wait *newr = NULL;

    /* Search for a continuation */

    for (i = host_record->sequence - 1; i >= 0 && newr == NULL; i--)
      {
      sprintf(buffer, "%s:%d", hostname, i);
      newr = db_read(dbm_file, buffer);
      }

    /* If no continuation, delete the current and break the loop */

    if (newr == NULL)
      {
      db_delete(dbm_file, hostname);
      break;
      }

    /* Else replace the current with the continuation */

    db_delete(dbm_file, buffer);
    store_free(host_record);
    host_record = newr;
    host_length = host_record->count * MESSAGE_ID_LENGTH;
    }


  /* If we found an existing message, break the continuation loop. */

  if (found) break;


  /* If host_length <= 0 we have emptied a record and not found a good message,
  and there are no continuation records. Otherwise there is a continuation
  record to process. */

  if (host_length <= 0)
    {
    db_close(dbm_file);
    DEBUG(4) debug_printf("waiting messages already delivered\n");
    return FALSE;
    }
  }


/* Control gets here when an existing message has been encountered; its
id is in new_message_id, and host_length is the revised length of the
host record. If it is zero, the record has been removed. Update the
record if required, and then close the database. */

if (host_length > 0)
  {
  host_record->count = host_length/MESSAGE_ID_LENGTH;
  db_write(dbm_file, hostname, host_record, (int)sizeof(db_wait) + host_length);
  }
db_close(dbm_file);


/* Now we need to fork a new exim process to deliver the message, and do
a re-exec, both to get a clean delivery process, and to regain root privilege
in cases where it has been given away. */

if ((pid = vfork()) == 0)
  {
  int fd; 
  int i = 0;
  char *argv[8];

  /* Disconnect entirely from the parent process. */

  if (vfork() != 0) _exit(EXIT_SUCCESS);

  /* Set up the calling arguments */

  argv[i++] = exim_path;
  if (debug_level > 0)
    argv[i++] = string_sprintf("-d%d", debug_level);
  argv[i++] = "-MC";
  argv[i++] = transport_name;
  argv[i++] = hostname;
  argv[i++] = string_sprintf("%d", continue_sequence + 1);
  argv[i++] = new_message_id;
  argv[i++] = NULL;

  /* Arrange for the channel to be on stdin and stdout */

  for (fd = mac_maxfd; fd >= 0; fd--)
    if (fd != fd_in && fd != fd_out &&
      (debug_file == NULL || fd != fileno(debug_file)))
        close(fd);

  if (fd_in != 0)
    {
    dup2(fd_in, 0);
    close(fd_in);
    }

  if (fd_out != 1)
    {
    dup2(fd_out, 1);
    close(fd_out);
    }

  DEBUG(4) debug_printf("fork %s %s %s %s %s %s %s\n", argv[0], argv[1],
    argv[2], argv[3], argv[4], argv[5], (argv[6] == NULL)? "" : argv[6]);

  execv(argv[0], argv);

  /* Failed to execv, or getrlimit failed */

  DEBUG(4) debug_printf("fork failed: %s\n", strerror(errno));

  _exit(errno);         /* Note: must be _exit(), NOT exit() */
  }

/* If the process creation succeeded, wait for the first-level child, which
immediately exits, leaving the second level process entirely disconnected from
this one. */

if (pid > 0)
  {
  while (wait(&status) != pid);
  return TRUE;
  }
else return FALSE;
}

/* End of transport.c */
