/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions for sending messages to sender or to mailmaster. */


#include "exim.h"



/*************************************************
*          Send message to sender                *
*************************************************/

/* This function is called when errors are detected during the receipt of a
message. Delivery failures are handled separately in deliver.c.

If there is a valid sender_address, and the failing message is not a local 
error message, then this function calls moan_send_message to send a message to
that person. If the sender's address is null, then an error has occurred with a
message that was generated by a mailer daemon. All we can do is to write
information to log files. The same action is taken if local_error_message is 
set - this can happen for non null-senders in certain configurations where exim 
doesn't run setuid root. */

BOOL moan_to_sender(int ident, error_block *eblock, header_line *headers,
  FILE *message_file)
{
if (sender_address != NULL && sender_address[0] != 0 && !local_error_message)
  return moan_send_message(sender_address, ident, eblock, headers,
    message_file);

switch(ident)
  {
  case ERRMESS_BADARGADDRESS:
  case ERRMESS_BADNOADDRESS:
  case ERRMESS_BADADDRESS:
  log_write(LOG_MAIN, "Error while handling error message: "
    "at least one malformed recipient address: "
    "%s - %s", eblock->text1, eblock->text2);
  break;

  case ERRMESS_NOADDRESS:
  log_write(LOG_MAIN, "Error while handling error message %s: "
    "no recipient addresses", message_id);
  break;

  case ERRMESS_SPOOLWRITE:
  log_write(LOG_MAIN|LOG_PANIC_DIE, 
    "Error while handling error message %s: write to spool failed", message_id);
  break;

  case ERRMESS_MESSAGEREAD:
  log_write(LOG_MAIN, "Error while handling error message %s: "
    "read from input file failed", message_id);
  break;
  
  case ERRMESS_VLONGHEADER:
  log_write(LOG_MAIN, "Error while handling error message: "
    "excessively long header line read");
  break;    

  case ERRMESS_TOOBIG:
  log_write(LOG_MAIN, "Error while handling error message: "
    "message too big (limit set to %d)", message_size_limit);
  break;    

  default:
  log_write(LOG_MAIN|LOG_PANIC, "Error while handling error message %s: "
    "unknown error number %d", message_id, ident);
  break;
  }
return FAIL;
}



/*************************************************
*              Send error message                *
*************************************************/

/* This function sends an error message by opening a pipe to
a new process running the specified MTA, and writing a message
to it using the "-t" option. This is not used for delivery failures,
which have their own code for handing failed addresses. */

BOOL moan_send_message(char *recipient, int ident, error_block *eblock,
  header_line *headers, FILE *message_file)
{
int written = 0;
int fd;
int status;
int count = 0;
int size_limit = return_size_limit;
FILE *f;
int pid = child_open(mailer_argv, NULL, 077, -1, -1, &fd, 
    (debug_file != NULL)? fileno(debug_file) : -1);

/* Creation of child failed */

if (pid < 0)
  {
  DEBUG(2) debug_printf("Failed to create child to send message\n");
  return FALSE;
  }
else DEBUG(9) debug_printf("Child process %d for sending message\n", pid);   

/* Creation of child succeeded */

f = fdopen(fd, "w");
if (errors_reply_to != NULL) fprintf(f, "Reply-to: %s\n", errors_reply_to);
fprintf(f, "From: Mail Delivery System <Mailer-Daemon@%s>\n", 
  qualify_domain_sender);
fprintf(f, "To: %s\n", recipient);

switch(ident)
  {
  case ERRMESS_BADARGADDRESS:
  fprintf(f,
  "Subject: Mail failure - malformed recipient address\n\n");
  fprintf(f,
  "A message that you sent contained a recipient address that was incorrectly\n"
  "constructed:\n\n");
  fprintf(f, "  %s  %s\n", eblock->text1, eblock->text2);
  count = (int)strlen(eblock->text1); 
  if (count > 0 && eblock->text1[count-1] == '.')
    fprintf(f,
    "\nRecipient addresses must not end with a '.' character.\n");
  fprintf(f,
  "\nThe message has not been delivered to any recipients.\n");
  break;

  case ERRMESS_BADNOADDRESS:
  case ERRMESS_BADADDRESS:
  fprintf(f,
  "Subject: Mail failure - malformed recipient address\n\n");
  fprintf(f,
  "A message that you sent contained one or more recipient addresses that were\n"
  "incorrectly constructed:\n\n");

  while (eblock != NULL)
    {
    fprintf(f, "  %s: %s\n", eblock->text1, eblock->text2);
    count++;
    eblock = eblock->next;
    }

  fprintf(f, (count == 1)? "\nThis address has been ignored. " :
    "\nThese addresses have been ignored. ");

  fprintf(f, (ident == ERRMESS_BADADDRESS)?
  "The other addresses in the message were\n"
  "syntactically valid and have been passed on for an attempt at delivery.\n" :

  "There were no other addresses in your\n"
  "message, and so no attempt at delivery was possible.\n");
  break;

  case ERRMESS_NOADDRESS:
  fprintf(f, "Subject: Mail failure - no recipient addresses\n\n");
  fprintf(f,
  "A message that you sent contained no recipient addresses, and so no\n"
  "delivery could be attempted.\n");
  break;

  case ERRMESS_SPOOLWRITE:
  fprintf(f, "Subject: Mail failure - system failure\n\n");
  fprintf(f,
  "A system failure (spool write failure) was encountered while processing\n"
  "a message that you sent, so it has not been possible to deliver it.\n");
  break;

  case ERRMESS_MESSAGEREAD:
  fprintf(f, "Subject: Mail failure - system failure\n\n");
  fprintf(f,
  "A system failure (input read failure) was encountered while processing\n"
  "a message that you sent, so it has not been possible to deliver it.\n");
  break;
  
  case ERRMESS_VLONGHEADER:
  fprintf(f, "Subject: Mail failure - overlong header\n\n");
  fprintf(f,
  "A message that you sent contained a header line that was excessively\n"
  "long and could not be handled by the mail transmission software. The\n"
  "message has not been delivered to any recipients.\n");
  break;       

  case ERRMESS_TOOBIG:
  fprintf(f, "Subject: Mail failure - message too big\n\n");
  fprintf(f,
  "A message that you sent was longer than the maximum size allowed on this\n"
  "system. It was not delivered to any recipients.\n");
  break;       

  default:
  fprintf(f, "Subject: Mail failure\n\n");
  fprintf(f,
  "A message that you sent has caused the error routine to be entered with\n"
  "an unknown error number (%d).\n");
  break;
  }

/* Now copy the message - headers then the rest of the input if
available, up to the configured limit. */

if (size_limit == 0 || size_limit > message_size_limit) 
  size_limit = message_size_limit;
  
if (size_limit > 0)
  {
  int x = size_limit;
  char *k = ""; 
  if ((x & 1023) == 0)
    {
    k = "K";
    x >>= 10; 
    }
  fprintf(f, "\n"
  "------ This is a copy of your message, including all the headers.\n"
  "------ No more than %d%s characters of the body are included.\n\n", x, k);
  }   
else fprintf(f, "\n"
  "------ This is a copy of your message, including all the headers. ------"
  "\n\n");

while (headers != NULL)
  {
  fprintf(f, "%s", headers->text);
  headers = headers->next;
  }

if (ident != ERRMESS_VLONGHEADER) fputc('\n', f);

/* After early detection of an error, the message file may be STDIN,
in which case we might have to terminate on a line containing just "."
as well as on EOF. */

if (message_file != NULL)
  {
  int ch;
  int state = 1;
  BOOL enddot = dot_ends && message_file == stdin;
  while ((ch = fgetc(message_file)) != EOF)
    {
    fputc(ch, f);
    if (size_limit > 0 && ++written > size_limit) break;
    if (enddot)
      {
      if (state == 0) { if (ch == '\n') state = 1; }
      else if (state == 1)
        { if (ch == '.') state = 2; else if (ch != '\n') state = 0; }
      else
        { if (ch == '\n') break; else state = 0; }
      }
    }
  }

/* Close the file, which should send an EOF to the child process
that is receiving the message. Wait for it to finish. */

fclose(f);
status = child_close(pid);  /* Waits for child to close */
if (status != 0)
  {
  DEBUG(2) debug_printf("Child mail process returned status %d\n", status);
  log_write(LOG_MAIN, "Child mail process returned status %d", status);
  return FALSE; 
  }
      
return TRUE;
}



/*************************************************
*            Send message to mailmaster          *
*************************************************/

/* This is called when exim is configured to tell the mailmaster about some
incident. */

void moan_tell_mailmaster(char *subject, char *format, ...)
{
FILE *f;
va_list ap;
int fd;
int pid = child_open(mailer_argv, NULL, 077, -1, -1, &fd, -1);

if (pid < 0)
  {
  DEBUG(2) debug_printf("Failed to create child to send message\n");
  return;
  }

f = fdopen(fd, "w");
fprintf(f, "From: Mail Delivery System <Mailer-Daemon@%s>\n", 
  qualify_domain_sender);
fprintf(f, "To: %s\n", errors_address);
fprintf(f, "Subject: %s\n\n", subject);
va_start(ap, format);
vfprintf(f, format, ap);
va_end(ap);

fclose(f);
child_close(pid);  /* Waits for child to close */
}



/*************************************************
*            Handle SMTP batch error             *
*************************************************/

/* This is called when exim is configured to tell somebody when something
goes wrong in batched (-bS) SMTP input. If the caller is not trusted, tell the 
caller. Otherwise tell the mailmaster. */

void moan_smtp_batch(char *format, ...)
{
FILE *f;
va_list ap;
int fd;
int pid = child_open(mailer_argv, NULL, 077, -1, -1, &fd, -1);
char *recipient = trusted_caller? errors_address : sender_address;

if (pid < 0)
  {
  DEBUG(2) debug_printf("Failed to create child to send message\n");
  log_write(LOG_MAIN, "Failed to create child process to send error message"); 
  return;
  }

f = fdopen(fd, "w");
fprintf(f, "From: Mail Delivery System <Mailer-Daemon@%s>\n", 
  qualify_domain_sender);
fprintf(f, "To: %s\n", recipient);
fprintf(f, "Subject: Error in batched SMTP input\n\n");
fprintf(f, 
  "While reading a batch of messages using the -bS option, the\n"
  "following error was detected:\n\n");
va_start(ap, format);
vfprintf(f, format, ap);
va_end(ap);

fprintf(f, "\n");   

fclose(f);
child_close(pid);  /* Waits for child to close */
}




/*************************************************
*         Check for error copies                 *
*************************************************/

/* This function is passed the recipient of an error message, and
must check the error_copies string to see whether there is an additional
recipient list to which errors for this recipient must be bcc'd. The incoming
recipient is always fully qualified. */

char *moan_check_errorcopy(char *recipient)
{
char *item, *localpart, *domain;
char *yield = NULL;
int llen;
re_block *reblock = NULL; 

if (errors_copy == NULL) return NULL;

/* Set up pointer to the local part and domain, and compute the
length of the local part. */

localpart = recipient;
domain = strchr(recipient, '@');  
if (domain == NULL) return NULL;  /* should not occur, but avoid crash */
llen = domain++ - recipient;

/* Scan through the configured items */

for (item = string_nextinlist(errors_copy, ':'); item != NULL;
     item = string_nextinlist(NULL, ':'))
  {
  BOOL match = FALSE; 
  char *newaddress;

  /* Move over the first address item to the first non-quoted space */
   
  for (newaddress = item; *newaddress != 0 && !isspace(*newaddress); 
       newaddress++)
    {
    if (*newaddress == '\"')
      {
      while (*(++newaddress) != 0)
        {
        if (*newaddress == '\\' && newaddress[1] != 0) newaddress++;
          else if (*newaddress == '\"') { newaddress++; break; }
        }
      }  
    }  
    
  /* If no new address found, just skip this item. Otherwise, terminate the 
  test address - the yield of string_nextinlist() is always in a private 
  static buffer, so it is OK to do this. Then move on to the start of the new
  address (again, skip if not there). */
    
  if (*newaddress == 0) continue;
  *newaddress++ = 0; 
  while (isspace(*newaddress)) newaddress++;
  if (*newaddress == 0) continue;
  
  /* We now have an item to match as a string in item, and the additional
  address in newaddress. First handle a regular expression, which must match 
  the entire incoming address. As this is a rare occurrence, don't make
  any effort to cache compiled expressions. */ 

  expand_nmax = 0; 
  expand_nstring[0] = recipient;
  expand_nlength[0] = (int)strlen(recipient);

  if (item[0] == '^') 
    {
    if (reblock != NULL) { store_free(reblock); reblock = NULL; }
    match = match_string(recipient, item, &reblock);
    if (match) for (; expand_nmax < NSUBEXP; expand_nmax++)
      {
      expand_nstring[expand_nmax] = reblock->re->startp[expand_nmax];
      expand_nlength[expand_nmax] = reblock->re->endp[expand_nmax] - 
        expand_nstring[expand_nmax];
      }
    expand_nmax--;
    }
    
  /* If not a regular expression, either part may begin with an
  asterisk, and both parts must match. There must be an '@' in the
  address. */ 
    
  else
    {
    char *slocalpart = item; 
    char *sdomain = strchr(item, '@');
    int sllen; 

    /* No @ => complain and ignore */
     
    if (sdomain == NULL)
      {
      log_write(LOG_MAIN|LOG_PANIC, "domain missing in errors_copy entry: %s",
        item);
      continue;    
      }  
    else 
      {
      sllen = sdomain - item; 
      sdomain += 1;
      } 
      
    /* Check the local part */
    
    if (slocalpart[0] == '*')
      {
      int cllen = sllen - 1; 
      match = llen >= cllen &&
        strncmpic(localpart + llen - cllen, slocalpart + 1, cllen) == 0;
      if (match)
        { 
        expand_nstring[++expand_nmax] = localpart;
        expand_nlength[expand_nmax] = llen - cllen;     
        } 
      }
    else match = 
      llen == sllen && strncmpic(localpart, slocalpart, llen) == 0;
  
    /* If the local part matched, check the domain using the generalized
    function, which supports file lookups. */

    if (match) 
      {
      match = match_string(domain, sdomain, NULL);
      if (match && sdomain[0] == '*')
        {  
        expand_nstring[++expand_nmax] = domain;
        expand_nlength[expand_nmax] = 
          (int)strlen(domain) - (int)strlen(sdomain) + 1;
        } 
      } 
    }

  /* If we have found a match, expand the new address string and 
  return it. During expansion, make local part and domain available
  for insertion. This requires a temporary termination of the local part. */

  if (match)
    {
    deliver_domain = domain;
    deliver_localpart = localpart;
    localpart[llen] = 0; 
    yield = expand_string(newaddress);
    localpart[llen] = '@'; 
    deliver_domain = deliver_localpart = NULL;
    if (yield == NULL)
      log_write(LOG_MAIN|LOG_PANIC, "Failed to expand %s when processing "
        "errors_copy: %s", newaddress, expand_string_message);
    break;
    }    
  }       
  
DEBUG(5) debug_printf("errors_copy check returned %s\n", 
  (yield == NULL)? "NULL" : yield);
   
if (reblock != NULL) store_free(reblock);
expand_nmax = -1;
return yield;
}

/* End of moan.c */
