/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions for writing log files. */


#include "exim.h"



/*************************************************
*           Local static variables               *
*************************************************/

char mainlog_name[256];

static int mainlogfd = -1;
static int mainlog_inode = -1;
static int processlogfd = -1;
static int rejectlogfd = -1;

static BOOL panic_recurseflag = FALSE;




/*************************************************
*                Open a log file                 *
*************************************************/

/* This function opens one of a number of logs, which all reside in the same
spool directory, creating the log directory if necessary. This may be called
recursively on failure, in order to open the panic log.

If exim is configured to avoid running as root wherever possible, the log files
must be owned by the non-privileged user. To ensure this, first try an open
without O_CREAT - most of the time this will succeed. If it fails, try to
create the file, and adjust the owner if necessary. */

static void open_log(int *fd, char *name)
{
*fd = open(name, O_APPEND|O_WRONLY, LOG_MODE);
if (*fd >= 0) return;

/* Need to create */

*fd = open(name, O_CREAT|O_APPEND|O_WRONLY, LOG_MODE);
if (*fd < 0 && errno == ENOENT)
  {
  directory_make(spool_directory, "log", LOG_DIRECTORY_MODE);
  *fd = open(name, O_CREAT|O_APPEND|O_WRONLY, LOG_MODE);
  }

/* Creation succeeded; change owner if we are currently root, and ensure
the mode is as specified. */

if (*fd >= 0)
  {
  if (exim_uid > 0 && geteuid() == root_uid)
    chown(name, exim_uid, exim_gid);
  chmod(name, LOG_MODE);
  return;
  } 

/* Creation failed */

log_write(LOG_PANIC_DIE, "Cannot open %s: %s", name, strerror(errno));
/* Never returns */ 
}



/*************************************************
*            Write message to log file           *
*************************************************/

/* The message always gets '\n' added on the end of it, since more than one
process may be writing to the log at once and we don't want intermingling to
happen in the middle of lines. To be absolutely sure of this we write the data
into a private buffer and then put it out in a single write() call.

The flags determine which log(s) the message is written to, and in the
case of the panic log, whether the process should die afterwards. 

This function is called from some functions that are shared between exim and 
its utilities. The variable really_exim is TRUE only when it is exim itself 
that is running. If it is not, don't try to write to the log (permission will 
probably be denied) - just write to stderr instead.

In exim proper, the buffer for building the message is got at start-up, so that
nothing gets done if it can't be got. However, some functions that are shared
occasionally obey log_write calls in error situations, and it is simplest to
put a single malloc() here rather than put one in each utility. Malloc is 
used directly because the store functions may call log_write().

If a message_id exists, we include it after the timestamp. */

void log_write(int flags, char *format, ...)
{
char *ptr;
int paniclogfd;
va_list ap;

/* Ensure we have a buffer (see comment above); this should never be obeyed
when running exim proper.  */

if (log_buffer == NULL) 
  {
  log_buffer = (char *)malloc(LOG_BUFFER_SIZE);
  if (log_buffer == NULL)
    {
    fprintf(stderr, "exim: failed to get store for log buffer\n");
    exit(EXIT_FAILURE);
    }
  } 

/* If debugging, show all log entries, but don't show headers */

DEBUG(1)
  {
  fprintf(debug_file, "LOG:%s%s%s%s\n  ",
    ((flags & LOG_MAIN) != 0)?    " MAIN"   : "",
    ((flags & LOG_PANIC) != 0)?   " PANIC"  : "",
    ((flags & LOG_PROCESS) != 0)? " PROCESS": "",
    ((flags & LOG_REJECT) != 0)?  " REJECT" : ""); 
  va_start(ap, format);
  vfprintf(debug_file, format, ap);
  va_end(ap);
  fputc('\n', debug_file); 
  fflush(debug_file); 
  }

/* If no log file is specified, we are in a mess. */

if (flags == 0)
  log_write(LOG_PANIC_DIE, "log_write called with no flags set");

/* Create the main message in the log buffer, including the message
id except for the process log and when called by a utility. */

ptr = log_buffer;
if (really_exim && (flags & LOG_PROCESS) == 0 && message_id[0] != 0)
  sprintf(ptr, "%s %s ", tod_stamp(tod_log), message_id);
else sprintf(ptr, "%s ", tod_stamp(tod_log));
while(*ptr) ptr++;

va_start(ap, format);
vsprintf(ptr, format, ap);
while(*ptr) ptr++;
va_end(ap);

/* Add list of recipients to the message if required; the raw list,
before rewriting, was saved in raw_recipients. */

if ((flags & LOG_RECIPIENTS) != 0)
  {
  int i;
  sprintf(ptr, " for");
  while (*ptr) ptr++; 
  for (i = 0; i < recipients_count; i++)
    {
    char *s = raw_recipients[i];
    if (log_buffer + LOG_BUFFER_SIZE - ptr < (int)strlen(s) + 3) break;
    sprintf(ptr, " %s", s);
    while (*ptr) ptr++;  
    } 
  } 

sprintf(ptr, "\n");
while(*ptr) ptr++;

/* Handle loggable errors when running a utility. */

if (!really_exim)
  {
  fprintf(stderr, "%s", log_buffer);
  return;
  }  

/* Handle the main log. The log gets left open during a message's delivery
once it has been opened, but we don't want to keep on writing to it for too
long after it has been renamed. Therefore, do a stat() and see if the inode
has changed, and if so, re-open. */

if ((flags & LOG_MAIN) != 0)
  {
  struct stat statbuf;
  if (mainlogfd >= 0)
    {
    if (stat(mainlog_name, &statbuf) < 0 || statbuf.st_ino != mainlog_inode)
      {
      close(mainlogfd); 
      mainlogfd = -1;
      mainlog_inode = -1; 
      } 
    }   
  if (mainlogfd < 0) 
    {
    sprintf(mainlog_name, "%s/log/mainlog", spool_directory);
    open_log(&mainlogfd, mainlog_name);     /* No return on error */
    if (fstat(mainlogfd, &statbuf) >= 0) mainlog_inode = statbuf.st_ino; 
    }
  write(mainlogfd, log_buffer, ptr - log_buffer);
  }
  

/* Handle the log for rejected messages: log the headers if any, but
watch out for overflowing the buffer. Stick a separator between messages. */

if ((flags & LOG_REJECT) != 0)
  {
  header_line *h;
  if (rejectlogfd < 0) 
    {
    char buffer[256]; 
    sprintf(buffer, "%s/log/rejectlog", spool_directory);
    open_log(&rejectlogfd, buffer); /* No return on error */
    } 
  for (h = header_list; h != NULL; h = h->next)
    {
    if (log_buffer + LOG_BUFFER_SIZE - ptr < (int)strlen(h->text) + 100) break;
    sprintf(ptr, "%c %s", h->type, h->text);
    while(*ptr) ptr++;
    }   
  sprintf(ptr, "----------------------------------------"
    "--------------------------------------\n");    
  while(*ptr) ptr++;
  write(rejectlogfd, log_buffer, ptr - log_buffer);  
  }   
 

/* Handle the process log file, where exim processes can be made to dump 
details of what they are doing by sending them a USR1 signal. Note that
a message id is not automatically added above. */

if ((flags & LOG_PROCESS) != 0)
  {
  if (processlogfd < 0) 
    {
    char buffer[256];  
    sprintf(buffer, "%s/log/processlog", spool_directory);
    open_log(&processlogfd, buffer);  /* No return on error */
    } 
  write(processlogfd, log_buffer, ptr - log_buffer);  
  }


/* Handle the panic log, which is not kept open like the others. If it fails to
open, there will be a recursive call that ends up here. We detect this and
attempt to write to the system log as a last-ditch try at telling somebody. In
all cases, try to write to stderr and/or debug_file. */

if ((flags & LOG_PANIC) != 0)
  {
  char buffer[256]; 

  if (stderr != NULL && stderr != debug_file)
    fprintf(stderr, "%s", log_buffer);
  
  if (panic_recurseflag)
    {
    if (debug_file != NULL)
      fprintf(debug_file, "exim: could not open panic log: aborting\n");
    if (stderr != NULL && stderr != debug_file)
      fprintf(stderr, "exim: could not open panic log: aborting\n");
    syslog(LOG_MAIN|LOG_ERR, "exim: could not open panic log");
    exit(EXIT_FAILURE);
    }
  
  panic_recurseflag = TRUE; 
  sprintf(buffer, "%s/log/paniclog", spool_directory); 
  open_log(&paniclogfd, buffer);  /* Won't return on failure */
  panic_recurseflag = FALSE;
  write(paniclogfd, log_buffer, ptr - log_buffer); 
  close(paniclogfd);
   
  if ((flags & LOG_PANIC_DIE) != LOG_PANIC) exit(EXIT_FAILURE);
  }      
}



/*************************************************
*            Close any open log files            *
*************************************************/

void log_close(void)
{
if (mainlogfd >= 0) 
  { close(mainlogfd); mainlogfd= -1; }
if (processlogfd >= 0)
  { close(processlogfd); processlogfd = -1; }   
if (rejectlogfd >= 0)
  { close(rejectlogfd); rejectlogfd = -1; }   
}


/*************************************************
**************************************************
*             Stand-alone test program           *
**************************************************
*************************************************/

#ifdef STAND_ALONE
int main(void)
{
printf("spool directory = %s\n", spool_directory);
log_write(LOG_MAIN, "Test output to the log file");
return 0;
}

#endif

/* End of log.c */
