/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */


/* Functions for handling string expansion. */


#include "exim.h"


/* Recursively called function */

static char *expand_string_internal(char *, BOOL, char **, BOOL);



/*************************************************
*            Local statics and tables            *
*************************************************/

typedef struct {
  char *name;
  int   type;
  void *value;
} var_entry;

enum {
  vtype_int,            /* value is address of int */
  vtype_string,         /* value is address of string */
  vtype_stringptr,      /* value is address of pointer to string */
  vtype_msgbody,        /* as stringptr, but read when first required */
  vtype_todbsdin,       /* value not used; generate BSD inbox tod */
  vtype_todf,           /* value not used; generate full tod */
  vtype_todl,           /* value not used; generate log tod */
  vtype_reply           /* value not used; get reply from headers */
  };

/* This pointer gets set to values looked up in databases. It must
come before the following table. */

static char *lookup_value = NULL;

/* This table must be kept in alphabetical order. */

static var_entry var_table[] = {
  { "caller_gid",          vtype_int,         &real_gid },
  { "caller_uid",          vtype_int,         &real_uid },
  { "compile_date",        vtype_stringptr,   &version_date },
  { "compile_number",      vtype_stringptr,   &version_cnumber },
  { "domain",              vtype_stringptr,   &deliver_domain },
  { "home",                vtype_stringptr,   &deliver_home },
  { "host",                vtype_stringptr,   &deliver_host },
  { "local_part",          vtype_stringptr,   &deliver_localpart },
  { "message_body",        vtype_msgbody,     &message_body },
  { "message_id",          vtype_string,      &message_id_external },
  { "message_size",        vtype_int,         &message_size },
  { "primary_hostname",    vtype_stringptr,   &primary_hostname },
  { "received_protocol",   vtype_stringptr,   &received_protocol },
  { "reply_address",       vtype_reply,       NULL },
  { "return_path",         vtype_stringptr,   &return_path },
  { "sender_address",      vtype_stringptr,   &sender_address },
  { "sender_fullhost",     vtype_stringptr,   &sender_fullhost },
  { "sender_host_address", vtype_stringptr,   &sender_host_address },
  { "sender_host_name",    vtype_stringptr,   &sender_host_name },
  { "sender_ident",        vtype_stringptr,   &sender_ident },
  { "spool_directory",     vtype_stringptr,   &spool_directory },
  { "tod_bsdinbox",        vtype_todbsdin,    NULL },
  { "tod_full",            vtype_todf,        NULL },
  { "tod_log",             vtype_todl,        NULL },
  { "value",               vtype_stringptr,   &lookup_value },
  { "version_number",      vtype_stringptr,   &version_string }
};

static int var_table_size = sizeof(var_table)/sizeof(var_entry);
static char var_buffer[256];





/*************************************************
*             Pick out a name from a string      *
*************************************************/

/* The 3rd argument is pointing at the first alphabetic character.
The yield is its revised value. If the name is too long, it is
silently truncated. */

static char *read_name(char *name, int max, char *s)
{
int ptr = 0;
while (isalnum(*s) || *s == '_')
  {
  if (ptr < max-1) name[ptr++] = *s;
  s++;
  }
name[ptr] = 0;
return s;
}



/*************************************************
*     Pick out the rest of a header name         *
*************************************************/

/* A variable name starting $header_ (or just $h_ for those who like
abbreviations) might not be the complete header name because headers can
contain any printing characters in their names, except ':'. This function is
called to read the rest of the name, chop h[eader]_ off the front, and put ':'
on the end. */

static char *read_header_name(char *name, int max, char *s)
{
int prelen = strchr(name, '_') - name + 1;
int ptr = strlen(name) - prelen;
if (ptr > 0) memmove(name, name+prelen, ptr);
while (isgraph(*s) && *s != ':')
  {
  if (ptr < max-1) name[ptr++] = *s;
  s++;
  }
if (*s == ':') s++;
name[ptr++] = ':';
name[ptr] = 0;
return s;
}



/*************************************************
*           Pick out a number from a string      *
*************************************************/

/* The 2nd argument is pointing at the first digit.
The yield is its revised value. */

static char *read_number(int *n, char *s)
{
*n = 0;
while (isdigit(*s)) *n = *n * 10 + (*s++ - '0');
return s;
}



/*************************************************
*        Extract keyed subfield from a string    *
*************************************************/

/* This function is also used by the search routines when given a multi-level
key to search for. The yield is in dynamic store; NULL means that the key
was not found. */

char *expand_getkeyed(char *key, char *s)
{
int length = (int)strlen(key);
while (isspace(*s)) s++;

/* Loop to search for the key */

while (*s != 0)
  {
  int subkeylength, datalength;
  char *data;
  char *subkey = s;

  while (*s != 0 && *s != '=' && !isspace(*s)) s++;
  subkeylength = s - subkey;

  while (isspace(*s)) s++;
  if (*s == '=') while (isspace(*(++s)));

  /* For now, just find the end of the data field - interpret quoted
  string later if it is actually wanted. */

  data = s;
  if (*s == '\"')
    {
    while (*(++s) != 0 && *s != '\"')
      {
      if (*s == '\\' && s[1] != 0) s++;
      }
    if (*s == '\"') s++;
    }
  else while (*s != 0 && !isspace(*s)) s++;
  datalength = s - data;

  /* If keys match, set up the subfield as the yield and return. If
  the value is quoted, interpret the string (which cannot be longer than
  the original). */

  if (length == subkeylength && strncmp(key, subkey, length) == 0)
    {
    char *yield = store_malloc(datalength + 1);

    if (*data == '\"')
      {
      int i = 0;
      for (;;)
        {
        int ch = *(++data);
        if (ch == 0 || ch == '\"') break;
        if (ch == '\\')
          {
          ch = *(++data);
          if (isdigit(ch) && ch != '8' && ch != '9')
            {
            ch -= '0';
            if (isdigit(data[1]) && data[1] != '8' && data[1] != '9')
              {
              ch = ch * 8 + *(++data) - '0';
              if (isdigit(data[1]) && data[1] != '8' && data[1] != '9')
                ch = ch * 8 + *(++data) - '0';
              }
            }
          else switch (ch)
            {
            case 0:    data--;  /* Fall through */
            case '\\': ch = '\\'; break;
            case 'n':  ch = '\n'; break;
            case 'r':  ch = '\r'; break;
            case 't':  ch = '\t'; break;
            case 'x':
            ch = 0;
            if (isxdigit(data[1]))
              {
              ch = ch * 16 +
                strchr(hex_digits, tolower(*(++data))) - hex_digits;
              if (isxdigit(data[1])) ch = ch * 16 +
                strchr(hex_digits, tolower(*(++data))) - hex_digits;
              }
            break;
            }
          }
        yield[i++] = ch;
        }
      yield[i] = 0;
      }

    /* Not a quoted string */
    else
      {
      strncpy(yield, data, datalength);
      yield[datalength] = 0;
      }
    return yield;
    }

  /* Move on to next subkey */

  while (isspace(*s)) s++;
  }

return NULL;
}



/*************************************************
*               Find value of a variable         *
*************************************************/

/* The table of variables is kept in alphabetic order, so we
can search it using a binary chop. The "choplen" variable is
nothing to do with the binary chop. It can be set non-zero to
cause chars at the end of the returned string to be disregarded.
It should already be zero on entry. */

static char *find_variable(char *name, int *choplen)
{
int first = 0;
int last = var_table_size;

while (last > first)
  {
  header_line *h;
  char *s;
  char **ss;
  int middle = (first + last)/2;
  int c = strcmp(name, var_table[middle].name);

  if (c == 0) switch (var_table[middle].type)
    {
    case vtype_int:
    sprintf(var_buffer, "%d", *(int *)(var_table[middle].value)); /* Integer */
    return var_buffer;

    case vtype_string:                         /* String */
    return (char *)(var_table[middle].value);

    case vtype_stringptr:                      /* Pointer to string */
    s = *((char **)(var_table[middle].value));
    return (s == NULL)? "" : s;

    case vtype_msgbody:                        /* Pointer to msgbody string */
    ss = (char **)(var_table[middle].value);
    if (*ss == NULL && deliver_datafile >= 0)  /* Read body when needed */
      {
      char *body;
      int len = message_body_visible;
      if (len > message_size) len = message_size;
      *ss = body = store_malloc(len+1);
      body[0] = 0;
      lseek(deliver_datafile, 0, SEEK_SET);
      len = read(deliver_datafile, body, len);
      if (len >= 0) body[len] = 0;
      while (*body != 0) 
        {
        if (*body == '\n') *body = ' ';
        body++;
        }  
      }
    return (*ss == NULL)? "" : *ss;

    case vtype_todbsdin:                       /* BSD inbox time of day */
    return tod_stamp(tod_bsdin);

    case vtype_todf:                           /* Full time of day */
    return tod_stamp(tod_full);

    case vtype_todl:                           /* Log format time of day */
    return tod_stamp(tod_log);

    case vtype_reply:                          /* Get reply address */
    s = NULL;
    for (h = header_list; h != NULL; h = h->next)
      {
      if (h->type == htype_replyto) s = strchr(h->text, ':') + 1;
      if (h->type == htype_from && s == NULL) s = strchr(h->text, ':') + 1;
      }
    if (s == NULL) return "";
    while (isspace(*s)) s++; 
    if (choplen != NULL) *choplen = 1;     /* Disregard final \n in header */
    return s;
    }

  else if (c > 0) first = middle + 1;
  else last = middle;
  }

return NULL;
}




/*************************************************
*          Find the value of a header            *
*************************************************/

/* NULL is returned if the header does not exist. The common case
involves only one header, so no need to get store. Use a flag to
distinguish so the caller can free it. It's set FALSE on entry. */

static char *find_header(char *name, BOOL *freestore)
{
int len = (int)strlen(name);
char *yield = NULL;
header_line *h;
for (h = header_list; h != NULL; h = h->next)
  {
  if (h->type != htype_old)
    {
    if (len <= h->slen && strncmpic(name, h->text, len) == 0)
      {
      if (yield == NULL)
        {
        yield = h->text + len;
        while (isspace(*yield)) yield++;
        }
      else
        {
        char *newyield = store_malloc((int)strlen(yield) + h->slen - len);
        strcpy(newyield, yield);
        strcat(newyield, h->text + len);
        if (*freestore) store_free(yield); else *freestore = TRUE;
        yield = newyield;
        }
      }
    }
  }
return yield;
}




/*************************************************
*            Read a condition                    *
*************************************************/

static char *eval_condition(char *s, BOOL *yield)
{
BOOL testfor = TRUE;
char name[256];

for (;;)
  {
  while (isspace(*s)) s++;
  if (*s == '!') { testfor = !testfor; s++; } else break;
  }

s = read_name(name, 256, s);

/* def: tests for a non-zero or non-NULL variable */

if (strcmp(name, "def") == 0 && *s == ':')
  {
  char *value;
  s = read_name(name, 256, s+1);
  value = find_variable(name, NULL);
  if (value == NULL)
    {
    expand_string_message = string_sprintf("unknown variable: %s", name);
    return NULL;
    }
  *yield = (value[0] != 0 && strcmp(value, "0") != 0) == testfor;
  return s;
  }
  
/* exists: tests for file existence */

else if (strcmp(name, "exists") == 0)
  {
  char *filename; 
  struct stat statbuf; 
  while (isspace(*s)) s++;
  if (*s != '{') goto COND_FAILED_CURLY;
  filename = expand_string_internal(s+1, TRUE, &s, FALSE);
  if (filename == NULL) goto COND_FAILED;
  if (*s++ != '}') goto COND_FAILED_CURLY;
  *yield = (stat(filename, &statbuf) == 0) == testfor;
  store_free(filename); 
  return s;  
  }  

/* eq: tests for string equality */

else if (strcmp(name, "eq") == 0)
  {
  int i;
  char *sub[2];
  for (i = 0; i < 2; i++)
    {
    while (isspace(*s)) s++;
    if (*s != '{') goto COND_FAILED_CURLY;
    sub[i] = expand_string_internal(s+1, TRUE, &s, FALSE);
    if (sub[i] == NULL) goto COND_FAILED;
    if (*s++ != '}') goto COND_FAILED_CURLY;
    }
  *yield = (strcmp(sub[0], sub[1]) == 0) == testfor;
  store_free(sub[0]);
  store_free(sub[1]);
  return s;
  }

/* and/or: computes logical and/or of several conditions */

else if (strcmp(name, "or") == 0 || strcmp(name, "and") == 0)
  {
  BOOL and = strcmp(name, "and") == 0;
  int comb = and;
  while (isspace(*s)) s++;
  if (*s++ != '{') goto COND_FAILED_CURLY;

  for (;;)
    {
    BOOL temp;
    while (isspace(*s)) s++;
    if (*s != '{') break;
    s = eval_condition(s+1, &temp);
    if (s == NULL) return NULL;
    while (isspace(*s)) s++;
    if (*s++ != '}') goto COND_FAILED_CURLY;
    if (and) comb &= temp; else comb |= temp;
    }

  if (*s++ != '}') goto COND_FAILED_CURLY;
  *yield = (comb == testfor);
  return s;
  }


/* Unknown type of condition */

COND_FAILED:
expand_string_message = "unknown condition";
return NULL;

COND_FAILED_CURLY:
expand_string_message = "missing or misplaced { or }";
return NULL;
}




/*************************************************
*                 Expand string                  *
*************************************************/

/* Returns the expanded string in malloc'ed store. Interpreted
sequences are:

   \c                    where c is any character -> copies c literally
   $name                 substitutes the variable
   ${name}               ditto
   ${op:string}          operates on the string value
   ${extract {key} {string}}
                         extracts keyed substring; null if not found
   ${if cond {s1} {s2}}  conditional; the yielded string is expanded
                         {s2} can be replaced by "fail" or be omitted
   ${lookup{key}search-type{file}{found}{not-found}}
                         the key, file, & strings are expanded; $value available
                         and {not-found} can be replaced by "fail" or be
                         omitted.
                         The key can in fact consist of mainkey:subkey, in
                         which case a subfield is extracted from the found
                         string, which must consist of key=value pairs.

Operators:
   lc                      lowercase the string
   length_<n>              take first n chars only
   l_<n>                   ditto  

Conditions:
   !cond                   negates condition
   def:variable            variable is defined and not empty
   exists {string}         file exists 
   eq {string1}{string2}   strings are equal, case included
   or {{cond1}{cond2}...}  as it says
   and {{cond1}{cond2}...} ditto

We use an internal routine recursively to handle embedded substrings. The
external function follows. The yield is NULL if the expansion failed, and there
are two cases: if something collapsed syntactically, or if "fail" was given
as the action on a lookup failure. These can be distinguised by looking at the
variable expand_string_forcedfail, which is TRUE in the latter case.

The skipping flag is set true when expanding a substring that isn't actually
going to be used (after "if" or "lookup") and it prevents lookups from
happening lower down. */

static char *expand_string_internal(char *string, BOOL ket_ends, char **left,
  BOOL skipping)
{
int ptr = 0;
int size = sizeof(string) + 50;
char *yield = (char *)store_malloc(size);
char *s = string;

expand_string_forcedfail = FALSE;
expand_string_message = "";

while (*s != 0)
  {
  char *value;
  char name[256];

  /* \ escapes the next character, which must exist, or else
  the expansion fails. */

  if (*s == '\\')
    {
    if (*(++s) == 0)
      {
      expand_string_message = "\\ at end of string";
      goto EXPAND_FAILED;
      }
    yield = string_cat(yield, &size, &ptr, s++, 1);
    continue;
    }

  /* Anything other than $ is just copied verbatim, unless we are
  looking for a terminating } character. */

  if (ket_ends && *s == '}') break;

  if (*s != '$')
    {
    yield = string_cat(yield, &size, &ptr, s++, 1);
    continue;
    }

  /* No { after the $ - must be a plain name or a number for string
  match variable. There has to be a fudge for variables that are the
  names of header fields preceded by "$header_" because header field
  names can contain any printing characters except space and colon.
  For those that don't like typing this much, "$h_" is a synonym for
  "$header_". A non-existent header yields a NULL value; nothing is
  inserted. */

  if (isalpha(*(++s)))
    {
    BOOL freestore = FALSE;
    int choplen = 0;
    s = read_name(name, 256, s);
    if (strncmp(name, "header_", 7) == 0 || strncmp(name, "h_", 2) == 0)
      {
      s = read_header_name(name, 256, s);
      value = find_header(name, &freestore);
      choplen = 1;
      }
    else
      {
      value = find_variable(name, &choplen);
      if (value == NULL)
        {
        expand_string_message = string_sprintf("unknown variable: %s", name);
        goto EXPAND_FAILED;
        }
      }
    if (value != NULL)
      {
      int len = (int)strlen(value) - choplen;
      if (len < 0) len = 0;
      yield = string_cat(yield, &size, &ptr, value, len);
      if (freestore) store_free(value);
      }
    continue;
    }

  if (isdigit(*s))
    {
    int n;
    s = read_number(&n, s);
    if (n <= expand_nmax)
      yield = string_cat(yield, &size, &ptr, expand_nstring[n],
        expand_nlength[n]);
    continue;
    }

  /* Otherwise, if there's no '{' after $ it's an error. */

  if (*s != '{')
    {
    expand_string_message = "$ not followed by letter, digit, or {";
    goto EXPAND_FAILED;
    }

  /* After { there can be various things, but they all start with
  an initial word, except for a number for a string match variable. */

  if (isdigit(*(++s)))
    {
    int n;
    s = read_number(&n, s);
    if (*s++ != '}')
      {
      expand_string_message = "} expected after number";
      goto EXPAND_FAILED;
      }
    if (n <= expand_nmax)
      yield = string_cat(yield, &size, &ptr, expand_nstring[n],
        expand_nlength[n]);
    continue;
    }

  if (!isalpha(*s))
    {
    expand_string_message = "letter or digit expected after ${";
    goto EXPAND_FAILED;
    }
  s = read_name(name, 256, s);

  /* Handle conditionals */

  if (strcmp(name, "if") == 0)
    {
    BOOL cond;
    char *sub1, *sub2;
    s = eval_condition(s, &cond);
    if (s == NULL) goto EXPAND_FAILED;  /* message already set */

    /* The condition must be followed by one or two substrings,
    enclosed in braces, with optional space between them and
    after them. Then there must be a final } to end. */

    while (isspace(*s)) s++;
    if (*s != '{') goto EXPAND_FAILED_CURLY;
    sub1 = expand_string_internal(s+1, TRUE, &s, FALSE);
    if (sub1 == NULL) goto EXPAND_FAILED;
    if (*s++ != '}') goto EXPAND_FAILED_CURLY;

    while (isspace(*s)) s++;
    if (*s == '}') sub2 = NULL;
    else if (*s == '{')
      {
      sub2 = expand_string_internal(s+1, TRUE, &s, FALSE);
      if (sub2 == NULL) goto EXPAND_FAILED;
      if (*s++ != '}') goto EXPAND_FAILED_CURLY;
      }
    else
      {
      s = read_name(name, 256, s);
      if (strcmp(name, "fail") == 0)
        {
        if (!cond)
          {
          expand_string_message = "if failed and \"fail\" requested";
          expand_string_forcedfail = TRUE;
          goto EXPAND_FAILED;
          }
        else sub2 = NULL;
        }
      else
        {
        expand_string_message = "syntax error in \"else\" substring";
        goto EXPAND_FAILED;
        }
      }

    while (isspace(*s)) s++;
    if (*s++ != '}') goto EXPAND_FAILED_CURLY;

    /* Add the appropriate string to the result */

    if (cond)
      yield = string_cat(yield, &size, &ptr, sub1, (int)strlen(sub1));
    else if (sub2 != NULL)
      yield = string_cat(yield, &size, &ptr, sub2, (int)strlen(sub2));

    store_free(sub1);
    if (sub2 != NULL) store_free(sub2);
    continue;
    }

  /* Handle database lookups. If "skipping" is TRUE, we are expanding an
  internal string that isn't actually going to be used. All we need to
  do is check the syntax, so don't do a lookup at all. */

  if (strcmp(name, "lookup") == 0)
    {
    void *handle;
    char *key, *filename, *sub1, *sub2;
    int stype;

    /* Get the key we are to look up. */

    while (isspace(*s)) s++;
    if (*s != '{') goto EXPAND_FAILED_CURLY;
    key = expand_string_internal(s+1, TRUE, &s, FALSE);
    if (key == NULL) goto EXPAND_FAILED;
    if (*s++ != '}') goto EXPAND_FAILED_CURLY;
    while (isspace(*s)) s++;

    /* Find out the type of database */

    if (!isalpha(*s))
      {
      expand_string_message = "missing lookup type";
      goto EXPAND_FAILED;
      }
    s = read_name(name, 256, s);

    if (strcmp(name, "lsearch") == 0) stype = stype_lsearch;
    else if (strcmp(name, "dbm") == 0) stype = stype_dbm;
    else if (strcmp(name, "nis") == 0) 
      {
      if (have_nis) stype = stype_nis; else
        {
        expand_string_message = "lookup type \"nis\" unavailable";
        goto EXPAND_FAILED;  
        }  
      } 
    else if (strcmp(name, "nis0") == 0) 
      {
      if (have_nis) stype = stype_nis0; else
        {
        expand_string_message = "lookup type \"nis0\" unavailable";
        goto EXPAND_FAILED;  
        }  
      } 
    else
      {
      expand_string_message = string_sprintf("unknown lookup type \"%s\"",
        name);
      goto EXPAND_FAILED;
      }

    /* Get the file name */

    while (isspace(*s)) s++;
    if (*s != '{') goto EXPAND_FAILED_CURLY;
    filename = expand_string_internal(s+1, TRUE, &s, FALSE);
    if (filename == NULL) goto EXPAND_FAILED;
    if (*s++ != '}') goto EXPAND_FAILED_CURLY;
    while (isspace(*s)) s++;

    /* If skipping, don't do the next bit - it has the result of
    leaving lookup_value == NULL, as if the entry was not found. */

    if (!skipping)
      {
      handle = search_open(filename, stype, 0, NULL, NULL,
        &expand_string_message, &expand_tree);
      if (handle == NULL) goto EXPAND_FAILED;
      lookup_value = search_find(handle, filename, key, stype);
      }

    /* The key and file name strings are no longer needed */

    store_free(key);
    store_free(filename);

    /* Expand the first substring; $value will get the looked up value. */

    while (isspace(*s)) s++;
    if (*s != '{') goto EXPAND_FAILED_CURLY;
    sub1 = expand_string_internal(s+1, TRUE, &s, lookup_value == NULL);
    if (sub1 == NULL) goto EXPAND_FAILED;
    if (*s++ != '}') goto EXPAND_FAILED_CURLY;

    /* If the lookup succeeded, add the new string to the output. */

    if (lookup_value != NULL)
      yield = string_cat(yield, &size, &ptr, sub1, (int)strlen(sub1));

    /* There now follows either another substring, or "fail", or nothing. */

    while (isspace(*s)) s++;
    if (*s == '}') sub2 = NULL;
    else if (*s == '{')
      {
      sub2 = expand_string_internal(s+1, TRUE, &s, lookup_value != NULL);
      if (sub2 == NULL) goto EXPAND_FAILED;
      if (*s++ != '}') goto EXPAND_FAILED_CURLY;
      if (lookup_value == NULL)
        yield = string_cat(yield, &size, &ptr, sub2, (int)strlen(sub2));
      }

    /* If the word "fail" is present, and lookup failed, set a flag indicating
    it was a forced failure rather than a syntactic error. */

    else
      {
      s = read_name(name, 256, s);
      if (strcmp(name, "fail") == 0)
        {
        if (lookup_value == NULL)
          {
          expand_string_message = "lookup failed and \"fail\" requested";
          expand_string_forcedfail = TRUE;
          goto EXPAND_FAILED;
          }
        else sub2 = NULL;
        }
      else
        {
        expand_string_message = "syntax error in \"not found\" substring";
        goto EXPAND_FAILED;
        }
      }

    while (isspace(*s)) s++;
    if (*s++ != '}') goto EXPAND_FAILED_CURLY;

    /* Free store and continue */

    store_free(sub1);
    if (sub2 != NULL) store_free(sub2);
    if (lookup_value != NULL)
      {
      store_free(lookup_value);
      lookup_value = NULL;
      }

    continue;
    }

  /* Handle keyed substring extraction */

  if (strcmp(name, "extract") == 0)
    {
    int i;
    char *sub[3];
    for (i = 0; i < 2; i++)
      {
      while (isspace(*s)) s++;
      if (*s != '{') goto EXPAND_FAILED_CURLY;
      sub[i] = expand_string_internal(s+1, TRUE, &s, FALSE);
      if (sub[i] == NULL) goto EXPAND_FAILED;
      if (*s++ != '}') goto EXPAND_FAILED_CURLY;
      }
    while (isspace(*s)) s++;
    if (*s++ != '}') goto EXPAND_FAILED_CURLY;
    sub[2] = expand_getkeyed(sub[0], sub[1]);
    if (sub[2] != NULL)
      {
      yield = string_cat(yield, &size, &ptr, sub[2], (int)strlen(sub[2]));
      store_free(sub[2]);
      }
    store_free(sub[0]);
    store_free(sub[1]);
    continue;
    }

  /* Handle operations on a subsequent string */

  if (*s == ':')
    {
    char *sub = expand_string_internal(s+1, TRUE, &s, FALSE);
    if (sub == NULL) goto EXPAND_FAILED;
    s++;

    /* lc lowercases */
     
    if (strcmp(name, "lc") == 0)
      {
      int count = 0;
      char *t = sub - 1;
      while (*(++t) != 0) { *t = tolower(*t); count++; }
      yield = string_cat(yield, &size, &ptr, sub, count);
      store_free(sub);
      continue;
      }
      
    /* length_n or l_n takes just the first n characters or the whole
    string, whichever is the shorter. */
    
    if (strncmp(name, "length_", 7) == 0 || strncmp(name, "l_", 2) == 0)
      {
      int len = 0; 
      int sublen = (int)strlen(sub); 
      char *num = strchr(name, '_') + 1;
      while (*num != 0)
        {
        if (!isdigit(*num))
          {
          expand_string_message =
            string_sprintf("non-digit after underscore in \"%s\"", name);
          goto EXPAND_FAILED;
          }       
        len = len*10 + *num++ - '0';
        }  
      if (sublen > len) sub[len] = 0; else len = sublen;
      yield = string_cat(yield, &size, &ptr, sub, len);
      store_free(sub);
      continue; 
      }   

    /* Unknown operator */
     
    expand_string_message =
      string_sprintf("unknown expansion operator \"%s\"", name);
    goto EXPAND_FAILED;
    }

  /* Handle a plain name */

  if (*s++ == '}')
    {
    BOOL freestore = FALSE;
    int choplen = 0;
    if (strncmp(name, "header_", 7) == 0 || strncmp(name, "h_", 2) == 0)
      {
      s = read_header_name(name, 256, s);
      value = find_header(name, &freestore);
      choplen = 1;
      }
    else
      {
      value = find_variable(name, &choplen);
      if (value == NULL)
        {
        expand_string_message = string_sprintf("unknown variable: %s", name);
        goto EXPAND_FAILED;
        }
      }
    if (value != NULL)
      {
      yield = string_cat(yield, &size, &ptr, value,
        (int)strlen(value) - choplen);
      if (freestore) store_free(value);
      }
    continue;
    }

  /* Else there's something wrong */

  goto EXPAND_FAILED;
  }

/* Expansion succeeded; add a terminating zero and return the
expanded string. If left != NULL, return a pointer to the terminator. */

yield[ptr] = 0;
if (left != NULL) *left = s;
return yield;

/* This is the failure exit: easiest to program with a goto. */

EXPAND_FAILED_CURLY:
expand_string_message = "missing or misplaced { or }";

EXPAND_FAILED:
store_free(yield);
return NULL;
}

/* This is the external function call. */

char *expand_string(char *string)
  { return expand_string_internal(string, FALSE, NULL, FALSE); }



/*************************************************
**************************************************
*             Stand-alone test program           *
**************************************************
*************************************************/

#ifdef STAND_ALONE
int main(void)
{
char buffer[256];

printf("Testing string expansion\n");

expand_nstring[1] = "string 1....";
expand_nlength[1] = 8;
expand_nmax = 1;

while (fgets(buffer, 256, stdin) != NULL)
  {
  char *yield = expand_string(buffer);
  if (yield != NULL)
    {
    printf("%s", yield);
    store_free(yield);
    }
  else printf("Failed: %s\n", expand_string_message);
  }

return 0;
}

#endif

/* End of expand.c */
