/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */


#include "../exim.h"
#include "aliasfile.h"



/* Options specific to the aliasfile director. */

optionlist aliasfile_director_options[] = {
  { "*expand_group",      opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, expand_gid)) },
  { "*expand_user",       opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, expand_uid)) },
  { "directory",          opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, directory)) },      
  { "errors_to",          opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, errors_to)) },
  { "file",               opt_stringptr,
      (void *)(offsetof(aliasfile_director_options_block, file)) },
  { "forbid_file",        opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, forbid_file)) },
  { "forbid_pipe",        opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, forbid_pipe)) },
  { "freeze_missing_include", opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, freeze_missing_include)) },
  { "group",              opt_expand_gid,
      (void *)(offsetof(aliasfile_director_options_block, gid)) },
  { "modemask",           opt_octint,
      (void *)(offsetof(aliasfile_director_options_block, modemask)) },
  { "optional",           opt_bool,
      (void *)(offsetof(aliasfile_director_options_block, optional)) },
  { "owners",             opt_uidlist,
      (void *)(offsetof(aliasfile_director_options_block, owners)) },
  { "owngroups",          opt_gidlist,
      (void *)(offsetof(aliasfile_director_options_block, owngroups)) },
  { "search_type",        opt_searchtype,
      (void *)(offsetof(aliasfile_director_options_block, search_type)) },
  { "user",               opt_expand_uid,
      (void *)(offsetof(aliasfile_director_options_block, uid)) }
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int aliasfile_director_options_count =
  sizeof(aliasfile_director_options)/sizeof(optionlist);

/* Default private options block for the aliasfile director. */

aliasfile_director_options_block aliasfile_director_option_defaults = {
  NULL,     /* handle */
  NULL,     /* file */
  NULL,     /* directory */ 
  NULL,     /* errors_to */
  NULL,     /* expand_uid */
  NULL,     /* expand_gid */
  -1,       /* uid */
  -1,       /* gid */
  022,      /* modemask */
  -1,       /* search_type */
  NULL,     /* owners */
  NULL,     /* owngroups */
  FALSE,    /* optional */
  FALSE,    /* forbid_file */
  FALSE,    /* forbid_pipe */
  TRUE      /* freeze_missing_include */
};



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up. */

void aliasfile_director_init(director_instance *dblock)
{
aliasfile_director_options_block *ob =
  (aliasfile_director_options_block *)(dblock->options_block);

/* If a fixed uid field is set, then a gid field must also be set. */

if (ob->uid >= 0 && ob->gid < 0)
  log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
    "user set without group for the %s director", dblock->name);

/* A search type is mandatory */

if (ob->search_type < 0)
  log_write(LOG_PANIC_DIE, "Exim configuration error for %s director:\n  "
    "a search type option is required", dblock->name);

/* An absolute file name is mandatory for lsearch and dbm; but can't check for
absoluteness if the name is being looked up. */

if (ob->file == NULL)
  log_write(LOG_PANIC_DIE, "Exim configuration error for %s director:\n  "
    "no file name specified", dblock->name);

if ((ob->search_type == stype_lsearch || ob->search_type == stype_dbm) &&
     ob->file[0] != '/' && ob->file[0] != '$')
  log_write(LOG_PANIC_DIE, "Exim configuration error for %s director:\n  "
    "an absolute file path name is required for lsearch or dbm", dblock->name);
}




/*************************************************
*             Tidyup entry point                 *
*************************************************/

/* Called when all directing is finished, to enable this director
to leave files open until this time, for efficiency. */

void aliasfile_director_tidyup(director_instance *dblock)
{
aliasfile_director_options_block *ob =
  (aliasfile_director_options_block *)(dblock->options_block);

if (ob->handle != NULL)
  {
  search_close(ob->handle, ob->search_type);
  ob->handle = NULL;
  }

DEBUG(9) debug_printf("aliasfile tidyup run\n");
}




/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for interface description. */

int aliasfile_director_entry(
  director_instance *dblock,      /* data for this instantiation */
  address_item *addr,             /* address we are working on */
  address_item **addr_local,      /* add it to this if it's local */
  address_item **addr_remote,     /* add it to this if it's remote */
  address_item **addr_new,        /* put new addresses on here */
  BOOL verify)                    /* true if verifying */
{
aliasfile_director_options_block *ob =
  (aliasfile_director_options_block *)(dblock->options_block);
address_item *generated = NULL;
char *errors_to = addr->errors_address;
char *filename;
char *aliastext;
char *error;
int  uid = ob->uid;
int  gid = ob->gid;
int  extracted;
BOOL uid_ok = FALSE;
BOOL gid_ok = FALSE;

/* Do file existence tests */

if (!match_exists(dblock->require_files))
  {
  DEBUG(9) debug_printf("%s director skipped: file existence failure\n",
    dblock->name);
  return FAIL;   
  }

/* Get the required file name and expand it. If the expansion fails, log the
incident and indicate an internal error. The file name has already been checked
for absoluteness, at initialization time, but only if it did not start with an
expansion, so we double check here. */

filename = expand_string(ob->file);
if (filename == NULL)
  {
  log_write(LOG_MAIN|LOG_PANIC, "%s director failed to expand %s: %s",
    dblock->name, ob->file, expand_string_message);
  addr->message = string_sprintf("%s director failed to expand %s: %s",
    dblock->name, ob->file, expand_string_message);
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }
else if ((ob->search_type == stype_lsearch || ob->search_type == stype_dbm) &&
         filename[0] != '/')
  {
  log_write(LOG_MAIN|LOG_PANIC, "%s director requires absolute file name "
    "for lsearch or dbm: "
    "%s generated from expanding %s", dblock->name, filename, ob->file);
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }

DEBUG(2) debug_printf("%s director: file = %s search type = %d\n",
  dblock->name, filename, ob->search_type);


/* Open the file (or whatever) for searching, according to the search type that
is set. The search functions return a handle identifying the search. For files
this is a FILE * or a DBM *; for other things is is < 0. If this director
has been called earlier for this message, the search database may already be
open, in which case no checks are necessary. If the optional flag is set,
failure to open is not an error; we just fail to direct. */

if (ob->handle == NULL)
  {
  ob->handle = search_open(filename, ob->search_type,
    ob->modemask, ob->owners, ob->owngroups, &error, NULL);
  if (ob->handle == NULL)
    {
    if (ob->optional)
      {
      DEBUG(2) debug_printf("%s director skipped: file missing and optional "
        "flag set\n", dblock->name);
      return FAIL;
      }
    addr->errno = ERRNO_BADALIAS;
    addr->message = string_sprintf("%s director: %s", dblock->name, error);
    log_write(LOG_MAIN|LOG_PANIC, "%s director: %s", dblock->name, error);
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }
  }

/* Now search the file (or whatever) for the entry we are interested in.
The text is returned in dynamic store. */

aliastext = search_find(ob->handle, filename, addr->local_part, 
  ob->search_type);

if (aliastext == NULL)
  {
  DEBUG(2) debug_printf("%s director failed for %s\n", dblock->name,
    addr->local_part);
  return FAIL;
  }

/* If there is no fixed uid set, see if there's a dynamic one that can
be expanded and possibly looked up. */

if (uid < 0 && ob->expand_uid != NULL)
  {
  struct passwd *pw = direct_find_expanded_user(ob->expand_uid,
    dblock->name, "director", &uid);
  if (pw != NULL) gid = pw->pw_gid;

  if (gid < 0 && ob->expand_gid != NULL)
    (void) direct_find_expanded_group(ob->expand_gid, dblock->name,
      "director", &gid);

  /* If no gid has been set, it is a disaster. */

  if (gid < 0)
    log_write(LOG_PANIC_DIE, "User set without group for %s director",
      dblock->name);
  }

/* If this director has a local errors_to setting for where to send error
messages for its children, expand it, and then check that it is a valid
address before using it, except when just verifying an address. Otherwise
there could be directing loops if someone sets up a silly configuration. */

if (ob->errors_to != NULL)
  {
  char *s = expand_string(ob->errors_to);
  if (s == NULL)
    {
    log_write(LOG_MAIN, "%s director failed to expand %s", dblock->name,
      ob->errors_to);
    addr->special_action = SPECIAL_FREEZE;
    return ERROR;
    }

  if (verify) errors_to = s; else
    {
    char *snew;  
    if (verify_address(s, TRUE, TRUE, NULL, NULL, &snew, FALSE, FALSE) == OK)
      errors_to = snew;
    }   
  }

/* If there is a transport specified for the director, then set up this
address to use that transport. Ignore the alias text. */

if (dblock->transport != NULL)
  {
  addr->transport = dblock->transport;

  if (errors_to != NULL) addr->errors_address = errors_to;
  addr->director = dblock;
  addr->home_dir = ob->directory; 
  if (uid >= 0) addr->uid = uid;
  if (gid >= 0) addr->gid = gid;

  if (addr->transport->info->local)
    {
    addr->next = *addr_local;
    *addr_local = addr;
    }

  else
    {
    addr->next = *addr_remote;
    *addr_remote = addr;
    }

  DEBUG(2) debug_printf("  queued for %s transport uid=%d gid=%d\n",
    dblock->transport->name, addr->uid, addr->gid);

  return OK;
  }

/* There is a common function for use by aliasing and aliasing directors that
extracts a list of addresses from a text string. Setting the second-last
argument FALSE makes generating no addresses an error. Setting the last
argument NULL causes no checks to be made on :include: files. */

extracted = parse_extract_addresses(aliastext, &generated, &error, FALSE, NULL);
store_free(aliastext);

/* If extraction failed, return error and freeze, unless it was a missing
include file and no_freeze_missing_include is set. */

if (extracted != 0)
  {
  addr->errno = ERRNO_BADALIAS;
  addr->message =
    string_sprintf("<%s> - error in alias file: %s", addr->orig, error);
  if (extracted > 0 && !ob->freeze_missing_include) return DEFER;
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }


/* Add the new addresses to the list of new addresses, copying in the
uid, gid and permission flags for use by pipes and files, setting
the parent, and or-ing its ignore_error flag. */

while (generated != NULL)
  {
  address_item *next = generated;
  generated = next->next;
  next->parent = addr;
  next->ignore_error |= addr->ignore_error; 
  addr->child_count++;

  next->next = *addr_new;
  *addr_new = next;

  if (errors_to != NULL) next->errors_address = errors_to;

  if (next->orig[0] == '|' || next->orig[0] == '/')
    {
    next->director = dblock;
    if (uid >= 0) next->uid = uid;
    if (gid >= 0) next->gid = gid;
    next->home_dir = ob->directory; 
    next->allow_pipe = !ob->forbid_pipe;
    next->allow_file = !ob->forbid_file;
    }

  DEBUG(2) 
    {
    debug_printf("%s director generated %s%s%s%s\n",
      dblock->name,
      next->orig,
      (errors_to != NULL)? " (errors to " : "",
      (errors_to != NULL)? errors_to : "",
      (errors_to != NULL)? ")" : "");
    debug_printf("   uid=%d gid=%d home=%s\n", next->uid, next->gid,
      (next->home_dir == NULL)? "null" : next->home_dir);    
    }   
  }

/* If the alias file generated no addresses, it is an error. Compare
forward file, where it is not. */

if (addr->child_count <= 0)
  {
  addr->errno = ERRNO_BADALIAS;
  addr->message =
    string_sprintf("<%s> - no addresses generated by alias file (%s director)\n",
    addr->orig, dblock->name);
  addr->special_action = SPECIAL_FREEZE;
  return ERROR;
  }

return OK;
}

/* End of director/aliasfile.c */
