/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions that operate on the input queue. */


#include "exim.h"



/*************************************************
*              Perform a queue run               *
*************************************************/

/*The arguments give the messages to start and stop at; NULL means start at the
beginning or stop at the end. If the given start message doesn't exist, we
start at the next lexically greater one, and likewise we stop at the after the
previous lexically lesser one if the given stop message doesn't exist. Because
a queue run can take some time, stat each file before forking, in case it has 
been delivered in the meantime by some other means. 

If deliver_selectstring is not NULL, the deliver_message() function skips
messages whose recipients do not contain the string. As this option is used
when a machine comes back online, we want to ensure that at least one
delivery attempt takes place, so force the first one. */

void queue_run(char *start_id, char *stop_id)
{
BOOL force_delivery = deliver_selectstring != NULL;
queue_filename *f;
log_write(LOG_MAIN, "Start queue run: pid=%d", getpid());

/* Turn off the flag that causes SMTP deliveries not to happen. */

queue_smtp = FALSE;

/* Run the queue. If a start or finish id is given, we must take the queue in 
its natural order. Otherwise "randomize" it so we don't always do things in the 
same order. */

for (f = queue_get_spool_list(start_id == NULL && stop_id == NULL); 
     f != NULL; f = f->next)
  {
  f->text[SPOOL_NAME_LENGTH-2] = 0;
  if (stop_id != NULL && strcmp(f->text, stop_id) > 0) break;
  if (start_id == NULL || strcmp(f->text, start_id) >= 0)
    {
    int pid, status, rc;
    struct stat statbuf;
    char buffer[256];
     
    sprintf(buffer, "%s/input/%s-H", spool_directory, f->text);
    if (stat(buffer, &statbuf) < 0) continue; 
     
    set_process_info("running queue: %s", f->text);
    if ((pid = fork()) == 0)
      {
      if (deliver_message(f->text, force_delivery, FALSE)) 
        _exit(EXIT_SUCCESS);
      else _exit(EXIT_FAILURE);   
      }
    if (pid < 0)
      log_write(LOG_PANIC_DIE, "fork of delivery process failed\n"); 
       
    set_process_info("running queue: waiting for %s (%d)", f->text, pid);
    while (wait(&status) != pid);
    set_process_info("running queue");

    /* A successful return means a delivery was attempted; turn off the
    force flag for any subsequent calls. */
       
    if ((status & 0xff00) == 0) force_delivery = FALSE;
    }
  }
log_write(LOG_MAIN, "End queue run: pid=%d", getpid());   
}



/*************************************************
*             Get list of spool files            *
*************************************************/

/* Scan the spool directory and return a list of the relevant file names
therein. If the argument is TRUE, they are returned in "randomized" order.
Actually, the order is anything but random, but the algorithm is cheap, and
the point is simply to ensure that the same order doesn't occur every time, in 
case a particular message is causing a remote MTA to barf - we would like to 
try other messages to that MTA first. */

queue_filename *queue_get_spool_list(BOOL randomize)
{
int flags = 0;
int resetflags = -1;
queue_filename *yield = NULL;
queue_filename *last;
struct dirent *ent;
DIR *dd;
char buffer[256];

sprintf(buffer, "%s/input", spool_directory);
dd = opendir(buffer);
if (dd == NULL) return NULL;

/* The file names are added onto the start or end of the list according to the 
bits of the flags variable. When randomizing, get a collection of bits from the 
current time. Use the bottom 16 and just keep re-using them if necessary. */

if (randomize) resetflags = time(NULL) & 0xFFFF;

/* Now scan the directory. */

while ((ent = readdir(dd)) != NULL)
  {
  char *name = ent->d_name;
  if ((int)strlen(name) == SPOOL_NAME_LENGTH && 
      strcmp(name + SPOOL_NAME_LENGTH - 2, "-H") == 0)
    {
    queue_filename *next =
      store_malloc(sizeof(queue_filename) + (int)strlen(name));
    strcpy(next->text, name);
    if (yield == NULL)
      {
      next->next = NULL;
      yield = last = next;  
      }
    else
      {       
      if (flags == 0) flags = resetflags; 
      if ((flags & 1) == 0)
        {
        next->next = yield;
        yield = next;  
        }
      else
        {      
        next->next = NULL;
        last->next = next;
        last = next;
        }
      flags = flags >> 1;    
      } 
    }
  }
closedir(dd);
return yield;
}




/************************************************
*          List messages on the queue           *
************************************************/

/* We get a list of file names as quickly as possible, then scan each one for
information to output. If any disappear while we are processing, just leave
them out. This function is a top-level function that is obeyed as a result of
the -bp argument. There is no point wasting resources in freeing the store used
for the list of names at the end. However, as there may be a lot of messages on
the queue, we must tidy up the store after reading the headers for each one. */

void queue_list(void)
{
int i;
int now = (int)time(NULL);
queue_filename *f = queue_get_spool_list(FALSE);

for (; f != NULL; f = f->next)
  {
  int rc = spool_read_header(f->text, FALSE); 
  int save_errno = errno; 
  BOOL env_read = (rc == spool_read_OK || rc == spool_read_hdrerror); 
 
  if (env_read)
    { 
    i = (now - received_time)/60;  /* minutes on queue */   
    if (i > 90) 
      {
      i = (i + 30)/60;
      if (i > 72) printf("%2dd ", (i + 12)/24); else printf("%2dh ", i);
      }
    else printf("%2dm ", i);
    } 

  for (i = 0; i < 16; i++) fputc(f->text[i], stdout); 

  if (env_read && sender_address != NULL)
    {
    printf(" <%s>", sender_address);
    store_free(sender_address);
    }
    
  if (rc != spool_read_OK)
    {
    printf("\n    ");  
    if (save_errno == ERRNO_SPOOLFORMAT)
      {
      struct stat statbuf;
      sprintf(big_buffer, "%s/input/%s", spool_directory, f->text);
      if (stat(big_buffer, &statbuf) == 0)
        printf("*** spool format error: size=%d ***", statbuf.st_size); 
      else printf("*** spool format error ***");       
      }  
    else printf("*** spool read error: %s ***", strerror(save_errno));
    if (rc != spool_read_hdrerror)
      {
      printf("\n");   
      continue;
      } 
    }  

  if (deliver_freeze) printf(" *** frozen ***");

  printf("\n");
   
  if (recipients_list != NULL)
    {
    for (i = 0; i < recipients_count; i++)
      {
      printf("  %s %s\n", 
        (tree_search_addr(tree_nonrecipients, recipients_list[i]) == NULL)? 
          " ":"*", recipients_list[i]);
      store_free(recipients_list[i]);
      }
    printf("\n");
    store_free(recipients_list);
    }

  tree_free(tree_nonrecipients);

  /*** Not currently using headers
  while (header_list != NULL)
    {
    header_line *this = header_list;
    header_list = header_list->next;
    printf("%s", this->text); 
    store_free(this);
    }
  ***/
  }
}



/*************************************************
*             Act on a specific message          *
*************************************************/

/* Yield FALSE if there was any problem. Actions that require a list of
addresses make use of argv/argc/recipients_arg. Other actions do not. */

BOOL queue_action(char *id, int action, FILE *f, char **argv, int argc, 
  int recipients_arg)
{
BOOL yield = TRUE;
struct passwd *pw;
char *doing = NULL;
char *username;
char spoolname[256];

/* Open and lock the data file to ensure that no other process is working on 
this message. If the file does not exist, continue only if the action is remove
and the user is an admin user, to allow for tidying up broken states. */

if (!spool_open_datafile(id))
  {
  if (errno == ENOENT)
    {
    yield = FALSE; 
    if (f != NULL) fprintf(f, "Spool data file for %s does not exist\n", id); 
    if (action != MSG_REMOVE || !admin_user) return FALSE; 
    if (f != NULL) fprintf(f, "Continuing to ensure all files removed\n"); 
    }
  else
    {    
    if (f != NULL) fprintf(f, "Message %s is locked\n", id);
    return FALSE;
    }  
  } 

/* Read the spool header file for the message. Again, continue after an
error only in the case of deleting by an administrator. */

sprintf(spoolname, "%s-H", id);
if (spool_read_header(spoolname, TRUE) != spool_read_OK)
  {
  yield = FALSE; 
  if (f != NULL) 
    {
    if (errno != ERRNO_SPOOLFORMAT)  
      fprintf(f, "Spool read error for %s: %s\n", spoolname, strerror(errno));
    else
      fprintf(f, "Spool format error for %s\n", spoolname);
    } 
  if (action != MSG_REMOVE || !admin_user)
    {    
    close(deliver_datafile);
    deliver_datafile = -1; 
    return FALSE;
    } 
  if (f != NULL) fprintf(f, "Continuing to ensure all files removed\n"); 
  }

/* Check that the user running this process is entitled to operate on this
message. Only admin users may freeze/thaw, add/cancel recipients, or otherwise 
mess about, but the original sender is permitted to remove a message. */

if (!admin_user && (action != MSG_REMOVE || real_uid != originator_uid))
  {
  if (f != NULL) fprintf(f, "Permission denied\n");
  close(deliver_datafile); 
  deliver_datafile = -1; 
  return FALSE;
  }   
  
/* Set up the user name for logging. */

pw = getpwuid(real_uid); 
username = (pw != NULL)? pw->pw_name : string_sprintf("uid %d", real_uid);

/* Take the necessary action. */

if (f != NULL) fprintf(f, "Message %s ", id);

switch(action)
  {
  case MSG_FREEZE:
  if (deliver_freeze) 
    { 
    yield = FALSE; 
    if (f != NULL) fprintf(f, "is already frozen\n"); 
    }
  else
    {
    deliver_freeze = TRUE;
    deliver_frozen_at = time(NULL); 
    if (spool_write_header(id) > 0)
      { 
      if (f != NULL) fprintf(f, "is now frozen\n");
      log_write(LOG_MAIN, "%s frozen by %s", id, username);
      }
    else
      {
      yield = FALSE; 
      if (f != NULL) fprintf(f, "could not be frozen: failed to rewrite "
        "spool header file\n");
      log_write(LOG_MAIN, "failed to rewrite header file for %s", id);   
      }      
    }
  break;              

  
  case MSG_THAW:
  if (!deliver_freeze) 
    { 
    yield = FALSE; 
    if (f != NULL) fprintf(f, "is not frozen\n"); 
    }
  else
    {
    deliver_freeze = FALSE;
    if (spool_write_header(id) > 0)
      { 
      if (f != NULL) fprintf(f, "is no longer frozen\n"); 
      log_write(LOG_MAIN, "%s unfrozen by %s", id, username);
      } 
    else
      {
      yield = FALSE; 
      if (f != NULL) fprintf(f, "could not be unfrozen: failed to rewrite "
        "spool header file\n");
      log_write(LOG_MAIN, "failed to rewrite header file for %s", id);   
      }      
    }
  break;              

  
  case MSG_REMOVE:
  sprintf(spoolname, "%s/msglog/%s", spool_directory, id);
  if (unlink(spoolname) < 0)
    {
    if (errno != ENOENT)
      {
      yield = FALSE; 
      if (f != NULL)
        fprintf(f, "Error while removing %s: %s\n", spoolname, 
          strerror(errno));
      }    
    }   
  sprintf(spoolname, "%s/input/%s-D", spool_directory, id);
  if (unlink(spoolname) < 0)
    {
    if (errno != ENOENT)
      {
      yield = FALSE; 
      if (f != NULL)
        fprintf(f, "Error while removing %s: %s\n", spoolname, 
          strerror(errno));
      }     
    }    
  sprintf(spoolname, "%s/input/%s-H", spool_directory, id);
  if (unlink(spoolname) < 0)
    {
    if (errno != ENOENT)
      {
      yield = FALSE; 
      if (f != NULL)
        fprintf(f, "Error while removing %s: %s\n", spoolname, 
          strerror(errno));
      }     
    }    
   
  /* In the common case, the datafile is open (and locked), so give the
  obvious message. Otherwise be more specific. */
    
  if (f != NULL) 
    {
    if (deliver_datafile >= 0) fprintf(f, "has been removed\n"); 
      else fprintf(f, "has been removed or did not exist\n"); 
    } 
     
  log_write(LOG_MAIN, "%s removed by %s", id, username);
  break;
  

  case MSG_EDIT_SENDER:
  if (recipients_arg < argc - 1)
    {
    yield = FALSE;
    if (f != NULL)
      fprintf(f, "- only one sender address can be specified\n");
    break; 
    }
  doing = "editing sender";  
  /* Fall through */ 

  case MSG_ADD_RECIPIENT:
  if (doing == NULL) doing = "adding recipient";
  /* Fall through */
    
  case MSG_MARK_DELIVERED: 
  if (doing == NULL) doing = "marking as received";
 
  /* Common code for EDIT_SENDER, ADD_RECIPIENT, & MARK_DELIVERED */
   
  if (recipients_arg >= argc)
    {
    yield = FALSE;
    if (f != NULL)
      fprintf(f, "- error while %s: no address given\n", doing);
    }
  else 
    {
    for (; recipients_arg < argc; recipients_arg++)
      {
      int start, end, domain;
      char *errmess;
      char *receiver =
        parse_extract_address(argv[recipients_arg], &errmess, &start, &end, 
          &domain, (action == MSG_EDIT_SENDER));
           
      if (receiver == NULL)
        {
        yield = FALSE;
        if (f != NULL)
          fprintf(f, "- error while %s:\n  bad address %s: %s\n", 
            doing, argv[recipients_arg], errmess);
        }
      else if (receiver[0] != 0 && domain == 0)
        {
        yield = FALSE;
        if (f != NULL)
          fprintf(f, "- error while %s:\n  bad address %s: "
            "domain missing\n", doing, argv[recipients_arg]);
        }
      else 
        {
        if (action == MSG_ADD_RECIPIENT) 
          {
          accept_add_recipient(receiver);   
          log_write(LOG_MAIN, "%s recipient <%s> added by %s", id, 
            receiver, username);
          }
        else if (action == MSG_MARK_DELIVERED)
          {
          int i;
          for (i = 0; i < recipients_count; i++)
            if (strcmp(recipients_list[i], receiver) == 0) break;
          if (i >= recipients_count)
            {
            if (f != NULL)
              fprintf(f, "- error while %s:\n  %s is not a recipient:"
                " message not updated\n", doing, receiver);   
            yield = FALSE;     
            }       
          else
            { 
            tree_add_nonrecipient(receiver);    
            log_write(LOG_MAIN, "%s address <%s> marked delivered by %s", id, 
              receiver, username);
            }   
          } 
        else  /* MSG_EDIT_SENDER */
          {
          sender_address = receiver;
          log_write(LOG_MAIN, "%s sender address changed to <%s> by %s", id,
            receiver, username); 
          }     
        } 
      } 

 
    if (yield)
      {
      if (spool_write_header(id) > 0)
        { 
        if (f != NULL) fprintf(f, "has been modified\n"); 
        } 
      else
        {
        yield = FALSE; 
        if (f != NULL) fprintf(f, "- failed to rewrite spool header file "
          "while %s\n", doing);
        log_write(LOG_MAIN, "%s failed to rewrite header file while %s", id,
          doing);   
        }      
      }    
    }               
  break; 
  

  case MSG_EDIT_BODY:
  if (recipients_arg < argc)
    {
    yield = FALSE;
    if (f != NULL)
      fprintf(f, "- only one message can be edited at once\n");
    }
  else
    {    
    sprintf(spoolname, "/bin/sh -c \"${VISUAL:-${EDITOR:-vi}} %s/input/%s-D\"", 
      spool_directory, id);
    if ((system(spoolname) & 0xff00) == 0) 
      {
      if (f != NULL) fprintf(f, "has been modified\n");
      log_write(LOG_MAIN, "%s body edited by %s", id, username);  
      }
    else
      {
      if (f != NULL) fprintf(f, "not modified: editing failed\n");
      yield = FALSE; 
      } 
    }   
  break;  
  }    


/* Closing the datafile releases the lock and permits other processes
to operate on the message (if it still exists). */

close(deliver_datafile);
deliver_datafile = -1; 
return yield;
}

/* End of queue.c */
