/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions for reading the configuration file, and for displaying
overall configuration values. */

#include "exim.h"


static char time_buffer[24];


/*************************************************
*           Main configuration options           *
*************************************************/

/* The list of options that can be set in the main configuration file. This
must be in alphabetic order because it is searched by binary chop. */

static optionlist optionlist_config[] = {
  { "accept_timeout",          opt_time,        &accept_timeout },
  { "address_file_transport",  opt_stringptr,   &address_file_transport },
  { "address_pipe_transport",  opt_stringptr,   &address_pipe_transport },
  { "address_reply_transport", opt_stringptr,   &address_reply_transport }, 
  { "auto_thaw",               opt_time,        &auto_thaw }, 
  { "debug_transport",         opt_stringptr,   &debug_transport_file },
  { "delay_warning",           opt_time,        &delay_warning },
  { "delivery_date_remove",    opt_bool,        &delivery_date_remove }, 
  { "errors_address",          opt_stringptr,   &errors_address },
  { "errors_copy",             opt_stringptr,   &errors_copy }, 
  { "errors_reply_to",         opt_stringptr,   &errors_reply_to },
  { "exim_group",              opt_gid,         &exim_gid },
  { "exim_path",               opt_stringptr,   &exim_path },
  { "exim_user",               opt_uid,         &exim_uid },
  { "finduser_retries",        opt_int,         &finduser_retries },
  { "freeze_tell_mailmaster",  opt_bool,        &freeze_tell_mailmaster },
  { "gecos_name",              opt_stringptr,   &gecos_name },
  { "gecos_pattern",           opt_stringptr,   &gecos_pattern },  
  { "ignore_errmsg_errors",    opt_bool,        &ignore_errmsg_errors }, 
  { "keep_malformed",          opt_time,        &keep_malformed }, 
  { "local_domains",           opt_lcstringptr, &local_domains },
  { "local_domains_include_host", opt_bool,     &local_domains_include_host }, 
  { "log_received_recipients", opt_bool,        &log_received_recipients },
  { "message_body_visible",    opt_mkint,       &message_body_visible }, 
  { "message_id_header_text",  opt_stringptr,   &message_id_text }, 
  { "message_size_limit",      opt_mkint,       &message_size_limit }, 
  { "never_users",             opt_uidlist,     &never_users },
  { "nobody_group",            opt_gid,         &nobody_gid },
  { "nobody_user",             opt_uid,         &nobody_uid },
  { "percent_hack_domains",    opt_lcstringptr, &percent_hack_domains },
  { "preserve_message_logs",   opt_bool,        &preserve_message_logs }, 
  { "primary_hostname",        opt_lcstringptr, &primary_hostname },
  { "qualify_domain",          opt_lcstringptr, &qualify_domain_sender },
  { "qualify_recipient",       opt_lcstringptr, &qualify_domain_recipient },
  { "queue_only",              opt_bool,        &queue_only },
  { "queue_run_max",           opt_int,         &queue_run_max },
  { "queue_smtp",              opt_bool,        &queue_smtp },
  { "received_header_text",    opt_stringptr,   &received_header_text },
  { "received_headers_max",    opt_int,         &received_headers_max },
  { "receiver_try_verify",     opt_bool,        &receiver_try_verify }, 
  { "receiver_unqualified_hosts", opt_stringptr,&receiver_unqualified_hosts }, 
  { "receiver_unqualified_nets", opt_stringptr, &receiver_unqualified_nets }, 
  { "receiver_verify",         opt_bool,        &receiver_verify }, 
  { "retry_interval_max",      opt_time,        &retry_interval_max },
  { "return_path_remove",      opt_bool,        &return_path_remove },
  { "return_size_limit",       opt_mkint,       &return_size_limit }, 
  { "rfc1413_except_hosts",    opt_lcstringptr, &rfc1413_except_hosts },
  { "rfc1413_except_nets",     opt_stringptr,   &rfc1413_except_nets },
  { "rfc1413_query_timeout",   opt_time,        &rfc1413_query_timeout },
  { "security",                opt_lcstringptr, &security_type },
  { "sender_host_accept",      opt_lcstringptr, &sender_host_accept },
  { "sender_host_reject",      opt_lcstringptr, &sender_host_reject },
  { "sender_net_accept",       opt_stringptr,   &sender_net_accept },
  { "sender_net_reject",       opt_stringptr,   &sender_net_reject },  
  { "sender_reject",           opt_stringptr,   &sender_reject }, 
  { "sender_try_verify",       opt_bool,        &sender_try_verify }, 
  { "sender_unqualified_hosts", opt_stringptr,  &sender_unqualified_hosts }, 
  { "sender_unqualified_nets", opt_stringptr,   &sender_unqualified_nets }, 
  { "sender_verify",           opt_bool,        &sender_verify },
  { "sender_verify_except_hosts",opt_lcstringptr,&sender_verify_except_hosts },
  { "sender_verify_except_nets",opt_stringptr,  &sender_verify_except_nets },
  { "sender_verify_fixup",     opt_bool,        &sender_verify_fixup },
  { "sender_verify_log_details", opt_bool,      &sender_verify_log_details }, 
  { "sender_verify_reject",    opt_bool,        &sender_verify_reject }, 
  { "smtp_accept_max",         opt_int,         &smtp_accept_max },
  { "smtp_accept_queue",       opt_int,         &smtp_accept_queue },
  { "smtp_accept_reserve",     opt_int,         &smtp_accept_reserve }, 
  { "smtp_banner",             opt_stringptr,   &smtp_banner },
  { "smtp_connect_backlog",    opt_int,         &smtp_connect_backlog },
  { "smtp_receive_timeout",    opt_time,        &smtp_receive_timeout },
  { "smtp_reserve_hosts",      opt_stringptr,   &smtp_reserve_hosts }, 
  { "smtp_reserve_nets",       opt_stringptr,   &smtp_reserve_nets }, 
  { "smtp_verify",             opt_bool,        &smtp_verify },
  { "spool_directory",         opt_stringptr,   &spool_directory },
  { "trusted_groups",          opt_gidlist,     &trusted_groups },
  { "trusted_users",           opt_uidlist,     &trusted_users },
  { "unknown_login",           opt_stringptr,   &unknown_login },
  { "unknown_username",        opt_stringptr,   &unknown_username }  
};

static int optionlist_config_size =
  sizeof(optionlist_config)/sizeof(optionlist);



/*************************************************
*             Read a name                        *
*************************************************/

/* The yield is the pointer to the next char. Names longer than the
output space are silently truncated. */

char *readconf_readname(char *name, int len, char *s)
{
int p = 0;
while (isspace(*s)) s++;
if (isalpha(*s))
  {
  while (isalnum(*s) || *s == '_')
    {
    if (p < len-1) name[p++] = *s;
    s++;
    }
  }
name[p] = 0;
while (isspace(*s)) s++;
return s;
}




/*************************************************
*          Read a time value                     *
*************************************************/

/* This function is also called from outside, to read argument
time values. The format of a time value is:

  [<n>w][<n>d][<n>h][<n>m][<n>s]

as long as at least one is present. If a format error is encountered,
return a negative value. The value must be terminated by the given
terminator. */

int readconf_readtime(char *s, int terminator)
{
int yield = 0;
for (;;)
  {
  int value, count;
  if (!isdigit(*s)) return -1;
  (void)sscanf(s, "%d%n", &value, &count);
  s += count;
  switch (*s)
    {
    case 'w': value *= 7;
    case 'd': value *= 24;
    case 'h': value *= 60;
    case 'm': value *= 60;
    case 's': s++;
    break;

    default: return -1;
    }
  yield += value;
  if (*s == terminator) return yield;
  }
/* Control never reaches here. */
}



/*************************************************
*          Read a fixed point value              *
*************************************************/

/* The value is returned *1000 */

static int readconf_readfixed(char *s, int terminator)
{
int yield = 0;
int value, count;
if (!isdigit(*s)) return -1;
(void)sscanf(s, "%d%n", &value, &count);
s += count;
yield = value * 1000;
if (*s == '.')
  {
  int m = 100;
  while (isdigit(*(++s)))
    {
    yield += (*s - '0') * m;
    m /= 10;
    }
  }

return (*s == terminator)? yield : (-1);
}



/*************************************************
*            Find option in list                 *
*************************************************/

/* The lists are always in order, so binary chop can be used. */

static optionlist *find_option(char *name, optionlist *ol, int last)
{
int first = 0;
while (last > first)
  {
  int middle = (first + last)/2;
  int c = strcmp(name, ol[middle].name);
  if (c == 0) return ol + middle;
    else if (c > 0) first = middle + 1;
      else last = middle;
  }
return NULL;
}



/*************************************************
*            Handle option line                  *
*************************************************/

/* This function is called from several places to process a line containing the
setting of an option. The first argument is the line to be decoded; it has been
checked not to be empty and not to start with '#'. Trailing newlines and white
space have been removed. The second argument is a pointer to the list of
variable names that are to be recognized, together with their types and
locations, and the third argument gives the number of entries in the list.

The fourth argument is a pointer to a data block. If it is NULL, then the data
values in the options list are absolute addresses. Otherwise, they are byte
offsets in the data block.

The next argument is a FILE variable, to be used if the option contains a
string setting that continues onto subsequent input lines. Any failure in
decoding results in a panic crash. The final argument is the logical name of
the file, to be included in the panic message.

The yield of this function is normally zero. If a string continues onto
multiple lines, then the data value is permitted to be followed by a comma
or a semicolon (for use in drivers) and the yield is that character. */

static int readconf_handle_option(char *buffer, optionlist *oltop, int last,
  void *data_block)
{
int yield = 0;
int ptr = 0;
int offset = 0;
int uid, gid;
BOOL boolvalue = TRUE;
optionlist *ol, *ol2;
char *s = buffer;
char name[64];
char name2[64];

/* There may be leading spaces */

while (isspace(*s)) s++;

/* Read the name of the option, and skip any subsequent white space. */

while (isalnum(*s) || *s == '_')
  {
  if (ptr < sizeof(name)-1) name[ptr++] = *s;
  s++;
  }
name[ptr] = 0;
while (isspace(*s)) s++;

/* Deal with "no_" or "not_" here for booleans */

if (strncmp(name, "no_", 3) == 0)
  {
  boolvalue = FALSE;
  offset = 3;
  }

if (strncmp(name, "not_", 4) == 0)
  {
  boolvalue = FALSE;
  offset = 4;
  }

/* Search the list for the given name. */

ol = find_option(name + offset, oltop, last);
if (ol != NULL)
  {
  int n, count, value;
  BOOL lc = FALSE;
  BOOL freesptr = TRUE;
  transport_instance *tp;
  struct passwd *pw;
  struct group *gr;
  char *sptr;

  /* Types with data values must be followed by '='; the "no[t]_" prefix 
  applies only to boolean values. */

  if (ol->type != opt_bool)
    {
    if (offset != 0)
      log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
        "negation prefix applied to a non-boolean option in line %d",
        config_lineno);
    if (*s != '=')
      log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
        "\"=\" expected after %s in line %d", name, config_lineno);
    }
    
  /* If a boolean wasn't preceded by "no[t]_" it can be followed by = and
  true/false/yes/no */
    
  else if (*s != 0)
    {
    if (offset != 0) 
      log_write(LOG_PANIC_DIE,
        "Exim configuration error:\n  extra characters follow boolean value "
        "for variable %s in line %d", name, config_lineno);
    if (*s != '=')
      log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
        "\"=\" expected after %s in line %d", name, config_lineno);
    }
  
  /* Skip white space after = */
    
  while (isspace(*(++s)));

  /* Now get the data according to the type. */

  switch (ol->type)
    {
    /* If a string value is not enclosed in quotes, it consists of
    the rest of the current line, verbatim. Otherwise, string escapes
    are processed, and the value can be continued onto subsequent lines
    by terminating it with a \, in which case white space at the start of
    the next line is skipped.

    A transport is specified as a string, which is then looked up in the
    list of transports. A search type is specified as one of a number of
    known strings.

    Uids and gids are specified as strings which are then looked up in the
    passwd file. Lists of uids and gids are similarly specified as colon-
    separated strings. */

    case opt_lcstringptr:
    lc = TRUE;

    case opt_stringptr:
    freesptr = FALSE;

    case opt_uid:
    case opt_gid:
    case opt_expand_uid:
    case opt_expand_gid:
    case opt_uidlist:
    case opt_gidlist:
    case opt_searchtype:
    case opt_transportptr:
    case opt_local_smtp: 

    n = (int)strlen(s) + 1;
    sptr = store_malloc(n);
    *sptr = 0;                   /* insurance */ 

    if (*s == '\"')
      {
      int count = 0;

      while (*(++s) != 0)
        {
        /* End of string. Trailing spaces should have been removed, but
        there may be trailing ; or , characters if the string has been
        split over several lines. */

        if (*s == '\"')
          {
          while (isspace(*(++s)));
          if (*s == ';' || *s == ',') yield = *s++;
          if (*s == 0) break;
          log_write(LOG_PANIC_DIE,
            "Exim configuration error:\n  extra characters follow string "
            "for variable %s in line %d", name, config_lineno);
          }

        /* Still in the string. Handle escape characters. */

        else if (*s == '\\')
          {
          if (s[1] == 0)
            {
            /* If we hit eof, nothing is done here, the main loop then
            retries and finds *(++s) == 0 and so exits. */

            while (fgets(buffer, 256, config_file) != NULL)
              {
              int n = (int)strlen(buffer);
              config_lineno++;
              while (n > 0 && isspace(buffer[n-1])) n--;
              buffer[n] = 0;
              s = buffer;
              while (isspace(*s)) s++;  
              if (*s == '#') continue;
              s = buffer - 1;
              while (isspace(s[1])) s++;
              break;
              }
            }

          /* Handle '\' not at end of line */

          else
            {
            char ch = *(++s);
            if (isdigit(ch) && ch != '8' && ch != '9')
              {
              ch -= '0';
              if (isdigit(s[1]) && s[1] != '8' && s[1] != '9')
                {
                ch = ch * 8 + *(++s) - '0';
                if (isdigit(s[1]) && s[1] != '8' && s[1] != '9')
                  ch = ch * 8 + *(++s) - '0';
                }
              }
            else switch(ch)
              {
              case 'n':  ch = '\n'; break;
              case 'r':  ch = '\r'; break;
              case 't':  ch = '\t'; break;
              case 'x':
              ch = 0;
              if (isxdigit(s[1]))
                {
                ch = ch * 16 +
                  strchr(hex_digits, tolower(*(++s))) - hex_digits;
                if (isxdigit(s[1])) ch = ch * 16 +
                  strchr(hex_digits, tolower(*(++s))) - hex_digits;
                }
              break;
              }
            sptr = string_cat(sptr, &n, &count, &ch, 1);
            }
          }

        /* Neither " nor \ so just add to the string */

        else
          {
          if (lc) *s = tolower(*s);
          sptr = string_cat(sptr, &n, &count, s, 1);
          }
        }

      /* End of string scanning loop. Terminate the string; string_cat always
      leaves room */

      sptr[count] = 0;
      }

    /* If not quoted, just take the rest of the line. */

    else
      {
      if (lc)
        {
        int i = 0;
        while (*s != 0) sptr[i++] = tolower(*s++);
        sptr[i] = 0;
        }
      else strcpy(sptr, s);
      }

    /* Having read a string, we now have several different ways of using it,
    depending on the data type, so do another switch. */

    switch (ol->type)
      {
      /* If this was a string, set the variable to point to the new string. */

      case opt_lcstringptr:
      case opt_stringptr:
      if (data_block == NULL)
        *((char **)(ol->value)) = sptr;
      else
        *((char **)((char *)data_block + (long int)(ol->value))) = sptr;
      break;

      /* If it was a transport - look it up in the transports list. */

      case opt_transportptr:
      for (tp = transports; tp != NULL; tp = tp->next)
        {
        if (strcmp(tp->name, sptr) == 0)
          {
          if (data_block == NULL)
            *((transport_instance **)(ol->value)) = tp;
          else
            *((transport_instance **)((char *)data_block + 
              (long int)(ol->value))) = tp;
          break;
          }
        }
      if (tp == NULL)
        log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
          "  %s transport, referred to in line %d, "
          "was not found", sptr, config_lineno);
      break;

      /* If it was an expanded uid, see if there is any expansion to be
      done by checking for the presence of a $ character. If there is, save it
      in the corresponding *expand_user option field. Otherwise, fall through
      to treat it as a fixed uid. Ensure mutual exclusivity of the two kinds
      of data. */

      case opt_expand_uid:
      sprintf(name2, "*expand_%s", name);
      ol2 = find_option(name2, oltop, last);
      if (ol2 != NULL && ol2->type == opt_stringptr)
        {
        char *ss = (strchr(sptr, '$') != NULL)? sptr : NULL;
        if (data_block == NULL)
          *((char **)(ol2->value)) = ss;
        else
          *((char **)((char *)data_block + (long int)(ol2->value))) = ss;
        if (ss != NULL)
          {
          if (data_block == NULL)
            *((int *)(ol->value)) = -1;
          else
            *((int *)((char *)data_block + (long int)(ol->value))) = -1;
          freesptr = FALSE;
          break;
          }
        }

      /* Look up a fixed uid, and also make use of the corresponding gid
      if a passwd entry is returned and the gid has not been set. */

      case opt_uid:
      pw = direct_finduser(sptr, &uid);
      if (uid < 0)
        log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
          "  user %s, referred to in line %d, "
          "was not found", sptr, config_lineno);
      if (data_block == NULL)
        *((int *)(ol->value)) = uid;
      else
        *((int *)((char *)data_block + (long int)(ol->value))) = uid;

      /* Handle matching gid if we have a passwd entry: done by finding the
      same name with terminating "user" changed to "group"; if not found,
      ignore. Also ignore if the value is already set. */

      if (pw == NULL) break;
      strcpy(name+(int)strlen(name)-4, "group");
      ol2 = find_option(name, oltop, last);
      if (ol2 != NULL && (ol2->type == opt_gid || ol2->type == opt_expand_gid))
        {
        if (data_block == NULL)
          {
          if (*((int *)(ol2->value)) < 0)
            *((int *)(ol2->value)) = pw->pw_gid;
          }
        else
          {
          if (*((int *)((char *)data_block + (long int)(ol2->value))) < 0)
            *((int *)((char *)data_block + (long int)(ol2->value))) = pw->pw_gid;
          }
        }
      break;

      /* If it was an expanded gid, see if there is any expansion to be
      done by checking for the presence of a $ character. If there is, save it
      in the corresponding *expand_user option field. Otherwise, fall through
      to treat it as a fixed gid. Ensure mutual exclusivity of the two kinds
      of data. */

      case opt_expand_gid:
      memmove(name+8, name, (int)strlen(name)+1);
      strncpy(name, "*expand_", 8);
      ol2 = find_option(name, oltop, last);
      if (ol2 != NULL && ol2->type == opt_stringptr)
        {
        char *ss = (strchr(sptr, '$') != NULL)? sptr : NULL;
        if (data_block == NULL)
          *((char **)(ol2->value)) = ss;
        else
          *((char **)((char *)data_block + (long int)(ol2->value))) = ss;
        if (ss != NULL)
          {
          if (data_block == NULL)
            *((int *)(ol->value)) = -1;
          else
            *((int *)((char *)data_block + (long int)(ol->value))) = -1;
          freesptr = FALSE;
          break;
          }
        }

      /* Handle freestanding gid */

      case opt_gid:
      gr = direct_findgroup(sptr, &gid);
      if (gid < 0)
        log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
          "  group %s, referred to in line %d, "
          "was not found", sptr, config_lineno);
      if (data_block == NULL)
        *((int *)(ol->value)) = gid;
      else
        *((int *)((char *)data_block + (long int)(ol->value))) = gid;
      break;

      /* If it was a uid or gid list, look up each individual entry, and build
      a vector of uids, terminated by -1. */

      case opt_uidlist:
      case opt_gidlist:
        {
        int count = 2;
        int *list;
        int ptr = 0;
        char *p = sptr;

        while (*p != 0) if (*p++ == ':') count++;
        list = store_malloc(count*sizeof(int));
        if (data_block == NULL)
          *((int **)(ol->value)) = list;
        else
          *((int **)((char *)data_block + (long int)(ol->value))) = list;

        p = sptr;
        while (count-- > 1)
          {
          char *q = strchr(p, ':');
          if (q == NULL) q = p + (int)strlen(p); else *q = 0;

          if (ol->type == opt_uidlist)
            {
            pw = direct_finduser(p, &uid);
            if (uid < 0)
              log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
                "  user %s, referred to in line %d, "
                "was not found", p, config_lineno);
            list[ptr++] = uid;
            }
          else
            {
            gr = direct_findgroup(p, &gid);
            if (gid < 0)
              log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
                "  group %s, referred to in line %d, "
                "was not found", p, config_lineno);
            list[ptr++] = gid;
            }
          p = q+1;
          }
        list[ptr] = -1;
        }
      break;

      /* Search types are known names */

      case opt_searchtype:
        {
        int type;
        if (strcmp(sptr, "lsearch") == 0) type = stype_lsearch;
        else if (strcmp(sptr, "dbm") == 0) type = stype_dbm;
        else if (strcmp(sptr, "nis") == 0) 
          {
          if (have_nis) type = stype_nis;
            else log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
              "search type \"nis\" unavailable (not in binary - see HAVE_NIS) "
              "in line %d", config_lineno); 
          } 
        else if (strcmp(sptr, "nis0") == 0) 
          {
          if (have_nis) type = stype_nis0;
            else log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
              "search type \"nis0\" unavailable (not in binary - see HAVE_NIS) "
              "in line %d", config_lineno); 
          } 
        else log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
          "search type %s unknown in line %d", sptr, config_lineno);
        if (data_block == NULL)
          *((int *)(ol->value)) = type;
        else
          *((int *)((char *)data_block + (long int)(ol->value))) = type;
        }
      break;
      
      /* Local smtp options are known names */
      
      case opt_local_smtp:
        {
        int type; 
        if (strcmp(sptr, "one") == 0) type = local_smtp_one; 
        else if (strcmp(sptr, "domain") == 0) type = local_smtp_domain;
        else if (strcmp(sptr, "all") == 0) type = local_smtp_all;
        else if (strcmp(sptr, "none") == 0) type = local_smtp_off; 
        else log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
          "  local smtp type %s unknown in line %d", sptr, config_lineno);    
        if (data_block == NULL)
          *((int *)(ol->value)) = type;
        else
          *((int *)((char *)data_block + (long int)(ol->value))) = type;
        }
      break;       
      }

    /* Release store for temporary strings */

    if (freesptr) store_free(sptr);
    break;

    /* Boolean: if no characters follow, the value is boolvalue. Otherwise
    look for yes/not/true/false. */

    case opt_bool:
    if (*s != 0)
      {
      s = readconf_readname(name2, 64, s);
      if (strcmpic(name2, "true") == 0 || strcmpic(name2, "yes") == 0)
        boolvalue = TRUE;   
      else if (strcmpic(name2, "false") == 0 || strcmpic(name2, "no") == 0)
        boolvalue = FALSE;   
      else log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
        "\"%s\" is not a valid value for the \"%s\" option in line %d",
        name2, name, config_lineno);     
      }  
    if (data_block == NULL)
      *((BOOL *)(ol->value)) = boolvalue;
    else
      *((BOOL *)((char *)data_block + (long int)(ol->value))) = boolvalue;
    break;

    /*  Integer: a simple(ish) case; allow octal and hex formats, and
    suffixes K and M. The different types affect output, not input. */

    case opt_mkint:
    case opt_octint:
    case opt_int:
    if (sscanf(s, "%i%n", &value, &count) != 1)
      log_write(LOG_PANIC_DIE,
        "Exim configuration error:\n  integer expected "
        "for variable %s in line %d", name, config_lineno);
        
    if (tolower(s[count]) == 'k') { value *= 1024; count++; }
    if (tolower(s[count]) == 'm') { value *= 1024*1024; count++; }
    while (isspace(s[count])) count++; 
 
    if (s[count] != 0)
      log_write(LOG_PANIC_DIE,
        "Exim configuration error:\n  extra characters follow integer value "
        "for variable %s in line %d", name, config_lineno);

    if (data_block == NULL)
      *((int *)(ol->value)) = value;
    else
      *((int *)((char *)data_block + (long int)(ol->value))) = value;
    break;

    /* There's a special routine to read time values. */

    case opt_time:
    value = readconf_readtime(s, 0);
    if (value < 0)
      log_write(LOG_PANIC_DIE,
        "Exim configuration error:\n  invalid time value "
        "for variable %s in line %d", name, config_lineno);
    if (data_block == NULL)
      *((int *)(ol->value)) = value;
    else
      *((int *)((char *)data_block + (long int)(ol->value))) = value;
    break;
    }
  }

/* A non-existent name is a disaster. */

else log_write(LOG_PANIC_DIE,
  "Exim configuration error:\n  \"%s\" unknown in line %d", name,
  config_lineno);

return yield;
}



/*************************************************
*               Print a time value               *
*************************************************/

/* A time given in seconds is returned as a string in the
format of readconf_readtime(). A fixed buffer is used. */

char *readconf_printtime(int t)
{
int s, m, h, d, w;
char *p = time_buffer;

s = t % 60;
t /= 60;
m = t % 60;
t /= 60;
h = t % 24;
t /= 24;
d = t % 7;
w = t/7;

if (w > 0) { sprintf(p, "%dw", w); while (*p) p++; }
if (d > 0) { sprintf(p, "%dd", d); while (*p) p++; }
if (h > 0) { sprintf(p, "%dh", h); while (*p) p++; }
if (m > 0) { sprintf(p, "%dm", m); while (*p) p++; }
if (s > 0 || p == time_buffer) sprintf(p, "%ds", s);

return time_buffer;
}



/*************************************************
*      Print an individual option value          *
*************************************************/

static void print_ol(optionlist *ol, char *name, void *options_block,
  optionlist *oltop, int last)
{
struct passwd *pw;
struct group *gr;
transport_instance *t;
void *value;
char *s;
int  *i;

if (ol == NULL)
  {
  printf("%s is not a known option\n", name);
  return;
  }
  

value = ol->value;
if (options_block != NULL)
  {
  value = (void *)((char *)options_block + (long int)value);
  }

switch(ol->type)
  {
  case opt_lcstringptr:
  case opt_stringptr:
  s = *((char **)value);
  printf("%s = %s\n", name, (s == NULL)? "" : string_printing(s, FALSE));
  break;

  case opt_transportptr:
  t = *((transport_instance **)value);
  printf("%s = %s\n", name, (t == NULL)? "" : t->name);
  break;

  case opt_int:
  printf("%s = %d\n", name, *((int *)value));
  break;
  
  case opt_mkint:
    {
    int x = *((int *)value);
    if (x != 0 && (x & 1023) == 0)
      {
      int c = 'K';
      x >>= 10; 
      if ((x & 1023) == 0)
        {
        c = 'M';
        x >>= 10;
        }
      printf("%s = %d%c\n", name, x, c);        
      }  
    else printf("%s = %d\n", name, x);
    }   
  break;  

  case opt_octint:
  printf("%s = %#o\n", name, *((int *)value));
  break;

  /* If the numerical value is unset, try for the string value */

  case opt_expand_uid:
  if (*((int *)value) < 0)
    {
    char name2[64];
    optionlist *ol2;
    sprintf(name2, "*expand_%s", name);
    ol2 = find_option(name2, oltop, last);
    if (ol2 != NULL && ol2->type == opt_stringptr)
      {
      void *value2 = ol2->value;
      if (options_block != NULL)
        {
        value2 = (void *)((char *)options_block + (long int)value2);
        }
      s = *((char **)value2);
      printf("%s = %s\n", name, (s == NULL)? "" : string_printing(s, FALSE));
      break;
      }
    }
  /* Else fall through */

  case opt_uid:
  pw = getpwuid(*((int *)value));
  if (pw == NULL)
    {
    int uid = *((int *)value);
    if (uid < 0) printf("%s =\n", name); 
      else printf("%s = %d\n", name, uid);
    } 
  else printf("%s = %s\n", name, pw->pw_name);
  break;

  /* If the numerical value is unset, try for the string value */

  case opt_expand_gid:
  if (*((int *)value) < 0)
    {
    char name2[64];
    optionlist *ol2;
    sprintf(name2, "*expand_%s", name);
    ol2 = find_option(name2, oltop, last);
    if (ol2 != NULL && ol2->type == opt_stringptr)
      {
      void *value2 = ol2->value;
      if (options_block != NULL)
        {
        value2 = (void *)((char *)options_block + (long int)value2);
        }
      s = *((char **)value2);
      printf("%s = %s\n", name, (s == NULL)? "" : string_printing(s, FALSE));
      break;
      }
    }
  /* Else fall through */

  case opt_gid:
  gr = getgrgid(*((int *)value));
  if (gr == NULL)
    {
    int gid = *((int *)value);
    if (gid < 0) printf("%s =\n", name);   
      else printf("%s = %d\n", name, gid);
    }   
  else printf("%s = %s\n", name, gr->gr_name);
  break;

  case opt_uidlist:
  case opt_gidlist:
  i = *((int **)value);
  printf("%s =", name);
  if (i != NULL)
    {
    int j = 0;
    while (i[j] >= 0)
      {
      char *name = NULL;
      if (ol->type == opt_uidlist)
        {
        pw = getpwuid(i[j]);
        if (pw != NULL) name = pw->pw_name;
        }
      else
        {
        gr = getgrgid(i[j]);
        if (gr != NULL) name = gr->gr_name;
        }
      if (name != NULL) printf(" %s", name); else printf(" %d", i[j]);
      j++;
      }
    }
  printf("\n");
  break;

  case opt_time:
  printf("%s = %s\n", name, readconf_printtime(*((int *)value)));
  break;

  case opt_bool:
  printf("%s%s\n", (*((BOOL *)value))? "" : "no_", name);
  break;

  case opt_searchtype:
    {
    int type = *((int *)value);
    printf("%s = %s\n", name, 
      (type == stype_lsearch)? "lsearch" :
      (type == stype_dbm)? "dbm" : 
      (type == stype_nis)? "nis" : 
      (type == stype_nis0)? "nis0" : 
      (type == -1)? "" : "?");
    }
  break;
  
  case opt_local_smtp:
    {
    int type = *((int *)value);
    printf("%s = %s\n", name, (type == local_smtp_one)? "one" :
      (type == local_smtp_domain)? "domain" :
      (type == local_smtp_all)? "all" : "none");
    }
  break;         
  }
}



/*************************************************
*        Print value from main configuration     *
*************************************************/

/* This function, called as a result of encountering the -bP option,
causes the value of any main configuration variable to be output if the
second argument is NULL. Otherwise, it must be one of "director", "router",
or "transport", in which case the first name identifies the driver whose
options are to be printed. */

void readconf_print(char *name, char *type)
{
BOOL names_only = FALSE;
optionlist *ol, *ol2;
driver_instance *d;
int size;

if (type == NULL)
  {
  if (strcmp(name, "configure_file") == 0)
    {
    printf("%s\n", config_filename);
    return;
    }

  if (strcmp(name, "all") == 0)
    {
    for (ol = optionlist_config;
      ol < optionlist_config + optionlist_config_size; ol++)
        print_ol(ol, ol->name, NULL, optionlist_config, optionlist_config_size);
    return;
    }
    
  if (strcmp(name, "directors") == 0)
    {
    type = "director";
    name = NULL;
    }     
  else if (strcmp(name, "routers") == 0)
    {
    type = "router";
    name = NULL;
    }     
  else if (strcmp(name, "transports") == 0)
    {
    type = "transport";
    name = NULL;
    }     
  else if (strcmp(name, "director_list") == 0)
    {
    type = "director";
    name = NULL;
    names_only = TRUE; 
    }     
  else if (strcmp(name, "router_list") == 0)
    {
    type = "router";
    name = NULL;
    names_only = TRUE; 
    }     
  else if (strcmp(name, "transport_list") == 0)
    {
    type = "transport";
    name = NULL;
    names_only = TRUE; 
    }     
  else
    { 
    print_ol(find_option(name, optionlist_config, optionlist_config_size),
      name, NULL, optionlist_config, optionlist_config_size);
    return;
    } 
  }

/* Handle the options for a director, router, or transport. Skip options
with names starting with '*' as those are used for internal alternative
representations of other options (which the printing function will sort
out). */

if (strcmp(type, "director") == 0)
  {
  d = (driver_instance *)directors;
  ol2 = optionlist_directors;
  size = optionlist_directors_size;
  }
else if (strcmp(type, "router") == 0)
  {
  d = (driver_instance *)routers;
  ol2 = optionlist_routers;
  size = optionlist_routers_size;
  }
else if (strcmp(type, "transport") == 0)
  {
  d = (driver_instance *)transports;
  ol2 = optionlist_transports;
  size = optionlist_transports_size;
  }
  
if (names_only)
  {
  for (; d != NULL; d = d->next) printf("%s\n", d->name);
  return;    
  }  

/* Either search for a given driver, or print all of them */

for (; d != NULL; d = d->next)
  {
  if (name == NULL)
    printf("\n%s %s:\n", d->name, type); 
  else if (strcmp(d->name, name) != 0) continue;

  for (ol = ol2; ol < ol2 + size; ol++)
    print_ol(ol, ol->name, d, ol2, size); 

  for (ol = d->info->options;
       ol < d->info->options + *(d->info->options_count); ol++)
    {
    if (ol->name[0] != '*')
      print_ol(ol, ol->name, d->options_block, d->info->options,
        *(d->info->options_count));
    }
  if (name != NULL) return;
  }
if (name != NULL) printf("%s %s not found\n", type, name);
}



/*************************************************
*             Syntax check a list of nets        *
*************************************************/

static void check_net_list(regexp *netregexp, char *list, char *name)
{
char *s;
if (list == NULL) return;

for (s = string_nextinlist(list, ':'); s != NULL;
     s = string_nextinlist(NULL, ':'))
  {
  if (!regexec(netregexp, s))
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
      "error in IP net specification %s in %s option", s, name);  
  }
} 



/*************************************************
*         Read main configuration options        *
*************************************************/

/* This function is the first to be called for configuration reading.
It opens the configuration file and reads general configuration settings
until it reaches a line containing "end". The file is then left open so
that delivery configuration data can subsequently be read if needed.

The configuration file must be owned either by root or exim, and be writeable
only by root or uid/gid exim. The values for Exim's uid and gid can be changed
in the config file, so the test is done on the compiled in values. A slight
anomaly, to be carefully documented.

The name of the configuration file is included in the binary of Exim. It can
be altered from the command line, but if that is done, root privilege is
immediately withdrawn.

For use on multiple systems that share file systems, first look for a
configuration file whose name has the current node name on the end. If that
is not found, try the generic name. For really contorted configurations, that 
run multiple Exims with different uid settings, first try adding the effective
uid before the node name. These complications are going to waste resources
on most systems. Therefore they are available only when requested by 
compile-time options. */

void readconf_main(void)
{
struct passwd *pw;
struct utsname s;
struct stat statbuf;
regexp *netregexp;
char buffer[256];

/* Try for the node-specific file if a node name exists */

#ifdef CONFIGURE_FILE_USE_NODE
if (uname(&s) >= 0)
  {
  #ifdef CONFIGURE_FILE_USE_EUID 
  sprintf(buffer, "%s.%d.%s", config_filename, geteuid(), s.nodename);
  config_file = fopen(buffer, "r");
  if (config_file == NULL) 
  #endif 
    {
    sprintf(buffer, "%s.%s", config_filename, s.nodename);
    config_file = fopen(buffer, "r");
    }
  }
#endif   

/* Otherwise, try the generic name, possibly with the euid added */

#ifdef CONFIGURE_FILE_USE_EUID
if (config_file == NULL) 
  {
  sprintf(buffer, "%s.%d", config_filename, geteuid());
  config_file = fopen(buffer, "r");
  }
#endif   

/* Finally, try the unadorned name */

if (config_file == NULL) config_file = fopen(config_filename, "r");

/* Failure to open the configuration file is a serious disaster. */

if (config_file == NULL)
  log_write(LOG_PANIC_DIE, "Failed to open configuration file %s",
    config_filename);

/* Check the status of the file we have opened, unless it was specified on
the command line, in which case privilege was given away at the start. */

if (!config_changed)
  {
  if (fstat(fileno(config_file), &statbuf) != 0)
    log_write(LOG_PANIC_DIE, "Failed to stat configuration file %s",
      config_filename);
  
  if ((statbuf.st_uid != root_uid && statbuf.st_uid != exim_uid) ||
      (statbuf.st_gid != exim_gid && (statbuf.st_mode & 020) != 0) ||
      ((statbuf.st_mode & 2) != 0))
    log_write(LOG_PANIC_DIE, "Exim configuration file %s has the wrong "
      "owner, group, or mode", config_filename);
  }     

/* Process the main configuration settings */

while (fgets(buffer, 256, config_file) != NULL)
  {
  char *s = buffer; 
  int n = (int)strlen(buffer);
  config_lineno++;
  while (n > 0 && isspace(buffer[n-1])) n--;
  buffer[n] = 0;
  while (isspace(*s)) s++; 
  if (*s == 0 || *s == '#') continue;
  if (strcmpic(s, "end") == 0) break;
  readconf_handle_option(s, optionlist_config, optionlist_config_size,
    NULL);
  }

/* If any of the options that specify lists of IP networks are given,
syntax-check their values. */

netregexp = regcomp(
  "^[0-9][0-9]?[0-9]?\\.[0-9][0-9]?[0-9]?\\."
  "[0-9][0-9]?[0-9]?\\.[0-9][0-9]?[0-9]?"
  "/[0-9][0-9]?[0-9]?\\.[0-9][0-9]?[0-9]?\\."
  "[0-9][0-9]?[0-9]?\\.[0-9][0-9]?[0-9]?$");

check_net_list(netregexp, sender_net_accept, "sender_net_accept");
check_net_list(netregexp, sender_net_reject, "sender_net_reject");
check_net_list(netregexp, sender_verify_except_nets, 
  "sender_verify_except_nets");
check_net_list(netregexp, smtp_reserve_nets, "smtp_reserve_nets");

/* If certain items were not given in the configuration file,
we must set them by other means. */

/* The primary host name is obtained from uname(), but if that yields
an unqualified value, make a FQDN by using gethostbyname to canonize it. */

if (primary_hostname == NULL)
  {
  struct utsname s;
  if (uname(&s) < 0)
    log_write(LOG_PANIC_DIE, "Failed to compute host name");
  if (strchr(s.nodename, '.') == NULL)
    {
    struct hostent *host = gethostbyname(s.nodename);
    primary_hostname = string_copy((char *)host->h_name);  
    }  
  else primary_hostname = string_copy(s.nodename);
  }

/* The qualify domains default to the primary host name */

if (qualify_domain_sender == NULL) 
  qualify_domain_sender = primary_hostname;
if (qualify_domain_recipient == NULL)
  qualify_domain_recipient = qualify_domain_sender;

/* Default local domains to the qualify domain, plus the primary
host name if different and requested. */

if (local_domains == NULL) 
  {
  local_domains = (local_domains_include_host && 
    strcmp(qualify_domain_recipient, primary_hostname) != 0)?
      string_sprintf("%s:%s", qualify_domain_recipient, primary_hostname) :
        qualify_domain_recipient;
  } 
  
/* Otherwise, add the primary host name if requested. */
 
else if (local_domains_include_host)
  local_domains = string_sprintf("%s:%s", local_domains, primary_hostname);   

/* Setting exim_user in the configuration sets the gid as well if a name is 
given, but a numerical value does not. Also, we may have a compiled-in uid and
no gid. */

if (exim_uid >= 0 && exim_gid < 0)
  {
  struct passwd *pw = getpwuid(exim_uid);
  if (pw == NULL)
    log_write(LOG_PANIC_DIE, "Failed to look up uid %d", exim_uid);
  exim_gid = pw->pw_gid;
  }

/* If exim_uid is set at runtime to be root, it has the effect of unsetting
it. */

if (exim_uid == root_uid)
  {
  exim_uid = -1;
  exim_gid = -1;
  }

/* If no security type was specified in the configuration file, use level
2 or 3 if an exim uid has been defined. */

if (security_type == NULL && exim_uid >= 0)
  security_type = have_seteuid? "setuid+seteuid" : "setuid";

/* If a security type is set and there is no exim_uid, complain. */

if (security_type != NULL && exim_uid < 0)
  log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
    "%s security requested in configuration file but no exim user defined");

/* Check that the security type is valid, and set the int version of it. If
seteuid is requested, check that this operating system has it (or at least,
exim has been configured to say that it has!) */

if (exim_uid >= 0)
  {
  if (strcmp(security_type, "setuid") == 0) security_level = 2;
  else if (strcmp(security_type, "seteuid") == 0) security_level = 1;
  else if (strcmp(security_type, "setuid+seteuid") == 0) security_level = 3;
  else log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
    "Unknown security setting: %s", security_type);

  if (!have_seteuid && (security_level == 1 || security_level == 3))
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
      "seteuid requested in configuration file but not configured in Make.os");
  }

/* If the errors_reply_to field is set, check that it is syntactically valid
and ensure it contains a domain. */

if (errors_reply_to != NULL)
  {
  char *errmess;
  int start, end, domain;
  char *recipient = parse_extract_address(errors_reply_to, &errmess,
    &start, &end, &domain, FALSE);

  if (recipient == NULL)
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
      "error in errors_reply_to (%s): %s", errors_reply_to, errmess);

  if (domain == 0)
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
      "errors_reply_to (%s) does not contain a domain", errors_reply_to);

  store_free(recipient);
  }

/* The debug_transport option is supported only when Exim is compiled
with the debut transport available. */

#ifndef TRANSPORT_DEBUG
if (debug_transport_file != NULL)
  log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
    "debug_transport is available only when TRANSPORT_DEBUG is defined");
#endif
}



/*************************************************
*          Initialize one driver                 *
*************************************************/

/* This is called once the driver's generic options, if any, have been read.
We can now find the driver, and set up for the private options. Also, save
the initialization entry point so that it can be called when all the options
have been read. */

static driver_info *init_driver(driver_instance *d,
  driver_info *drivers_available, int size_of_info, char *class)
{
driver_info *dd;

if (d->driver_name == NULL)
  log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
    "  no driver name given for %s %s", class, d->name);

for (dd = drivers_available; dd->driver_name[0] != 0;
     dd = (driver_info *)(((char *)dd) + size_of_info))
  {
  if (strcmp(d->driver_name, dd->driver_name) == 0)
    {
    int len = dd->options_len;
    d->info = dd;
    d->options_block = store_malloc(len);
    memcpy(d->options_block, dd->options_block, len);
    return dd;
    }
  }

log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
  "  cannot find %s driver \"%s\"", class, d->driver_name);
return NULL;   /* never obeyed */
}




/*************************************************
*             Initialize driver list             *
*************************************************/

/* This function is called for directors, routers, and transports. It reads the
data from the current point in the configuration file up to a line containing
"end", and sets up a chain of instance blocks according to the file's contents.
The file will already have been opened by a call to readconf_main, and must be
left open for subsequent reading of further data.

Any errors cause a panic crash. Note that the blocks with names driver_info and
driver_instance must map the first portions of all the _info and _instance
blocks for this shared code to work. */

void readconf_driver_init(
  char *class,                   /* "director", "router", or "transport" */
  driver_instance **anchor,      /* &directors, &routers, or &transports */
  driver_info *drivers_available,/* available drivers */
  int size_of_info,              /* size of each info block */
  void *instance_default,        /* points to default data */
  int  instance_size,            /* size of instance block */
  optionlist *driver_optionlist, /* generic option list */
  int  driver_optionlist_count)  /* count of same */
{
int state = 0;
BOOL generic;
driver_info *this_info;
driver_instance **p = anchor;
driver_instance *d = NULL;
char buffer[256];

while (fgets(buffer, 256, config_file) != NULL)
  {
  char *s = buffer;
  BOOL was_semicolon = FALSE;
  int n = (int)strlen(buffer);
  char name[24];

  /* Remove trailing newlines and spaces, ignore comments and blank
  lines, and stop when we reach "end". */

  config_lineno++;
  while (n > 0 && isspace(buffer[n-1])) n--;
  buffer[n] = 0;
  while (isspace(*s)) s++; 
  if (*s == '#' || *s == 0) continue;
  if (strcmpic(s, "end") == 0) break;

  /* If the terminating character is a comma, just remove it; if it is
  a semicolon, remove it and set a flag. */

  if (buffer[n-1] == ',') buffer[n-1] = 0;
    else if (buffer[n-1] == ';')
      { buffer[n-1] = 0; was_semicolon = TRUE; }

  /* If the line starts with a name terminated by a colon, we are at the
  start of the definition of a new driver. The first option may be on the
  same line, or it may follow on the next line. */

  s = readconf_readname(name, 24, buffer);
  if (*s++ == ':')
    {
    /*If there was a previous driver and we have not yet looked up its
    details (which will be the case if no private options were given) then
    do so now. */

    if (d != NULL && d->info == NULL)
      this_info = init_driver(d, drivers_available, size_of_info, class);

    /* Finish off initializing the previous driver. */

    if (d != NULL) (d->info->init)(d);

    /* Set up a new driver instance data block on the chain, with
    its default values installed. */

    d = store_malloc(instance_size);
    memcpy(d, instance_default, instance_size);
    *p = d;
    p = &(d->next);
    d->name = string_copy(name);
    generic = TRUE;

    /* If nothing more on this line, do the next loop iteration. */

    while (isspace(*s)) s++;
    if (*s == 0) continue;
    }

  /* If not the start of a new driver, scan from start of line. */

  else s = buffer;

  /* Give an error if we have not set up a current driver yet. */

  if (d == NULL)
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
      "  %s name missing in line %d", class, config_lineno);

  /* Handle generic options, and finish with them if terminator was
  a semicolon. */

  if (generic)
    {
    if (readconf_handle_option(s, driver_optionlist,
        driver_optionlist_count, d) == ';') was_semicolon = TRUE;
    if (was_semicolon)
      {
      this_info = init_driver(d, drivers_available, size_of_info, class);
      generic = FALSE;
      }
    }

  /* Handle private options */

  else readconf_handle_option(s, this_info->options,
    *(this_info->options_count), d->options_block);
  }

/* Initialize the final driver if we haven't already done so, and then
finish off its initialization. */

if (d != NULL && d->info == NULL)
  this_info = init_driver(d, drivers_available, size_of_info, class);
if (d != NULL) (d->info->init)(d);
}



/*************************************************
*            Check driver dependency             *
*************************************************/

/* This function is passed a driver instance and a string. It checks whether 
any of the string options for the driver contains the given string as an 
expansion variable. */

BOOL readconf_depends(driver_instance *d, char *s)
{
int count = *(d->info->options_count);
void *options_block = d->options_block;
optionlist *ol;
char *ss;

for (ol = d->info->options; ol < d->info->options + count; ol++)
  {
  char *value;
  if (ol->type != opt_stringptr && ol->type != opt_lcstringptr)
    continue;
  value = *(char **)((char *)d->options_block + (long int)(ol->value));
  if (value != NULL && (ss = strstr(value, s)) != NULL)
    {
    if (ss <= value || (ss[-1] != '$' && ss[-1] != '{') ||
      isalnum(ss[(int)strlen(s)])) continue;
    DEBUG(9) debug_printf("driver %s option %s depends on %s\n",
      d->name, ol->name, s);
    return TRUE;     
    }  
  } 

DEBUG(9) debug_printf("driver %s does not depend on %s\n", d->name, s);  
return FALSE; 
}




/*************************************************
*                Read retry information          *
*************************************************/

/* Each line of retry information contains:

.  A domain name, possibly a regexp, or a name starting with *; for local
   retry data, local_part@domain is used. 

.  An error name, possibly with additional data, or *;

.  An optional sequence of retry items, each consisting of an identifying
   letter, a cutoff time, and optional parameters.

All this is decoded and placed into a control block. */


/* Subroutine to read an argument, preceded by a comma and terminated
by comma, semicolon, whitespace, or newline. The types are: 0 = time value,
1 = fixed point number (returned *1000). */

static int retry_arg(char **paddr, int type)
{
char *p = *paddr;
char *pp;

if (*p++ != ',')
  log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
    "  comma expected in line %d", config_lineno);

while (isspace(*p)) p++;
pp = p;
while (isalnum(*p)|| (type == 1 && *p == '.')) p++;

if (*p != 0 && !isspace(*p) && *p != ',' && *p != ';')
  log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
    "  comma or semicolon expected in line %d", config_lineno);

*paddr = p;
switch (type)
  {
  case 0:
  return readconf_readtime(pp, *p);
  break;

  case 1:
  return readconf_readfixed(pp, *p);
  break;
  }
}


void readconf_retries(void)
{
retry_config **chain = &retries;
retry_config *next;
char buffer[256];

while (fgets(buffer, 256, config_file) != NULL)
  {
  retry_rule **rchain;
  char *p = buffer;
  char *pp, *q;
  int n = (int)strlen(buffer);
  int len;

  config_lineno++;
  while (n > 0 && isspace(buffer[n-1])) n--;
  buffer[n] = 0;
  while (isspace(*p)) p++;
  if (*p == 0 || *p == '#') continue;
  if (strcmpic(p, "end") == 0) break;

  next = store_malloc(sizeof(retry_config));
  next->next = NULL;
  *chain = next;
  chain = &(next->next);
  next->errno = next->more_errno = 0;
  next->rules = NULL;
  rchain = &(next->rules);

  pp = p;
  while (isgraph(*p)) p++;
  if (p - pp <= 0)
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
      "missing domain name in line %d", config_lineno);
  next->destination = string_copyn(pp, p-pp);

  if (next->destination[0] == '^')
    {
    regexp_compiling = next->destination;
    next->re = regcomp(next->destination);
    regexp_compiling = NULL;
    }
    
  else
    {
    char *at;
    if ((at = strchr(next->destination, '@')) != NULL)
      {
      if (!match_isinlist(at+1, local_domains, &re_local_domains))
        log_write(LOG_PANIC_DIE, "Exim configuration error:\n  " 
          "key %s in rewrite rules contains a local part with a non-local "
          "domain", next->destination);  
      }    
    }  

  while (isspace(*p)) p++;
  pp = p;
  while (isgraph(*p)) p++;
  if (p - pp <= 0)
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
      "  missing error type in line %d", config_lineno);

  /* Test error names for things we understand. */

  q = pp;
  while (q < p && *q != '_') q++;
  len = q - pp;

  if (len == 5 && strncmpic(pp, "quota", len) == 0)
    {
    next->errno = errno_quota;
    if (q != p)
      {
      if ((next->more_errno = readconf_readtime(q+1, *p)) < 0)
        log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
          "  bad time value in line %d", config_lineno);
      }
    }

  else if (len == 7 && strncmpic(pp, "refused", len) == 0)
    {
    next->errno = ECONNREFUSED;
    if (q != p)
      {
      if (strncmpic(q+1, "MX", p-q-1) == 0) next->more_errno = 'M';
      else if (strncmpic(q+1, "A", p-q-1) == 0) next->more_errno = 'A';
      else log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
        "  A or MX expected after \"refused\" in line %d", config_lineno);
      }
    }

  else if (len == 7 && strncmpic(pp, "timeout", len) == 0)
    {
    next->errno = ETIMEDOUT;
    if (q != p)
      {
      if (strncmpic(q+1, "DNS", p-q-1) == 0) next->more_errno = 'D';
      else if (strncmpic(q+1, "connect", p-q-1) == 0) next->more_errno = 'C';
      else log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
        "  DNS or CONNECT expected after \"timeout\" "
        "in line %d", config_lineno);
      }
    }

  else if (len != 1 || strncmp(pp, "*", 1) != 0)
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
      "  unknown error name in line %d", config_lineno);


  /* Now the retry rules. Keep the maximum timeout encountered. */

  while (isspace(*p)) p++;
  while (*p != 0)
    {
    retry_rule *rule = store_malloc(sizeof(retry_rule));
    *rchain = rule;
    rchain = &(rule->next);
    rule->next = NULL;

    rule->rule = toupper(*p++);
    rule->timeout = retry_arg(&p, 0);
    if (rule->timeout > retry_maximum_timeout)
      retry_maximum_timeout = rule->timeout;  

    switch (rule->rule)
      {
      case 'F':   /* Fixed interval */
      rule->p1 = retry_arg(&p, 0);
      break;

      case 'G':   /* Geometrically increasing intervals */
      rule->p1 = retry_arg(&p, 0);
      rule->p2 = retry_arg(&p, 1);
      if (rule->p1 == 0 || rule->p2 < 1000)
        log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
          "  bad parameters for geometric retry rule in line %d",
          config_lineno);
      break;

      default:
      log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
        "  unknown retry rule letter in line %d", config_lineno);
      break;
      }

    while (isspace(*p)) p++;
    if (*p == ';')
      {
      *p++;
      while (isspace(*p)) p++;
      }
    else if (*p != 0)
      log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
      "  semicolon expected in line %d", config_lineno);
    }
  }
}



/*************************************************
*              Read rewrite information          *
*************************************************/

/* Each line of rewrite information contains:

.  A complete address in the form user@domain, possibly with
   leading * for each part; or alternatively, a regexp.

.  A replacement string (which will be expanded).

.  An optional sequence of one-letter flags, indicating which
   headers etc. to apply this rule to.

All this is decoded and placed into a control block. The global variable
rewrite_existflags is the OR of all the flag words. */

void readconf_rewrites(void)
{
rewrite_rule **chain = &rewrite_rules;
rewrite_rule *next;
char buffer[1024];

while (fgets(buffer, 1024, config_file) != NULL)
  {
  char *p = buffer;
  char *pp;
  int n = (int)strlen(buffer);

  config_lineno++;
  while (n > 0 && isspace(buffer[n-1])) n--;
  buffer[n] = 0;
  while (isspace(*p)) p++;
  if (*p == 0 || *p == '#') continue;

  while (buffer[n-1] == '\\')
    {
    int c = fgetc(config_file);
    while (c == ' ' || c == '\t') c = fgetc(config_file);
    if (c == '\n' || c == EOF) break;
    buffer[n-1] = c;
    if (fgets(buffer+n, 1024-n, config_file) == NULL) break;
    n = (int)strlen(buffer);
    config_lineno++;
    while (n > 0 && isspace(buffer[n-1])) n--;
    buffer[n] = 0;
    }

  while (isspace(*p)) p++;
  if (strcmpic(p, "end") == 0) break;

  next = store_malloc(sizeof(rewrite_rule));
  next->next = NULL;
  *chain = next;
  chain = &(next->next);

  pp = p;
  while (*p != 0 && !isspace(*p)) p++;

  next->key = string_copyn(pp, p-pp);
  if (pp[0] == '^')
    {
    regexp_compiling = next->key;
    next->compiled_key = regcomp(next->key);
    regexp_compiling = NULL;
    }
  else
    {
    next->compiled_key = NULL;
    pp = strchr(next->key, '@');
    if (pp == NULL)
      log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
        "  domain missing in rewrite string in line %d", config_lineno);
    next->lenuser = pp - next->key;
    next->lendomain = (int)strlen(next->key) - next->lenuser - 1;
    }

  while (isspace(*p)) p++;
  if (*p == 0)
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
      "  missing rewrite replacement string in line %d", config_lineno);

  next->flags = 0;
  pp = p;
  while (*p != 0 && !isspace(*p)) p++;
  next->replacement = string_copyn(pp, p-pp);

  while (*p != 0) switch (*p++)
    {
    case ' ': case '\t': break;

    case 'h': next->flags |= rewrite_sender | rewrite_from | 
      rewrite_to | rewrite_cc | rewrite_bcc | rewrite_replyto;
    break;

    case 's': next->flags |= rewrite_sender; break;
    case 'f': next->flags |= rewrite_from; break;
    case 't': next->flags |= rewrite_to;   break;
    case 'c': next->flags |= rewrite_cc;   break;
    case 'b': next->flags |= rewrite_bcc;  break;
    case 'r': next->flags |= rewrite_replyto; break;
     
    case 'E': next->flags |= rewrite_envfrom | rewrite_envto; break;
    case 'F': next->flags |= rewrite_envfrom; break;
    case 'T': next->flags |= rewrite_envto; break;

    default:
      log_write(LOG_PANIC_DIE, "Exim configuration error:\n"
        "  unknown rewrite flag character (%c) in line %d",
        p[-1], config_lineno);
    }

  if (next->flags == 0) next->flags = rewrite_all;
  rewrite_existflags |= next->flags;
  }
}

/* End of readconf.c */
