/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */


#include "exim.h"



/*************************************************
*              Create a child process            *
*************************************************/

/* This function creates a child process and runs the given command in it. It
sets up a pipe to the standard input of the new process, and returns that to
the caller via fdptr. An optional file descriptor may be supplied for the
standard output and standard error of the subprocess. A value < 0 implies none,
in which case we set up /dev/null. The function returns the pid of the new
process, or -1 if things go wrong.

A new umask is supplied for the process, and an optional new uid and gid are
also available. These are used by the queryprogram router to set an
unprivileged id. */

int child_open(char **argv, char **envp, int newumask, int newuid, int newgid,
  int *fdptr, int outfd)
{
int pfd[2];

/* If no output provided, use /dev/null; ensure outfd is not zero. */

if (outfd < 0) outfd = open("/dev/null", O_WRONLY);

if (outfd == 0)
  {
  int fd = dup(outfd);
  close(outfd);
  outfd = fd;
  }

/* Create the pipe and fork the process */

if (pipe(pfd) == 0)
  {
  int pid = vfork();

  /* The child process makes the reading end of the pipe into the
  standard input, and outfd into the standard output and error,
  then execs, having closed all other file descriptors. */

  if (pid == 0)
    {
    int fd;

    close(pfd[pipe_write]);     /* Be absolutely sure this one is closed */

    for (fd = mac_maxfd; fd >= 0; fd--)
      if (fd != pfd[pipe_read] && fd != outfd) close(fd);
       
    if (pfd[pipe_read] != 0)
      {
      dup2(pfd[pipe_read], 0);  /* Make the standard input */
      close(pfd[pipe_read]);
      }
       
    if (outfd > 0)
      {
      if (outfd != 1) dup2(outfd, 1);
      if (outfd != 2) dup2(outfd, 2);
      if (outfd != 1 && outfd != 2) close(outfd);
      }

    /* Recover the power to setuid if necessary. We shouldn't be
    here not as root if it isn't available. */

    if (geteuid() != root_uid) mac_seteuid(root_uid);

    /* Set the required environment, and exec */

    if (newgid >= 0) setgid(newgid);
    if (newuid >= 0) setuid(newuid);
    umask(newumask);

    if (envp == NULL) execv(argv[0], argv);
      else execve(argv[0], argv, envp);

    /* Failed to execv */
    
    _exit(errno);         /* Note: must be _exit(), NOT exit() */
    }

  /* Parent */

  close(pfd[pipe_read]);
  if (pid > 0)            /* fork succeeded */
    {
    *fdptr = pfd[pipe_write];
    return pid;
    }
  else                    /* fork failed */
    {
    close(pfd[pipe_write]);
    return -1;
    }
  }
else return -1;
}



/*************************************************
*           Close down child process             *
*************************************************/

/* Wait for the given process to finish. */

int child_close(int pid)
{
int rc, status;
while ((rc = wait(&status)) != pid && rc >= 0);
return (status >> 8) & 255;
}

/* End of child.c */

