/* Auxiliary routines to special  ls  program at  nic.funet.fi */

/* Makefile CAN define:
   FTP_UID_NAME FTP_GID_NAME FTP_MODE
   Default values below.  */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <errno.h>
#include "ls.h"
#include <pwd.h>
#include <grp.h>
#include <fcntl.h>

#ifndef	FTP_UID_NAME
# define FTP_UID_NAME "ftp"
#endif
#ifndef	FTP_GID_NAME
# define FTP_GID_NAME "ftp"
#endif
#ifndef	FTP_MODE
# define FTP_MODE 0666
#endif

#define FALSE 0
#define TRUE  1

/* Directory data is stored into the file with
   first a block of   STAT data, then with
   a block of dirent data */
   

extern int errno;
extern void *debug;

extern int f_listdot, f_listalldot, f_needstat;
extern int f_nonprint, f_ignorelink;

static void dirpath_canon(char *);

#ifdef DO_FCNTL_LOCK
#include <fcntl.h>

#define LOCK_UN 0
#define LOCK_EX 1
#define LOCK_SH 2

int flock(int fd, int cmd)
{
  struct flock64 fl;
  int rc;

  memset(&fl, 0, sizeof(fl));

  switch(cmd) {
  case LOCK_SH:
    fl.l_type = F_RDLCK;
    break;
  case LOCK_EX:
    fl.l_type = F_WRLCK;
    break;
  case LOCK_UN:
    fl.l_type = F_UNLCK;
    break;
  default:
    return -1;
  }
  fl.l_whence = SEEK_SET;

  
  rc = fcntl(fd, F_SETLKW, &fl);
  return rc;
}

#endif

LSDIR *ls_opendir(dirname,fsp,advicep)
char *dirname;
int fsp;
int *advicep;	/* Advice refresh! */
{
  LSDIR *lsdir = malloc(sizeof(LSDIR));
  char fpath[1024];
  char magicbuf[32];
  char *endp;
  struct stat64 fstats, dstats;
  FILE *dirstats;
  int fd;

  lsdir->use_dir = 1;
  /*  if (!fsp) {  MGET fix kludge?*/ 
  {
    lsdir->dir = opendir(*dirname == 0 ? ".":dirname);	/* Normal opendir()			*/
#ifdef WITHIN_FTPDLS
    if (debug) fprintf(stderr,"Non-cache mode, normal opendir, result: %d\n",lsdir->dir != 0);
#endif
    return lsdir;
  }
  

  if (advicep)
    *advicep = 0;

  strcpy(fpath,dirname);
  dirpath_canon(fpath);		/* Canonized filename (directory),	*/
  endp = fpath+strlen(fpath);	/* terminates with an "/"		*/

  strcpy(endp,FTP_LS_DATA);	/* Append ".FTP_LS_DATA" (or whatever)	*/

  dirstats = NULL;
  fd = open(fpath,O_RDWR,0);
  if (fd >= 0)
    dirstats = fdopen(fd,"r+");

  if (!dirstats) {		/* Cache file open failed..		*/
#ifdef WITHIN_FTPDLS
    if (debug) fprintf(stderr,"Cache open of %s failed,",fpath);
#endif
    strcpy(endp,".");
    lsdir->dir = opendir(*fpath == 0 ? "." : fpath);
#ifdef WITHIN_FTPDLS
    if (debug) fprintf(stderr," normal opendir of %s yields %d\n",fpath,lsdir->dir!=0);
#endif
    return lsdir;
  }
  flock(fd, LOCK_SH);
  fstat64(fd, &fstats);		/* MUST exist, as it was ..	*/
  if (fstats.st_size == 0) {	/*  ... opened successfully!	*/
    fclose(dirstats);		/* (should not happen, "." is not 0 size!) */
#ifdef WITHIN_FTPDLS
    if (debug) fprintf(stderr,"Cache open gave 0-size file %s,",fpath);
#endif
    strcpy(endp,".");
    lsdir->dir = opendir(*fpath == 0 ? "." : fpath);	/* But MIGHT be at some point!....	*/
#ifdef WITHIN_FTPDLS
    if (debug) fprintf(stderr," normal opendir of %s yields %d\n",fpath,lsdir->dir!=0);
#endif
    return lsdir;
  }

  if (fgets(magicbuf,sizeof(magicbuf),dirstats) == NULL) return NULL; /* Umm.. */
  if (*magicbuf != '#') { /* Old-style.. */
    fseek(dirstats,0,0);
    if (advicep)	/* Old style, advice update! */
      *advicep = 1;
  } else {
    int longsize, statsize;
    if (strncmp("#cachels",magicbuf,8) != 0 ||
	sscanf(magicbuf+8,"%d %d",&longsize,&statsize) != 2 ||
	longsize != sizeof(long) || statsize != sizeof(struct stat64)) {
      if (advicep)	/* Some mismatch, advice update ! */
	*advicep = 1;
    }
  }

  strcpy(endp,".");	/* "." is ALWAYS available, as cache-file is there.. */
  stat64(fpath,&dstats);
  if (dstats.st_size == 0) { 
    fclose(dirstats);
    /*strcpy(endp,".");*/
    lsdir->dir = opendir(*fpath == 0 ? "." : fpath);
    return lsdir;
  }
  if (dstats.st_mtime > fstats.st_mtime)
    /* Directory changed since creation of this cache file! */
    if (advicep) {
      *advicep = 1;
#ifdef WITHIN_FTPDLS
      if (debug) fprintf(stderr,"ADVICE: refresh this dir: %s\n",fpath);
#endif
    }

  lsdir->use_dir = 0;
  lsdir->dirfil = dirstats;
#ifdef WITHIN_FTPDLS
  if (debug) {
    strcpy(endp,FTP_LS_DATA);	/* Append ".FTP_LS_DATA" (or whatever)	*/
    fprintf(stderr,"Successfull ls_opendir(%s)\n",fpath);
  }
#endif
  return lsdir;
}

#ifdef WITHIN_FTPDLS
static int ftp_uid  = 0;
static int ftp_gid  = 0;
static int ftp_mode = FTP_MODE;

LSDIR *ls_rewritedir(dirname,do_truncate)
char *dirname;
int do_truncate;
{
  char fpath[1024];
  LSDIR *lsdir = malloc(sizeof(LSDIR));
  int fd;

  strcpy(fpath,dirname);
  dirpath_canon(fpath);
  strcat(fpath,FTP_LS_DATA);

  lsdir->use_dir = 1;

  if (ftp_uid == 0 && ftp_gid == 0) {
#ifndef NO_GETPW
    struct passwd *pwd = getpwnam(FTP_UID_NAME);
    struct group  *grp = getgrnam(FTP_GID_NAME);

    ftp_uid = -1; /* No defined uid.. */
    if (pwd)  ftp_uid = pwd->pw_uid;
    ftp_gid = -1; /* No defined gid.. */
    if (grp)  ftp_gid = grp->gr_gid;
#else
    ftp_uid = namemap_name2uid(FTP_UID_NAME);
    ftp_gid = namemap_name2gid(FTP_GID_NAME);
#endif
  }
  
  umask(0);


  if (unlink(fpath)!=0) {
    if (errno!=ENOENT) {
      struct stat64 stbuf;
      lstat64(fpath,&stbuf);
      if (!S_ISREG(stbuf.st_mode)) {
	fprintf(stderr,"readdir(%s) reported error %s, report this to problems@ftp.funet.fi if it persists\n",fpath,strerror(errno)); 
	return NULL;
      }
    } 
  } 
  lsdir->dirfil = NULL;
  fd = open(fpath,O_RDWR|O_CREAT,FTP_MODE);		/* fopen(fpath,"r+") didn't do it! */
  if (fd >= 0)
    lsdir->dirfil = fdopen(fd,"w+");
  if (!lsdir->dirfil) { free(lsdir); return NULL; }
  lseek(fd,0,0);
  flock(fd,LOCK_EX);
  if (do_truncate)
    ftruncate(fd, 0);

  if (lsdir->dirfil) {
    /*    fchown(fileno(lsdir->dirfil),ftp_uid,ftp_gid);
    fchmod(fileno(lsdir->dirfil),ftp_mode); */
    fchown(fd,ftp_uid,ftp_gid);
    fchmod(fd,ftp_mode);
  }
  lsdir->use_dir = 0;
  return lsdir;
}
#endif

struct DIRENT_T *ls_readdir(dirp,statp)
LSDIR *dirp;
struct stat64 *statp;
{
  int rc;
  static struct DIRENT_T *dent = NULL;

  if (dirp->use_dir)
    return readdir(dirp->dir);
  if (!dent)
    dent = (struct DIRENT_T*)malloc(DENTSIZE+512);

  rc = fread(statp, sizeof(struct stat64), 1, dirp->dirfil);
  if (rc != 1) return NULL;
  rc = fread(dent, DENTSIZE, 1, dirp->dirfil);
  if (rc != 1) return NULL;
  rc = fread(dent->d_name, dent->d_reclen-DENTSIZE, 1, dirp->dirfil);
  if (rc != 1) return NULL;

  return dent;
}

void ls_closedir(dirp)
LSDIR *dirp;
{
  if (dirp->use_dir)
    closedir(dirp->dir);
  else {
    fflush(dirp->dirfil);
    lseek(fileno(dirp->dirfil),0,0);
    flock(fileno(dirp->dirfil),LOCK_UN);
    fclose(dirp->dirfil);
    free(dirp);
  }
}

void ls_rewinddir(dirp)
LSDIR *dirp;
{
  if (dirp->use_dir)
    rewinddir(dirp->dir);
  else
    fseek(dirp->dirfil,0,0);
}

#ifdef WITHIN_FTPDLS
int ls_writedir(dirp,dent,statp)
LSDIR *dirp;
struct DIRENT_T *dent;
struct stat64   *statp;
{
  if (dirp->use_dir != 0) {
    errno = EIO;
    return -1;
  }

  if (!(fwrite(statp, sizeof(struct stat64), 1, dirp->dirfil) &&
	fwrite(dent, dent->d_reclen, 1, dirp->dirfil))) {
    errno = EIO;
    return -1;
  }
  return 0;
}


/* Regenerate, and open the dirname pointed directory */
LSDIR *
regen_FTP_LS_DATA(dirp,lp,dirname)
LSDIR *dirp;
LS *lp;
char *dirname;
{
	LS *stats = NULL;
	int num;
	char *names = NULL;
	int o_f_listdot, o_f_listalldot, o_f_needstat;
	int o_f_nonprint, o_f_ignorelink;

	int i;


	if (dirp)
	  ls_closedir(dirp);
	dirp = ls_rewritedir(dirname,TRUE);

	if (!dirp) {				/* Could not open it!	 */
		dirp = malloc(sizeof(LSDIR));
		dirp->use_dir = 1;
		dirp->dir = opendir(*dirname == 0 ? "." : dirname);	/* Do ordinary opendir() */
		if (!dirp->dir) {
		  free(dirp);
		  return NULL;
		}
		return dirp;
	}

	fprintf(dirp->dirfil,"#cachels %d %d\n",sizeof(long), sizeof(struct stat64));

	o_f_listdot    = f_listdot;  /* Make sure the environment is proper */
	o_f_listalldot = f_listalldot;
	o_f_needstat   = f_needstat;
	o_f_nonprint   = f_nonprint;
	o_f_ignorelink = f_ignorelink;
	f_listdot = f_listalldot = f_needstat = 1;
	f_nonprint = f_ignorelink = 0;
	if ((num = tabdir(lp, &stats, &names, 0))) {
		/* Now write them out! */
		struct DIRENT_T *dent = (struct DIRENT_T*)malloc(DENTSIZE+512);;
		for (i=0; i<num; ++i) {
			long namlen = stats[i].len;
			if (namlen > 512) namlen = 512;
			dent->d_reclen = DENTSIZE+namlen+1;
			dent->d_reclen += sizeof(long);
			dent->d_reclen -= (dent->d_reclen % sizeof(long));
			strncpy(dent->d_name,stats[i].name,namlen+1);
			ls_writedir(dirp,dent,&stats[i].lstat);
		}
		free(dent);
	}
	f_listdot    = o_f_listdot;
	f_listalldot = o_f_listalldot;
	f_needstat   = o_f_needstat;
	f_nonprint   = o_f_nonprint;
	f_ignorelink = o_f_ignorelink;

	if (stats)  free(stats);
	if (names)  free(names);

	ls_closedir(dirp);
	return ls_opendir(dirname,1,&i);

	fflush(dirp->dirfil);
	ls_rewinddir(dirp);
	return dirp;
}

#endif
static void dirpath_canon(fpath)
char *fpath;
{
	/* Append `/' at the end of the file path,
	   if there is no `/' already in there. */
	int len = strlen(fpath);
	char *s = fpath+len-1;

	if (len == 1 && *fpath == '.') {
	  *fpath = 0; /* Don't present "./" there at all.. */
	  return;
	}

	if (len == 0 || *s != '/') {
	  *++s = '/';
	  *++s = 0;
	}
}
