/*
 * Copyright (c) 1989 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Michael Fischbein.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char sccsid[] = "@(#)print.c	5.14 (Berkeley) 6/28/89";
#endif /* not lint */

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <grp.h>
#include <pwd.h>
#include <utmp.h>
#include <tzfile.h>
#ifdef USE_MKDEV
#include <sys/mkdev.h>
#endif
#include "ls.h"
#include "namemap.h"

extern void *malloc(), *realloc();
extern char path[];

static int printaname(FILE *fil, LS *stats);
static void printlink(FILE *fil, const char *name);
static void printperms(FILE*, mode_t);
static void printowner(FILE *, uid_t);
static void printgrp(FILE *, gid_t);
static void printtime(FILE *, time_t);
static int  printtype(FILE *, mode_t);


void
printscol(fil,stats, num)
	register FILE *fil;
	register LS *stats;
	register int num;
{
	for (; num--; ++stats) {
		if (ferror(fil)) exit (0);

		printaname(fil,stats);
		fputc('\n',fil);
	}
}

/* [mea@nic.funet.fi] Print binary(hex) bobbelygoo of struct stat information. */

/* A sample output line:
736.D82B.41ED.2.2A9D.65.FFFFD306.400.25AE0526.2000.2 .
dev.inode.mode.nlnk.uid.gid.rdev.size.mtime.blksize.blocks name
*/
static void printgobbelyinfo(fil,stats)
	register FILE *fil;
	register LS *stats;
{
	fprintf(fil,"%lX.",(long)stats->lstat.st_dev);
	fprintf(fil,"%lX.",(long)stats->lstat.st_ino);
	fprintf(fil,"%lX.",(long)stats->lstat.st_mode);
	fprintf(fil,"%lX.",(long)stats->lstat.st_nlink);
	fprintf(fil,"%lX.",(long)stats->lstat.st_uid);
	fprintf(fil,"%lX.",(long)stats->lstat.st_gid);
	fprintf(fil,"%lX.",(long)stats->lstat.st_rdev & 0xFFFF);
	fprintf(fil,"%lX.",(long)stats->lstat.st_size);
	fprintf(fil,"%lX.",(long)stats->lstat.st_mtime);
	fprintf(fil,"%lX.",(long)stats->lstat.st_blksize);
	fprintf(fil,"%lX ",(long)stats->lstat.st_blocks);

	if (f_printpath && *path && strcmp(".",path))
	  fprintf(fil,"%s/",path );
	fprintf(fil,"%s",stats->name);
	if (S_ISLNK(stats->lstat.st_mode))
	  printlink(fil,stats->name);

	if(ferror(fil)) exit (0);
}

void
printgobbely(fil,stats, num)
	register FILE *fil;
	register LS *stats;
	register int num;
{
	for (; num--; ++stats) {
		/* Invisible file ? */
		if (f_noreadables && ((stats->lstat.st_mode & S_IROTH) == 0))
			continue;

		printgobbelyinfo(fil,stats);
		fputc('\n',fil);
	}
}



/* Special internal processing at nic.funet.fi:
   mode(dec) size(dec) mtime(dec) own(dec) grp(dec) \t filepath	*/

void
printallfil(fil,stats, num)
	FILE *fil;
	LS *stats;
	register int num;
{
	for (; num--; ++stats) {
		/* Invisible file ? */
		if (f_noreadables && ((stats->lstat.st_mode & S_IROTH) == 0))
			continue;
		fprintf(fil,"%d %lld %ld %d %d\t",
			(int)stats->lstat.st_mode,
			stats->lstat.st_size,
			(long)stats->lstat.st_mtime,
			(int)stats->lstat.st_uid,
			(int)stats->lstat.st_gid);
		if (*path && strcmp(".",path) != 0)
		  fprintf(fil,"%s/",path);
		fprintf(fil,"%s\n",stats->name);
	}
}

void
printlong(fil,stats, num)
	FILE *fil;
	LS *stats;
	register int num;
{
	if (f_total)
		fprintf(fil,"total %lu\n", stats[0].lstat.st_btotal);
	for (; num--; ++stats) {
		/* Invisible file ? */
		if (f_noreadables && ((stats->lstat.st_mode & S_IROTH) == 0))
			continue;
		if (f_inode)
			fprintf(fil,"%6lu ", (long)stats->lstat.st_ino);
		if (f_size)
			fprintf(fil,"%4ld ", (long)stats->lstat.st_blocks);
		printperms(fil,stats->lstat.st_mode);
		fprintf(fil," %3d ", (int)stats->lstat.st_nlink);
		printowner(fil,stats->lstat.st_uid);
		if (f_group)
			printgrp(fil,stats->lstat.st_gid);
		if (S_ISCHR(stats->lstat.st_mode) ||
		    S_ISBLK(stats->lstat.st_mode))
			fprintf(fil," %3ld,%3ld ",
				major(stats->lstat.st_rdev),
				minor(stats->lstat.st_rdev));
		else
			fprintf(fil," %7lld ", stats->lstat.st_size);
		if (f_accesstime)
			printtime(fil,stats->lstat.st_atime);
		else if (f_statustime)
			printtime(fil,stats->lstat.st_ctime);
		else
			printtime(fil,stats->lstat.st_mtime);
		if (f_printpath && *path && strcmp(".",path))
			fprintf(fil,"%s/",path );
		fprintf(fil,"%s", stats->name);
		if (f_type)
			printtype(fil,stats->lstat.st_mode);
		if (S_ISLNK(stats->lstat.st_mode))
			printlink(fil,stats->name);
		fputc('\n',fil);
	}
}

#define		TAB	8

void
printcol(fil,stats, num)
	FILE *fil;
	LS *stats;
	int num;
{
	extern int termwidth;
	register int base, chcnt, cnt, col, colwidth;
	int endcol, numcols, numrows, row;

	colwidth = stats[0].lstat.st_maxlen;
	if (f_inode)
		colwidth += 6;
	if (f_size)
		colwidth += 5;
	if (f_type)
		colwidth += 1;
	colwidth = (colwidth + TAB) & ~(TAB - 1);


	numcols = termwidth / colwidth;
	if( numcols < 2 ) numcols = 2;
	numrows = num / numcols;
	if (num % numcols)
		++numrows;
	

	if (f_size && f_total)
		fprintf(fil,"total %lu\n", stats[0].lstat.st_btotal);
	for (row = 0; row < numrows; ++row) {

		if(ferror(fil)) exit (0);

		endcol = colwidth;
		for (base = row, chcnt = col = 0; col < numcols; ) {
		  /* Invisible file ? */
		  if (f_noreadables && ((stats->lstat.st_mode & S_IROTH) == 0))
		    continue;
		  chcnt += printaname(fil,stats + base);
		  if ((base += numrows) >= num)
		    break;
		  while ((cnt = ((chcnt + TAB) & ~(TAB - 1))) <= endcol) {
		    fputc('\t',fil);
		    chcnt = cnt;
		  }
		  endcol += colwidth;
		  ++col;
		}
		fputc('\n',fil);
	}
}

/*
 * print [inode] [size] name
 * return # of characters printed, no trailing characters
 */
int
printaname(fil,lp)
	FILE *fil;
	LS *lp;
{
	int chcnt;

	if(ferror(fil)) exit (0);

	chcnt = 0;	
	if (f_inode)
		chcnt += fprintf(fil,"%5lu ", (long)lp->lstat.st_ino);
	if (f_size)
		chcnt += fprintf(fil,"%4ld ", (long)lp->lstat.st_blocks);
	if (f_printpath && *path && strcmp(".",path))
	  chcnt += fprintf(fil,"%s/",path);
	chcnt += fprintf(fil,"%s", lp->name);
	if (f_type)
		chcnt += printtype(fil,lp->lstat.st_mode);
	return(chcnt);
}

#define	NCACHE	128		/* power of 2 */
#define	LSMASK	(NCACHE - 1)	/* bits to store with */
static void
printowner(fil,uid)
	FILE *fil;
	uid_t uid;
{
#ifndef NO_GETPW
	static struct ncache {
		uid_t	uid;
		char	name[UT_NAMESIZE+1];
	} c_uid[NCACHE];
	register struct passwd *pw;
	register struct ncache *cp;
#else
	char *pwname;
	extern char *namemap_uid2name();
#endif

	if (ferror(fil)) exit (0);

#ifdef	NO_GETPW
	pwname = namemap_uid2name(uid);
	if (pwname != NULL)
	  fprintf(fil,"%-*s ", UT_NAMESIZE, pwname);
	else
	  fprintf(fil,"%-*lu ", UT_NAMESIZE, uid);
	return;
#else
	    
	cp = c_uid + (uid & LSMASK);
	if ((cp->uid != uid) || !*(cp->name)) {
		/* if can't find owner, print out number instead */
		if (f_numericnames ||
		    !(pw = getpwuid(uid)) ) {
			fprintf(fil,"%-*u ", UT_NAMESIZE, uid);
			return;
		}
		cp->uid = uid;
		strncpy(cp->name, pw->pw_name, UT_NAMESIZE);
	}
	fprintf(fil,"%-*.*s ", UT_NAMESIZE, UT_NAMESIZE, cp->name);
#endif
}

static void
printgrp(fil,gid)
	FILE *fil;
	gid_t gid;
{
#ifndef NO_GETPW
	static struct ncache {
		gid_t	gid;
		char	name[UT_NAMESIZE];
	} c_gid[NCACHE];
	register struct group *gr;
	register struct ncache *cp;
#else
	char *grname;
	extern char *namemap_gid2name();
#endif

	if(ferror(fil)) exit (0);

#ifdef NO_GETPW
	grname = namemap_gid2name(gid);

	if (grname != NULL)
	  fprintf(fil,"%-*s ", UT_NAMESIZE, grname);
	else
	  fprintf(fil,"%-*lu ", UT_NAMESIZE, gid);

	return;
#else

	cp = c_gid + (gid & LSMASK);
	if (cp->gid != gid || !*cp->name) {
		/* can't find group, print out number instead */
		if (f_numericnames ||
		    !(gr = getgrgid(gid)) ) {
			fprintf(fil,"%-*u ", UT_NAMESIZE, gid);
			return;
		}
		cp->gid = gid;
		strncpy(cp->name, gr->gr_name, UT_NAMESIZE);
	}
	fprintf(fil,"%-*.*s ", UT_NAMESIZE, UT_NAMESIZE, cp->name);
#endif
}

static void 
printtime(fil,ftime)
	FILE *fil;
	time_t ftime;
{
	int i;
	char *longstring, *ctime();
	extern time_t time();
	static time_t epoch = 0;

	if (epoch == 0)
	  epoch = time(NULL);

	if(ferror(fil)) exit (0);

	longstring = ctime((long *)&ftime);
	for (i = 4; i < 11; ++i)
		fputc(longstring[i],fil);

#define	SIXMONTHS	((DAYSPERNYEAR / 2) * SECSPERDAY)
	if (ftime + SIXMONTHS > epoch)
		for (i = 11; i < 16; ++i)
			fputc(longstring[i],fil);
	else {
		fputc(' ',fil);
		for (i = 20; i < 24; ++i)
			fputc(longstring[i],fil);
	}
	fputc(' ',fil);
}

/*
 * do the permissions printing, passed the mode
 */
static void
printperms(fil,mode)
	FILE *fil;
	mode_t mode;
{

	if(ferror(fil)) exit (0);

	 /* print type */
	switch (mode & S_IFMT) {
	case S_IFDIR:			/* directory */
		fputc('d',fil);
		break;
	case S_IFCHR:			/* character special */
		fputc('c',fil);
		break;
	case S_IFBLK:			/* block special */
		fputc('b',fil);
		break;
	case S_IFREG:			/* regular */
		fputc('-',fil);
		break;
	case S_IFLNK:			/* symbolic link */
		fputc('l',fil);
		break;
	case S_IFSOCK:			/* socket */
		fputc('s',fil);
		break;
#ifdef S_IFIFO
	case S_IFIFO:			/* fifo */
		fputc('p',fil);
		break;
#endif
	default:			/* unknown */
		fputc('?',fil);
		break;
	}
	/* usr */
	if (mode & S_IRUSR)
		fputc('r',fil);
	else
		fputc('-',fil);
	if (mode & S_IWUSR)
		fputc('w',fil);
	else
		fputc('-',fil);
	switch (mode & (S_IXUSR | S_ISUID)) {
	case 0:
		fputc('-',fil);
		break;
	case S_IXUSR:
		fputc('x',fil);
		break;
	case S_ISUID:
		fputc('S',fil);
		break;
	case S_IXUSR | S_ISUID:
		fputc('s',fil);
		break;
	}
	/* group */
	if (mode & S_IRGRP)
		fputc('r',fil);
	else
		fputc('-',fil);
	if (mode & S_IWGRP)
		fputc('w',fil);
	else
		fputc('-',fil);
	switch (mode & (S_IXGRP | S_ISGID)) {
	case 0:
		fputc('-',fil);
		break;
	case S_IXGRP:
		fputc('x',fil);
		break;
	case S_ISGID:
		fputc('S',fil);
		break;
	case S_IXGRP | S_ISGID:
		fputc('s',fil);
		break;
	}
	/* other */
	if (mode & S_IROTH)
		fputc('r',fil);
	else
		fputc('-',fil);
	if (mode & S_IWOTH)
		fputc('w',fil);
	else
		fputc('-',fil);
	switch (mode & (S_IXOTH | S_ISVTX)) {
	case 0:
		fputc('-',fil);
		break;
	case S_IXOTH:
		fputc('x',fil);
		break;
	case S_ISVTX:
		fputc('T',fil);
		break;
	case S_IXOTH | S_ISVTX:
		fputc('t',fil);
		break;
	}
}

static int
printtype(fil,mode)
	FILE *fil;
	mode_t mode;
{
	switch(mode & S_IFMT) {
	case S_IFDIR:
		fputc('/',fil);
		return(1);
	case S_IFLNK:
		fputc('@',fil);
		return(1);
	case S_IFSOCK:
		fputc('=',fil);
		return(1);
	}
	if (mode & (S_IXUSR | S_IXGRP | S_IXOTH)) {
		fputc('*',fil);
		return(1);
	}
	return(0);
}

static void
printlink(fil,name)
	FILE *fil;
	const char *name;
{
	int lnklen;
	char path[MAXPATHLEN + 1];

	if(ferror(fil)) exit (0);

	if ((lnklen = readlink(name, path, MAXPATHLEN)) == -1) {
		fprintf(stderr, "\nls: %s: %s\n", name, strerror(errno));
		return;
	}
	path[lnklen] = '\0';
	fprintf(fil," -> %s", path);
}
