#---------------------------------------------------------------------------
#
#	Create a Tree/DAG Display
#
#	The DAG coordinate data are generated by a layouter function given
#	by "init".
#
#	Each node of the displayed tree can be selected in two possible
#	ways: the left mouse button positively selects an item; the right
#	mouse button negatively selects an item. Shift plus the left or
#	right mouse button deselects an item. If "type" is "select", only
#	single, positive selections are allowed. If "type" is "posneg",
#	multiple, positive or negative selections are allowed.
#
#	Positive selections invoke the action "State+".
#	Negative selections invoke the action "State-".
#	Deselections invoke the action "State0".
#
#---------------------------------------------------------------------------

defwidget DAG Window {
	{cw 600}
	{ch 300}
}

defmethod DAG new {name args} {

  args	layout actions init text embedded type buttons nodefont

  # Add default help action
  if { [assoc Help $actions] == {} } {
    lappend actions [list Help {} DAG/help]
  }

  if { $nodefont == {} } {
    set nodefont small
  }

  # Create toplevel window
  if { $embedded == "true" } {
    Frame new $name -relief flat
  } {
    Toplevel new $name \
	-title $text \
	-resizable true \
	-buttons $buttons \
	-handler [list $name _do] \
	-actions $actions
  }
  defsuper $name DAG

  $name slot _init $init
  $name slot _nodefont $nodefont
  $name slot _actions $actions

  $name redisplay
  set canvas [$name.c canvas]

  case $type {
  {posneg} {
	set positive pos
	set negative neg
    }
  default {
	set positive select
	set negative select
    }
  }
  $canvas bind dag <Button-1> [list $name _status $canvas $positive]
  $canvas bind dag <Button-3> [list $name _status $canvas $negative]
  $canvas bind dag <Shift-Button-1> [list $name _status $canvas deselect]
  $canvas bind dag <Shift-Button-3> [list $name _status $canvas deselect]

  $name layout $layout
}

#---------------------------------------------------------------------------
#
#	Draw a tree onto a canvas
#

defmethod DAG redisplay {} {

  set nodefont [$self slot _nodefont]
  set sizes [list [Font slot $nodefont,w] [Font slot $nodefont,h]]
  set answer [eval [concat [$self slot _init] $sizes]]

  set win_w [lindex $answer 0]
  set win_h [lindex $answer 1]
  set items [lindex $answer 2]
  set links [lindex $answer 3]

  set cw [$self slot cw]
  set ch [$self slot ch]
  set win_x [expr {($win_w > $cw) ? $cw : $win_w}]
  set win_y [expr {($win_h > $ch) ? $ch : $win_h}]

  if { [winfo exists $self.c] } then {
    set canvas [$self.c canvas]
    $canvas delete dag
    $self.c adjust [list $win_w $win_h]
    $self slot _last {}
  } else {
    Canvas new $self.c -layout {top expand fill} \
	-width $win_x -height $win_y \
	-scroll [list $win_w $win_h]
    set canvas [$self.c canvas]
    $canvas bind dag <Any-Enter> [list $self _status $canvas movein]
    $canvas bind dag <Any-Leave> [list $self _status $canvas moveout]
  }

  set fg [Color slot fg]
  set bg [Color slot bg]
  set bb [Color slot bg,button]

  # Draw the links first, so the boxes may obscure them
  foreach link $links {
    $canvas create line [lindex $link 0] [lindex $link 1] \
	[lindex $link 2] [lindex $link 3] \
	-arrow last -arrowshape {4 4 2} -width 1 -fill $fg \
	-tags {dag dag(link)}
  }

  # Draw the nodes of the DAG (possibly over some links)
  foreach node $items {
    set x [lindex $node 0]
    set y [lindex $node 1]
    set xe [lindex $node 2]
    set ye [lindex $node 3]

    $canvas create rectangle \
	[expr $x+3] [expr $y+3] [expr $xe+3] [expr $ye+3] \
	-outline $fg -fill $fg -width 1 -tags {dag dag(rectbox)}

    $canvas create rectangle $x $y $xe $ye \
	-outline $bg -fill $bg -width 1 \
	-tags {dag dag(rectout)}
    $canvas create rectangle [expr $x+2] [expr $y+2] \
	[expr ${xe}-2] [expr ${ye}-2] \
	-outline $fg -fill $bb -width 1 \
	-tags {dag dag(rect)}

    $canvas create text [expr ($x+$xe)/2] [expr ($y+$ye)/2] \
	-anchor center -text [join [lrange $node 4 end] "\n"] \
	-font [Font slot $nodefont] -tags {dag dag(text)}
  }

  return $canvas
}

defmethod DAG _do {button action} {
  global system

  case $button {
  {Help Dismiss Ok} {
	$self $button $action
    }
  {Print} {
    $self.c print -printer $system(printer) -colormode gray -layout best
    }
  default {
	if { $action != "" } {
	  eval [concat $action [list $self]]
	}
    }
  }
}

#--	Change item status
#
defmethod DAG _status {canvas state} {

  # get the outer rectangle
  set id [$canvas find withtag current]
  foreach tag [$canvas gettags $id] {
    if { $tag == "dag(text)" } {
      incr id -2; break
    }
    if { $tag == "dag(rect)" } {
      incr id -1; break
    }
    if { $tag == "dag(rectout)" } {
      break
    }
    set tag {}
  }
  if { $tag != {} } {
    # Now, id is the canvas item id of the enclosing rectangle
    $self _select $canvas $id $state
  }
}

#--	Select or deselect an item. The 'id' arg is the id of the enclosing
#	rectangle. id+1 is the inner rectangle. id+2 is the text item.
#
defmethod DAG _select {canvas id state} {
  global color info

  case $state {
  {movein} {
	$canvas itemconfigure [expr $id+1] -width 3
	return
    }
  {moveout} {
	$canvas itemconfigure [expr $id+1] -width 1
	return
    }
  {set} {
	$self slot _last $id
	return
    }
  {select} {
	foreach node [$canvas find withtag dag(positive)] {
	  $self _select $canvas $node deselect
	}
	$self _select $canvas $id pos
	return
    }
  {deselect} {
	# inactive item: background color, thin border line
	$self slot _last $id
	$canvas itemconfigure $id -outline [Color slot bg]
	$canvas dtag $id dag(negative)
	$canvas dtag $id dag(positive)
	incr id
	$canvas itemconfigure $id -fill [Color slot bg,button]
	incr id
	$canvas itemconfigure $id -fill [Color slot fg]
	set state 0
    }
  {pos} {
	# positive selection: active color, thick border line
	$self slot _last $id
	$canvas itemconfigure $id -outline [Color slot fg]
	$canvas dtag $id dag(negative)
	$canvas addtag dag(positive) withtag $id
	incr id
	$canvas itemconfigure $id -fill [Color slot bg,active]
	incr id
	$canvas itemconfigure $id -fill [Color slot fg,active]
	set state +
    }
  {neg} {
	$self slot _last $id
	$canvas itemconfigure $id -outline [Color slot bg]
	$canvas dtag $id dag(positive)
	$canvas addtag dag(negative) withtag $id
	incr id
	$canvas itemconfigure $id -fill [Color slot bg,active]
	incr id
	$canvas itemconfigure $id -fill [Color slot fg,active]
	set state -
    }
  }

  set action [lrange [assoc State$state [$self slot _actions]] 1 end]
  if { $action != "" } {
    set val [lindex [$canvas itemconfigure $id -text] 4]
    eval [concat $action [list $self $id $val]]
  }
}

defmethod DAG canvas {} {

  $self.c canvas
}

#--	Confirm the selection and dismiss the tree box
#
defmethod DAG Ok {action} {

  set canvas [$self.c canvas]

  set action [lrange [assoc Ok [$self slot _actions]] 1 end]
  if { $action != "" && [Yesno new *$self -help DAG/confirm] == "Yes" } {
    set pos {}
    foreach id [$canvas find withtag dag(positive)] {
      set item [lindex [$canvas itemconfigure [expr ${id}+2] -text] 4]
      lappend pos $item
    }

    set neg {}
    foreach id [$canvas find withtag dag(negative)] {
      set item [lindex [$canvas itemconfigure [expr ${id}+2] -text] 4]
      lappend neg $item
    }
    eval [concat $action [list $pos $neg]]
  }

  $self Dismiss
}

#---------------------------------------------------------------------------
#
#	Simple DAG Layouter Function
#
#	This procedure receives a list of precedences and builds a DAG
#	structure with coordinates suitable for the DAG widget. It is
#	derived from the respective LISP counterpart and should probably
#	be implemented in C for efficiency.
#
#	List: { {node succ...} ...}
#

defmethod DAG computeLayout {roots list args} {
  global __dag

  args	{width 1} {height 1} {minchars 4} {imargin 4} {link 25} {sep 10} \
	{margin 10} {depth 200}

  # Initialize the global variables (ugh! these should be slot values).
  catch {unset __dag}
  set __dag(n) -1
  set __dag(list) $list
  set __dag(margin) $margin
  set __dag(minchars) $minchars
  set __dag(width) $width
  set __dag(height) $height
  set __dag(imargin) $imargin
  set __dag(link) $link
  set __dag(sep) $sep

  # Build the graph structure
  set rids {}
  foreach root $roots {
    lappend rids [$self __traverse [assoca $root $list] $depth]
  }

  # Compute a topological order
  set __dag(order) {}
  foreach rootid $rids {
    $self __topological $rootid
  }

  # Compute the Y layout (vertical arrangement)
  set __dag(F) 0
  set __dag(y) $margin
  foreach rootid $rids {
    $self __layoutY $rootid
  }

  # Compute the X layout (horizontal arrangement)
  set __dag(x) 0
  foreach n $__dag(order) {
    set x [$self __layoutX $n]
    if { $x > $__dag(x) } {
      set __dag(x) $x
    }
  }

  set result [list [expr $__dag(x)+$__dag(margin)] \
		   [expr $__dag(y)+$__dag(margin)-$__dag(sep)]]

  set res {}
  foreach n $__dag(order) {
    lappend res \
	[list $__dag(X$n) $__dag(Y$n) $__dag(XE$n) $__dag(YE$n) $__dag(N$n)]
  }
  lappend result $res

  set res {}
  foreach n $__dag(order) {
    foreach m $__dag(S$n) {
      lappend res \
	[list $__dag(XE$n) [expr ($__dag(Y$n)+$__dag(YE$n))/2] \
	      $__dag(X$m) [expr ($__dag(Y$m)+$__dag(YE$m))/2]]
    }
  }
  lappend result $res

  unset __dag
  return $result
}

defmethod DAG __traverse {node depth} {
  global __dag

  if { $depth <= 0} {
    return
  } elseif { $depth != {} } {
    incr depth -1
  }
  set name [lindex $node 0]

  if { [info exists __dag(I$name)] } {
    return
  }

  set succ [lrange $node 1 end]
  set n [incr __dag(n)]
  set __dag(N$n) $name
  set __dag(P$n) {}
  set __dag(S$n) {}
  set __dag(F$n) 0
  set __dag(X$n) {}
  set __dag(Y$n) {}
  set __dag(I$name) $n

  foreach child $succ {
    if { [info exists __dag(I$child)] } {
      set m $__dag(I$child)
    } {
      set next [assoca $child $__dag(list)]
      if { $next == {} } {
	set next [list $child]
      }
      set m [$self __traverse $next $depth]
    }
    lappend __dag(S$n) $m
    lappend __dag(P$m) $n
  }

  return $n
}

defmethod DAG __topological {n} {
  global __dag

  if { $__dag(F$n) } {
    return
  }

  foreach pred $__dag(P$n) {
    if { ! $__dag(F$pred) } {
      return
    }
  }

  lappend __dag(order) $n
  set __dag(F$n) 1

  foreach succ $__dag(S$n) {
    $self __topological $succ
  }

  return
}

defmethod DAG __layoutY {n} {
  global __dag

  if { $__dag(Y$n) == {} } {
    set pred [lindex $__dag(P$n) 0]
    if { $pred != {} && $__dag(F$pred) } {
      set __dag(F$pred) 0
      set __dag(Y$n) $__dag(Y$pred)
    } {
      set __dag(Y$n) $__dag(y)
    }

    set __dag(YE$n) [expr $__dag(Y$n)+$__dag(height)+$__dag(imargin)]
    set y [expr $__dag(YE$n)+$__dag(sep)]
    if { $y > $__dag(y) } {
      set __dag(y) $y
    }

    foreach succ $__dag(S$n) {
      $self __layoutY $succ
    }
  }
}

defmethod DAG __layoutX {n} {
  global __dag

  set max $__dag(margin)
  foreach pred $__dag(P$n) {
    if { $__dag(XE$pred) > $max } {
      set max $__dag(XE$pred)
    }
  }
  set __dag(X$n) [expr $max+$__dag(link)]

  set chars $__dag(minchars)
  set len [string length $__dag(N$n)]
  if { $len < $chars } {
    set len $chars
  }
  set __dag(XE$n) [expr $__dag(X$n)+$__dag(width)*$len+$__dag(imargin)]

  return $__dag(XE$n)
}

#---------------------------------------------------------------------------

Window addDemo DAG

defmethod DAG demo_compute {w h} {
  DAG computeLayout {{Activity 1}} {
	{{Activity 1} {Activity 5} {Activity 7} {Activity 1093} {SPT}}
	{{Activity 5} {Activity 3} {Activity 8}}
	{{Activity 7} {Activity 8} {Activity 2}}
	{{Activity 1093} {Development Test}}
	{{SPT} {T55} {S98}}
	{{T55} {Development Test} {q-9 Test}}
	{{S98} {S98.1} {S98.2} {S98.3}}
	{{S98.1} {S99}}
	{{S98.2} {S99}}
	{{S98.3} {S99}}
	{{q-9 Test} {S99}}
	{{Development Test} {Activity 4} {Process 10.x}}
	{{Activity 2} {Activity 4}}
	{{Activity 3} {Activity 9}}
	{{Activity 8} {Activity 9}}
	{{Activity 9} {Final Test}}
	{{Activity 4} {Final Test}}
	{{Process 10.x} {Final Test}}
  } -width $w -height $h
}

defmethod DAG demo {} {

  DAG new * \
	-text "Non-Linear Plan of Activities" \
	-layout +10+50 \
	-init		{DAG demo_compute} \
	-buttons {Print {} Ok} \
	-actions {
	  {Print	Drucken}
	  {Ok		{} action ok}
	  {State+	{} action pos}
	  {State0	{} action out}
	}

  DAG new * \
	-text "Select Activities" \
	-layout -10-50 \
	-init		{DAG demo_compute} \
	-type posneg \
	-buttons {Print {} Ok} \
	-actions {
	  {Print	Drucken}
	  {Ok		{} action ok}
	  {State+	{} action pos}
	  {State-	{} action neg}
	  {State0	{} action out}
	}
}
