/* bosh.c - BOS shell (adapted from tclsh/tclTest)
 *
 * Copyright 1992 (C) Sean Levy
 *
 * Copyright 1987-1991 Regents of the University of California
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#ifndef lint
static char *_RCSId =
  "/afs/cs/project/edrc/ndim/source/bos/bosh/bosh.c,v 1.2 1992/07/31 20:12:22 snl Exp";
#endif

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/file.h>
#include <bos/bos.h>
#ifdef NEED_STRTOL
#include <ctype.h>
#endif

#ifndef DEFAULT_LOAD_ROOT
#define DEFAULT_LOAD_ROOT "."
#endif
#ifndef DEFAULT_LOAD_LIST
#define DEFAULT_LOST_LIST "base"
#endif

extern int exit();
extern int Tcl_DumpActiveMemory();

Tcl_Interp *interp;
Tcl_CmdBuf buffer;
char dumpFile[100];
int quitFlag = 0;

char *initCmd =
    "if [file exists ~/.boshrc] {source ~/.boshrc}";

int
cmdCheckmem(clientData, interp, argc, argv)
    char *clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" fileName\"", (char *) NULL);
	return TCL_ERROR;
    }
    strcpy(dumpFile, argv[1]);
    quitFlag = 1;
    return TCL_OK;
}

int
cmdEcho(clientData, interp, argc, argv)
    char *clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int i;

    for (i = 1; ; i++) {
	if (argv[i] == NULL) {
	    if (i != argc) {
		echoError:
		sprintf(interp->result,
		    "argument list wasn't properly NULL-terminated in \"%s\" command",
		    argv[0]);
	    }
	    break;
	}
	if (i >= argc) {
	    goto echoError;
	}
	fputs(argv[i], stdout);
	if (i < (argc-1)) {
	    printf(" ");
	}
    }
    printf("\n");
    return TCL_OK;
}

void
deleteProc(clientData)
    char *clientData;
{
    printf("Deleting command with clientData \"%s\".\n", clientData);
}

int
main(argc, argv)
  int argc;
  char **argv;
{
    char line[1000], *cmd, *load_list, *load_root, *set_def();
    int result, gotPartial, no_prompt, no_init, i, load_up();
    Bos_World *world;

    load_list = load_root = (char *)0;
    no_prompt = no_init = 0;
    for (i = 1; i < argc; i++) {
      char *arg = argv[i];

      if (!strcmp(arg, "-q"))
	no_prompt = 1;
      else if (!strcmp(arg, "-n"))
	no_init = 1;
      else if (!strcmp(arg, "-l") && (i+1) < argc)
	load_list = argv[++i];
      else if (!strcmp(arg, "-r") && (i+1) < argc)
	load_root = argv[++i];
      else {
	fprintf(stderr, "%s: unknown option\n", arg);
	fprintf(stderr, "usage: %s [-n|-q]\n", argv[0]);
	fprintf(stderr, "\t-n\tdo not process ~/.boshrc\n");
	fprintf(stderr, "\t-q\tdo not print prompts\n");
	fprintf(stderr,
      "\t-l list\tload modules listed (list must be quoted; default is %s)\n",
		DEFAULT_LOAD_LIST);
	fprintf(stderr, "\t-r root\tset root for load list (default is %s)\n",
		DEFAULT_LOAD_ROOT);
	exit(1);
      }
    }
    load_list = set_def(load_list, "BOS_LOAD_LIST", DEFAULT_LOAD_LIST);
    load_root = set_def(load_root, "BOS_LOAD_ROOT", DEFAULT_LOAD_ROOT);

    interp = Tcl_CreateInterp();
#ifdef TCL_MEM_DEBUG
    Tcl_InitMemory(interp);
#endif
    Bos_InitializeBuiltins();
    world = Bos_InitInterp(interp, 0);
    Bos_CreateBuiltinObjects(world, interp);
    if (no_prompt)
      Tcl_SetVar(interp, "QUIET", "1", TCL_GLOBAL_ONLY);
    Tcl_CreateCommand(interp, "echo", cmdEcho, (ClientData) "echo",
	    (void (*)()) NULL);
    Tcl_CreateCommand(interp, "checkmem", cmdCheckmem, (ClientData) 0,
	    (void (*)()) NULL);
    load_up(interp, load_root, load_list);

    buffer = Tcl_CreateCmdBuf();
    if (!no_init) {
      result = Tcl_Eval(interp, initCmd, 0, (char **) NULL);
      if (result != TCL_OK) {
	printf("%s\n", interp->result);
	exit(1);
      }
    }
    gotPartial = 0;
    while (1) {
	clearerr(stdin);
	if (!gotPartial) {
	    if (!no_prompt)
	      fputs("% ", stdout);
	    fflush(stdout);
	}
	if (fgets(line, 1000, stdin) == NULL) {
	    if (!gotPartial) {
		exit(0);
	    }
	    line[0] = 0;
	}
	cmd = Tcl_AssembleCmd(buffer, line);
	if (cmd == NULL) {
	    gotPartial = 1;
	    continue;
	}

	gotPartial = 0;
	result = Tcl_RecordAndEval(interp, cmd, 0);
	if (result == TCL_OK) {
	    if (*interp->result != 0) {
		printf("%s\n", interp->result);
	    }
	    if (quitFlag) {
		Tcl_DeleteInterp(interp);
		Tcl_DeleteCmdBuf(buffer);
#ifdef TCL_MEM_DEBUG
		Tcl_DumpActiveMemory(dumpFile);
#endif
		exit(0);
	    }
	} else {
	    if (result == TCL_ERROR) {
		printf("Error");
	    } else {
		printf("Error %d", result);
	    }
	    if (*interp->result != 0) {
		printf(": %s\n", interp->result);
	    } else {
		printf("\n");
	    }
	}
    }
}

/* set_def - set value of variable if it needs to be set
 *
 * if the value we're given in val is NULL, then
 *    if the environment has a variable named envar then
 *       return the value of it
 *    else
 *       return the default we were given
 * else return val
 */

char *
set_def(val,
	envar,
	def)
  char *val;
  char *envar;
  char *def;
{
  if (val == (char *)0) {
    val = def;
    if (envar != (char *)0) {
      char *enval, *getenv();

      enval = getenv(envar);
      if (enval != (char *)0)
	val = enval;
    }
  }
  return val;
}

#ifdef NEED_STRTOL
/* %W% */
/* %Q% */
/* %G% */

/*
 * HISTORY:
 * bosh.c,v
 * Revision 1.2  1992/07/31  20:12:22  snl
 * Massive checkin
 *
 * Revision 1.1.1.1  1992/05/08  19:46:18  snl
 * bos 1.2
 *
 * Revision 1.2  92/03/09  01:03:12  snl
 * added QUIET
 * 
 * Revision 1.1  92/03/08  23:25:33  snl
 * Initial revision
 * 
 * Revision 1.2  92/03/03  16:24:59  snl
 * Added -n,-q
 * 
 * Revision 2.2  89/07/11  01:34:51  mbj
 * 	Check into cs libc sources.
 * 	[89/07/10  17:17:53  mbj]
 * 
 */

/*LINTLIBRARY*/
#include <ctype.h>
#define DIGIT(x)	(isdigit(x) ? (x) - '0' : \
			islower(x) ? (x) + 10 - 'a' : (x) + 10 - 'A')
#define MBASE	('z' - 'a' + 1 + 10)

long
strtol(str, ptr, base)
register char *str;
char **ptr;
register int base;
{
	register long val;
	register int c;
	int xx, neg = 0;

	if (ptr != (char **)0)
		*ptr = str; /* in case no number is formed */
	if (base < 0 || base > MBASE)
		return (0); /* base is invalid -- should be a fatal error */
	if (!isalnum(c = *str)) {
		while (isspace(c))
			c = *++str;
		switch (c) {
		case '-':
			neg++;
		case '+': /* fall-through */
			c = *++str;
		}
	}
	if (base == 0)
		if (c != '0')
			base = 10;
		else if (str[1] == 'x' || str[1] == 'X')
			base = 16;
		else
			base = 8;
	/*
	 * for any base > 10, the digits incrementally following
	 *	9 are assumed to be "abc...z" or "ABC...Z"
	 */
	if (!isalnum(c) || (xx = DIGIT(c)) >= base)
		return (0); /* no number formed */
	if (base == 16 && c == '0' && isxdigit(str[2]) &&
	    (str[1] == 'x' || str[1] == 'X'))
		c = *(str += 2); /* skip over leading "0x" or "0X" */
	for (val = -DIGIT(c); isalnum(c = *++str) && (xx = DIGIT(c)) < base; )
		/* accumulate neg avoids surprises near MAXLONG */
		val = base * val - xx;
	if (ptr != (char **)0)
		*ptr = str;
	return (neg ? val : -val);
}
#endif /* NEED_STRTOL */

#ifdef NEED_SINDEX
/*
 * Copyright (c) 1990 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND CARNEGIE MELLON UNIVERSITY
 * DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT
 * SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Carnegie Mellon any
 * improvements or extensions that they make and grant Carnegie the
 * rights to redistribute these changes.
 *
 * Export of this software is permitted only after complying with the
 * regulations of the U.S. Deptartment of Commerce relating to the
 * Export of Technical Data.
 */
/*  sindex  --  find index of one string within another
 *
 *  Usage:  p = sindex (big,small)
 *	char *p,*big,*small;
 *
 *  Sindex searches for a substring of big which matches small,
 *  and returns a pointer to this substring.  If no matching
 *  substring is found, 0 is returned.
 *
 * HISTORY
 * bosh.c,v
 * Revision 1.2  1992/07/31  20:12:22  snl
 * Massive checkin
 *
 * Revision 1.1.1.1  1992/05/08  19:46:18  snl
 * bos 1.2
 *
 * Revision 1.2  92/03/09  01:03:12  snl
 * added QUIET
 * 
 * Revision 1.1  92/03/08  23:25:33  snl
 * Initial revision
 * 
 * Revision 1.2  92/03/03  16:24:59  snl
 * Added -n,-q
 * 
 * Revision 1.4  92/01/27  16:22:02  snl
 * Port to new TCL
 * 
 * Revision 1.2  90/12/11  17:58:59  mja
 * 	Add copyright/disclaimer for distribution.
 * 
 * 26-Jun-81  David Smith (drs) at Carnegie-Mellon University
 *	Rewritten to avoid call on strlen(), and generally speed up things.
 *
 * 20-Nov-79  Steven Shafer (sas) at Carnegie-Mellon University
 *	Adapted for VAX from indexs() on the PDP-11 (thanx to Ralph
 *	Guggenheim).  The name has changed to be more like the index()
 *	and rindex() functions from Bell Labs; the return value (pointer
 *	rather than integer) has changed partly for the same reason,
 *	and partly due to popular usage of this function.
 *
 *  Originally from rjg (Ralph Guggenheim) on IUS/SUS UNIX.
 */


char *sindex (big,small) char *big,*small;
    {
    register char *bp, *bp1, *sp;
    register char c = *small++;

    if (c==0) return(0);
    for (bp=big;  *bp;  bp++)
	if (*bp == c)
	    {
	    for (sp=small,bp1=bp+1;   *sp && *sp == *bp1++;  sp++)
		;
	    if (*sp==0) return(bp);
	    }
    return 0;
    }
#endif NEED_SINDEX

#ifdef NEED_STRDUP
char *strdup(str)
     char *str;
{
  if (str == (char *)0)
    return (char *)0;
  else {
    int length = strlen(str);
    char *buf;

    buf = (char *)ckalloc(length + 1);
    if (buf != (char *)0)
      strcpy(buf, str);
    return buf;
  }
}
#endif /* NEED_STRDUP */

/*
 *---------------------------------------------------------------------------
 * load_up --
 *
 *     For each module on the load_list, we search the directory load_root
 *     for a load file for that module. Module load files are named
 *     <module>/load, e.g. for util it would be <load_root>/util/load.
 *
 *---------------------------------------------------------------------------
 */

int
load_up(interp,
	load_root,
	load_list
	)
  Tcl_Interp *interp;
  char *load_root;
  char *load_list;
{
  int n_load, return_val;
  char **loads = (char **)0;

  return_val = 0;
  if (Tcl_SplitList(interp, load_list, &n_load, &loads) != TCL_OK) {
    fprintf(stderr,
	    "WARNING: SplitList(%s) failed during load_up - nothing loaded.\n",
	    load_list? load_list: "<nil>");
    return_val = -1;
  } else if (!n_load) {
    fprintf(stderr, "WARNING: load_list is empty - nothing loaded.\n");
    return_val = -2;
  } else {
    int i, n_ok;

    for (i = n_ok = 0; i < n_load; i++) {
      char load_filename[400];
      int ok = 1;
      
      sprintf(load_filename, "%s/%s/load", load_root, loads[i]);
      if (!access(load_filename, F_OK)) {
	char dir[300];

	sprintf(dir, "%s/%s", load_root, loads[i]);
	Tcl_SetVar(interp, "LOAD_DIR", dir, TCL_GLOBAL_ONLY);
	if (Tcl_VarEval(interp, "source ", load_filename, 0) != TCL_OK) {
	  fprintf(stderr, "WARNING: error loading %s (%s): %s\n",
		  loads[i], load_filename,
		  Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY));
	  ok = 0;
	} else
	  ok = 1;
	Tcl_UnsetVar(interp, "LOAD_DIR", TCL_GLOBAL_ONLY);
      } else
	ok = 0;
      if (ok)
	n_ok++;
    }
    if (!n_ok) {
      fprintf(stderr, "WARNING: all of {%s} failed to load.\n", load_list);
      return_val = -3;
    } else
      return_val = 0;
  }
  if (loads != (char **)0)
    ckfree((char *)loads);
  return return_val;
}
#ifdef NEED_PANIC
void
panic(str)
  char *str;
{
  fprintf(stderr, "PANIC: %s\n", str);
  fflush(stderr);
  abort();
}
#endif NEED_PANIC
