;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: util.sc,v 1.3 91/09/15 01:13:32 johani Exp $

;;; util.sc -- Utilities used by various SCIX routines.
;;;

(module scixutil) ; (top-level))

(include "../include/types.sch")
(include "../include/lowlevel.sch")
(define-external msg-handler scixmh)

;;; byte-order -- returns #x6c if the processor has least significant byte
;;;               first, #x42 otherwise. The choice of returned values is
;;;               due to the fact that these values are the ones sent to
;;;               the X-server when a connection is requested representing
;;;               LSB first and MSB first respectively.
(define (byte-order)
  (let* ((indian 1) (old-val (scheme-int-ref 'indian 0)))
    (scheme-int-set! 'indian 0 1)	; Change the value...
    (let ((result (scheme-byte-ref 'indian 0))) ; Get first byte.
      (scheme-int-set! 'indian 0 old-val) ; ...and restore it.
      (if (zero? result)		; MSB is zero, thus if the
	  #x42				; First byte is zero we have MSB first
	  #x6c ))))			; else LSB first.

;;; Routines for handling length-tagged c-strings and the schemeified version
;;; consisting of a pair ((<current position> . <total length>) . <c-pointer>)

;;; selectors.
(define c-input-string-string cdr)
(define c-input-string-pos caar)
(define c-input-string-length cdar)
(define c-input-string-info car)

;;; mutator
(define (set-c-input-string-pos! the-pair new-pos)
  (set-car! (car the-pair) new-pos) )

;;; predicate
(define (end-of-input-string? the-pair)
  (>= (caar the-pair)
      (cdar the-pair) ))

;;; get-next-byte! -- returns bytes sequentially from a c-pointer. Takes an 
;;;                   argument created by 'make-c-input-string'.
(define (get-next-byte! the-pair)
  (if (end-of-input-string? the-pair)
      #f
      (let ((result (c-byte-ref (c-input-string-string the-pair)
				(c-input-string-pos the-pair) )))
	(set-car! (c-input-string-info the-pair)
		  (+ (c-input-string-pos the-pair) 1) )
	result )))

;;; get-next-int! -- returns next int from a c-input-string.
(define (get-next-int! the-pair)
  (if (end-of-input-string? the-pair)
      #f
      (let ((result (c-int-ref (c-input-string-string the-pair)
			       (c-input-string-pos the-pair) )))
	(set-car! (c-input-string-info the-pair)
		  (+ (c-input-string-pos the-pair) 4) )
	result )))

;;; get-next-unsigned! -- returns next unsigned from a c-input-string.
(define (get-next-unsigned! the-pair)
  (if (end-of-input-string? the-pair)
      #f
      (let ((result (c-unsigned-ref (c-input-string-string the-pair)
				    (c-input-string-pos the-pair) )))
	(set-car! (c-input-string-info the-pair)
		  (+ (c-input-string-pos the-pair) 4) )
	result )))

;;; peek-next-unsigned -- returns next unsigned that will be read from a
;;;                       c-input-string without updating position.
(define (peek-next-unsigned the-pair)
  (if (end-of-input-string? the-pair)
      #f
      (c-unsigned-ref (c-input-string-string the-pair)
		      (c-input-string-pos the-pair) )))

;;; get-next-short! -- returns next unsigned from a c-input-string.
(define (get-next-short! the-pair)
  (if (end-of-input-string? the-pair)
      #f
      (let ((result (c-shortint-ref (c-input-string-string the-pair)
				 (c-input-string-pos the-pair) )))
	(set-car! (c-input-string-info the-pair)
		  (+ (c-input-string-pos the-pair) 2) )
	result )))

;;; get-next-short-unsigned! -- returns next unsigned from a c-input-string.
(define (get-next-short-unsigned! the-pair)
  (if (end-of-input-string? the-pair)
      #f
      (let ((result (c-shortunsigned-ref (c-input-string-string the-pair)
					 (c-input-string-pos the-pair) )))
	(set-car! (c-input-string-info the-pair)
		  (+ (c-input-string-pos the-pair) 2) )
	result )))

;;; peek-byte -- returns byte nr from c-input-string str without side-effects.
(define (peek-byte str nr)
  (let ((the-pos (+ (c-input-string-pos str) nr)))
    (if (> the-pos (c-input-string-length str))
	#f
	(c-byte-ref (c-input-string-string str) the-pos)  )))

;;; peek-next-byte -- returns the next byte that would be returned by
;;;                   get-next-byte! Does not have any side-effects.
(define (peek-next-byte the-pair)
  (peek-byte the-pair 0) )

;;; make-c-input-string -- given a c-pointer and a length, returns a structure
;;;                        appropriate for get-c-byte.
(define (make-c-input-string pntr len)
  (cons (cons 0 len) pntr) )

;;; convert-length-tagged-string -- given a c-pointer pointing to a buffer 
;;;                                 containing <amount of bytes><the bytes>,
;;;                                 returns a structure appropriate for
;;;                                 next-c-byte.
(define (convert-length-tagged-string pntr)
  (make-c-input-string (+ pntr 4)
		       (c-int-ref pntr 0) ))

;;; lookup-constant -- return the value c as found in the alist.
;;; Note: The warning should probably be an error instead.
(define (lookup-constant c alist)
  (if (symbol? c)
      (let ((r (assoc c alist)))
	(if r
	    (cdr r)
	    (begin 
	      (format #t
      "Warning: lookup-constant: Substituting 0 for unknown constant ~a~%" c)
	      0)))
      c))

;;; s8->s16 -- converts a string8 to a string16 (I think)
(define (s8->s16 string)
  (let ((l (string->list string)))
    (list->string (apply append (map (lambda (e)
				       (list (integer->char 0) e))
				     l)))))

(define (remove-unused als)
  (let loop ((ls als))
    (let ((the-pair (assq 'unused ls)))
      (if the-pair
	  (loop (remq the-pair ls))
	  ls) )))	

;;; flatmap
(define (flatmap fun l)
  (flatten (map fun l)))

(define (flatten l)
  (if (null? l)
      '()
      (append (car l) (flatten (cdr l))) ))

;;; Used in HWOOPS
(define (filter pred l)
  (if (null? l)
      '()
      (let ((el (car l)))
	(if (pred el)
	    (cons el (filter pred (cdr l)))
	    (filter pred (cdr l))))))

;;; Used in HWOOPS
(define (remove-dup l)
  (let loop ((l l) (result '()))
    (if (null? l)
	result
	(let* ((f (car l))
	       (r (cdr l))
	       (m (car f)))
	  (if (assq m result)
	      (loop r result)
	      (if (assq m r)
		  (loop r (append result (list (cons m #f))))
		  (loop r (append result (list f)))))))))

;;; Used in HWOOPS
(define (common-id? l)
  (define (f l)
    (if (null? l)
	#f
	(let ((val (memq (car l) (cdr l))))
	  (if val
	      (car val)
	      (f (cdr l))))))
  (f (flatten l)) )

;;; Tree-copy and list-copy are not included in Scheme->C.
(define (tree-copy tr)
  (if (not (pair? tr))
      tr
      (cons (tree-copy (car tr))
	    (tree-copy (cdr tr)) )))

;;; This list-copy can deal with improper lists.
(define (list-copy ls)
  (cond ((null? ls) '())
	((pair? ls) (cons (car ls)
			  (list-copy (cdr ls)) ))
	(else ls) ))

;;; rem-dupl-name -- used by define-class
(define (rem-dupl-name l1 l2)
    (define (rem-dup l rem-all)
	(if (null? l)
	    '()
	    (let ((m (car l)))
		(if (memq m (cdr l))
		    (let ((r (rem-dup (remq m (cdr l)) rem-all)))
			(if rem-all r (cons m r)))
		    (cons m (rem-dup (cdr l) rem-all))))))
    (rem-dup (append l1 (rem-dup l2 #t)) #f))

;;; list-pad4 -- pads the list l with (pad (length l)) zeroes at the end.
(define (list-pad4 l)
  (let ((p (modulo (- 4 (modulo (length l) 4)) 4)))
    (append l
	    (cdr (assoc p 
			'((0 . ()) (1 . (0)) (2 . (0 0)) (3 . (0 0 0)))))) ))

;;; pad -- returns the amount of padding needed after a string of length n
(define (pad n)
  (modulo (- 4 (modulo n 4)) 4) )

;;; make-textitem -- a new solution to the format of the textitems in requests
;;;                  #74 (PolyText8) and #75 (PolyText16).
;;; 
;;; Format of call: (make-textitem 'string8 0 "Allan" font <font> ...)
;;; If no number precedes a string zero is assumed as delta-value. If the first
;;; argument is either 'string8 or 'string16 it is taken as the type, otherwise
;;; 'string8 is assumed.
;;; Please note that these textitems are only necessary when dealing with
;;; PolyTexts, as ImageTexts use ordinary strings.
;;;
;;; Note1: Not converted to new assembler format in a sensible way. Still deals
;;;        with lists instead of strings.
;;; Note2: Should it not be possible to unify make-polytext and make-textitem
;;;        into one constructor? The current use is kind of stupid...
(define make-textitem
  (lambda l
    (let ((type #f) (list-of-items '()))
      (if (memq (car l) '(string8 string16))
	  (begin
	    (set! type (car l))
	    (set! l (cdr l)) )
	  (set! type 'string8) )
      (let loop ((l l))
	(cond ((null? l))		; Do nothing
	      ((string? (car l))	; Is arg a string?
	       (set! list-of-items
		     (append list-of-items
			     (list (if (eq? type 'string8)
				       (string-length (car l))
				       (/ (string-length (car l)) 2) )
				   0)
			     (map (lambda (x)
				    (bit-and 255 (char->integer x)) )
				  (string->list (car l)) )))
	       (loop (cdr l)) )
	      ((< (length l) 2)		; If not a string then we need 2 args
	       (error 'make-textitem "Number of parameters even.")) ; incl type
	      ((eq? (car l) 'font)
	       (let ((font-id ((cadr l) 'id)))
		 (set! list-of-items
		       (append list-of-items
			       '(255)
			       (let loop ((id font-id) (res '()) (nr 0))
				 (if (= nr 4)
				     res
				     (loop (inexact->exact (quotient id 256))
					   (cons (inexact->exact
						  (modulo id 256)) res)
					   (+ 1 nr))))
			       ))
;			       (if (= (byte-order) #x6c) ; LSB first
;				   (reverse font-id)
;				   font-id)))
		 (loop (cddr l)) ))
	      ((number? (car l))
	       (set! list-of-items
		     (append list-of-items
			     (list (if (eq? type 'string8)
				       (string-length (cadr l))
				       (/ (string-length (cadr l)) 2) )
				   (car l) )
			     (map (lambda (x)
				    (bit-and 255 (char->integer x)) )
				  (string->list (cadr l)) )))
	       (loop (cddr l)) )
	      (else (error 'make-textitem "Unknown argument: ~a" (car l))) ))

      (set! list-of-items (list-pad4 list-of-items))

      (define (me msg . args)		; What args could be useful?
	(cond ((eq? msg 'type) type)
	      ((eq? msg 'object-class) 'textitem)
	      ((eq? msg 'object-system) 'light-weight)
	      ((eq? msg 'supported-messages) '(type items length))
	      ((eq? msg 'me) (lambda () me))
	      ((eq? msg 'object-desc)
	       (let ((ml `((type   . ,type)
			   (items  . ,list-of-items)
			   (length . ,(/ (length list-of-items) 4)) )))
		 (list 'textitem
		       ml
		       'dont-care 
		       (cons (cons 'me (lambda () me)) ml)
		       'dont-care)))
	      ((eq? msg 'items) list-of-items)
	      ((eq? msg 'length) (/ (length list-of-items) 4))
	      (else (error 'make-textitem "Unknown message: ~a" msg)) ))
      me)))

;;; make-atombox -- contructs a container for atoms. It takes two arguments:
;;;                 a list of initial atoms (as pairs of name and id) and a
;;;                 display. The list is intended to contain the "known"
;;;                 standard atoms of the protocol and it is accepted on faith;
;;;                 no check is done with the server.
;;;
;;;                 The atombox accepts the following messages:
;;;                 (abox 'lookup-name <id>) returns the name if the id is
;;;                                          known. If the id isn't known the
;;;                                          server is queried for the name.
;;;                                          If that also fails #f is returned
;;;                                          otherwise the name as a symbol is.
;;;                 (abox 'lookup-id <name>) same as for lookup-name.
;;;                 (abox 'intern! <name>)   interns the name <name> as a new
;;;                                          atom and returns the id.
;;;                 (abox 'object-class)       ==> 'ATOMBOX
;;;                 (abox 'object-system)      ==> 'LIGHT-WEIGHT
;;;                 (abox 'supported-messages) ==>
;;;                                     (lookup-name lookup-id intern! insert!)
;;;                 The use for the atombox is typically to lookup used
;;;                 symbols before sending them to the server (lookup-id, used
;;;                 in the assembler) and lookup found ids before returning
;;;                 them to the caller (lookup-name, used in the disassembler).
;;;
;;;                 Note1: in the disassembler the dpy is a parameter to d-atom
;;;                 (the function that parses an atom) so it is possible to get
;;;                 the name corresponding to the id from the atombox inside
;;;                 the dpy. But on th assembler side that is not possible as
;;;                 a-atom does not have the dpy as a parameter. It is not 
;;;                 probable that we will introduce the dpy as an extra param-
;;;                 eter in all the assembler type functions just to suit
;;;                 a-atom, so it will instead be up to the individual requests
;;;                 to do the lookup-id to get the id corresponding to the
;;;                 name.
;;;                 Note2: To optimize the disassembler we use a vector indexed
;;;                 by id to store names. But the assembler side still uses
;;;                 an ordinary list. When we have a good hash-function for
;;;                 symbols that should be changed.
;;;                 Note3: If the id's grow beyond the size of the vector it
;;;                 should be replaced with a larger one. It isn't currently.
;;;                 Note4: It is NOT possible to query the server for the name
;;;                 associated with an id (with GetAtomName) in 'lookup-name
;;;                 because it is used in the disassembler. It is rather 
;;;                 obvious: if one sent a round-trip request for the name
;;;                 while parsing another reply it would be difficult to resume
;;;                 parsing the first reply after having parsed the second.
;;;                 Therefore lookup-name! is provided. It does the same thing,
;;;                 only it queries the server when an unknown id is used.
(define-external send-getatomname scixreq1)
(define-external send-internatom scixreq1)
(define-external x-reply? scixmu)

(define (make-atombox initial-atoms dpy)
  (let ((a-atoms '()) (d-atoms (make-vector 300 '()))
		      (have-used-set! #f))

    (define (lookup-name id)		; Used by the X disassembler
      (if (< id (vector-length d-atoms))
	  (let ((p (vector-ref d-atoms id)))
	    (if p
		p
		id))			; This means the id is unknown and the
					; server should be queried for it.
	  (error 'atombox "Id out of bounds: ~a Atom-vector to small." id) ))

    (define (lookup-name! id rest)
      (if (< id (vector-length d-atoms))
	  (let ((p (vector-ref d-atoms id)))
	    (if p
		p
		(let ((repl (send-getatomname id dpy rest)))
		  (if (x-reply? repl)
		      (let ((name (string->symbol (repl 'name))))
			(vector-set! d-atoms id name)
			(set! a-atoms (cons (cons name id) a-atoms))
			name)
		      (error 'atombox
			     "Server returned error on GetAtomName: ~a"
			     (repl 'error-name) )))))
	  (error 'atombox "Id out of bounds: ~a Atom-vector to small." id) ))

    ;; Perhaps lookup-id shouldn't do an InternAtom at all? I think it should.
    (define (lookup-id name rest)		; Used by the X assembler
      (let ((pare (assq name a-atoms)))
	(if (pair? pare)
	    (cdr pare)
	    (let ((repl (send-internatom #t (symbol->string name) dpy rest)))
	      (if (x-reply? repl)
		  (let ((id (repl 'atom)))
		    (if (not (eq? id 'None))
			(begin
			  (vector-set! d-atoms id name)
			  (set! a-atoms (cons (cons name id) a-atoms)) ))
		    id)
		  (error 'atombox "Server returned error on InternAtom: ~a"
			 (repl 'error-name) ) )))))
    
    (define (intern! name rest)
      (let ((repl (send-internatom #f (symbol->string name) dpy rest)))
	(if (x-reply? repl)
	    (let ((id (repl 'atom)))
	      (vector-set! d-atoms id name)
	      (set! a-atoms (cons (cons name id) a-atoms))
	      id)
	    (error 'atombox "Server returned error on InternAtom: ~a"
		   (repl 'error-name)) )))

    (define (setbox! l)
      (if (not have-used-set!)
	  (begin
	    (for-each (lambda (p)
			(vector-set! d-atoms (cdr p) (car p)) )
		      l)
	    (set! a-atoms (tree-copy l))
	    (set! have-used-set! #t) )))
    
    (setbox! initial-atoms)

    (define (me msg . args)
      (cond ((eq? msg 'lookup-id) (lookup-id (car args) (cdr args)))
	    ((eq? msg 'lookup-name) (lookup-name (car args)))
	    ((eq? msg 'lookup-name!) (lookup-name! (car args) (cdr args) ))
	    ((eq? msg 'intern!) (intern! (car args) (cdr args)))
	    ((eq? msg 'supported-messages)
	     '(lookup-id lookup-name lookup-name! intern!))
	    ((eq? msg 'object-class) 'atombox)
	    ((eq? msg 'object-system) 'light-weight)
	    ((eq? msg 'me) (lambda () me))
	    ((eq? msg 'object-desc)
	     (let ((ml `((lookup-id    . ,lookup-id)
			 (lookup-name  . ,lookup-name)
			 (lookup-name! . ,lookup-name!)
			 (intern!      . ,intern!) )))
	       (list 'atombox
		     ml
		     'dont-care 
		     (cons (cons 'me (lambda () me)) ml)
		     'dont-care)))))
    me))

;;; A small handy predicate that extracts the color status of the hardware.
(define (color-screen? scr)
  (let* ((a-d (scr 'allowed-depths))
	 (visuals (flatmap (lambda (d)
			     (d 'visuals) )
			   a-d))
	 (classes (map (lambda (o)
			 (o 'class) )
		       visuals)))
    (or (memq 'directcolor classes)
	(memq 'pseudocolor classes)
	(memq 'truecolorcolor classes) )))
  
;;; strip-object is used to extract the optional unparsed object sent through
;;; the X assembler (in the display object).
(define (strip-object obj)
  (if (and (pair? obj) (null? (cdr obj)))
      (strip-object (car obj))
      obj))

(define (match-argument arg-name default arg-list)
  (let ((rest-args (memq arg-name arg-list)))
    (if rest-args
	(cadr rest-args)
	default)))

;;; match-arg -- searches the list arg-list for the tag tag with an eq?. If
;;;              the tag is found then the NEXT element is returned. If the
;;;              tag isn't found and default is bound to 'no-default an error
;;;              results. If default is not bound to 'no-default then it will
;;;              be used instead of the missing value. match-arg has to be a
;;;              macro, because default can be impossible to evaluate under
;;;              some circumstances. Therefore it must only be evaluated when
;;;              it is to be used. The macros match-arg andd match-arg! are
;;;              located in ../macros/hwoops.sc and are only shown here.
;(extend-syntax (match-arg)
;  ((match-arg tag default arg-list)
;   (match-arg-aux tag (lambda () default) arg-list) ))

(define (match-arg-aux tag default arg-list)
  (let ((rest-args (memq tag arg-list)))
    (if rest-args
	(cadr rest-args)
	(let ((default (default)))
	  (if (eq? default 'no-default)
	      (error 'match-arg
		     "Did not find tag ~a and no default provided." tag)
	      default) ))))

;;; match-arg! -- a destructive match-arg. This is sloppy code and could be
;;;               improved. The remq! can be avoided.
;(extend-syntax (match-arg!)
;  ((match-arg! tag default arg-list)
;   (match-arg-aux! tag (lambda () default) arg-list) ))

(define (match-arg-aux! tag default arg-list)
  (let ((rest-args (memq tag arg-list)))
    (if rest-args
	(if (null? (cdr rest-args))
	    (error 'match-arg! "No value found after tag ~a in arglist: ~a"
		   tag arg-list)
	    (let ((result (cadr rest-args)))
	      (set-cdr! rest-args (cddr rest-args))
	      (remq! tag arg-list)
	      result) )
	(let ((default (default)))
	  (if (eq? default 'no-default)
	      (error 'match-arg!
		     "Did not find tag ~a and no default provided." tag)
	      default) ))))

;;; cons* -- it is included in Scheme->C, but not necessarily in other Schemes.
;;;          Therefore we include the code but have it commented out.
;(define (cons* x . y)
;  (letrec ((cons*1 (lambda (x)
;		     (cond ((null? (cdr x)) (car x))
;			   (else (cons (car x) (cons*1 (cdr x))))))))
;    (if (not (null? y))
;	(cons x (cons*1 y))
;	x)))

(define (evaluate class-name msg arg-list methods)
  (let ((pare (assq msg methods)))
    (cond ((not pare)
	   (error class-name "INSTANCE: Unknown message ~a with args: ~a"
		  msg arg-list))
	  ((not (cdr pare))
	   (error class-name "INSTANCE: Ambigous message ~a with args: ~a"
		  msg arg-list))
	  ((not (procedure? (cdr pare))) (cdr pare))
	  (else (begin
		  (apply (cdr pare) arg-list))))))

(define (scix-msg . args)
  (apply format (cons #t args))
  (flush-buffer) )

;;; This one is REALLY useful.
(define (scix-process-events . screens)
  (for-each (lambda (scr)
	      (define-system-file-task
		(scr 'scix-socket)
		#f 
		(lambda () (msg-handler 'scix-process-events scr)) ))
	    screens))

;;; Stupid name. But is it really worth it to hack the top-level function?
(define (scix-reset)
  (enable-system-file-tasks #t))

;;; getprop-type -- used in the 'getprop method in window. Experimental!

;;; The procedure GETPROP-TYPE gets an object SIMPLE-OBJ (or #F) qualified
;;; by KEY for the type TYPE associated by the display DPY. + convenience
;;; routines for encoding/decoding with no assigned DPY

(define (getprop-type type key dpy)
  (let ((obj (getprop type key)))
    (if (pair? obj)
	(let ((obj-pair (assq dpy obj)))
	  (if obj-pair (cdr obj-pair) #f))
	obj)))
