;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: objects.sc,v 1.5 91/09/15 01:13:26 johani Exp $

;;; Some SCIX Objects.

;;; Object types closely coupled to the X protocol: drawable pixmap window gc
;;; pointer (in lw) cursor keyboard (in lw) display screen colormap
;;; color (in lw).
;;;

(module scixobj (top-level screen visual drawable pixmap font cursor))

(include "../include/lw-objs.sch")
(include "../include/requests.sch")
(include "../include/msgutil.sch")
(include "../include/util.sch")
(include "../include/masks.sch")

(include "../macros/extsyntax.sc")
(include "../macros/oos.sc")

;;; The SCIX Screen Object.

;;; Screens are a special subclass of display in that it inherits a special
;;; instance of display rather than the display class. This facilitates several
;;; screens to use and change the same display object's local state, which is
;;; exactly what is needed to make it possible to replace the dpy parameter
;;; everywhere with a screen ditto.

;;; Note: We want the screen object to inherit more or less all the display
;;; object methods but it should also export the contents of the root-data
;;; object. The really beautiful way to achieve that would be if it was
;;; possible to inherit both the display instance and the root-data instance.
;;; That is possible now...

(define-class (screen root-data dpy)
  (locals
   (whitepixel (color 'make #xffff #xffff #xffff))
   (blackpixel (color 'make 0 0 0))
   (gcache #f)
   (visuals #f)
   (root-visual #f)
   (default-colormap ((root-data 'default-colormap)))
   (root ((root-data 'root))) )
  (inherit root-data dpy)
  (methods
   (whitepixel (lambda () whitepixel))
   (blackpixel (lambda () blackpixel))
   (root (lambda () root))
   (default-colormap (lambda () default-colormap))
   (root-visual (lambda () root-visual))
   (visuals (lambda () visuals))
   (scix-root-data (lambda () root-data))
   
   ;; Messages mapped directly on X requests
   (listinstalledcolormaps (lambda rest	         ; #83: ListInstalledColormaps
			     (let ((r (send-listinstalledcolormaps me rest)))
			       (if (x-reply? r)
				   (r 'cmaps)
				   r))))
   ;; #92: LookupColor -- returns a list of two lists - the first list contains
   ;; the exact RGB values, and the second list contains the visual RGB values.
   (lookupcolor (lambda (name . rest)
		  (let ((reply (send-lookupcolor name me rest)))
		    (if (x-reply? reply)
			(list
			 (list (reply 'exact-red)
			       (reply 'exact-green)
			       (reply 'exact-blue) )
			 (list (reply 'visual-red)
			       (reply 'visual-green)
			       (reply 'visual-blue) ))
			#f))))
   (gcache (lambda ()
	     (if gcache
		 gcache
		 (begin
		   (set! gcache (graphics-cache 'make 'screen me))
		   gcache)))) )
  (init
   (root 'set-screen! me)		; A bit of a kludge, I'm afraid.
   (whitepixel 'set-pixel! (me 'white-pixel))
   (blackpixel 'set-pixel! (me 'black-pixel))
   (root 'set-width! (root-data 'width-in-pixels))
   (root 'set-height! (root-data 'height-in-pixels))
   (root 'set-depth! (root-data 'root-depth))
   
   ;; Insert the visuals in the "known id space"
   (set! visuals (map (lambda (d)
			(cons (d 'depth)
			      (map (lambda (v)
				     (visual 'make v me (v 'id) ))
				   (d 'visuals) )))
		      (me 'allowed-depths) ))
   (set! root-visual ((dpy 'scix-id-vector) 'lookup (root-data 'root-visual)))
   ))

;;; Visuals -- does not use optional parameters.
(define-class (visual visual-data scr id)
  (inherit visual-data)
  (methods)
  (init
   ((scr 'scix-id-vector) 'insert-with-key! me id) ))

;;; Drawables -- windows and pixmaps.
(define-class (drawable . args)
  (locals
   (scr (match-arg 'screen 'no-default args))
   (width  (match-arg 'width 300 args))
   (height (match-arg 'height 300 args))
   (depth  (match-arg 'depth 1 args))
   (callbacks '()) )
  (inherit
   (resource args) )
  (methods
   ;; Simple selectors...
   (depth (lambda () depth))
   (width (lambda () width))
   (height (lambda () height))
   ;; ...and mutators. Be careful with these. There be dragons here!
   (set-depth! (lambda (arg) (set! depth arg)))
   (set-width! (lambda (arg) (set! width arg)))
   (set-height! (lambda (arg) (set! height arg)))
   (screen (lambda () scr))
   
   ;; A GC below denotes a member of the disjoint union between the set
   ;; of all gc object instances and {#F} - #F denotes a "nil" gc.
   ;; A graphic object is either simple (a primitive object or a view)
   ;; or a pair P ((car P) is a graphic object and (cdr P) is a GC).
   ;; The object P is called a GC CLOSURE. The graphic object P is always
   ;; interpreted using the outermost non "nil" bound GC (starting with
   ;; the gc parameter in the DRAW message).
   
   (draw (lambda (obj-list the-gc . rest)
	   (define (draw-graphic o the-gc)
	     (if (and (procedure? o) (eq? (o 'object-class) 'view))
		 (for-each (lambda (o)
			     (draw-graphic o the-gc) )
			   (o 'contents) )
		 (if (pair? o)
		     (draw-graphic (car o) (if the-gc the-gc (cdr o)))
		     (o 'draw the-gc me rest) )))
	   ;(display (format "DRAWABLE: 'draw obj-list: ~a~%" obj-list))
	   (for-each (lambda (o)
		       (draw-graphic o the-gc) )
		     obj-list) ))
   
   ;; Event-handling and callbacks
   (callbacks (lambda () callbacks))
   
   (add-callback! (lambda (event-kind handler)
		    (let ((cb-l (assq event-kind callbacks)))
		      (if cb-l
			  (set-cdr! (last-pair cb-l) (list handler))
			  (set! callbacks (cons (list event-kind handler)
						callbacks)) ))))
   (set-callbacks! (lambda (l)
		     (set! callbacks l) ))
   
   (remove-callback! (lambda (event-kind handler)
		       (let ((cb-l (assq event-kind callbacks)))
			 (if cb-l
			     (set! cb-l (remq! handler cb-l))
			     #f))))
   
   ;; How to detect which events are mine and which belong to other windows?
   ;; The problem is that usually it won't be "me" that is the event-window,
   ;; but rather something that has inherited me. But this something (like a
   ;; toggle-button) should always be the same (in a given widget inheritance
   ;; hierarchy only one object can be the top-level one). That is not
   ;; necessarily true in theory, but it must be in practice for this scheme
   ;; to work. Therefore the top-level object is supplied by the event-
   ;; handler in obj. Also note that the message 'my-events! returns a list
   ;; of all consecutive events belonging to this window.
   ;; Note1: for now the callbacks are called with one event at a time, not
   ;;        the msg-handler.
   ;; Note2: the ev-source can be either a list of events or the msg-handler.
   ;;        This is necessary to take care of derived classes with local
   ;;        event-handlers that extract the events from the msg-handler.
   
   (scix-event-dispatch (lambda (ev-source obj)
			  (let ((ev-l (if (pair? ev-source)
					  ev-source
					  (ev-source 'my-events! obj) )))
			    (let loop ((e (car ev-l)) (ev-l (cdr ev-l)))
			      (let ((pare (assq (e 'event-name) callbacks)))
				(if pare
				    ((cadr pare) e obj)
				    #f))
			      (if (not (null? ev-l))
				  (loop (car ev-l) (cdr ev-l)) )))))
   
   ;; Methods mapped directly on X requests.
   (getgeometry (lambda rest		                    ; #14: GetGeometry
		  (send-getgeometry me scr rest) ))
   ;; #62: Copyarea
   (copyarea (lambda (src-d the-gc src-x src-y dst-x dst-y width height . rest)
	       (send-copyarea src-d me the-gc src-x src-y
			      dst-x dst-y width height scr rest)))
   (copyplane (lambda (src-d the-gc src-x src-y dst-x dst-y   ; #63: CopyPlane
			     width height bit-plane . rest)
		(send-copyplane src-d me the-gc src-x src-y dst-x dst-y
				width height bit-plane scr rest)))
   (putimage (lambda (fmt the-gc width height dst-x dst-y     ; #72: PutImage
			  left-pad byte-list . rest)
	       (send-putimage fmt me the-gc width height dst-x dst-y
			      left-pad byte-list scr rest) ))
   (getimage (lambda (fmt x y width height plane-mask . rest) ; #73: GetImage
	       (send-getimage fmt me x y width height
			      plane-mask scr rest) ))
   (querybestsize (lambda (class width height . rest)    ; #97: QueryBestSize
		    (let ((reply (send-querybestsize class me width height
						     scr rest)))
		      (if (x-reply? reply)
			  (list (reply 'width) (reply 'height))
			  reply))))
   ))

;;; Pixmaps.
(define-class (pixmap . args)
  (locals
   (scr      (match-arg 'screen 'no-default args))
   (width    (match-arg 'width 100 args))
   (height   (match-arg 'height 100 args))
   (depth    (match-arg 'depth (scr 'root-depth) args))
   (dble     (scr 'root)) )
  (inherit
   (drawable args) )
  (methods
   (drawable (lambda () dble))
   (createpixmap (lambda rest			; #53: CreatePixmap
		   (send-createpixmap me scr rest) ))
   (freepixmap (lambda rest			; #54: FreePixmap
		 (me 'scix-denounce-id!)
		 (send-freepixmap me scr rest) ))
   )
  (init
   (me 'createpixmap) ))

;;; The SCIX Font Object.
(define-class (font . args)
  (locals
   (scr (match-arg 'screen 'no-default args))
   (charinfo-vec #f)
   (fontname (match-arg! 'fontname "fixed" args)) )
  (inherit
   (resource args) )
  (methods
   (fontname (lambda () fontname))
   (openfont (lambda (font-name . rest)	; #45: OpenFont
	       (set! fontname font-name)
	       (send-openfont me font-name scr rest) ))
   (closefont (lambda rest			; #46: CloseFont
		(me 'scix-denounce-id!)
		(set! fontname #f)
		(send-closefont me scr rest) ))
   (queryfont (lambda rest			; #47: QueryFont (also in gc)
		(let ((r (send-queryfont me scr rest)))
		  (if (x-reply? r)
		      (if (and (zero? (r 'min-byte1)) (zero? (r 'max-byte1)))
			  (begin
			    (set! charinfo-vec
				  (make-vector (+ (r 'max-char-or-byte2) 1)) )
			    (let loop ((i (r 'min-char-or-byte2)) (cis (r 'charinfos)))
			      (vector-set! charinfo-vec i (car cis))
			      (if (not (null? (cdr cis)))
				  (loop (+ i 1) (cdr cis)) ))))))))

   ;; #48: QueryTextExtents (also in gc)
   (querytextextents (lambda (string . rest)
		       (send-querytextextents me string scr rest) ))
   )
  ;; The small init routine
  (init
   (me 'openfont fontname) )
  )

;;; The SCIX Cursor Object.
;;;
;;; Note: Now the cursor is automatically allocated in the X server as well
;;;       as in SCIX. As a cursor can be created through two different X
;;;       requests it is necessary to indicate which one is desired through
;;;       a parameter to the constructor rather than through the choice of
;;;       creator-method which was used before. The current protocol is that
;;;       if the symbol 'glyph is present among the args, then the cursor is
;;;       created with CreateGlyphCursor, otherwise with CreateCursor.
;;;
(define-class (cursor . args)
  (locals
   (scr        (match-arg 'screen 'no-default args))
   (fore-color (match-arg! 'fore-color (scr 'blackpixel) args))
   (back-color (match-arg! 'back-color (scr 'whitepixel) args))
   (src        (match-arg! 'source 'no-default args))
   (mask       (match-arg! 'mask 'None args))
   (src-char   (match-arg! 'source-char 0 args)) ; Useless default, but
   (mask-char  (match-arg! 'mask-char 0 args))   ; it doesn't matter.
   (x-hot      (match-arg! 'x-hot 0 args))
   (y-hot      (match-arg! 'y-hot 0 args)) )
  (inherit
   (resource args) )
  (methods
   (fore-color (lambda () fore-color))
   (back-color (lambda () back-color))
   
   (createcursor (lambda (src mask x-hot y-hot . rest) ; #93: CreateCursor
		   (send-CreateCursor me src mask fore-color
				      back-color x-hot y-hot scr rest) ))   
   (createglyphcursor (lambda (src mask src-char mask-char . rest) ; #94
			(send-createglyphcursor me src mask
						src-char mask-char fore-color
						back-color scr rest) ))   
   (freecursor (lambda rest			        ; #95: FreeCursor
		 (me 'scix-denounce-id!)
		 (send-freecursor me scr rest) ))
   (recolorcursor (lambda (f-color b-color . rest)	; #96: RecolorCursor
		    (set! fore-color f-color)
		    (set! back-color b-color)
		    (send-recolorcursor me fore-color
					back-color scr rest) ))
   )
  (init
   (if (memq 'glyph args)
       (me 'createglyphcursor src mask src-char mask-char)
       (me 'createcursor src mask x-hot y-hot) ))
  ) ;; End of define-class
