;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: dtypes.tmpl,v 1.1 91/07/25 16:49:46 johani Exp $

;;; types.sc -- converters from the different datatypes used by the X
;;;             protocol to lists of integers. The integers should be in the
;;;             range 0..255 for subsequent conversion to characters.
;;;

(module scixdtypes); (top-level))

(include "../include/util.sch")
(include "../include/masks.sch")
(include "../include/objects.sch")
(include "../include/lw-objs.sch")
(include "../include/lowlevel.sch")

(include "../macros/extsyntax.sc")		; Needed by mask-object.sc
(include "../macros/mask-obj.sc")

;;;
;;; X disassembler type functions
;;;

;;; d-identity is not actually a type function, but a constructor for such.
;;; It is used solely to fit the name of a reply or event into the object.
(define (d-identity arg)
  (lambda (str dpy)
    arg))

(define (d-card8 str dpy)
  (get-next-byte! str) )

;;; NOTE! The following require sizeof(int) and sizeof(unsigned) to be 32,
;;;       sizeof(short) and sizeof(short unsigned) to be 16.

;;; d-const -- tries to parse a constant using the assoc-list clist. If
;;;            always-const is #t, an error is signaled if no constant
;;;            for the numeric value is found, otherwise it is parsed
;;;            normally. The latter case is often used when an id can
;;;            either be a resource or a constant eg, None.
;(define (d-const always-const func clist)
;  (lambda (str dpy)
;    (if always-const			      ; Can it be an id or a constant?
;	(let* ((val (func str dpy))
;	       (pare (assv val clist)))
;	  (if pare
;	      (cdr pare)
;	      (error 'd-const "No symbolic constant for value ~a" val) ))
;	(let* ((val (peek-next-unsigned str)) ; Only card32 can have both.
;	       (pare (assv val clist)) )
;	  (if pare
;	      (begin
;		((d-unused 4) str dpy)         ; Read past the value.
;		(cdr pare) )
;	      (func str dpy) )))))

(define (d-const always-const func clist)
  (let ((val #f) (pare #f))
    (lambda (str dpy)
      (if always-const			      ; Can it be an id or a constant?
	  (begin
	    (set! val (func str dpy))
	    (set! pare (assv val clist))
	    (if pare
		(cdr pare)
		(error 'd-const "No symbolic constant for value ~a" val) ))
	  (begin
	    (set! val (peek-next-unsigned str)) ; Only card32 can have both.
	    (set! pare (assv val clist))
	    (if pare
		(begin
		  ((d-unused 4) str dpy)         ; Read past the value.
		  (cdr pare) )
		(func str dpy) ))))))

(define (d-card16 str dpy)
  (get-next-short-unsigned! str) )

(define (d-card32 str dpy)
  (get-next-unsigned! str) )

(define (d-int16 str dpy)
  (get-next-short! str) )

(define (d-int32 str dpy)
  (get-next-int! str) )

(define d-drawable
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond (((dpy 'scix-id-vector) 'lookup id))
	    (else (error 'd-drawable "Unknown drawable id: ~a" id)) ))))

;;; d-window 
;;; 910729 johani -- New design: the window constructor has a new optional
;;;                  parameter 'mask where we now supply an empty mask.
;;;                  Also, as we now automatically do a CreateWindow when
;;;                  instantiatiating windows a method to avoid doing that
;;;                  for windows "owned" by other clients was needed. The
;;;                  current (ad hoc) solution is to have 'no-auto-create
;;;                  among the args to the window constructor, which is
;;;                  checked in the init-part of the window just before sending
;;;                  the CreateWindow request.
;;;                  Another problem is how to find out the correct screen for
;;;                  a window not previously known. The present solution is
;;;                  obviously not correct in all situations and to have a
;;;                  special type for root windows isn't pretty...
(define d-rootwindow
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond ((zero? id) 'None)
	    ((= id 1) 'PointerRoot)
	    (((dpy 'scix-id-vector) 'lookup id))    ; Is this a known window?
	    (else (window 'make 'screen dpy 'parent #f 'no-auto-create
			  'depth #f 'id id 'mask (make-window-value-mask)))))))

(define d-window
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond ((zero? id) 'None)
	    ((= id 1) 'PointerRoot)
	    (((dpy 'scix-id-vector) 'lookup id))    ; Is this a known window?
	    (else (window 'make 'screen (dpy 'defaultscreen) 'parent #f
			  'no-auto-create
			  'depth #f 'id id 'mask (make-window-value-mask)))))))

(define d-pixmap
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond (((dpy 'scix-id-vector) 'lookup id))      ; Is this a known pixmap?
	    (else (error 'd-pixmap "Unknown pixmap id: ~a" id)) ))))

(define d-colormap
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond ((zero? id) 'None)		         ; Colormaps can be 'None
	    (((dpy 'scix-id-vector) 'lookup id)) ; Is this a known colormap?
	    (else (colormap 'make 'screen dpy 'id id)) ))))

(define d-cursor
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond (((dpy 'scix-id-vector) 'lookup id))      ; Is this a known cursor?
	    (else (error 'd-cursor "Unknown cursor id: ~a" id)) ))))

(define d-font
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond (((dpy 'scix-id-vector) 'lookup id))       ; Is this a known font?
	    (else (error 'd-font "Unknown font id: ~a" id)) ))))

;;; Never used in the core protocol, but why bother...
(define d-gcontext
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond (((dpy 'scix-id-vector) 'lookup id))	 ; Is this a known gc?
	    (else (error 'd-gcontext "Unknown gcontext id: ~a" id)) ))))

(define d-fontable
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond (((dpy 'scix-id-vector) 'lookup id))  ; Is this a known "fontable"?
	    (else (error 'd-fontable "Unknown fontable id: ~a" id))))))

(define d-atom
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (if (zero? id)
	  'None
	  ((dpy 'atombox) 'lookup-name id) ))))

(define d-visualid
  (let ((id #f))
    (lambda (str dpy)
      (set! id (d-card32 str dpy))
      (cond (((dpy 'scix-id-vector) 'lookup id))  ; Is this a known visualid?
	    (else id) ))))

(define d-timestamp
  (let ((t #f))
    (lambda (str dpy)
      (set! t (d-card32 str dpy))
      (if (zero? t)
	  'CurrentTime
	  t) )))

;;; Several types are really CARD32's...
(define d-keysym  d-card32)

;;; ...or CARD8's
(define d-byte    d-card8)
(define d-keycode d-card8)
(define d-button  d-card8)

;;; This is cheating a bit...
(define d-int8 d-card8)

(define (d-bool str dpy)
  (not (zero? (d-card8 str dpy))) )

;;; define-enum-type --- macro for enumerated types

;;; syntax (define-enum-type type name alist)
;;; a-name and d-name are automaticly defined

(eval-when (eval compile load)

(define (turn alist) ; reverses the pairs in alist - Obs alist must be quoted
  (let loop ((alist (cadr alist)) (res '()))
    (if (null? alist)
        res
        (let ((el (car alist)))
          (loop (cdr alist) (append res (list (cons (cdr el) (car el)))))))))

) ; end of eval-when

(extend-syntax (define-enum-type)
  ((define-enum-type type name alist)
   (with ((dlist (turn 'alist))
	  (a-name (string->symbol (format "A-~a" 'name)))
	  (d-name (string->symbol (format "D-~a" 'name)))
	  (a-type (string->symbol (format "A-~a" 'type)))
	  (d-type (string->symbol (format "D-~a" 'type))) )
	 (begin
	   (define a-name (lambda (x buffer dpy)

                            (pad4 3 buffer)

			    (a-type (lookup-constant x alist) buffer dpy)

                            ))
	   (define d-name (lambda (str dpy)
			    ((d-const #t d-type 'dlist) str dpy)))
	   ))))

(define-enum-type CARD8 bitgravity '((Forget    . 0)
				     (NorthWest . 1)
				     (North     . 2)
				     (NorthEast . 3)
				     (West      . 4)
				     (Center    . 5)
				     (East      . 6)
				     (SouthWest . 7)
				     (South     . 8)
				     (SouthEast . 9)
				     (Static    . 10) ))

(define-enum-type CARD8 wingravity '((Unmap     . 0)
				     (NorthWest . 1)
				     (North     . 2)
				     (NorthEast . 3)
				     (West      . 4)
				     (Center    . 5)
				     (East      . 6)
				     (SouthWest . 7)
				     (South     . 8)
				     (SouthEast . 9)
				     (Static    . 10) ))

(define-enum-type CARD8 window-stack-mode '((Above    . 0)
					    (Below    . 1)
					    (TopIf    . 2)
					    (BottomIf . 3)
					    (Opposite . 4) ))

(define-enum-type CARD8 gc-function '((Clear        . 0)
				      (And          . 1)
				      (AndReverse   . 2)
				      (Copy         . 3)
				      (AndInverted  . 4)
				      (NoOp         . 5)
				      (Xor          . 6)
				      (Or           . 7)
				      (Nor          . 8)
				      (Equiv        . 9)
				      (Invert       . 10)
				      (OrReverse    . 11)
				      (CopyInverted . 12)
				      (OrInverted   . 13)
				      (Nand         . 14)
				      (Set          . 15) ))

(define-enum-type CARD8 gc-line-style '((Solid      . 0)
					(OnOffDash  . 1)
					(DoubleDash . 2) ))

(define-enum-type CARD8 gc-cap-style '((NotLast    . 0)
				       (Butt       . 1)
				       (Round      . 2); RoundCap and RoundJoin
				       (Projecting . 3) ))

(define-enum-type CARD8 gc-join-style '((Miter . 0)
					(Round . 1); RoundCap and RoundJoin
					(Bevel . 2) ))

(define-enum-type CARD8 gc-fill-style '((Solid          . 0)
					(Tiled          . 1)
					(Stippled       . 2)
					(OpaqueStippled . 3) ))

(define-enum-type CARD8 gc-fill-rule '((EvenOdd . 0)
				       (Winding . 1) ))

(define-enum-type CARD8 gc-subwindow-mode '((ClipByChildren  . 0)
					    (IcludeInferiors . 1) ))

(define-enum-type CARD8 gc-arc-mode '((Chord    . 0)
				      (PieSlice . 1) ))


(define-enum-type CARD8 onoff-mode '((Off     . 0)
				     (On      . 1)
				     (Default . 2) ))



;(define (d-bitgravity str dpy)
;  ((d-const #t d-card8
;	    '((0 . Forget)
;	      (1 . NorthWest)
;	      (2 . North)
;	      (3 . NorthEast)
;	      (4 . West)
;	      (5 . Center)
;	      (6 . East)
;	      (7 . SouthWest)
;	      (8 . South)
;	      (9 . SouthEast)
;	      (10 . Static) ))
;   str dpy))

;(define (d-wingravity str dpy)
;  ((d-const #t d-card8
;	    '((0 . Unmap)
;	      (1 . NorthWest)
;	      (2 . North)
;	      (3 . NorthEast)
;	      (4 . West)
;	      (5 . Center)
;	      (6 . East)
;	      (7 . SouthWest)
;	      (8 . South)
;	      (9 . SouthEast)
;	      (10 . Static) ))
;   str dpy))

;;; masks... 

;;; johani -- What should we do about this one?
;(define (d-bitmask str dpy)		; Only used in event ConfigureRequest
;  (let ((val (d-card32 str dpy))	; (event #23)
;	(mask-obj (make-mask 
;		   (x y width height border-width sibling stack-mode) )))
;    (mask-obj 'set-mask! val)
;    mask-obj))

;;; johani -- What should we do about this one?
(define (d-bitmask str dpy)		; Only used in event ConfigureRequest
  (let ((val (d-card32 str dpy))	; (event #23)
	(mask-obj (make-bitmask 'configurerequest)) )
    (mask-obj 'set-mask! val)
    mask-obj))

;(define (d-setofevent str dpy)
;  (let ((val (d-card32 str dpy))
;	(mask-obj (make-event-mask)) )
;    (mask-obj 'set-mask! val)
;    mask-obj))

(define (d-setofevent str dpy)
  (let ((val (d-card32 str dpy))
	(mask-obj (make-bitmask 'event)) )
    (mask-obj 'set-mask! val)
    mask-obj))

;(define (d-setofpointerevent str dpy)
;  (let ((val (d-card32 str dpy))
;	(mask-obj (make-pointerevent-mask)) )
;    (mask-obj 'set-mask! val)
;    mask-obj))

(define (d-setofpointerevent str dpy)
  (let ((val (d-card32 str dpy))
	(mask-obj (make-bitmask 'pointerevent)) )
    (mask-obj 'set-mask! val)
    mask-obj))

;(define (d-setofdeviceevent str dpy)
;  (let ((val (d-card32 str dpy))
;	(mask-obj (make-deviceevent-mask)) )
;    (mask-obj 'set-mask! val)
;    mask-obj))

(define (d-setofdeviceevent str dpy)
  (let ((val (d-card32 str dpy))
	(mask-obj (make-bitmask 'deviceevent)) )
    (mask-obj 'set-mask! val)
    mask-obj))

;(define (d-setofkeybutmask str dpy)
;  (let ((val (d-card16 str dpy))
;	(mask-obj (make-keybutmask-mask)) )
;    (mask-obj 'set-mask! val)
;    mask-obj))

(define (d-setofkeybutmask str dpy)
  (let ((val (d-card16 str dpy))
	(mask-obj (make-bitmask 'keybutmask)) )
    (mask-obj 'set-mask! val)
    mask-obj))

(define (d-string8 len str dpy)
  (let* ((string-pos (c-input-string-pos str))         ; Position of string
	 (end-pos (+ string-pos len))                  ; End of string
	 (tmpbyte (c-byte-ref (c-input-string-string str)
			      end-pos))                ; Byte after string
	 (pad (modulo (- 4 (modulo len 4)) 4))         ; Amount of padding
	 (result '()))
    (c-byte-set! (c-input-string-string str)
		 end-pos
		 0)                                    ; Add NUL-byte
    (set! result (c-string->string
		  (+ (c-input-string-string str)
		     string-pos )))                    ; Get the string
    (c-byte-set! (c-input-string-string str)
		 end-pos
		 tmpbyte)                              ; Reset last byte
    (set-c-input-string-pos! str (+ pad end-pos))      ; Update position
    result ))                                          ; Return the string

(define (d-str str dpy)
  (let* ((len (get-next-byte! str))
	 (string-pos (c-input-string-pos str))
	 (end-pos (+ string-pos len))
	 (tmpbyte (c-byte-ref (c-input-string-string str)
			      end-pos))
	 (result '()) )
    (c-byte-set! (c-input-string-string str)
		 end-pos
		 0)                                    ; Add NUL-byte
    (set! result (c-string->string
		  (+ (c-input-string-string str)
		     string-pos )))                    ; Get the string
    (c-byte-set! (c-input-string-string str)
		 end-pos
		 tmpbyte)                              ; Reset last byte
    (set-c-input-string-pos! str end-pos)              ; Update position
    result ))                                          ; Return the string

;;; d-host -- Only used in ListHosts-reply, so it could preferrably be
;;;           declared inline. It is correct for IP addresses, but not for
;;;           DECnet ditto.
(define (d-host str dpy)
  (let* ((domain (lookup-constant
		  (get-next-byte! str)
		  '((0 . Internet)
		    (1 . DECnet)
		    (2 . Chaos) )))
	 (unused (get-next-byte! str))
	 (addr-len (d-card16 str dpy))
	 (addr (let loop ((n addr-len) (r '()))
		 (if (zero? n)
		     r
		     (loop (- n 1) 
			   (append r (list (get-next-byte! str))) )))) )
    (if (eq? domain 'DECnet)
	(let ((node-nr (bit-or (bit-lsh (bit-and 3 (cadr addr)) 8)
			       (car addr) ))
	      (area (bit-rsh (bit-and (cadr addr) 252) 2)) )
	  (set! addr (list addr (list area node-nr)) )))
    (list domain addr) ))

;;; d-list -- a compound disassembler type constructor.
(define (d-list type len)
  (lambda (str dpy)
    (let loop ((n len) (result '()))
      (if (or (end-of-input-string? str)
	      (zero? n) )
	  result
	  (loop (- n 1) (append result (list (type str dpy)))) ))))

;;; 
(define (d-object type)
  (lambda (str dpy)
    (let ((val (type str dpy)))
      (lambda ()
	val))))

;;; d-unused -- creates a procedure which strips off n bytes from its argument
;;;             when called.
(define (d-unused n)
  (lambda (str dpy)
    (set-c-input-string-pos! str (+ (c-input-string-pos str) n)) ))

