;;;
;;;              Copyright 1991 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; The game "Battleships" for two X servers. Johan Ihren.

;;; Warning: This code was written in haste and contains many strange and
;;; unclear spots. Especially dangerous is the various structure sharing
;;; that takes place between the two pairs of charts. But, given that, it
;;; does show rather more of SCIX capabilities than the other demo apps.

;;; $Id: ship.sc,v 2.4 91/09/15 01:01:41 johani Exp $

(module demoship (top-level demo-ship))

(include "../include/stox.sch")
(include "../include/scix.sch")

(include "../macros/extsyntax.sc")
(include "../macros/oos.sc")

(define-external view-textfile demoutil)

;;; chart -- this is the chequered square that is used for almost all
;;;          interaction when playing: deployment of ships and shooting.
(define-class (chart . args)
  (locals
   (console (match-arg 'console 'no-default args))
   (scr     (console 'screen))
   (gcache  (match-arg 'gcache (apply graphics-cache (cons 'make args)) args))
   (is-master (match-arg 'master #f args))
   (gc-draw (if is-master
		(gcache 'gc-reverse)
		(gcache 'gc-draw) ))
   (gc-rev (if is-master
		(gcache 'gc-draw)
		(gcache 'gc-reverse) ))
   (partner #f)
   (rows (match-arg 'rows 10 args))
   (columns (match-arg 'columns 10 args))
   (status #f)				; chart status variable
   (matrix #f)				; where ship layout is stored.
   (hits #f)				; a graphic obj indicating hits.
   (shots #f)				; a polyfillrectangle of missed shots
   (ships #f)				; a polyfillrectangle of ship blocks
   (ship-blocks 0)			; # of currently deployed "blocks"
   (net (if is-master
	    (polysegment
	     'make
	     (append (let loop ((x 20) (seg-list '()))
		       (if (< x (* 20 columns))
			   (loop (+ x 20) (cons (list x 0 x (* 20 rows))
						seg-list))
			   seg-list))
		     (let loop ((y 20) (seg-list '()))
		       (if (< y (* 20 rows))
			   (loop (+ y 20)
				 (cons (list 0 y (* 20 columns) y) seg-list))
			   seg-list)) ))
	    #f))
   (dpy-list (list net ships shots)) )
  (inherit
   (widget (cons* 'width (* 20 columns) 'height (* 20 rows) 'border-width 1
		  'value-mask (make-window-value-mask
			       `(background-pixel ,(if is-master
						       (scr 'blackpixel)
						       (scr 'whitepixel) ))
			       `(event-mask ,(make-mask 'event
							'Exposure
							'ButtonPress
							'ButtonRelease)))
		  args)) )
  (methods
   (stox-expose (lambda (e)
		  (me 'draw (list net ships shots) gc-draw)
		  (if is-master
		      (me 'draw (list hits) gc-rev)
		      (me 'draw (list hits) gc-draw))
		  (scr 'flush!) ))
   (register-chart (lambda (s)
		     (set! partner s)
		     (list rows columns shots net) ))
   (set-status! (lambda (s)
		 (set! status s) ))

   ;; 'shoot -- this is sent from the "shooting" chart to its partner.
   (shoot (lambda (x y)
	    (let ((hit (matrix-ref matrix x y)))
	      (if hit
		  (begin
		    (hits 'add! (list (list (* 20 x) (* 20 y)
					    (+ 20 (* 20 x)) (+ 20 (* 20 y)) )
				      (list (+ 20 (* 20 x)) (* 20 y)
					    (* 20 x) (+ 20 (* 20 y)) )))
		    (me 'draw (list hits) gc-rev)
		    (matrix-set! matrix x y 'hit)
		    (scr 'flush!)
		    (set! ship-blocks (- ship-blocks 1)) )
		  (console 'system 'our-turn) ) ; Our turn to shoot.
	      (cond ((not hit) #f)
		    ((<= ship-blocks 0)
		     (console 'system 'finito)
		     'finito)
		    ((ship-sunk? matrix x y) 'ship-sunk)
		    (else hit) ))))

   (deployed-ok? (lambda ()
		   (let ((r (check-deployment matrix '#(4 3 2 1))))
		     (if (string? r) (console 'console-msg r))
		     (eq? r #t) )))

   ;; 'activate -- used by the console to connect one "ship-chart" with
   ;;              the corresponding "shot-chart".
   (activate (let ((first-time #t))
	       (lambda (p)
		 (set! ship-blocks 0)
		 (set! partner p)
		 (cond (is-master (set! matrix (make-matrix rows columns #f))
				  (set! hits (polysegment 'make '()))
				  (set! shots (polyfillrectangle 'make '()))
				  (set! ships (polyfillrectangle 'make '()))
				  (set! dpy-list (list shots net)) )
		       (else (partner 'activate me)
			     (let ((data (partner 'register-chart me)))
			       (set! shots (caddr data))
			       (set! net (cadddr data))
			       (set! ships (polyfillrectangle 'make '()))
			       (set! rows (car data))
			       (set! columns (cadr data))
			       (set! matrix (make-matrix rows columns #f))
			       (set! hits (polyfillrectangle 'make '()))
			       (set! dpy-list (list shots net)) )))
		 (if first-time
		     (begin
		       (me '(widget stox-activate))
		       (set! first-time #f) )
		     (begin
		       (me 'unmapwindow)
		       (me 'mapwindow) )))))
   
   ;; 'update-display -- used by the chart itself whenever its visual
   ;;                    appearance has changed.
   (update-display (lambda ()
		     (me 'draw (list ships shots) gc-draw)
		     (if is-master
			 (me 'draw (list hits) gc-rev)
			 (me 'draw (list hits) gc-draw))
		     (scr 'flush!))) )
  (init
   (me 'add-callback! 'ButtonPress
       (lambda (e w)
	 (case status
	   ((placement)
	    (let ((x (inexact->exact (/ (e 'event-x) 20)))
		  (y (inexact->exact (/ (e 'event-y) 20))) )
	      (cond ((matrix-ref matrix x y)
		     (ships 'remove! (list (+ 3 (* 20 x)) (+ 3 (* 20 y))
					   15 15))
		     (me 'draw (list (polyfillrectangle
				      'make (list (list (+ 3 (* 20 x))
							(+ 3 (* 20 y))
							15 15))))
			 gc-rev)
		     (matrix-set! matrix x y #f)
		     (set! ship-blocks (- ship-blocks 1))
		     (console 'console-msg "Deploy ~a more blocks."
			      (- 20 ship-blocks)) )
		    ((not (deploy-ok? matrix x y))
		     (console 'console-msg "Illegal position.") )
		    ((< ship-blocks 20)
		     (ships 'add! (list (+ 3 (* 20 x)) (+ 3 (* 20 y))
					15 15))
		     (matrix-set! matrix x y 'ship)
		     (set! ship-blocks (+ ship-blocks 1))
		     (if (< ship-blocks 20)
			 (console 'console-msg "Deploy ~a more blocks."
				  (- 20 ship-blocks))
			 (console 'console-msg "All blocks deployed.") ))
		    (else
		     (console 'console-msg "No more ships.")
		     (scr 'bell 40)))
	      (me 'update-display)
	      (scr 'flush!)))
	   ((shoot)
	    (let ((x (inexact->exact (/ (e 'event-x) 20)))
		  (y (inexact->exact (/ (e 'event-y) 20))) )
	      (if (not (matrix-ref matrix x y))
		  (begin
		    (matrix-set! matrix x y #t)
		    (let ((hit (partner 'shoot x y)))
		      (cond (hit
			     (sound 5 scr)
			     (hits 'add!
				   (list (list (+ 3 (* 20 x))
					       (+ 3 (* 20 y)) 15 15)))
			     (cond ((eq? hit 'ship-sunk)
				    (console 'console-msg
					     "Ship sunk. Still your turn.") )
				   ((not (eq? hit 'finito))
				    (console 'console-msg
					     "A hit. Still your turn.") )))
			    (else
			     (shots 'add!
				    (list (list (+ 7 (* 20 x))
						(+ 7 (* 20 y)) 6 6)))
			     (set! status #f)
			     (console 'system 'no-hit) ))))
		  (console 'console-msg "You have already hit that square.")))
	    (me 'update-display)
	    (partner 'update-display) )
	   ((#f) (scr 'bell 40) (scr 'flush!)) )))))

;;; console -- this is a container for two charts and most of the game logic.
;;;            It is also a mess.
(define-class (console . args)
  (locals
   (scr (match-arg 'screen 'no-default args))
   (gcache (graphics-cache 'make 'screen scr 'fontname "variable"))
   (opponent #f)
   (our-name "")
   (opponent-name "opponent")
   (namef #f) (msg-label #f) (ship-chart #f) (shot-chart #f) (ngb #f)
   (our-tag #f) (opp-tag #f) (our-box #f) (opp-box #f) (db #f)
   (resources '())
   (shipblocks 0)
   (ready-first #f) )
  (inherit
   (container (cons* 'parent (scr 'root) 'title "SCIX Battleships" args)))
  (methods

   ;; 'opponent-msg -- used for communication between the two consoles.
   (opponent-msg (lambda (msg . args)
		    (cond ((eq? msg 'quit)
			   (set! opponent #f)
			   (namef 'unmapwindow)
			   (me 'console-msg "~a just quit." 
			       (if (equal? opponent-name "opponent")
				   "Your opponent"
				   opponent-name)))
			  ((eq? msg 'name)
			   (set! opponent-name (car args))
			   (opp-tag 'set-label! opponent-name)
			   (opp-box 'stox-arrange) )
			  ((eq? msg 'ship-chart) ship-chart)
			  ((eq? msg 'placement-done)
			   (if (not ready-first)
			       (set! ready-first 'opponent) ))
			  ((eq? msg 'begin)
			   (me 'console-msg "You begin.")
			   (shot-chart 'set-status! 'shoot))
			  ((eq? msg 'we-are-sunk) ; msg from other console
			   (me 'console-msg "You won. Congratulations!")
			   (ngb 'stox-activate) )
			  (else #f) )))

   (restart (lambda (deployment)
	      (set! ready-first #f)
	      (shot-chart 'activate (opponent 'opponent-msg 'ship-chart))
	      (if deployment
		  (me 'system 'new-game) )))

   ;; 'system -- keep track of how far the user have got.
   (system (lambda (msg)
	     (cond ((eq? msg 'deployed)	      ; msg from the "Deployed"-button.
		    (ship-chart 'set-status! #f)
		    (cond ((not opponent)
			   (me 'console-msg "No opponent. Sorry."))
			  (ready-first	      ; I.e. opponent was ready first.
			   (begin
			     (me 'console-msg "~a begins." opponent-name)
			     (opponent 'opponent-msg 'begin) ))
			  (else
			   (me 'console-msg "Waiting on ~as deployment."
			       opponent-name)
			   (set! ready-first #t)
			   (opponent 'opponent-msg 'placement-done) )))
		   ((eq? msg 'our-turn)	      ; msg from ship-chart
		    (me 'console-msg "Your turn.")
		    (shot-chart 'set-status! 'shoot))
		   ((eq? msg 'no-hit)	      ; msg from shot-chart
		    (me 'console-msg "~as turn." opponent-name) )
		   ((eq? msg 'finito)	      ; msg from ship-chart
		    (me 'console-msg "You lost.")
		    (opponent 'opponent-msg 'we-are-sunk) )
		   ((eq? msg 'new-game)
		    (ship-chart 'set-status! 'placement)
		    (me 'console-msg "Deploy your ships.")
		    (db 'mapwindow) )
		   (else #f) )))

   ;; 'activate -- create various widgets and get hold of the opponent console.
   (activate
    (lambda (o)
      (if (not opponent)
	  (begin
	    (set! opponent o)
	    (set! db (button 'make 'gcache gcache 'parent me 'label "Deployed"
			     'on-action (lambda (e)
					  (if (ship-chart 'deployed-ok?)
					      (begin
						((e 'event) 'unmapwindow)
						(me 'system 'deployed) )))))
	    (me 'restart #f)
	    (set! ngb (button 'make 'parent me 'gcache gcache
			      'label "New game" 'on-action
			      (lambda (e)
				((e 'event) 'stox-free)
				(opponent 'restart #t)
				(me 'restart #t) )))
	    (set! msg-label (label 'make 'parent me 'gcache gcache))
	    (set! our-tag (label 'make 'parent me 'gcache gcache))
	    (set! opp-tag (label 'make 'parent me 'gcache gcache))
	    (map (lambda (o) (o 'stox-activate))
		 (list msg-label our-tag opp-tag))
	    (set! namef (stringeditor 'make 'parent me 'width 150
				    'gcache gcache))
	    ;; Set up the StringEditor to return result here:
	    (namef 'stox-notify-result
		   (lambda (str)
		     (set! our-name str)
		     (our-tag 'set-label! str)
		     (our-box 'stox-arrange)
		     (namef 'unmapwindow)
		     (if opponent (opponent 'opponent-msg 'name str))
		     (me 'system 'new-game) ))
	    (let ((qb (button 'make 'parent me 'label "Quit"
			      'gcache gcache 'on-action (lambda (e)
							  (me 'cleanup)) ))
		  (rb (button 'make 'parent me 'label "Rules"
			      'gcache gcache 'on-action
			      (lambda (e)
				(view-textfile
				 'filename "demo/ship.rules" 'screen scr
				 'title "Short Battleship Rules")))) )
	      (map (lambda (o) (o 'stox-activate))
		   (list qb rb our-tag opp-tag namef msg-label))
	      (me 'console-msg "Enter name:")
	      (set! our-box
		    (vbox (hbox-to (ship-chart 'width) (hss) our-tag (hss))
			  (vskip 3) ship-chart))
	      (set! opp-box
		    (vbox (hbox-to (shot-chart 'width) (hss) opp-tag (hss))
			  (vskip 3) shot-chart))
	      (me 'set-box!
		  (vbox (vfill) (vskip 5) (hbox (hskip 3) our-box (hskip 5)
						(hfill) opp-box (hskip 3))
			(vskip 3)
			(hbox-to (+ 11 (* 2 (our-box 'width))) (hskip 3) qb
				 (hskip 3) msg-label (hskip 3)
				 (hbox-to 0 namef (hss)) (hfill) db
				 (hbox-to 0 (hss) ngb) (hskip 3) rb (hskip 3) )
			(vskip 3) (vfill) ))
	       (me 'mapwindow)
	       (scr 'flush!)
	       (o 'activate me) )))))

   ;; 'cleanup -- free resources and destroy console.
   (cleanup (lambda ()
	      (if opponent (opponent 'opponent-msg 'quit))
	      (gcache 'free)
	      (me 'stox-free) ))
   
   ;; 'setup -- used by the main program to get the console to allocate some
   ;;           resources needed BEFORE activation.
   (setup (lambda ()
	    (set! ship-chart (chart 'make 'parent me 'master #t 'console me
				      'x 5 'y 30 'gcache gcache))
	    (set! shot-chart (chart 'make 'parent me 'x 220 'y 30
				   'gcache gcache 'console me))))

   ;; 'console-msg -- displays the argument message in the message area.
   (console-msg (lambda args
		  (msg-label 'set-label! (apply format args)) ))
   ))

(define (demo-ship scr1 scr2)
  (let ((console1 (console 'make 'width 430 'height 260 'screen scr1))
	(console2 (console 'make 'width 430 'height 260 'screen scr2)) )
    (console1 'setup)
    (console2 'setup)
    (console1 'activate console2)	; Inform the consoles of each other.
    (scix-process-events scr1 scr2) 
    (list console1 console2) ))		; Return the consoles for debugging.

;;; A trivial matrix type.
(define (make-matrix rows cols contents)
  (list 'matrix
	rows
	cols
	(list->vector (let loop ((rows rows) (ls '()))
			(if (= rows -1)
			    ls
			    (loop (- rows 1)
				  (cons (make-vector (+ cols 1) contents)
					ls)))))))

(define matrix-rows cadr)
(define matrix-cols caddr)

(define (matrix-ref m x y)
  (vector-ref (vector-ref (cadddr m) y) x) )

(define (matrix-set! m x y v)
  (vector-set! (vector-ref (cadddr m) y) x v) )

(define (safe-matrix-ref m x y)
  (if (or (< x 0) (> x (caddr m)) (< y 0) (> y (cadr m)))
      #f
      (matrix-ref m x y) ))

;;; deploy-ok? -- check whether it would be legal or not to put a ship
;;;               (or part of a ship) at (x,y).
(define (deploy-ok? m x y)
  (not (or (safe-matrix-ref m (- x 1) (- y 1))
	   (safe-matrix-ref m (- x 1) (+ y 1))
	   (safe-matrix-ref m (+ x 1) (- y 1))
	   (safe-matrix-ref m (+ x 1) (+ y 1)) )))

;;; ship-sunk? -- return #t if the ship at (x,y) is completely sunk.
(define (ship-sunk? m x y)
  (define (ship? x y) (safe-matrix-ref m x y))
  (define (aux x y dx dy)
    (cond ((eq? (safe-matrix-ref m x y) 'hit)
	   (aux (+ x dx) (+ y dy) dx dy))
	  ((eq? (safe-matrix-ref m x y) 'ship)
	   #f)
	  (else #t)))
  (cond ((or (ship? (- x 1) y) (ship? (+ x 1) y))
	 (and (aux (- x 1) y -1 0) (aux (+ x 1) y 1 0)) )
	((or (ship? x (- y 1)) (ship? x (+ y 1)))
	 (and (aux x (- y 1) 0 -1) (aux x (+ y 1) 0 1)) )
	(else #t) ))

;;; check-deployment -- examine the entire chart for illegal ship cominations.
(define (check-deployment m ships-allowed)
  ;; Copy the supplied ship-info.
  (let ((ships-allowed (list->vector (vector->list ships-allowed)))
	(rows (matrix-rows m)) (cols (matrix-cols m)) )
    (let ((val
	   (call-with-current-continuation
	    (lambda (throw)
	      (let oloop ((i 0) (j 0))
		(if (> j rows)
		    (if (equal? ships-allowed '#(0 0 0 0))
			(throw #t)
			(throw "You haven't deployed all ships.") ))
		(if (eq? (safe-matrix-ref m i j) 'ship)
		    (if (safe-matrix-ref m (+ i 1) j) 
			(let iloop ((n (+ i 1)) (found 0)) ; Horizontal ship
			  (if (= found 4)
			      (throw "Too long ship detected.") )
			  (if (safe-matrix-ref m n j)
			      (iloop (+ n 1) (+ found 1))
			      (begin
				(if (zero? (vector-ref ships-allowed found))
				    (throw
				     (format "Too many ships of length ~a."
					     (+ found 1) )))
				(vector-set! ships-allowed found
					     (- (vector-ref ships-allowed
							    found)
						1))
				(if (>= n cols)
				    (oloop 0 (+ j 1))
				    (oloop (+ n 1) j) ))))
			(begin			; Vertical ship or submarine
			  (if (not (safe-matrix-ref m i (+ j 1)))
			      (begin		; Submarine
				(if (zero? (vector-ref ships-allowed 0))
				    (throw
				     (format "Too many ships of length 1.")))
				(vector-set! ships-allowed 0
					     (- (vector-ref ships-allowed 0)
						1) )
				(if (= i cols)
				    (oloop 0 (+ j 1))
				    (oloop (+ i 1) j) ))
			      (let iloop ((n (+ j 1)) (found 0))
				(if (= found 4)
				    (throw "Too long ship detected.") )
				(if (safe-matrix-ref m i n)
				    (begin
				      (matrix-set! m i n 'already-checked)
				      (iloop (+ n 1) (+ found 1)) )
				    (begin
				      (if (zero? (vector-ref ships-allowed
							     found))
					  (throw
					   (format
					    "Too many ships of length ~a."
					    (+ found 1) )))
				      (vector-set! ships-allowed found
						   (- (vector-ref ships-allowed
								  found)
						      1))
				      (if (= i cols)
					  (oloop 0 (+ j 1))
					  (oloop (+ i 1) j) )))))))
		    (if (= i cols)
			(oloop 0 (+ j 1))
			(oloop (+ i 1) j) )))))))
      ;; Restore matrix.
      (let loop ((i 0) (j 0))
	(if (<= j (matrix-rows m))
	    (begin
	      (if (eq? (matrix-ref m i j) 'already-checked)
		  (matrix-set! m i j 'ship) )
	      (if (= i (matrix-cols m))
		  (loop 0 (+ j 1))
		  (loop (+ i 1) j) ))))
      val)))

;;; small utility
(define (sound count scr)
  (let loop ((c count))
    (scr 'bell 50)
    (if (positive? c) (loop (- c 1)))))
