;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: msg-util.sc,v 1.2 90/06/26 10:03:56 johani Exp $

;;; The SCIX message formats.

(module scixmu)

(include "../include/util.sch")
(include "../include/lowlevel.sch")
(include "../include/types.sch")

(define-external send-GetInputFocus scixreq1)

(include "formats.sc")

(define *current-reply* '())		 ; Used for backtracking during parsing
					 ; of replies

;;; make-reply -- Fetches format for 'name' from *reply-formats* and
;;;               unpacks 'reply' according to this. This is done in the
;;;               following way:
;;;               *reply-formats* is an association list of the form
;;;               '((reply-name . ((name type) ...)
;;;                               ...) ...)
;;;               The appropriate field is copied (so as to not destroy the
;;;               list itself), whereafter a
;;;               (map (lambda (ls)
;;;                      (set-cdr! (car ls) ((cadar ls) str))
;;;                     <local copy of format>))
;;;               essentially is performed. Thus, the local format-copy is
;;;               transformed into an association list of the form
;;;               '((id . val) ...), in which we then can search for messages
;;;               using assq.
;;;               (Isn't functional programming fun? :-)
;;;               The fiddling with *current-reply* appears to be inevitable,
;;;               as the binding of symbols is, to say the least, subtle. The
;;;               (let ((safe *current-reply*) ...) (set! *current-reply* safe)
;;;               is a push/pop version needed for parsing replies which 
;;;               themselves parse replies (initial reply and replies with
;;;               charinfos). Hopefully these will be remade, and we won't
;;;               have this problem...

(define (make-reply reply str dpy)
  (let* ((safe *current-reply*)         ; This is ugly!!!
	 (fl (let ((format (tree-copy reply)))
	       (set! *current-reply* format)  
	       format))
	 (the-reply (let loop ((ls fl))	; Change cdr of each sub-list
		      (if (null? ls)
			  (remove-unused fl)
			  (begin
			    (set-cdr! (car ls) ((cdar ls) str dpy))
			    (loop (cdr ls)) )))))
    (set! *current-reply* safe)		; See above...

    ;; Note: the args are there to conform with the protocol used when sub-
    ;;       classing. It is not used for anything within a reply.
    (define (me msg . args)
      (cond ((eq? msg 'supported-messages)
	     (cons 'display (map car the-reply)))
	    ((eq? msg 'object-class) 'reply)
	    ((eq? msg 'object-system) 'light-weight)
	    ((eq? msg 'me) (lambda () me))
	    ((eq? msg 'object-desc)
	     (list 'reply
		   the-reply
		   'dont-care
		   (cons (cons 'me (lambda () me)) the-reply)
		   'dont-care))
	    ((eq? msg 'display) dpy)
	    (else
	     (let ((the-pair (assq msg the-reply)))
	       (if the-pair
		   (cdr the-pair)
		   (error (cdr (assq 'reply the-reply)) 
			  "No such message: ~a" msg) )))))
    me))

;;; make-event --- creates an event of type event-no from the input-string str.
;;;                The string is parsed sequentially, and each local binding
;;;                "strips off" as many bytes as it needs. The format (i.e,
;;;                the definition of the event) is fetched from the vector
;;;                (dpy 'scix-event-formats), which contains name and type of
;;;                each field in the object as a pair. This information is
;;;                copied, and the type is replaced by the value obtained by
;;;                applying it to the input string when parsing. Example of
;;;                usage:
;;;                      (define the-event (make-event event-type reply))
;;;                for construction, and eg,
;;;                       (the-event 'event)
;;;                to obtain the name of the event.
(define (make-event event-no str dpy)
  ;; sent-event is true if the event is created through a SendEvent request
  ;; from some client.
  (let* ((sent-event (not (zero? (bit-and event-no #x80))))
	 (event-code (bit-and event-no #x7f))
	 (evf (let loop ((ls (vector-ref (dpy 'scix-event-formats) event-code))
			 (result '()))
		(if (null? ls)
		    result
		    (loop (cdr ls) (append result (list (cons (caar ls)
							      (cdar ls) )))))))
	 (the-event (let lop ((ls evf))
		      (if (null? ls)
			  (remove-unused evf)
			  (begin
			    (set-cdr! (car ls) ((cdar ls) str dpy))
			    (lop (cdr ls)) ))))
	 (valid-msgs (append '(display sent-event?)
			     (map car the-event) )))
    (define (me msg . args)
      (let ((the-pair (assq msg the-event)))
	(if the-pair
	    (cdr the-pair)
	    (cond ((eq? msg 'supported-messages) valid-msgs)
		  ((eq? msg 'object-class) 'event)
		  ((eq? msg 'object-system) 'light-weight)
		  ((eq? msg 'me) (lambda () me))
		  ((eq? msg 'sent-event?) sent-event)
		  ((eq? msg 'display) dpy)
		  ((eq? msg 'object-desc) #f) ; Not meaningful.
		  (else (error 'event "No such message ~a" msg)) ))))
    me))

;;; make-error -- Creates an error object from a byte stream. Not entirely
;;;               unlike make-event, only errors are easier, as they only have
;;;               one field that may vary from error to error. The message for
;;;               obtaining the error-name is 'error-name.

(define (make-error str dpy)
  (let* ((type (d-card8 str dpy))
	 (the-err (vector-ref (dpy 'scix-error-names) type))
	 (err-name (car the-err))
	 (info-type (cdr the-err))
	 (seq-nr (d-card16 str dpy))
	 (info (let ((the-info (d-card32 str dpy)))
		 (case info-type
		   ('bad-resource-id
		    ((dpy 'scix-id-vector) 'lookup the-info) )
		   ('bad-atom-id
		    ((dpy 'atombox) 'lookup-name the-info) )
		   (else the-info) )))
	 (minor-opcode (d-card16 str dpy))
	 (major-opcode (d-card8 str dpy))
	 (unused ((d-unused 21) str dpy))
	 (valid-msgs (remove 'unused `(error-name seq-nr display
				       ,info-type minor-opcode
				       major-opcode))))
    (define (me msg . args)
      (cond ((eq? msg 'supported-messages) valid-msgs)
            ((eq? msg 'object-class) 'error)
	    ((eq? msg 'object-system) 'light-weight)
	    ((eq? msg 'me) (lambda () me))
            ((eq? msg 'error-name) err-name)
	    ((eq? msg 'seq-nr) seq-nr)
	    ((and (eq? msg info-type)
		  (not (eq? msg 'unused)) ) info)
	    ((eq? msg 'minor-opcode) minor-opcode)
	    ((eq? msg 'major-opcode) major-opcode)
	    ((eq? msg 'display) dpy)
	    ((eq? msg 'object-desc) #f)	; Not meaningful
	    (else (error 'error "No such message ~a" msg)) ))
    me))

;;; Create the queues.
(define *error-queue* '())	  

(define (empty-error-queue)
  (set! *error-queue* '()))

(define (reply-seq-nr reply)
  (reply 'seq-nr) )

(define (error-seq-nr error)
  (error 'seq-nr) )

(define (x-error? message)
  (eq? 'error (message 'object-class)) )

(define (x-event? message)
  (eq? 'event (message 'object-class)) )

(define (x-reply? message)
  (eq? 'reply (message 'object-class)) )

;;; type-or-zero -- Used only in ListFontsWithInfo (qv).
(define (type-or-zero func)
  (lambda (str dpy)
    (if (zero? (backtrack 'length-of-name))
	0
	(func str dpy) )))

;;; backtrack -- Used for getting a previously parsed value. Needed since
;;;              make-reply can't bind the symbols. See make-reply for an
;;;              explanation of *current-reply*.
(define (backtrack sym)
  (cdr (assq sym *current-reply*)) )
