#!/usr/bin/env perl

use strict;
use warnings;

use Graph::Easy::Marpa;

use Getopt::Long;

use Pod::Usage;

# -----------------------------------------------

my($option_parser) = Getopt::Long::Parser -> new();

my(%option);

if ($option_parser -> getoptions
(
 \%option,
 'cooked_file=s',
 'description=s',
 'dot_input_file=s',
 'format=s',
 'help',
 'input_file=s',
 'logger=s',
 'maxlevel=s',
 'minlevel=s',
 'output_file=s',
 'parsed_tokens_file=s',
 'rankdir=s',
 'report_items=i',
 'report_stt=i',
 'stt_file=s',
 'timeout=i',
 'type=s',
) )
{
	pod2usage(1) if ($option{'help'});

	# Return 0 for success and 1 for failure.

	exit Graph::Easy::Marpa -> new(%option) -> run;
}
else
{
	pod2usage(2);
}

__END__

=pod

=head1 NAME

gem.pl - Run Graph::Easy::Marpa::Lexer and Graph::Easy::Marpa::Parser.

=head1 SYNOPSIS

gem.pl [options]

	Options:
	-cooked_file aCookedFileName
	-description graphDescription
	-dot_input_file aDotInputFileName
	-format outputGraphFormat
	-help
	-input_file aRawFileName
	-logger loggerOrEmptyString
	-maxlevel logOption1
	-minlevel logOption2
	-output_file aGraphOutputFile
	-parsed_tokens_file aTokenFileName
	-rankdir LR or RL or TB or BT
	-report_items 0 or 1
	-report_stt 0 or 1
	-stt_file sttFileName
	-timeout seconds
	-type '' or csv or ods

Exit value: 0 for success, 1 for failure. Die upon error.

=head1 OPTIONS

=over 4

=item -cooked_file aCookedFileName

Specify the name of a CSV file of cooked tokens to write. This file can be input to the parser.

There is no default value.

=item -description graphDescription

Specify a graph description string for the DFA to process.

You are strongly encouraged to surround this string with '...' to protect it from your shell.

See also the -input_file option to read the description from a file.

The -description option takes precedence over the -input_file option.

There is no default value.

=item -dot_input_file aDotInputFileName

Specify the name of a file that the rendering engine can write to, which will contain the input
to dot (or whatever). This is good for debugging.

Default: ''.

If '', the file will not be created.

=item -format outputGraphFormat

The format (e.g. 'svg') to pass to the rendering engine.

=item -help

Print help and exit.

=item -input_file aRawFileName

Read the graph description string from a file.

See also the -description option to read the graph description from the command line.

The whole file is slurped in as 1 graph.

The first lines of the file can start with /\s*#/, and will be discarded as comments.

The -description option takes precedence over the -input_file option.

There is no default value.

=item -logger loggerOrEmptyString

Specify a logger to pass to the lexer and the parser.

In scripts/gem.sh, use -l '' to stop logging.

=item -maxlevel logOption1

This option affects Log::Handler.

See the Log::handler docs.

The default maxlevel is 'info'. Another typical value is 'debug'.

=item -minlevel logOption2

This option affects Log::Handler.

See the Log::handler docs.

The default minlevel is 'error'.

No lower levels are used.

=item o -output_file aGraphOutputFile

A file to which the graph is written.

If not specified (the default), the graph is not saved.

The default is ''.

=item o -parsed_tokens_file aTokenFileName

The list of tokens generated by the parser will be written to this file.

If not specified (the default), the tokens are not saved.

The default is ''.

=item -rankdir LR or RL or TB or BT

Specify the rankdir of the graph as a whole.

Default: TB (top to bottom).

=item -report_items 0 or 1

Report the items recognized in the cooked file.

The default value is 0.

=item -report_stt 0 or 1

Call Set::FA::Element.report(). Set min and max log levels to 'info' for this.

The default value is 0.

=item -stt_file sttFileName

Specify which file contains the state transition table.

Default: ''.

The default value means the STT is read from the source code of Graph::Easy::Marpa::Lexer.

Candidate files are '', 'data/default.stt.csv' and 'data/default.stt.ods'.

The type of this file must be specified by the -type option.

Note: If you use stt_file => your.stt.ods and type => 'ods', L<Module::Load>'s load() will be used to
load L<OpenOffice::OODoc>. This module is no longer listed in Build.PL and Makefile.PL as a pre-req,
so you will need to install it manually.
 
=item -timeout seconds

Run the DFA for at most this many seconds.

Default: 3.

=item -type '' or cvs or ods

Specify the type of the stt_file: '' for internal STT, csv for CSV or ods for Open Office Calc spreadsheet.

Default: ''.

The default value means the STT is read from the source code of Graph::Easy::Marpa::Lexer.

This option must be used with the -stt_file option.

=back

=cut
