package File::Rename;

use 5.032;  # use strict; use warnings;

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw( rename );

our $VERSION = '2.01_99';

sub import {
    my $pack = shift;
    my($args, $config) = &_config;  # sees @_
    $pack->export_to_level(1, $pack, @$args);
    require File::Rename::Options;
    File::Rename::Options->import(@$config);
}

sub rename_files {
    my $code = shift;
    my $options = shift;
    _default(\$options);

    my $sub = $code;
    if ( $options->{unicode_strings} ) {
        require File::Rename::Unicode;
        $sub = File::Rename::Unicode::code($code,
                            $options->{encoding});
    }
    my $errors;
    for (@_) {
        my $was = $_;
        if ( $options->{filename_only} ) {
            require File::Spec;
            my($vol, $dir, $file) = File::Spec->splitpath($_);
            $sub->() for ($file);
            $_ = File::Spec->catpath($vol, $dir, $file);
        }
        else {
            $sub->();
        }

        if( $was eq $_ ){ }     # ignore quietly
        elsif( -e $_ and not $options->{over_write} ) {
            if (/\s/ or $was =~ /\s/ ) {
                warn  "'$was' not renamed: '$_' already exists\n";
            }
            else {
                warn  "$was not renamed: $_ already exists\n";
            }
            $errors ++;
        }
        elsif( $options->{no_action} ) {
            print "rename($was, $_)\n";
        }
        elsif( CORE::rename($was,$_)) {
            print "$was renamed as $_\n" if $options->{verbose};
        }
        else {  warn  "Can't rename $was $_: $!\n"; $errors ++; }
    }
    return !$errors;
}

sub rename_list {
    my($code, $options, $fh, $file) = @_;
    _default(\$options);
    print "Reading filenames from ",
      ( defined $file ?                 $file
        : defined *{$fh}{SCALAR} and
          defined ${*{$fh}{SCALAR}} ?   ${*{$fh}{SCALAR}}
        :                               "file handle ($fh)"
      ),
      "\n" if $options->{verbose};
    my @file;
    {
        local $/ = "\0" if $options->{input_null};
        chop(@file = <$fh>);
    }
    rename_files $code, $options,  @file;
}

sub rename {
    my($argv, $code, $verbose) = @_;
    if( ref $code ) {
        if( 'HASH' eq ref $code ) {
            if(defined $verbose ) {
                require Carp;
                Carp::carp(<<CARP);
File::Rename::rename: third argument ($verbose) ignored
CARP
            }
            $verbose = $code;
            $code = delete $verbose->{_code};
            unless ( $code ) {
                require Carp;
                Carp::carp(<<CARP);
File::Rename::rename: no _code in $verbose
CARP
            }

        }
    }
    unless( ref $code ) {
        if( my $eval = eval <<CODE )
sub {
$code
}
CODE
        {
            $code = $eval;
        }
        else {
            my $error = $@;
            $error =~ s/\b(at\s+)\(eval\s+\d+\)\s/$1/g;
            $error =~ s/(\s+line\s+)(\d+)\b/$1 . ($2-1)/eg;
            $error =~ s/\.?\s*\z/, in:\n$code\n/;
            die $error;
        }
    }
    if( @$argv ) { rename_files $code, $verbose, @$argv }
    else { rename_list $code, $verbose, \*STDIN, 'STDIN' }
}

sub _default {
    my $ref = shift;
    return if ref $$ref;
    my $verbose = $$ref;
    $$ref = { verbose => $verbose }
}

sub _config {
    # copied from GetOpt::Long::import
    my @syms = ();              # symbols to import
    my @config = ();            # configuration
    my $dest = \@syms;          # symbols first
    for ( @_ ) {
        if ( $_ eq ':config' ) {
            $dest = \@config;   # config next
            next;
        }
        push(@$dest, $_);       # push
    }
    return (\@syms, \@config);
}
    
1;

__END__

=head1 NAME

File::Rename - Perl extension for renaming multiple files

=head1 SYNOPSIS

  use File::Rename qw(rename);          # hide CORE::rename
  rename \@ARGV, sub { s/\.pl\z/.pm/ }, 1;

  use File::Rename;
  File::Rename::rename \@ARGV, '$_ = lc';

  use File::Rename qw(:config no_require_order);

=head1 DESCRIPTION

=head2 USE OPTIONS

Parameters to C<use File::Rename> consists of
functions to be imported and configuration options.

The only exported function is C<rename()>.  The
configuation options are preceded by :config,
and are passed to File::Rename::Options.

=head2 FUNCTIONS

=over 4

=item C<rename( FILES, CODE [, VERBOSE])>

rename FILES using CODE,
if FILES is empty read list of files from stdin

=item C<rename_files( CODE, VERBOSE, FILES)>

rename FILES using CODE

=item C<rename_list( CODE, VERBOSE, HANDLE [, FILENAME])>

rename a list of file read from HANDLE, using CODE

=back

=head2 OPTIONS

=over 8

=item FILES

List of files to be renamed,
for C<rename> must be an ARRAY reference

=item CODE

Subroutine to change file names,
for C<rename> can be a string,
otherwise it is a code reference

=item VERBOSE

Flag for printing names of files successfully renamed,
optional for C<rename>

=item HANDLE

Filehandle to read file names to be renames

=item FILENAME (Optional)

Name of file that HANDLE reads from

=back

=head2 HASH

Either CODE or VERBOSE can be a HASH of options.

If CODE is a HASH, VERBOSE is ignored 
and CODE is supplied by the B<_code> key.

Other options are 

=over 16

=item B<verbose>

As VERBOSE above, provided by B<-v>.

=item B<input_null>

Input separator \0 when reading file names from stdin.

=item B<no_action>

Print names of files to be renamed, but do not rename
(i.e. take no action), provided by B<-n>.

=item B<over_write>

Allow files to be over-written by the renaming, provided by B<-f>. 

=item B<filename_only>

Only apply renaming to the filename component of the path, 
provided by B<-d>.

=item B<show_help>

Print help, provided by B<-h>.

=item B<show_manual> 

Print manual page, provided by B<-m>.

=item B<show_version> 

Print version number, provided by B<-V>.

=item B<unicode_strings> 

Enable unicode_strings feature, provided by B<-u>.

=item B<encoding> 

Encoding for filenames, provided by B<-u>.

=back

=head2 EXPORT

rename

=head1 ENVIRONMENT

No environment variables are used.

=head1 SEE ALSO

mv(1), perl(1), rename(1)

=head1 HISTORY

Revision history for Perl extension File::Rename.

=over

=item Larry Wall

Revision: 4.1   Date: 92/08/07 17:20:30 

=item Robin Barker

    # Revision 1.1  1997/02/27  15:48:51  rmb1
    # Initial revision

    # Revision 1.2  1997/02/27  16:15:40  rmb1

    # Revision 1.3  1997/02/27  16:39:07  rmb1
    # added -v

    # Revision 1.4  1997/02/27  17:19:26  rmb1
    # corrected usage string

    #Revision 1.5  1998/12/18 16:16:31  rmb1
    # moved to perl/source
    # changed man documentation to POD

=item 0.01  Mon Dec 13 17:54:05 2004

original version; created by h2xs 1.23 with options
S< -XAn File::Rename >

=item 0.02	Robin Barker 2006-01-13

Added t/pod*.t, and extended POD to pass tests

=item 0.03	Robin Barker 2007-09-26

Added --force and --nono options (over_write, no_action)

=item 0.04	Robin Barker 2007-09-27

Replaced depencies on perl versions by explicit
requirements on modules in Build.PL/Makefile.PL

=item 0.05	Robin Barker 2007-10-03

Removed perl 5.6.0 dependencies and successfully
tested on perl 5.005_05 (with patched Temp::File).

=item 0.06	Robin Barker 2011-09-23

Added example/rename.pl, dealt with other Kwalitee metrics.

Updated META files

=item 0.09	(beta for 0.10) Robin Barker 2006-06-26 - not released

Added options -e, -f, -n and -V (version). 

Options -e, -f, -n suggested by code
written by Aristotle Pagaltzis.

=item 0.10	Robin Barker 2013-04-29

Merged "0.09 (beta for 0.10)" from 2006-06-26

Added option -V (version). 

=item 0.20	Robin Barker 2013-04-30

Added option -E (statement): alternative to -e

=item 0.30	Robin Barker 2018-06-02

(tidied configure_requires)

removed typo from rename POD

options do not need to before code / files

allow null separated file names reading from STDIN

=item 0.31	Robin Barker 2018-06-05

Removed use of s///r in tests

Rewrote tests - more robust - use of testlib.pl

=item 0.32	Robin Barker 2018-06-08 - as 0.32-fix

Fixed syntax of rmtree() for perl5.16

=item 0.33	Robin Barker 2018-06-13

Added return code for File::Rename::rename

More tests: in preparation for v1.00

Add $File::Rename::Options::VERSION

=item 0.35	Robin Barker 2018-06-14

Add $File::Rename::Options::VERSION

=item 0.99_01	Robin Barker 2018-06-12

File::Rename::Options in separate file

=item 0.99_02	Robin Barker 2018-06-26

Remove spurious C<opendir> in t/File-Rename-script.t

=item 1.00	Robin Barker 2018-07-03

File::Rename::Options module 

=item 1.09_01	Robin Barker 2018-09-16

Option for renaming file component only: filename-only/-d

=item 1.09_02	Robin Barker 2018-09-16

Fix test failure for perl 5.14

=item 1.09_03	Robin Barker 2018-09-17

Change options to closer align to original feature request

=item 1.09_04 Robin Barker 2018-09-19

Added CONTRIBUTING

Fixed file 'log' which was supposed to be deleted

=item 1.10	Robin Barker 2018-09-25

Added option --filename [-d] to rename filename component only

Added option --fullpath [--path] to rename any part of path 

=item 1.11	Robin Barker 2020-06-16

Added more examples in the script POD, suggested by 'xavier'

Added a test script to test that examples in POD are valid

=item 1.12	Robin Barker 2020-06-17

Added new test script to MANIFEST

=item 1.13	Robin Barker 2020-06-17

Fix 'examples' test script to do file globbing

=item 1.19_01	Robin Barker 2021-03-17

Fix Makefile and tests to find script on darwin

=item 1.19_02 Robin Barker 2021-03-18

More robust OSType handling

=item 1.19_03 Robin Barker 2021-03-19

Fix META.* for Makefile.PL

=item 1.19_04 Robin Barker 2021-03-20

Fix distribution file format

=item 1.20	Robin Barker 2021-03-22

On darwin, when built with ExtUtils::MakeMaker, 
builds the correct script and passes tests; see

http://matrix.cpantesters.org/?dist=File-Rename%201.19_04;os=darwin;reports=1

=item 1.29_01 Robin Narker 2021-07-31

Add -u to do utf8::upgrade and unicode_strings

=item 1.29_02 Robin Narker 2021-08-01

Add File::Rename::Unicode to hide C<use feature>

=item 1.29_03 Robin Narker 2021-08-01

Fix tests which should skip

=item 1.29_04 Robin Narker 2021-08-02

Tests which should skip_all: BEGIN { plan skip_all => 

=item 1.29_05 Robin Narker 2021-08-03

Add encoding to --unicode

Reworked C<create()> in testlib.pl:

=over

=item

removed unnessary C<die>

=item    

simplified File-Rename-unicode.t

=back

=item 1.29_06 Robin Barker 2021-08-04 - not released

Documented approach to unicode strings in README

=item 1.30    Robin Barker 2021-08-16

Unicode support: --unicode and File::Rename::Unicode

=item 1.30_01 Robin Barker 2021-08-16

Fix tests for perl < 5.8.9 (on cygwin)

=item 1.30_02 Neil Bowers 2021-09-05

Add LICENCE=>perl to Makefile.PL

=item 1.30_03 Robin Barker 2022-04-23

Add explicit 'cygwin' in test lib and script

=item 1.31    Robin Barker 2022-05-07

Add explicit LICENCE field

Fix tests for perl 5.8.* and cygwin

=item 1.90_1  Robin Barker 2022-12-19

require_order in script

New script unsafe-rename with no_require_order

Rewrite examples to mark start of non-options

=item 1.90_2  Robin Barker 2022-12-20

Add :config to use File:Rename, to mimic GetOpt::Long

Fix cygwin tests - cygwin is not Windows

See http://www.cpantesters.org/cpan/report/41cc76dd-84fd-1014-94ee-444106f7812f

=item 1.90_3  Robin Barker 2022-12-22

Increase required version of ExtUtils::MakeMaker

Add test for unsafe script

=item 1.90_4  Robin Barker 2022-12-23

Remove Build.PL - so testers build and test unsafe-rename

More globbing in rename-examples.t

Test for warnings in rename-examples.t

=item 1.90_5  Robin Barker 2022-12-24

Add back Build.PL for 1.99 - to be removed for 2.00

README and POD for 1.99

=item 1.99    Robin Barker 2022-12-24

rename script requires options before code/files

New script unsafe-rename with no_require_order:
to recover the old behavior of the rename script

=item 1.991   Robin Barker 2022-12-27

Add required ExtUtils::MakeMaker version to 
BUILD_REQUIRES and CONFIGURE_REQUIRES

Separate out TEST_REQUIRES from PREREQ_PM

Fix typos in 1.99 POD and Changes file 

=item 1.992   Robin Barker 2022-12-27

Reinstate shebang line

https://rt.cpan.org/Ticket/Display.html?id=145711

=item 1.99_9  Robin Barker 2022-12-28

Remove Build.PL from distribution

Set MIN_PERL_VERSION in Makefile.PL

=item 2.00    Robin Barker 2022-12-31

rename script requires options before code/files

New script unsafe-rename with no_require_order

Removed Build.PL from distribution

=item 2.00_1  Robin Barker 2023-01-14

min perl version 5.6.1 

use warnings

use Getopt::Long 2.24

=item 2.00_3  Robin Barker 2023-01-15

Restore indirect print in t/testlib.pl

Change regex for $INC values

2.00_2 deleted

=item 2.00_4  Robin Barker 2023-01-16

Min perl version 5.8.0

Move rename source file to source/ - to avoid
rename being found when @INC includes '.'

Revert regex for $INC values

Initialize C<our $print> for v5.10

=item 2.01    Robin Barker 2023-01-17

min perl version 5.8.0

use warnings

use Getopt::Long 2.24

source/rename source of rename script   

=item 2.01_91 Robin Barker 2023-12-27

replace C<use 5.032;> on old perl

add evn var FILE_RENAME_TEST to test old perl

added example of using options in code

=item 2.01_99 Robin Barker 2023-12-28

Move Changes to HISTORY in Pod

=back

=head1 AUTHOR

Robin Barker <RMBarker@cpan.org>

=head1 Acknowledgements

Based on code from Larry Wall.

Options B<-e>, B<-f>, B<-n> suggested
by more recent code written by Aristotle Pagaltzis.

=head1 DIAGNOSTICS

Errors from the code argument are not trapped.

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2004, 2005, 2006, 2011, 2018, 2021, 2022, 2023
by Robin Barker

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.4 or,
at your option, any later version of Perl 5 you may have available.

=cut

