#===============================================================================
#      PODNAME:  Net::IP::Identifier::Plugin::Facebook
#     ABSTRACT:  identify Facebook (AS32934) owned IP addresses
#
#       AUTHOR:  Reid Augustin (REID)
#        EMAIL:  reid@hellosix.com
#      CREATED:  Thu Oct 30 14:28:30 PDT 2014
#===============================================================================

use 5.008;
use strict;
use warnings;

package Net::IP::Identifier::Plugin::Facebook;

use Role::Tiny::With;
with qw( Net::IP::Identifier_Role );

our $VERSION = '0.086'; # VERSION

sub new {
    my ($class, %opts) = @_;

    my $self = {};
    bless $self, (ref $class || $class);

    # List of known Facebook (AS32934) IP blocks as of Oct 2014
    $self->ips(qw(
        31.13.24.0 - 31.13.31.255
        31.13.64.0 - 31.13.127.255
        45.64.40.0 - 45.64.43.255
        66.220.144.0 - 66.220.159.255
        69.63.176.0 - 69.63.191.255
        69.171.224.0 - 69.171.255.255
        74.119.76.0 - 74.119.79.255
        103.4.96.0 - 103.4.99.255
        173.252.64.0 - 173.252.127.255
        179.60.192/22
        185.60.216.0 - 185.60.219.255
        204.15.20.0 - 204.15.23.255
    ));
    return $self;
}

sub name {
    return 'Facebook';
}

sub refresh {
    my ($self) = @_;

    my @fb = system `whois -h whois.radb.net '!gAS32934'`;
    my @cidrs = split(/\s+/, join('', map { m|/| } @fb));

    delete $self->{ips};
    $self->ips(@cidrs);
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Net::IP::Identifier::Plugin::Facebook - identify Facebook (AS32934) owned IP addresses

=head1 VERSION

version 0.086

=head1 SYNOPSIS

 use Net::IP::Identifier::Plugin::Facebook;

=head1 DESCRIPTION

Net::IP::Identifier::Plugin::Facebook identifies Facebook (AS32934) host IPs.

=head2 Methods

=over

=item new

Creates a new Net::IP::Identifier::Plugin::Facebook object.

=back

=head1 SEE ALSO

=over

=item IP::Net

=item IP::Net::Identifier

=item IP::Net::Identifier_Role

=back

=head1 AUTHOR

Reid Augustin <reid@hellosix.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Reid Augustin.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
