package Bencher::Scenario::LevenshteinModules;

our $DATE = '2017-01-13'; # DATE
our $VERSION = '0.12'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark various modules calculating the Levenshtein edit distance',
    modules => {
        'Text::Levenshtein' => {version => 0.11},
    },
    participants => [
        {
            fcall_template => "PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::fastdistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::XS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::Flexible::levenshtein(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::LevenshteinXS::distance(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            module => 'Text::Fuzzy',
            code_template => "Text::Fuzzy->new(<word1>)->distance(<word2>)",
        },
    ],
    datasets => [
        { name=>"a",       args => {word1=>"a"      , word2=>"aa"},      result => 1 },
        { name=>"foo",     args => {word1=>"foo"    , word2=>"bar"},     result => 3 },
        { name=>"program", args => {word1=>"program", word2=>"porgram"}, result => 2 },
        { name=>"reve"   , args => {word1=>"reve"   , word2=>"rêves"},   result => 2, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
        { name=>"euro"   , args => {word1=>"Euro"   , word2=>"€uro"},    result => 1, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
    ],
    on_result_failure => 'warn',
};

1;
# ABSTRACT: Benchmark various modules calculating the Levenshtein edit distance

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LevenshteinModules - Benchmark various modules calculating the Levenshtein edit distance

=head1 VERSION

This document describes version 0.12 of Bencher::Scenario::LevenshteinModules (from Perl distribution Bencher-Scenarios-LevenshteinModules), released on 2017-01-13.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LevenshteinModules

To run module startup overhead benchmark:

 % bencher --module-startup -m LevenshteinModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<PERLANCAR::Text::Levenshtein> 0.02

L<Text::Fuzzy> 0.25

L<Text::Levenshtein> 0.13

L<Text::Levenshtein::Flexible> 0.09

L<Text::Levenshtein::XS> 0.503

L<Text::LevenshteinXS> 0.03

=head1 BENCHMARK PARTICIPANTS

=over

=item * PERLANCAR::Text::Levenshtein::editdist (perl_code) [no_unicode_support]

Function call template:

 PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)



=item * Text::Levenshtein::fastdistance (perl_code)

Function call template:

 Text::Levenshtein::fastdistance(<word1>, <word2>)



=item * Text::Levenshtein::XS::distance (perl_code)

Function call template:

 Text::Levenshtein::XS::distance(<word1>, <word2>)



=item * Text::Levenshtein::Flexible::levenshtein (perl_code)

Function call template:

 Text::Levenshtein::Flexible::levenshtein(<word1>, <word2>)



=item * Text::LevenshteinXS::distance (perl_code) [no_unicode_support]

Function call template:

 Text::LevenshteinXS::distance(<word1>, <word2>)



=item * Text::Fuzzy (perl_code)

Code template:

 Text::Fuzzy->new(<word1>)->distance(<word2>)



=back

=head1 BENCHMARK DATASETS

=over

=item * a

=item * foo

=item * program

=item * reve [unicode]

=item * euro [unicode]

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.24.0 >>, CPU: I<< Intel(R) Core(TM) M-5Y71 CPU @ 1.20GHz (2 cores) >>, OS: I<< GNU/Linux LinuxMint version 17.3 >>, OS kernel: I<< Linux version 3.19.0-32-generic >>.

Benchmark with default options (C<< bencher -m LevenshteinModules >>):

 #table1#
 +------------------------------------------+---------+-----------+-----------+------------+---------+---------+
 | participant                              | dataset | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +------------------------------------------+---------+-----------+-----------+------------+---------+---------+
 | Text::Levenshtein::fastdistance          | program |     13000 |   77      |      1     | 1.1e-07 |      20 |
 | PERLANCAR::Text::Levenshtein::editdist   | program |     25000 |   41      |      1.9   | 1.6e-07 |      20 |
 | Text::Levenshtein::fastdistance          | reve    |     34000 |   30      |      2.6   | 5.3e-08 |      20 |
 | Text::Levenshtein::fastdistance          | euro    |     40000 |   25      |      3     | 2.6e-08 |      21 |
 | Text::Levenshtein::fastdistance          | foo     |     64000 |   16      |      4.9   | 2.5e-08 |      22 |
 | PERLANCAR::Text::Levenshtein::editdist   | foo     |    110000 |    9.13   |      8.41  | 3.3e-09 |      20 |
 | Text::Levenshtein::fastdistance          | a       |    210000 |    4.8    |     16     | 6.7e-09 |      20 |
 | PERLANCAR::Text::Levenshtein::editdist   | a       |    362000 |    2.76   |     27.8   | 7.3e-10 |      26 |
 | Text::Levenshtein::XS::distance          | program |    520000 |    1.9    |     40     | 2.5e-09 |      20 |
 | Text::Fuzzy                              | euro    |    550000 |    1.8    |     42     | 3.3e-09 |      20 |
 | Text::Levenshtein::XS::distance          | reve    |    670000 |    1.5    |     51     | 1.7e-09 |      20 |
 | Text::Levenshtein::XS::distance          | euro    |    680000 |    1.5    |     52     | 8.1e-09 |      33 |
 | Text::Levenshtein::XS::distance          | foo     |    820100 |    1.219  |     62.92  | 4.5e-11 |      20 |
 | Text::Fuzzy                              | program |    907230 |    1.1023 |     69.602 | 9.1e-12 |      20 |
 | Text::Fuzzy                              | reve    |    991300 |    1.009  |     76.05  | 4.6e-11 |      20 |
 | Text::Fuzzy                              | foo     |   1050000 |    0.95   |     80.8   | 1.1e-10 |      22 |
 | Text::Levenshtein::XS::distance          | a       |   1100000 |    0.93   |     82     | 1.3e-09 |      33 |
 | Text::Fuzzy                              | a       |   1100000 |    0.92   |     83     | 1.7e-09 |      20 |
 | Text::LevenshteinXS::distance            | program |   3040000 |    0.329  |    233     | 1.9e-10 |      27 |
 | Text::Levenshtein::Flexible::levenshtein | program |   4380000 |    0.228  |    336     | 3.5e-11 |      21 |
 | Text::Levenshtein::Flexible::levenshtein | euro    |   4900000 |    0.2    |    380     | 1.4e-09 |      20 |
 | Text::Levenshtein::Flexible::levenshtein | reve    |   6420000 |    0.156  |    492     | 1.1e-10 |      20 |
 | Text::Levenshtein::Flexible::levenshtein | foo     |   7900000 |    0.13   |    600     | 2.1e-10 |      20 |
 | Text::LevenshteinXS::distance            | foo     |   8300000 |    0.12   |    637     | 7.1e-11 |      20 |
 | Text::Levenshtein::Flexible::levenshtein | a       |   9320000 |    0.107  |    715     | 6.5e-11 |      25 |
 | Text::LevenshteinXS::distance            | a       |  11000000 |    0.0909 |    844     | 4.8e-11 |      31 |
 +------------------------------------------+---------+-----------+-----------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m LevenshteinModules --module-startup >>):

 #table2#
 +------------------------------+------------------------------+--------------------+----------------+-----------+------------------------+------------+---------+---------+
 | participant                  | proc_private_dirty_size (MB) | proc_rss_size (MB) | proc_size (MB) | time (ms) | mod_overhead_time (ms) | vs_slowest |  errors | samples |
 +------------------------------+------------------------------+--------------------+----------------+-----------+------------------------+------------+---------+---------+
 | Text::LevenshteinXS          | 1.3                          | 4.8                | 18             |      13   |      8.6               |        1   | 8.3e-05 |      20 |
 | Text::Levenshtein            | 0.98                         | 4.4                | 18             |      11   |      6.6               |        1.2 | 2.6e-05 |      20 |
 | Text::Levenshtein::Flexible  | 1.6                          | 5.1                | 19             |       9.8 |      5.4               |        1.3 | 1.4e-05 |      20 |
 | Text::Fuzzy                  | 0.82                         | 4.1                | 16             |       9.7 |      5.3               |        1.3 | 2.6e-05 |      20 |
 | Text::Levenshtein::XS        | 1.3                          | 4.7                | 18             |       7.9 |      3.5               |        1.6 | 5.8e-05 |      20 |
 | PERLANCAR::Text::Levenshtein | 1.4                          | 4.9                | 19             |       4.8 |      0.399999999999999 |        2.6 | 1.2e-05 |      20 |
 | perl -e1 (baseline)          | 0.85                         | 4.1                | 16             |       4.4 |      0                 |        2.9 | 9.6e-06 |      20 |
 +------------------------------+------------------------------+--------------------+----------------+-----------+------------------------+------------+---------+---------+


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-LevenshteinModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-LevenshteinModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-LevenshteinModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Bencher::Scenario::DamerauLevenshteinModules>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
