package Bencher::Scenario::LevenshteinModules;

our $DATE = '2016-01-14'; # DATE
our $VERSION = '0.09'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark various modules calculating the Levenshtein edit distance',
    modules => {
        'Text::Levenshtein' => 0.11,
    },
    participants => [
        {
            fcall_template => "PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::fastdistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::XS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::LevenshteinXS::distance(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
    ],
    datasets => [
        { name=>"a",       args => {word1=>"a"      , word2=>"aa"},      result => 1 },
        { name=>"foo",     args => {word1=>"foo"    , word2=>"bar"},     result => 3 },
        { name=>"program", args => {word1=>"program", word2=>"porgram"}, result => 2 },
        { name=>"reve"   , args => {word1=>"reve"   , word2=>"rêves"},   result => 2, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
        { name=>"euro"   , args => {word1=>"Euro"   , word2=>"€uro"},    result => 1, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
    ],
    on_result_failure => 'warn',
};

1;
# ABSTRACT: Benchmark various modules calculating the Levenshtein edit distance

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LevenshteinModules - Benchmark various modules calculating the Levenshtein edit distance

=head1 VERSION

This document describes version 0.09 of Bencher::Scenario::LevenshteinModules (from Perl distribution Bencher-Scenario-LevenshteinModules), released on 2016-01-14.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LevenshteinModules

To run module startup overhead benchmark:

 % bencher --module-startup -m LevenshteinModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<PERLANCAR::Text::Levenshtein> 0.02

L<Text::Levenshtein> 0.13

L<Text::Levenshtein::XS> 0.502

L<Text::LevenshteinXS> 0.03

=head1 BENCHMARK PARTICIPANTS

=over

=item * PERLANCAR::Text::Levenshtein::editdist (perl_code)

Function call template:

 PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)



=item * Text::Levenshtein::fastdistance (perl_code)

Function call template:

 Text::Levenshtein::fastdistance(<word1>, <word2>)



=item * Text::Levenshtein::XS::distance (perl_code)

Function call template:

 Text::Levenshtein::XS::distance(<word1>, <word2>)



=item * Text::LevenshteinXS::distance (perl_code)

Function call template:

 Text::LevenshteinXS::distance(<word1>, <word2>)



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default option:

 +-----+----------------------------------------+---------+-------------+------------+---------+---------+
 | seq | participant                            | dataset | rate        | time       | errors  | samples |
 +-----+----------------------------------------+---------+-------------+------------+---------+---------+
 | 5   | Text::Levenshtein::fastdistance        | program | 18717       | 53.4274μs  | 0       | 20      |
 | 2   | PERLANCAR::Text::Levenshtein::editdist | program | 3e+04       | 33.3μs     | 1.1e-08 | 31      |
 | 6   | Text::Levenshtein::fastdistance        | reve    | 4.341e+04   | 23.04μs    | 4.9e-09 | 37      |
 | 7   | Text::Levenshtein::fastdistance        | euro    | 5.024e+04   | 19.9μs     | 6e-09   | 25      |
 | 4   | Text::Levenshtein::fastdistance        | foo     | 8.77e+04    | 11.4μs     | 9.8e-09 | 21      |
 | 1   | PERLANCAR::Text::Levenshtein::editdist | foo     | 1.17e+05    | 8.58μs     | 1e-08   | 35      |
 | 3   | Text::Levenshtein::fastdistance        | a       | 2.82e+05    | 3.54μs     | 5e-09   | 20      |
 | 0   | PERLANCAR::Text::Levenshtein::editdist | a       | 451726      | 2.21373μs  | 0       | 20      |
 | 10  | Text::Levenshtein::XS::distance        | program | 648688      | 1.54157μs  | 0       | 20      |
 | 11  | Text::Levenshtein::XS::distance        | reve    | 845922      | 1.18214μs  | 0       | 20      |
 | 12  | Text::Levenshtein::XS::distance        | euro    | 912027      | 1.09646μs  | 0       | 20      |
 | 9   | Text::Levenshtein::XS::distance        | foo     | 1.0183e+06  | 0.982025μs | 0       | 31      |
 | 8   | Text::Levenshtein::XS::distance        | a       | 1.28859e+06 | 0.776044μs | 0       | 20      |
 | 15  | Text::LevenshteinXS::distance          | program | 4.21e+06    | 0.237μs    | 1e-10   | 20      |
 | 14  | Text::LevenshteinXS::distance          | foo     | 9.42933e+06 | 0.106052μs | 0       | 20      |
 | 13  | Text::LevenshteinXS::distance          | a       | 1.17e+07    | 0.0854μs   | 5.2e-11 | 20      |
 +-----+----------------------------------------+---------+-------------+------------+---------+---------+


Benchmark module startup overhead:

 +-----+------------------------------+--------+-------------------+---------+---------+
 | seq | participant                  | time   | mod_overhead_time | errors  | samples |
 +-----+------------------------------+--------+-------------------+---------+---------+
 | 4   | Text::LevenshteinXS          | 8.54ms | 6.09ms            | 1.5e-05 | 20      |
 | 2   | Text::Levenshtein            | 7.05ms | 4.6ms             | 1.6e-05 | 20      |
 | 3   | Text::Levenshtein::XS        | 4.89ms | 2.44ms            | 1.4e-05 | 20      |
 | 1   | PERLANCAR::Text::Levenshtein | 2.71ms | 0.259ms           | 8.1e-06 | 20      |
 | 0   | perl -e1 (baseline)          | 2.4ms  | 0ms               | 9e-06   | 20      |
 +-----+------------------------------+--------+-------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 SEE ALSO

L<Bencher::Scenario::DamerauLevenshteinModules>

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-LevenshteinModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-LevenshteinModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-LevenshteinModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
