#!/usr/bin/env perl 
#
# Before `make install' is performed this script should be runnable with
# `make test'.
# After `make install' it should work as `perl 400_sshare_list-parse.t`
#
# Various tests of sshare_list using  fake_sshare to test parsing of sshare output
#

use strict;
use warnings;

use Test::More;
use Cwd qw(abs_path);
use File::Basename qw(dirname);

use Slurm::Sshare;

my $testDir = dirname(abs_path($0));

our $num_tests_run = 0;
my $default_user = 'george';
my $default_cluster = 'yottascale';


require "${testDir}/helpers/parse-help.pl";

my $sa = 'Slurm::Sshare';
$sa->sshare("${testDir}/helpers/fake_sshare");

#---------------------------------------------------------
#	More helper routines
#---------------------------------------------------------

sub decluster_results($)
#Takes list ref of hash refs, and makes a clone with all cluster keys removed
{	my $olist = shift;

	my $nlist = [];
	foreach my $href (@$olist)
	{	my $nhash = { %$href };
		delete $nhash->{cluster};
		push @$nlist, $nhash;
	}
	return $nlist;
}


my (%filter, $results, $expected, $exp2, $arglist);
my ($cluster, $account, @ulist, @alist);

#---------------------------------------------------------
#	No arguments
#---------------------------------------------------------

%filter = ( cluster=>$default_cluster, user=>[undef, $default_user], );
$expected = filtered_test_data(%filter);
$exp2 = decluster_results($expected);
$results = $sa->sshare_list();
compare_lref_results($results, $exp2, 'sshare_list, no args' );

#---------------------------------------------------------
#	Tests on yottascale cluster
#---------------------------------------------------------
$cluster = 'yottascale';

# User list tests, no accounts

@ulist=( 'payerle' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, ys, single user (aref)' );

$arglist = join ',', @ulist;
$results = $sa->sshare_list(clusters=>$cluster, users=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, ys single user (CSV)' );

@ulist=( 'payerle', 'kevin', );
%filter = ( cluster => $cluster, user=>[undef, @ulist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, users=>[@ulist],);
compare_lref_results($results, $expected, 'sshare_list, ys 2 users (aref)' );

$arglist = join ',', @ulist;
$results = $sa->sshare_list(clusters=>$cluster, users=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, ys 2 users (CSV)' );

@ulist=( 'payerle', 'kevin', 'george' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, users=>[@ulist],);
compare_lref_results($results, $expected, 'sshare_list, ys 3 users (aref)' );

$arglist = join ',', @ulist;
$results = $sa->sshare_list(clusters=>$cluster, users=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, ys 3 users (CSV)' );

# Now for ALL users
$arglist='ALL';
%filter = ( cluster => $cluster, );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, users=>[ $arglist ],);
compare_lref_results($results, $expected, 'sshare_list, ys all users (aref)' );

$results = $sa->sshare_list(clusters=>$cluster, users=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, ys all users (CSV)' );

# Account list tests, no users

@alist=( 'abc124' );
%filter = ( cluster => $cluster, user=>[undef, $default_user], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist] );
compare_lref_results($results, $expected, 'sshare_list, ys single acct (aref)' );

$arglist = join ',', @alist;
$results = $sa->sshare_list(clusters=>$cluster, accounts=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, ys single acct (CSV)' );

@alist=( 'abc124', 'fbi' );
%filter = ( cluster => $cluster, user=>[undef, $default_user], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist] );
compare_lref_results($results, $expected, 'sshare_list, ys 2 accts (aref)' );

$arglist = join ',', @alist;
$results = $sa->sshare_list(clusters=>$cluster, accounts=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, ys 2 accts (CSV)' );

@alist=( 'abc124', 'fbi', 'nsa' );
%filter = ( cluster => $cluster, user=>[undef, $default_user], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist] );
compare_lref_results($results, $expected, 'sshare_list, ys 3 accts (aref)' );

$arglist = join ',', @alist;
$results = $sa->sshare_list(clusters=>$cluster, accounts=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, ys 3 accts (CSV)' );

# Account and user lists

@alist=( 'abc124', );
@ulist = ( 'payerle' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, ys single acct+user' );

@alist=( 'cia', );
@ulist = ( 'payerle' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, ys single acct+user not in acct' );

@alist=( 'fbi', 'cia', );
@ulist = ( 'payerle' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, ys 2 accts + 1 user' );

@alist=( 'abc124', );
@ulist = ( 'payerle', 'kevin' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, ys 1 acct + 2 users' );

@alist=( 'abc124', 'cia' );
@ulist = ( 'payerle', 'george' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, ys 2 accts + 2 users' );

@alist=( 'abc124', 'cia', 'fbi' );
@ulist = ( 'payerle', 'george', 'kevin' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, ys 3 accts + 3 users' );

@alist=( 'abc124', 'cia', 'fbi' );
$arglist = 'ALL';
%filter = ( cluster => $cluster,  account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, ys 3 accts + all users' );

#---------------------------------------------------------
#	Tests on test cluster
#---------------------------------------------------------

$cluster = 'test';

# User list tests, no accounts

@ulist=( 'payerle' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, testcl, single user (aref)' );

$arglist = join ',', @ulist;
$results = $sa->sshare_list(clusters=>$cluster, users=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, testcl single user (CSV)' );

@ulist=( 'payerle', 'george', );
%filter = ( cluster => $cluster, user=>[undef, @ulist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, users=>[@ulist],);
compare_lref_results($results, $expected, 'sshare_list, testcl 2 users (aref)' );

$arglist = join ',', @ulist;
$results = $sa->sshare_list(clusters=>$cluster, users=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, testcl 2 users (CSV)' );

# Now for ALL users
$arglist='ALL';
%filter = ( cluster => $cluster, );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, users=>[ $arglist ],);
compare_lref_results($results, $expected, 'sshare_list, testcl all users (aref)' );

$results = $sa->sshare_list(clusters=>$cluster, users=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, testcl all users (CSV)' );

# Account list tests, no users

@alist=( 'abc124' );
%filter = ( cluster => $cluster, user=>[undef, $default_user], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist] );
compare_lref_results($results, $expected, 'sshare_list, testcl single acct (aref)' );

$arglist = join ',', @alist;
$results = $sa->sshare_list(clusters=>$cluster, accounts=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, testcl single acct (CSV)' );

# Account and user lists

@alist=( 'abc124', );
@ulist = ( 'payerle' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, testcl single acct+user' );

@alist=( 'abc124', );
@ulist = ( 'payerle', 'george' );
%filter = ( cluster => $cluster, user=>[undef, @ulist], account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, testcl  1 acct+ 2 users' );

@alist=( 'abc124', );
$arglist = 'ALL';
%filter = ( cluster => $cluster,  account=>[@alist], );
$expected = filtered_test_data(%filter);
$results = $sa->sshare_list(clusters=>$cluster, accounts=>[@alist], users=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, testcl 1 accts + all users' );

#---------------------------------------------------------
#	Tests on both clusters
#---------------------------------------------------------

my @clist = ( 'test', 'yottascale' );

# No arguments except clusters
$expected = [];
foreach $cluster (@clist)
{	%filter = ( cluster=>$cluster, user=>[ undef, $default_user ], );
	$exp2 = filtered_test_data(%filter);
	push @$expected, @$exp2;
}
$results = $sa->sshare_list(clusters=>[ @clist ] );

compare_lref_results($results, $expected, 'sshare_list, 2 clusters (reversed) (aref)' );

$arglist = join ',', @clist;
$results = $sa->sshare_list(clusters=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, 2 clusters (reversed) (CSV)' );

@clist = ( 'yottascale', 'test' );
$expected = [];
foreach $cluster (@clist)
{	%filter = ( cluster=>$cluster, user=>[ undef, $default_user ], );
	$exp2 = filtered_test_data(%filter);
	push @$expected, @$exp2;
}
$results = $sa->sshare_list(clusters=>[ @clist ] );
compare_lref_results($results, $expected, 'sshare_list, 2 clusters (aref)' );

$arglist = join ',', @clist;
$results = $sa->sshare_list(clusters=>$arglist );
compare_lref_results($results, $expected, 'sshare_list, 2 clusters (CSV)' );

#Single user, no acct
@ulist = ( 'payerle' );
$expected = [];
foreach $cluster (@clist)
{	%filter = ( cluster=>$cluster,  user=>[undef, @ulist], );
	$exp2 = filtered_test_data(%filter);
	push @$expected, @$exp2;
}
$results = $sa->sshare_list(clusters=>[ @clist ], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, 2 clusters, single user' );

#Two users, no acct
@ulist = ( 'payerle', 'george' );
$expected = [];
foreach $cluster (@clist)
{	%filter = ( cluster=>$cluster,  user=>[undef, @ulist], );
	$exp2 = filtered_test_data(%filter);
	push @$expected, @$exp2;
}
$results = $sa->sshare_list(clusters=>[ @clist ], users=>[@ulist] );
compare_lref_results($results, $expected, 'sshare_list, 2 clusters, 2 users' );

#Single account, no user
@alist = ( 'abc124' );
$expected = [];
foreach $cluster (@clist)
{	%filter = ( cluster=>$cluster,  account=>[@alist], user=>[ undef, $default_user ], );
	$exp2 = filtered_test_data(%filter);
	push @$expected, @$exp2;
}
$results = $sa->sshare_list(clusters=>[ @clist ], accounts=>[@alist] );
compare_lref_results($results, $expected, 'sshare_list, 2 clusters, 1 acct' );

#Single account, 2 users
@alist = ( 'abc124' );
@ulist = ( 'payerle', 'george' );
$expected = [];
foreach $cluster (@clist)
{	%filter = ( cluster=>$cluster,  account=>[@alist], user=>[ undef, @ulist],  );
	$exp2 = filtered_test_data(%filter);
	push @$expected, @$exp2;
}
$results = $sa->sshare_list(clusters=>[ @clist ], accounts=>[@alist], users=>[@ulist], );
compare_lref_results($results, $expected, 'sshare_list, 2 clusters, 1 acct, 2 users' );

#Single account, all users
@alist = ( 'abc124' );
$arglist='ALL';
$expected = [];
foreach $cluster (@clist)
{	%filter = ( cluster=>$cluster,  account=>[@alist], );
	$exp2 = filtered_test_data(%filter);
	push @$expected, @$exp2;
}
$results = $sa->sshare_list(clusters=>[ @clist ], accounts=>[@alist], users=>$arglist, );
compare_lref_results($results, $expected, 'sshare_list, 2 clusters, 1 acct, all users' );


#---------------------------------------------------------
#	Finish
#---------------------------------------------------------


done_testing($num_tests_run);

