#!/usr/bin/perl

package Devel::Events;

use strict;
use warnings;

our $VERSION = "0.02";

__PACKAGE__;

__END__

=pod

=head1 NAME

Devel::Events - Extensible instrumentation framework.

=head1 SYNOPSIS

	use Devel::Events::Generator::Foo;
	use Devel::Events::Handler::Bar;
	use Devel::Events::Filter::Blort;

	my $h = Devel::Events::Generator->new(
		handler => Devel::Event::Filter->new(
			handler => Devel::Events::Handler::Bar->new(),
		),
	);

=head1 DESCRIPTION

L<Devel::Events> is an event generation, filtering and analaysis framework for
instrumenting and auditing perl code.

L<Devel::Events::Generator> object fire events, which can be mangled by
L<Devel::Event::Filter> objects. Eventually any number of
L<Devel::Event::Handler> objects can receive a given event, and perform
analysis based on it.

For example L<Devel::Event::Handler::ObjectTracker> can be used to detect
leaks.

=head1 COMPONENT OVERVIEW

There are two main types of components - generators and handlers.

Filters are special types of handlers that always delegate to another handler.

The multiplex handler may be used to delegate to any number of handlers.

Using these basic components complex chains of handlers can be built to
properly analyze the events you are interested in.

=head2 Generators

=item L<Devel::Events::Generator::Objects>

Generate C<object_bless> and C<object_destroy> events by overriding
C<CORE::GLOBAL::bless> and tracking every object using L<Variable::Magic>.

=item L<Devel::Events::Generator::SubTrace>

Uses the perl debugger hook to generate C<enter_sub> and C<leave_sub> events.

=item L<Devel::Events::Generator::LineTrace>

Fires an C<executing_line> event for every line using the perl debugger hook.

=head2 Handlers

=item L<Devel::Events::Handler::Multiplex>

Repeat events to multiple handlers.

=item L<Devel::Events::Handler::Callback>

Make a handler object out of a callback.

=item L<Devel::Events::Handler::Log::Memory>

Log all events to an array.

Provides filtering and slicing methods to ease analysis.

=item L<Devel::Events::Handler::ObjectTracker>

Handles C<object_bless> and C<object_destroy> events (as generated by
L<Devel::Events::Generator::Objects>) to track the lifetime of objects. When
objects lifetimes extend beyond the scope they're supposed to be destroyed in
leaks can be detected.}

=back

=head2 Filters

=over 4

=item L<Devel::Events::Filter::Stamp>

Add various stamp data (time, process ID, thread ID if threads are loaded) to
events.

=item L<Devel::Events::Filter::Stringify>

Stringify (by default using L<overload/StrVal>, to avoid side effects) all
event data to avoid leaking.

=item L<Devel::Events::Filter::Size>

Add size reports to events using L<Devel::Size>

=item L<Devel::Events::Filter::RemoveFields>

Remove certain fields from events.


=item L<Devel::Events::Filter::Warn>

Runs C<warn "@event"> before delegating to the sub handler. Useful for
debugging handler chains.

=back

=head1 AUTHOR

Yuval Kogman <nothingmuch@woobling.org>

=head1 COPYRIGHT & LICENSE

	Copyright (c) 2007 Yuval Kogman. All rights reserved
	This program is free software; you can redistribute it and/or modify it
	under the terms of the MIT license or the same terms as Perl itself.

=cut


