use strict;
use warnings;

# common setup for Forks::Super remote tests (t/49*.t)
#
# we must connect to a server (which may be the current host) through ssh
# and run some remote test commands
#
# options are, in order of preference:
#
#    1. $ENV{TEST_SSH} = user@host
#       specifies a server in an environment variable. Expects to use
#       passwordless, pubic key authentication, and expects to find the same
#       filesystem on the remote host that exists on the local host to build
#       this module
#
#    2. Test::SSH module, may depend on Net::OpenSSH.
#       constructs a local, temporary ssh server for the tests
#       t/forked_harness.pl, which is run if you "make fasttest",
#       might set $ENV{TEST_SSH_TARGET} which could be used to
#       construct a Test::SSH object.
#
#    3. if local machines is running sshd and user has an .ssh dir,
#       try to connect to local server with default publickey credentials
#

Forks::Super::POSTFORK_CHILD {
    # RT#117025 for Test::SSH workaround. Otherwise, the server
    # closes at the end of each child process.
    *Test::SSH::Backend::OpenSSH::_run_dir = sub { };
};

if (0) {
    $Forks::Super::Config::CONFIG{"Test::SSH"} = 0;
    $Forks::Super::Config::CONFIG{"Net::OpenSSH"} = 0;
}


# returns an object (Test::SSH or a mock) that
# describes an ssh server that our tests can access.
sub get_test_sshd {

    print STDERR "Identifying ssh connection\n";

    if (!$ENV{TEST_SSH_TARGET}) {
        # first, try public key authentication for the current user and host
        my $userathost = $ENV{USER} . '@' . $ENV{HOSTNAME};
        my $ssh = Forks::Super::Config::CONFIG_external_program("ssh");
        if ($ssh && $userathost =~ /.@./) {
            my @cmds = ("true", "echo", "dir");
            foreach my $cmd (@cmds) {
                local $SIG{ALRM} = sub { die "ssh timeout\n"; };
                alarm 15;
                eval {
                    if (0 == system($ssh,$userathost,$cmd)) {
                        $ENV{TEST_SSH_TARGET} = "ssh://$userathost";
                        last;
                    }
                };
                alarm 0;
            }
        }
    }

    if ($ENV{TEST_SSH_TARGET}) {
        use URI;
        my $uri = URI->new($ENV{TEST_SSH_TARGET});
        my $sshd = { host => $uri->host, uri => "$uri" };
        $sshd->{port} = $uri->port if $uri->port;
        if ($uri->password) {
            $sshd->{password} = $uri->password;
            $sshd->{auth_method} = 'password';
        } else {
            $sshd->{auth_method} = 'publickey';
        }

        my @u = split /;/, $uri->user;
        $sshd->{user} = shift @u;
        foreach my $u (@u) {
            my ($k,$v) = split /=/, $u, 2;
            $sshd->{$k} = $v;
        }
        bless $sshd, 'Mock::Test::SSH';
        return $sshd;
    }

    if (!$ENV{TEST_SSH} &&
        Forks::Super::Config::CONFIG_module('Test::SSH')) {
        my %opts;
        my $sshd = eval 'use Test::SSH; Test::SSH->new(%opts);';
        if ($sshd) {
            print STDERR " ... Test::SSH available\n";
            return $sshd;
        }
    }
    print STDERR " ... Test::SSH not available\n";
    if ($ENV{NO_SSHD}) {
        print STDERR " ... requested not to look for local ssh server\n";
        return;
    }

    # if Test::SSH is not installed, see if there is a local ssh server
    $Forks::Super::Config::CONFIG{"Test::SSH"} = 0;
    my @ps = grep /\bsshd\b/, `ps -ef`;
    if (@ps == 0) {
        print STDERR " ... no sshd found in process table\n";
        return;
    }

    my $sshd = {
        host => $ENV{HOSTNAME},
        user => $ENV{USER},
#       port => '',
        password => '',
        auth_method => 'publickey',
    };
    bless $sshd, 'Mock::Test::SSH';

    if ($ENV{TEST_SSH}) {
        if ($ENV{TEST_SSH} eq '1') {
            $ENV{TEST_SSH} = $ENV{USER} . '@' . $ENV{HOSTNAME};
        }
        $Forks::Super::Config::CONFIG{"Net::OpenSSH"} = 0;
        my @uph = split /\@/, $ENV{TEST_SSH};
        $sshd->{host} = pop @uph;
        my $user = join '@', @uph;
        if ($user =~ /:/) {
            my @up = split /:/, $user;
            $sshd->{password} = pop @up;
            $sshd->{auth_method} = 'password';
            $user = join ':', @up;
        }
        $sshd->{user} = $user;
        print STDERR " ... extracted credentials from ENV{TEST_SSH}\n";
    }

    if ($sshd->{auth_method} eq 'publickey') {
        my $sshdir = $ENV{HOME} . "/.ssh";
        if (! -d $sshdir) {
            print STDERR " ... no user .ssh dir found\n";
            return;
        }
        my (@cfg,@syscfg);
        if (-f "$sshdir/config") {
            open my $fh, '<', "$sshdir/config";
            @cfg = <$fh>;
            close $fh;
        }
        if (-f "/etc/ssh/ssh_config") {
            open my $fh, '<', "/etc/ssh/ssh_config";
            @syscfg = <$fh>;
            close $fh;
        }
        my @id = grep /IdentityFile/, @cfg;
        if (!@id) {
            @id = grep /IdentityFile/, @syscfg;
        }
        if (!@id) {
            print STDERR " ... no identity files for publickey auth found\n";
        }
        s/^\s*IdentityFile\s*//, s/\s+$//, s/~\/.ssh/$sshdir/ for @id;
        @id = grep { -f $_ } @id;
        if (@id) {
            $sshd->{key_path} = [ @id ];
        } else {
            print STDERR " ... identity files for publickey auth not found\n";
        }
    }

    my $host = $sshd->{host};
    my $job = { cmd => ["true"] };
    my $test_cmd = Forks::Super::Job::__build_ssh_command($host, $sshd, $job);
    print STDERR " ... test command is  '$test_cmd'\n";

    my $c1 = system($test_cmd);
    if ($c1) {
        print STDERR " ... test command failed: rc=$c1\n";
        return;
    }
    print STDERR " ... test command ok. We have a valid ssh server for testing\n";
    return $sshd;
}

sub Mock::Test::SSH::AUTOLOAD {
    my $self = shift;
    my $key = $Mock::Test::SSH::AUTOLOAD;
    $key =~ s/.*:://;
    return if $key eq 'DESTROY';
    return $self->{$key};
}

1;
