package Font::TTF::Tags;

=head1 NAME

Font::TTF::Tags - Utilities for TrueType/OpenType tags

=head1 SYNOPSIS

  use Font::TTF::Tags qw( %tttags %ttnames readtagsfile );
  
  # Look at built-in stuff:
  $script_tag = %tttags{'SCRIPT'}{'Cypriot Syllabary'};
  $lang_name = %ttnames{'LANGUAGE'}{'Afrikaans'}
  
  # Read latest tags file to add to built-in definitions
  readtagsfile ("C:\\Program Files\\Microsoft VOLT\\TAGS.txt");
  
First-level keys to %tttags and %ttnames are:

=over

'SCRIPT' -- retrieve script tag or name

'LANGUAGE' -- retrieve language tag or name

'FEATURE' -- retrieve feature tag or name

=back

=head1 METHODS

=cut

use strict;
use vars qw( %tttags %ttnames @EXPORT_OK );
use Exporter qw( import );
@EXPORT_OK = qw( %tttags %ttnames readtagsfile);


%tttags = (
'SCRIPT' => {
	 "Arabic" => "arab",
	 "Armenian" => "armn",
	 "Balinese" => "bali",
	 "Bengali" => "beng",
	 "Bopomofo" => "bopo",
	 "Braille" => "brai",
	 "Buginese" => "bugi",
	 "Buhid" => "buhd",
	 "Byzantine Music" => "byzm",
	 "Canadian Syllabics" => "cans",
	 "Cherokee" => "cher",
	 "CJK Ideographic" => "hani",
	 "Coptic" => "copt",
	 "Cypriot Syllabary" => "cprt",
	 "Cyrillic" => "cyrl",
	 "Default" => "dflt",
	 "Deseret" => "dsrt",
	 "Devanagari" => "deva",
	 "Ethiopic" => "ethi",
	 "Georgian" => "geor",
	 "Glagolitic" => "glag",
	 "Gothic" => "goth",
	 "Greek" => "grek",
	 "Gujarati" => "gujr",
	 "Gurmukhi" => "guru",
	 "Hangul Jamo" => "jamo",
	 "Hangul" => "hang",
	 "Hanunoo" => "hano",
	 "Hebrew" => "hebr",
	 "Hiragana" => "kana",
	 "Javanese" => "java",
	 "Kannada" => "knda",
	 "Katakana" => "kana",
	 "Kharosthi" => "khar",
	 "Khmer" => "khmr",
	 "Lao" => "lao ",
	 "Latin" => "latn",
	 "Limbu" => "limb",
	 "Linear B" => "linb",
	 "Malayalam" => "mlym",
	 "Mathematical Alphanumeric Symbols" => "math",
	 "Mongolian" => "mong",
	 "Musical Symbols" => "musc",
	 "Myanmar" => "mymr",
	 "N'ko" => "nko ",
	 "Ogham" => "ogam",
	 "Old Italic" => "ital",
	 "Old Persian Cuneiform" => "xpeo",
	 "Oriya" => "orya",
	 "Osmanya" => "osma",
	 "Phags-pa" => "phag",
	 "Phoenician" => "phnx",
	 "Runic" => "runr",
	 "Shavian" => "shaw",
	 "Sinhala" => "sinh",
	 "Sumero-Akkadian Cuneiform" => "xsux",
	 "Syloti Nagri" => "sylo",
	 "Syriac" => "syrc",
	 "Tagalog" => "tglg",
	 "Tagbanwa" => "tagb",
	 "Tai Le" => "tale",
	 "New Tai Lue" => "talu",
	 "Tamil" => "taml",
	 "Telugu" => "telu",
	 "Thaana" => "thaa",
	 "Thai" => "thai",
	 "Tibetan" => "tibt",
	 "Tifinagh" => "tfng",
	 "Ugaritic Cuneiform" => "ugar",
	 "Yi" => "yi  ",
    }, 

'LANGUAGE' => {
	 "Abaza" => "ABA ",
	 "Abkhazian" => "ABK ",
	 "Adyghe" => "ADY ",
	 "Afrikaans" => "AFK ",
	 "Afar" => "AFR ",
	 "Agaw" => "AGW ",
	 "Altai" => "ALT ",
	 "Amharic" => "AMH ",
	 "Arabic" => "ARA ",
	 "Aari" => "ARI ",
	 "Arakanese" => "ARK ",
	 "Assamese" => "ASM ",
	 "Athapaskan" => "ATH ",
	 "Avar" => "AVR ",
	 "Awadhi" => "AWA ",
	 "Aymara" => "AYM ",
	 "Azeri" => "AZE ",
	 "Badaga" => "BAD ",
	 "Baghelkhandi" => "BAG ",
	 "Balkar" => "BAL ",
	 "Baule" => "BAU ",
	 "Berber" => "BBR ",
	 "Bench" => "BCH ",
	 "Bible Cree" => "BCR ",
	 "Belarussian" => "BEL ",
	 "Bemba" => "BEM ",
	 "Bengali" => "BEN ",
	 "Bulgarian" => "BGR ",
	 "Bhili" => "BHI ",
	 "Bhojpuri" => "BHO ",
	 "Bikol" => "BIK ",
	 "Bilen" => "BIL ",
	 "Blackfoot" => "BKF ",
	 "Balochi" => "BLI ",
	 "Balante" => "BLN ",
	 "Balti" => "BLT ",
	 "Bambara" => "BMB ",
	 "Bamileke" => "BML ",
	 "Breton" => "BRE ",
	 "Brahui" => "BRH ",
	 "Braj Bhasha" => "BRI ",
	 "Burmese" => "BRM ",
	 "Bashkir" => "BSH ",
	 "Beti" => "BTI ",
	 "Catalan" => "CAT ",
	 "Cebuano" => "CEB ",
	 "Chechen" => "CHE ",
	 "Chaha Gurage" => "CHG ",
	 "Chattisgarhi" => "CHH ",
	 "Chichewa" => "CHI ",
	 "Chukchi" => "CHK ",
	 "Chipewyan" => "CHP ",
	 "Cherokee" => "CHR ",
	 "Chuvash" => "CHU ",
	 "Comorian" => "CMR ",
	 "Coptic" => "COP ",
	 "Cree" => "CRE ",
	 "Carrier" => "CRR ",
	 "Crimean Tatar" => "CRT ",
	 "Church Slavonic" => "CSL ",
	 "Czech" => "CSY ",
	 "Danish" => "DAN ",
	 "Dargwa" => "DAR ",
	 "Woods Cree" => "DCR ",
	 "German" => "DEU ",
	 "Default" => "dflt",
	 "Dogri" => "DGR ",
	 "Divehi" => "DIV ",
	 "Djerma" => "DJR ",
	 "Dangme" => "DNG ",
	 "Dinka" => "DNK ",
	 "Dungan" => "DUN ",
	 "Dzongkha" => "DZN ",
	 "Ebira" => "EBI ",
	 "Eastern Cree" => "ECR ",
	 "Edo" => "EDO ",
	 "Efik" => "EFI ",
	 "Greek" => "ELL ",
	 "English" => "ENG ",
	 "Erzya" => "ERZ ",
	 "Spanish" => "ESP ",
	 "Estonian" => "ETI ",
	 "Basque" => "EUQ ",
	 "Evenki" => "EVK ",
	 "Even" => "EVN ",
	 "Ewe" => "EWE ",
	 "French Antillean" => "FAN ",
	 "Farsi" => "FAR ",
	 "Finnish" => "FIN ",
	 "Fijian" => "FJI ",
	 "Flemish" => "FLE ",
	 "Forest Nenets" => "FNE ",
	 "Fon" => "FON ",
	 "Faroese" => "FOS ",
	 "French" => "FRA ",
	 "Frisian" => "FRI ",
	 "Friulian" => "FRL ",
	 "Futa" => "FTA ",
	 "Fulani" => "FUL ",
	 "Ga" => "GAD ",
	 "Gaelic" => "GAE ",
	 "Gagauz" => "GAG ",
	 "Galician" => "GAL ",
	 "Garshuni" => "GAR ",
	 "Garhwali" => "GAW ",
	 "Ge'ez" => "GEZ ",
	 "Gilyak" => "GIL ",
	 "Gumuz" => "GMZ ",
	 "Gondi" => "GON ",
	 "Greenlandic" => "GRN ",
	 "Garo" => "GRO ",
	 "Guarani" => "GUA ",
	 "Gujarati" => "GUJ ",
	 "Haitian" => "HAI ",
	 "Halam" => "HAL ",
	 "Harauti" => "HAR ",
	 "Hausa" => "HAU ",
	 "Hawaiin" => "HAW ",
	 "Hammer-Banna" => "HBN ",
	 "Hiligaynon" => "HIL ",
	 "Hindi" => "HIN ",
	 "High Mari" => "HMA ",
	 "Hindko" => "HND ",
	 "Ho" => "HO  ",
	 "Harari" => "HRI ",
	 "Croatian" => "HRV ",
	 "Hungarian" => "HUN ",
	 "Armenian" => "HYE ",
	 "Igbo" => "IBO ",
	 "Ijo" => "IJO ",
	 "Ilokano" => "ILO ",
	 "Indonesian" => "IND ",
	 "Ingush" => "ING ",
	 "Inuktitut" => "INU ",
	 "Irish" => "IRI ",
	 "Irish Traditional" => "IRT ",
	 "Icelandic" => "ISL ",
	 "Inari Sami" => "ISM ",
	 "Italian" => "ITA ",
	 "Hebrew" => "IWR ",
	 "Javanese" => "JAV ",
	 "Yiddish" => "JII ",
	 "Japanese" => "JAN ",
	 "Judezmo" => "JUD ",
	 "Jula" => "JUL ",
	 "Kabardian" => "KAB ",
	 "Kachchi" => "KAC ",
	 "Kalenjin" => "KAL ",
	 "Kannada" => "KAN ",
	 "Karachay" => "KAR ",
	 "Georgian" => "KAT ",
	 "Kazakh" => "KAZ ",
	 "Kebena" => "KEB ",
	 "Khutsuri Georgian" => "KGE ",
	 "Khakass" => "KHA ",
	 "Khanty-Kazim" => "KHK ",
	 "Khmer" => "KHM ",
	 "Khanty-Shurishkar" => "KHS ",
	 "Khanty-Vakhi" => "KHV ",
	 "Khowar" => "KHW ",
	 "Kikuyu" => "KIK ",
	 "Kirghiz" => "KIR ",
	 "Kisii" => "KIS ",
	 "Kokni" => "KKN ",
	 "Kalmyk" => "KLM ",
	 "Kamba" => "KMB ",
	 "Kumaoni" => "KMN ",
	 "Komo" => "KMO ",
	 "Komso" => "KMS ",
	 "Kanuri" => "KNR ",
	 "Kodagu" => "KOD ",
	 "Korean Old Hangul" => "KOH ",
	 "Konkani" => "KOK ",
	 "Kikongo" => "KON ",
	 "Komi-Permyak" => "KOP ",
	 "Korean" => "KOR ",
	 "Komi-Zyrian" => "KOZ ",
	 "Kpelle" => "KPL ",
	 "Krio" => "KRI ",
	 "Karakalpak" => "KRK ",
	 "Karelian" => "KRL ",
	 "Karaim" => "KRM ",
	 "Karen" => "KRN ",
	 "Koorete" => "KRT ",
	 "Kashmiri" => "KSH ",
	 "Khasi" => "KSI ",
	 "Kildin Sami" => "KSM ",
	 "Kui" => "KUI ",
	 "Kulvi" => "KUL ",
	 "Kumyk" => "KUM ",
	 "Kurdish" => "KUR ",
	 "Kurukh" => "KUU ",
	 "Kuy" => "KUY ",
	 "Koryak" => "KYK ",
	 "Ladin" => "LAD ",
	 "Lahuli" => "LAH ",
	 "Lak" => "LAK ",
	 "Lambani" => "LAM ",
	 "Lao" => "LAO ",
	 "Latin" => "LAT ",
	 "Laz" => "LAZ ",
	 "L-Cree" => "LCR ",
	 "Ladakhi" => "LDK ",
	 "Lezgi" => "LEZ ",
	 "Lingala" => "LIN ",
	 "Low Mari" => "LMA ",
	 "Limbu" => "LMB ",
	 "Lomwe" => "LMW ",
	 "Lower Sorbian" => "LSB ",
	 "Lule Sami" => "LSM ",
	 "Lithuanian" => "LTH ",
	 "Luba" => "LUB ",
	 "Luganda" => "LUG ",
	 "Luhya" => "LUH ",
	 "Luo" => "LUO ",
	 "Latvian" => "LVI ",
	 "Majang" => "MAJ ",
	 "Makua" => "MAK ",
	 "Malayalam Traditional" => "MAL ",
	 "Mansi" => "MAN ",
	 "Marathi" => "MAR ",
	 "Marwari" => "MAW ",
	 "Mbundu" => "MBN ",
	 "Manchu" => "MCH ",
	 "Moose Cree" => "MCR ",
	 "Mende" => "MDE ",
	 "Me'en" => "MEN ",
	 "Mizo" => "MIZ ",
	 "Macedonian" => "MKD ",
	 "Male" => "MLE ",
	 "Malagasy" => "MLG ",
	 "Malinke" => "MLN ",
	 "Malayalam Reformed" => "MLR ",
	 "Malay" => "MLY ",
	 "Mandinka" => "MND ",
	 "Mongolian" => "MNG ",
	 "Manipuri" => "MNI ",
	 "Maninka" => "MNK ",
	 "Manx Gaelic" => "MNX ",
	 "Moksha" => "MOK ",
	 "Moldavian" => "MOL ",
	 "Mon" => "MON ",
	 "Moroccan" => "MOR ",
	 "Maori" => "MRI ",
	 "Maithili" => "MTH ",
	 "Maltese" => "MTS ",
	 "Mundari" => "MUN ",
	 "Naga-Assamese" => "NAG ",
	 "Nanai" => "NAN ",
	 "Naskapi" => "NAS ",
	 "N-Cree" => "NCR ",
	 "Ndebele" => "NDB ",
	 "Ndonga" => "NDG ",
	 "Nepali" => "NEP ",
	 "Newari" => "NEW ",
	 "Norway House Cree" => "NHC ",
	 "Nisi" => "NIS ",
	 "Niuean" => "NIU ",
	 "Nkole" => "NKL ",
	 "N'ko" => "NK0 ",
	 "Dutch" => "NLD ",
	 "Nogai" => "NOG ",
	 "Norwegian" => "NOR ",
	 "Northern Sami" => "NSM ",
	 "Northern Tai" => "NTA ",
	 "Esperanto" => "NTO ",
	 "Nynorsk" => "NYN ",
	 "Oji-Cree" => "OCR ",
	 "Ojibway" => "OJB ",
	 "Oriya" => "ORI ",
	 "Oromo" => "ORO ",
	 "Ossetian" => "OSS ",
	 "Palestinian Aramaic" => "PAA ",
	 "Pali" => "PAL ",
	 "Punjabi" => "PAN ",
	 "Palpa" => "PAP ",
	 "Pashto" => "PAS ",
	 "Polytonic Greek" => "PGR ",
	 "Pilipino" => "PIL ",
	 "Palaung" => "PLG ",
	 "Polish" => "PLK ",
	 "Provencal" => "PRO ",
	 "Portuguese" => "PTG ",
	 "Chin" => "QIN ",
	 "Rajasthani" => "RAJ ",
	 "R-Cree" => "RCR ",
	 "Russian Buriat" => "RBU ",
	 "Riang" => "RIA ",
	 "Rhaeto-Romanic" => "RMS ",
	 "Romanian" => "ROM ",
	 "Romany" => "ROY ",
	 "Rusyn" => "RSY ",
	 "Ruanda" => "RUA ",
	 "Russian" => "RUS ",
	 "Sadri" => "SAD ",
	 "Sanskrit" => "SAN ",
	 "Santali" => "SAT ",
	 "Sayisi" => "SAY ",
	 "Sekota" => "SEK ",
	 "Selkup" => "SEL ",
	 "Sango" => "SGO ",
	 "Shan" => "SHN ",
	 "Sibe" => "SIB ",
	 "Sidamo" => "SID ",
	 "Silte Gurage" => "SIG ",
	 "Skolt Sami" => "SKS ",
	 "Slovak" => "SKY ",
	 "Slavey" => "SLA ",
	 "Slovenian" => "SLV ",
	 "Somali" => "SML ",
	 "Samoan" => "SMO ",
	 "Sena" => "SNA ",
	 "Sindhi" => "SND ",
	 "Sinhalese" => "SNH ",
	 "Soninke" => "SNK ",
	 "Sodo Gurage" => "SOG ",
	 "Sotho" => "SOT ",
	 "Albanian" => "SQI ",
	 "Serbian" => "SRB ",
	 "Saraiki" => "SRK ",
	 "Serer" => "SRR ",
	 "South Slavey" => "SSL ",
	 "Southern Sami" => "SSM ",
	 "Suri" => "SUR ",
	 "Svan" => "SVA ",
	 "Swedish" => "SVE ",
	 "Swadaya Aramaic" => "SWA ",
	 "Swahili" => "SWK ",
	 "Swazi" => "SWZ ",
	 "Sutu" => "SXT ",
	 "Syriac" => "SYR ",
	 "Tabasaran" => "TAB ",
	 "Tajiki" => "TAJ ",
	 "Tamil" => "TAM ",
	 "Tatar" => "TAT ",
	 "TH-Cree" => "TCR ",
	 "Telugu" => "TEL ",
	 "Tongan" => "TGN ",
	 "Tigre" => "TGR ",
	 "Tigrinya" => "TGY ",
	 "Thai" => "THA ",
	 "Tahitian" => "THT ",
	 "Tibetan" => "TIB ",
	 "Turkmen" => "TKM ",
	 "Temne" => "TMN ",
	 "Tswana" => "TNA ",
	 "Tundra Nenets" => "TNE ",
	 "Tonga" => "TNG ",
	 "Todo" => "TOD ",
	 "Turkish" => "TRK ",
	 "Tsonga" => "TSG ",
	 "Turoyo Aramaic" => "TUA ",
	 "Tulu" => "TUL ",
	 "Tuvin" => "TUV ",
	 "Twi" => "TWI ",
	 "Udmurt" => "UDM ",
	 "Ukrainian" => "UKR ",
	 "Urdu" => "URD ",
	 "Upper Sorbian" => "USB ",
	 "Uyghur" => "UYG ",
	 "Uzbek" => "UZB ",
	 "Venda" => "VEN ",
	 "Vietnamese" => "VIT ",
	 "Wa" => "WA  ",
	 "Wagdi" => "WAG ",
	 "West-Cree" => "WCR ",
	 "Welsh" => "WEL ",
	 "Wolof" => "WLF ",
	 "Tai Lue" => "XBD ",
	 "Xhosa" => "XHS ",
	 "Yakut" => "YAK ",
	 "Yoruba" => "YBA ",
	 "Y-Cree" => "YCR ",
	 "Yi Classic" => "YIC ",
	 "Yi Modern" => "YIM ",
	 "Chinese Hong Kong" => "ZHH ",
	 "Chinese Phonetic" => "ZHP ",
	 "Chinese Simplified" => "ZHS ",
	 "Chinese Traditional" => "ZHT ",
	 "Zande" => "ZND ",
	 "Zulu" => "ZUL ",
    },

'FEATURE' => {
	 "Access All Alternates" => "aalt",
	 "Above-Base Forms" => "abvf",
	 "Above-Base Mark Positioning" => "abvm",
	 "Above-Base Substitutions" => "abvs",
	 "Alternative Fractions" => "afrc",
	 "Akhands" => "akhn",
	 "Below-Base Forms" => "blwf",
	 "Below-Base Mark Positioning" => "blwm",
	 "Below-Base Substitutions" => "blws",
	 "Petite Capitals From Capitals" => "c2pc",
	 "Small Capitals From Capitals" => "c2sc",
	 "Contextual Alternates" => "calt",
	 "Case-Sensitive Forms" => "case",
	 "Glyph Composition/Decomposition" => "ccmp",
	 "Conjunct form after ro" => "cfar",
	 "Contextual Ligatures" => "clig",
	 "Conjuncts" => "cjct",
	 "Capital Spacing" => "cpsp",
	 "Contextual Swash" => "cswh",
	 "Cursive Positioning" => "curs",
	 "Default Processing" => "dflt",
	 "Distances" => "dist",
	 "Discretionary Ligatures" => "dlig",
	 "Denominators" => "dnom",
	 "Diphthongs" => "dpng",
	 "Expert Forms" => "expt",
	 "Final glyph Alternates" => "falt",
	 "Terminal Forms" => "fina",
	 "Terminal Forms #2" => "fin2",
	 "Terminal Forms #3" => "fin3",
	 "Fractions" => "frac",
	 "Full Width" => "fwid",
	 "Half Forms" => "half",
	 "Halant Forms" => "haln",
	 "Alternate Half Width" => "halt",
	 "Historical Forms" => "hist",
	 "Horizontal Kana Alternates" => "hkna",
	 "Historical Ligatures" => "hlig",
	 "Hangul" => "hngl",
	 "Half Width" => "hwid",
	 "Hojo Kanji Forms" => "hojo",
	 "Initial Forms" => "init",
	 "Isolated Forms" => "isol",
	 "Italics" => "ital",
	 "Japanese Forms" => "jajp",
	 "Justification Alternatives" => "jalt",
	 "JIS04 Forms" => "jp04",
	 "JIS78 Forms" => "jp78",
	 "JIS83 Forms" => "jp83",
	 "JIS90 Forms" => "jp90",
	 "Kerning" => "kern",
	 "Left Bounds" => "lfbd",
	 "Standard Ligatures" => "liga",
	 "Leading Jamo Forms" => "ljmo",
	 "Lining Figures" => "lnum",
	 "Localized Forms" => "locl",
	 "Mark Positioning" => "mark",
	 "Medial Forms" => "medi",
	 "Medial Forms #2" => "med2",
	 "Mathematical Greek" => "mgrk",
	 "Mark to Mark Positioning" => "mkmk",
	 "Mark Positioning via Substitution" => "mset",
	 "Alternate Annotation Forms" => "nalt",
	 "NLC Kanji Forms" => "nlck",
	 "Nukta Forms" => "nukt",
	 "Numerators" => "numr",
	 "Old Style Figures" => "onum",
	 "Optical Bounds" => "opbd",
	 "Ordinals" => "ordn",
	 "Ornaments" => "ornm",
	 "Proportional Alternate Width" => "palt",
	 "Petite Capitals" => "pcap",
	 "Proportional Figures" => "pnum",
	 "Pre-base Forms" => "pref",
	 "Pre-base Substitutions" => "pres",
	 "Post-base Forms" => "pstf",
	 "Post-base Substitutions" => "psts",
	 "Proportional Widths" => "pwid",
	 "Quarter Widths" => "qwid",
	 "Randomize" => "rand",
	 "Rakar Forms" => "rkrf",
	 "Required Ligatures" => "rlig",
	 "Reph Form" => "rphf",
	 "Right Bounds" => "rtbd",
	 "Right-To-Left Alternates" => "rtbd",
	 "Ruby Notation Forms" => "ruby",
	 "Stylistic Alternates" => "salt",
	 "Scientific Inferiors" => "sinf",
	 "Optical Size" => "size",
	 "Small Capitals" => "smcp",
	 "Simplified Forms" => "smpl",
	 "Sylistic Set 1" => "ss01",
	 "Sylistic Set 2" => "ss02",
	 "Sylistic Set 3" => "ss03",
	 "Sylistic Set 4" => "ss04",
	 "Sylistic Set 5" => "ss05",
	 "Sylistic Set 6" => "ss06",
	 "Sylistic Set 7" => "ss07",
	 "Sylistic Set 8" => "ss08",
	 "Sylistic Set 9" => "ss09",
	 "Sylistic Set 10" => "ss10",
	 "Sylistic Set 11" => "ss11",
	 "Sylistic Set 12" => "ss12",
	 "Sylistic Set 13" => "ss13",
	 "Sylistic Set 14" => "ss14",
	 "Sylistic Set 15" => "ss15",
	 "Sylistic Set 16" => "ss16",
	 "Sylistic Set 17" => "ss17",
	 "Sylistic Set 18" => "ss18",
	 "Sylistic Set 19" => "ss19",
	 "Sylistic Set 20" => "ss20",
	 "Subscript" => "subs",
	 "Superscript" => "sups",
	 "Swash" => "swsh",
	 "Titling" => "titl",
	 "Trailing Jamo Forms" => "tjmo",
	 "Traditional Name Forms" => "tnam",
	 "Tabular Figures" => "tnum",
	 "Traditional Forms" => "trad",
	 "Third Widths" => "twid",
	 "Unicase" => "unic",
	 "Alternate Vertical Metrics" => "valt",
	 "Vattu Variants" => "vatu",
	 "Vertical Writing" => "vert",
	 "Alternate Vertical Half Metrics" => "vhal",
	 "Vowel Jamo Forms" => "vjmo",
	 "Vertical Kana Alternates" => "vkna",
	 "Vertical Kerning" => "vkrn",
	 "Proportional Alternate Vertical Metrics" => "vpal",
	 "Vertical Rotation" => "vrt2",
	 "Slashed Zero" => "zero",
    }
);

{
    foreach my $s (qw ( SCRIPT LANGUAGE FEATURE ) )
    {
        map { $ttnames{$s}{$tttags{$s}{$_}} = $_ }  keys %{$tttags{$s}};
    }
}


=head2 readtagsfile ( filename )

Read a file in the syntax of Tags.txt (included with Microsoft VOLT) to obtain additional/replacement tag definitions.

Returns 0 if can't open the file; else 1.

=cut

sub readtagsfile
{
    my $fname = shift;
    open (TAGS, $fname) or return 0;
	my ($what, $name, $tag);
	while (<TAGS>)
	{
		($what, $name, $tag) = (m/"([^"]*)", "([^"]*)", "([^"]*)"/);  #"
		$ttnames{$what}{$tag} = $name;
		$tttags{$what}{$name} = $tag;
	}
	close TAGS;
    return 1;	
}



1;