package Math::DWT::Wavelet::Biorthogonal;

=head1 NAME

Math::DWT::Wavelet::Biorthogonal - FIR lo- & hi-pass filter coefficients for the Biorthogonal wavelet.

=head1 VERSION

Version 0.01_1

=cut

our $VERSION = '0.01_1';

=head1 SYNOPSIS

This module provides the lo- and hi-pass decomposition and reconstruction filter coefficients for the Biorthogonal wavelet.  It is meant to be used with other Math::DWT modules:

    use Math::DWT;
    use Math::DWT::UDWT;

    my $dwt = Math::DWT->new('Biorthogonal');
    my $udwt = Math::DWT::UDWT->new('Biorthogonal');


=cut

=head1 SUBROUTINES/METHODS

=head2 new(VAR)

Takes a single argument, var, which determines the particular filter set which the object will provide.  You can get a list of available filter sets with the vars() method.

    my $biorthogonal2_2 = Math::DWT::Wavelet::Biorthogonal->new('2.2');

This method returns a Math::DWT::Wavelet::Biorthogonal object;

=head2 vars()

This method returns a list of possible choices for VAR when creating a new object, e.g.:

    my @v = Math::DWT::Wavelet::Biorthogonal->vars();
    print scalar(@v); # 15


=head2 filters()

Depending on the context in which it is called, returns an array or an arrayref containing (lo_d, hi_d, lo_r, hi_r) - the set of filters which are defined with the instantiated object.


=head2 lo_d()

=head2 hi_d()

=head2 lo_r()

=head2 hi_r()

Returns the requested set of filter coefficients as either an array or arrayref, depending on calling context.

=head1 SEE ALSO

Math::DWT(3pm), Math::DWT::UDWT(3pm), Math::DWT::Wavelet::Haar(3pm), Math::DWT::Wavelet::Coiflet(3pm), Math::DWT::Wavelet::Symlet(3pm), Math::DWT::Wavelet::Daubechies(3pm), Math::DWT::Wavelet::ReverseBiorthogonal(3pm), Math::DWT::Wavelet::DiscreteMeyer(3pm), perl(1)


=head1 AUTHOR


Mike Kroh, C<< <kroh at cpan.org> >>

=head1 BUGS


Please report any bugs or feature requests to C<bug-math-dwt-wavelet-biorthogonal at rt.cpan.org>, or through the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Math-DWT-Wavelet-Biorthogonal>.  I will be notified, and then you'll automatically be notified of progress on your bug as I make changes.



=head1 ACKNOWLEDGEMENTS

These wavelet filter coefficients were scraped from this site: L<http://wavelets.pybytes.com/>.

=head1 LICENSE AND COPYRIGHT


Copyright 2016 Mike Kroh.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut




my @vars=qw/1.1 1.3 1.5 2.2 2.4 2.6 2.8 3.1 3.3 3.5 3.7 3.9 4.4 5.5 6.8/;

my %lo_d; my %hi_d; my %lo_r; my %hi_r;


$lo_d{1.1}=[qw/0.7071067811865476 0.7071067811865476/];
$hi_d{1.1}=[qw/-0.7071067811865476 0.7071067811865476/];
$lo_r{1.1}=[qw/0.7071067811865476 0.7071067811865476/];
$hi_r{1.1}=[qw/0.7071067811865476 -0.7071067811865476/];

$lo_d{1.3}=[qw/-0.08838834764831845 0.08838834764831845 0.7071067811865476 0.7071067811865476 0.08838834764831845 -0.08838834764831845/];
$hi_d{1.3}=[qw/0.0 0.0 -0.7071067811865476 0.7071067811865476 0.0 0.0/];
$lo_r{1.3}=[qw/0.0 0.0 0.7071067811865476 0.7071067811865476 0.0 0.0/];
$hi_r{1.3}=[qw/-0.08838834764831845 -0.08838834764831845 0.7071067811865476 -0.7071067811865476 0.08838834764831845 0.08838834764831845/];

$lo_d{1.5}=[qw/0.01657281518405971 -0.01657281518405971 -0.12153397801643787 0.12153397801643787 0.7071067811865476 0.7071067811865476 0.12153397801643787 -0.12153397801643787 -0.01657281518405971 0.01657281518405971/];
$hi_d{1.5}=[qw/0.0 0.0 0.0 0.0 -0.7071067811865476 0.7071067811865476 0.0 0.0 0.0 0.0/];
$lo_r{1.5}=[qw/0.0 0.0 0.0 0.0 0.7071067811865476 0.7071067811865476 0.0 0.0 0.0 0.0/];
$hi_r{1.5}=[qw/0.01657281518405971 0.01657281518405971 -0.12153397801643787 -0.12153397801643787 0.7071067811865476 -0.7071067811865476 0.12153397801643787 0.12153397801643787 -0.01657281518405971 -0.01657281518405971/];

$lo_d{2.2}=[qw/0.0 -0.1767766952966369 0.3535533905932738 1.0606601717798214 0.3535533905932738 -0.1767766952966369/];
$hi_d{2.2}=[qw/0.0 0.3535533905932738 -0.7071067811865476 0.3535533905932738 0.0 0.0/];
$lo_r{2.2}=[qw/0.0 0.3535533905932738 0.7071067811865476 0.3535533905932738 0.0 0.0/];
$hi_r{2.2}=[qw/0.0 0.1767766952966369 0.3535533905932738 -1.0606601717798214 0.3535533905932738 0.1767766952966369/];

$lo_d{2.4}=[qw/0.0 0.03314563036811942 -0.06629126073623884 -0.1767766952966369 0.4198446513295126 0.9943689110435825 0.4198446513295126 -0.1767766952966369 -0.06629126073623884 0.03314563036811942/];
$hi_d{2.4}=[qw/0.0 0.0 0.0 0.3535533905932738 -0.7071067811865476 0.3535533905932738 0.0 0.0 0.0 0.0/];
$lo_r{2.4}=[qw/0.0 0.0 0.0 0.3535533905932738 0.7071067811865476 0.3535533905932738 0.0 0.0 0.0 0.0/];
$hi_r{2.4}=[qw/0.0 -0.03314563036811942 -0.06629126073623884 0.1767766952966369 0.4198446513295126 -0.9943689110435825 0.4198446513295126 0.1767766952966369 -0.06629126073623884 -0.03314563036811942/];

$lo_d{2.6}=[qw/0.0 -0.006905339660024878 0.013810679320049757 0.046956309688169176 -0.10772329869638811 -0.16987135563661201 0.4474660099696121 0.966747552403483 0.4474660099696121 -0.16987135563661201 -0.10772329869638811 0.046956309688169176 0.013810679320049757 -0.006905339660024878/];
$hi_d{2.6}=[qw/0.0 0.0 0.0 0.0 0.0 0.3535533905932738 -0.7071067811865476 0.3535533905932738 0.0 0.0 0.0 0.0 0.0 0.0/];
$lo_r{2.6}=[qw/0.0 0.0 0.0 0.0 0.0 0.3535533905932738 0.7071067811865476 0.3535533905932738 0.0 0.0 0.0 0.0 0.0 0.0/];
$hi_r{2.6}=[qw/0.0 0.006905339660024878 0.013810679320049757 -0.046956309688169176 -0.10772329869638811 0.16987135563661201 0.4474660099696121 -0.966747552403483 0.4474660099696121 0.16987135563661201 -0.10772329869638811 -0.046956309688169176 0.013810679320049757 0.006905339660024878/];

$lo_d{2.8}=[qw/0.0 0.0015105430506304422 -0.0030210861012608843 -0.012947511862546647 0.02891610982635418 0.052998481890690945 -0.13491307360773608 -0.16382918343409025 0.4625714404759166 0.9516421218971786 0.4625714404759166 -0.16382918343409025 -0.13491307360773608 0.052998481890690945 0.02891610982635418 -0.012947511862546647 -0.0030210861012608843 0.0015105430506304422/];
$hi_d{2.8}=[qw/0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.3535533905932738 -0.7071067811865476 0.3535533905932738 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0/];
$lo_r{2.8}=[qw/0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.3535533905932738 0.7071067811865476 0.3535533905932738 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0/];
$hi_r{2.8}=[qw/0.0 -0.0015105430506304422 -0.0030210861012608843 0.012947511862546647 0.02891610982635418 -0.052998481890690945 -0.13491307360773608 0.16382918343409025 0.4625714404759166 -0.9516421218971786 0.4625714404759166 0.16382918343409025 -0.13491307360773608 -0.052998481890690945 0.02891610982635418 0.012947511862546647 -0.0030210861012608843 -0.0015105430506304422/];

$lo_d{3.1}=[qw/-0.3535533905932738 1.0606601717798214 1.0606601717798214 -0.3535533905932738/];
$hi_d{3.1}=[qw/-0.1767766952966369 0.5303300858899107 -0.5303300858899107 0.1767766952966369/];
$lo_r{3.1}=[qw/0.1767766952966369 0.5303300858899107 0.5303300858899107 0.1767766952966369/];
$hi_r{3.1}=[qw/-0.3535533905932738 -1.0606601717798214 1.0606601717798214 0.3535533905932738/];

$lo_d{3.3}=[qw/0.06629126073623884 -0.19887378220871652 -0.15467960838455727 0.9943689110435825 0.9943689110435825 -0.15467960838455727 -0.19887378220871652 0.06629126073623884/];
$hi_d{3.3}=[qw/0.0 0.0 -0.1767766952966369 0.5303300858899107 -0.5303300858899107 0.1767766952966369 0.0 0.0/];
$lo_r{3.3}=[qw/0.0 0.0 0.1767766952966369 0.5303300858899107 0.5303300858899107 0.1767766952966369 0.0 0.0/];
$hi_r{3.3}=[qw/0.06629126073623884 0.19887378220871652 -0.15467960838455727 -0.9943689110435825 0.9943689110435825 0.15467960838455727 -0.19887378220871652 -0.06629126073623884/];

$lo_d{3.5}=[qw/-0.013810679320049757 0.04143203796014927 0.052480581416189075 -0.26792717880896527 -0.07181553246425874 0.966747552403483 0.966747552403483 -0.07181553246425874 -0.26792717880896527 0.052480581416189075 0.04143203796014927 -0.013810679320049757/];
$hi_d{3.5}=[qw/0.0 0.0 0.0 0.0 -0.1767766952966369 0.5303300858899107 -0.5303300858899107 0.1767766952966369 0.0 0.0 0.0 0.0/];
$lo_r{3.5}=[qw/0.0 0.0 0.0 0.0 0.1767766952966369 0.5303300858899107 0.5303300858899107 0.1767766952966369 0.0 0.0 0.0 0.0/];
$hi_r{3.5}=[qw/-0.013810679320049757 -0.04143203796014927 0.052480581416189075 0.26792717880896527 -0.07181553246425874 -0.966747552403483 0.966747552403483 0.07181553246425874 -0.26792717880896527 -0.052480581416189075 0.04143203796014927 0.013810679320049757/];

$lo_d{3.7}=[qw/0.0030210861012608843 -0.009063258303782653 -0.01683176542131064 0.074663985074019 0.03133297870736289 -0.301159125922835 -0.026499240945345472 0.9516421218971786 0.9516421218971786 -0.026499240945345472 -0.301159125922835 0.03133297870736289 0.074663985074019 -0.01683176542131064 -0.009063258303782653 0.0030210861012608843/];
$hi_d{3.7}=[qw/0.0 0.0 0.0 0.0 0.0 0.0 -0.1767766952966369 0.5303300858899107 -0.5303300858899107 0.1767766952966369 0.0 0.0 0.0 0.0 0.0 0.0/];
$lo_r{3.7}=[qw/0.0 0.0 0.0 0.0 0.0 0.0 0.1767766952966369 0.5303300858899107 0.5303300858899107 0.1767766952966369 0.0 0.0 0.0 0.0 0.0 0.0/];
$hi_r{3.7}=[qw/0.0030210861012608843 0.009063258303782653 -0.01683176542131064 -0.074663985074019 0.03133297870736289 0.301159125922835 -0.026499240945345472 -0.9516421218971786 0.9516421218971786 0.026499240945345472 -0.301159125922835 -0.03133297870736289 0.074663985074019 0.01683176542131064 -0.009063258303782653 -0.0030210861012608843/];

$lo_d{3.9}=[qw/-0.000679744372783699 0.002039233118351097 0.005060319219611981 -0.020618912641105536 -0.014112787930175846 0.09913478249423216 0.012300136269419315 -0.32019196836077857 0.0020500227115698858 0.9421257006782068 0.9421257006782068 0.0020500227115698858 -0.32019196836077857 0.012300136269419315 0.09913478249423216 -0.014112787930175846 -0.020618912641105536 0.005060319219611981 0.002039233118351097 -0.000679744372783699/];
$hi_d{3.9}=[qw/0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 -0.1767766952966369 0.5303300858899107 -0.5303300858899107 0.1767766952966369 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0/];
$lo_r{3.9}=[qw/0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.1767766952966369 0.5303300858899107 0.5303300858899107 0.1767766952966369 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0/];
$hi_r{3.9}=[qw/-0.000679744372783699 -0.002039233118351097 0.005060319219611981 0.020618912641105536 -0.014112787930175846 -0.09913478249423216 0.012300136269419315 0.32019196836077857 0.0020500227115698858 -0.9421257006782068 0.9421257006782068 -0.0020500227115698858 -0.32019196836077857 -0.012300136269419315 0.09913478249423216 0.014112787930175846 -0.020618912641105536 -0.005060319219611981 0.002039233118351097 0.000679744372783699/];

$lo_d{4.4}=[qw/0.0 0.03782845550726404 -0.023849465019556843 -0.11062440441843718 0.37740285561283066 0.8526986790088938 0.37740285561283066 -0.11062440441843718 -0.023849465019556843 0.03782845550726404/];
$hi_d{4.4}=[qw/0.0 -0.06453888262869706 0.04068941760916406 0.41809227322161724 -0.7884856164055829 0.41809227322161724 0.04068941760916406 -0.06453888262869706 0.0 0.0/];
$lo_r{4.4}=[qw/0.0 -0.06453888262869706 -0.04068941760916406 0.41809227322161724 0.7884856164055829 0.41809227322161724 -0.04068941760916406 -0.06453888262869706 0.0 0.0/];
$hi_r{4.4}=[qw/0.0 -0.03782845550726404 -0.023849465019556843 0.11062440441843718 0.37740285561283066 -0.8526986790088938 0.37740285561283066 0.11062440441843718 -0.023849465019556843 -0.03782845550726404/];

$lo_d{5.5}=[qw/0.0 0.0 0.03968708834740544 0.007948108637240322 -0.05446378846823691 0.34560528195603346 0.7366601814282105 0.34560528195603346 -0.05446378846823691 0.007948108637240322 0.03968708834740544 0.0/];
$hi_d{5.5}=[qw/-0.013456709459118716 -0.002694966880111507 0.13670658466432914 -0.09350469740093886 -0.47680326579848425 0.8995061097486484 -0.47680326579848425 -0.09350469740093886 0.13670658466432914 -0.002694966880111507 -0.013456709459118716 0.0/];
$lo_r{5.5}=[qw/0.013456709459118716 -0.002694966880111507 -0.13670658466432914 -0.09350469740093886 0.47680326579848425 0.8995061097486484 0.47680326579848425 -0.09350469740093886 -0.13670658466432914 -0.002694966880111507 0.013456709459118716 0.0/];
$hi_r{5.5}=[qw/0.0 0.0 0.03968708834740544 -0.007948108637240322 -0.05446378846823691 -0.34560528195603346 0.7366601814282105 -0.34560528195603346 -0.05446378846823691 -0.007948108637240322 0.03968708834740544 0.0/];

$lo_d{6.8}=[qw/0.0 0.0019088317364812906 -0.0019142861290887667 -0.016990639867602342 0.01193456527972926 0.04973290349094079 -0.07726317316720414 -0.09405920349573646 0.4207962846098268 0.8259229974584023 0.4207962846098268 -0.09405920349573646 -0.07726317316720414 0.04973290349094079 0.01193456527972926 -0.016990639867602342 -0.0019142861290887667 0.0019088317364812906/];
$hi_d{6.8}=[qw/0.0 0.0 0.0 0.014426282505624435 -0.014467504896790148 -0.07872200106262882 0.04036797903033992 0.41784910915027457 -0.7589077294536541 0.41784910915027457 0.04036797903033992 -0.07872200106262882 -0.014467504896790148 0.014426282505624435 0.0 0.0 0.0 0.0/];
$lo_r{6.8}=[qw/0.0 0.0 0.0 0.014426282505624435 0.014467504896790148 -0.07872200106262882 -0.04036797903033992 0.41784910915027457 0.7589077294536541 0.41784910915027457 -0.04036797903033992 -0.07872200106262882 0.014467504896790148 0.014426282505624435 0.0 0.0 0.0 0.0/];
$hi_r{6.8}=[qw/0.0 -0.0019088317364812906 -0.0019142861290887667 0.016990639867602342 0.01193456527972926 -0.04973290349094079 -0.07726317316720414 0.09405920349573646 0.4207962846098268 -0.8259229974584023 0.4207962846098268 0.09405920349573646 -0.07726317316720414 -0.04973290349094079 0.01193456527972926 0.016990639867602342 -0.0019142861290887667 -0.0019088317364812906/];
;

sub new {
	my $class=shift;
	my $self={};
	my $var=shift;

	$self={lo_d=>$lo_d{$var},
		hi_d=>$hi_d{$var},
		lo_r=>$lo_r{$var},
		hi_r=>$hi_r{$var}
		};
	
	bless $self,$class;
	return $self;
};

sub vars {
	my $self=shift;
	if (wantarray()) {
		return (@vars);
	};
	return \@vars;
};

sub filters {
	my $self=shift;
	my $lo_d=$self->lo_d;
	my $hi_d=$self->hi_d;
	my $lo_r=$self->lo_r;
	my $hi_r=$self->hi_r;
	my @a=( $lo_d,$hi_d,$lo_r,$hi_r);
	if (wantarray()) {
		return (@a);
	};
	return \@a;
};

sub lo_d {
	my $self=shift;
	my $a=$self->{lo_d};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	
sub hi_d {
	my $self=shift;
	my $a=$self->{hi_d};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	
sub lo_r {
	my $self=shift;
	my $a=$self->{lo_r};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	
sub hi_r {
	my $self=shift;
	my $a=$self->{hi_r};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	

		
1;
