#
# Copyright (C) 1997 Ken MacLeod
# See the file COPYING for distribution terms.
#
# $Id: Notation.pm,v 1.2 1998/01/04 16:00:17 ken Exp $
#

package SGML::Notation;

use strict;
use Class::Visitor;

visitor_class 'SGML::Notation', 'Class::Visitor::Base', {
     'name' => '$',
     'system_id' => '$',
     'public_id' => '$',
     'generated_id' => '$',
};

=head1 NAME

SGML::Notation - an data type defined in SGML or XML

=head1 SYNOPSIS

  $name = $notation->name;
  $system_id = $notation->system_id;
  $public_id = $notation->public_id;
  $generated_id = $notation->generated_id;

  $notation->iter;

  $notation->accept($visitor, ...);

  The following are defined for type compatibilty:

  $notation->as_string([$context, ...]);
  $notation->accept_gi($visitor, ...);
  $notation->children_accept($visitor, ...);
  $notation->children_accept_gi($visitor, ...);

=head1 DESCRIPTION

C<SGML::Notation> objects are loaded by C<SGML::SPGrove>.  An
C<SGML::Notation> contains the type definition defined in a document
instance, with the possible `generated_id' generated by the parser.
Within a grove, any notation with the same name refers to the same
C<SGML::Notation> object.

C<SGML::Notation> objects occur as the value of element attributes or
as a member of external entities.

C<$notation-E<gt>name> returns the name of the notation.

C<$notation-E<gt>accept($visitor[, ...])> issues a call back to
S<C<$visitor-E<gt>visit_SGML_Notation($notation[, ...])>>.  Note that
C<SGML::Notation> objects are never primary children of an
C<SGML::Element> object and will not ordinarily occur while simply
visiting a grove.

C<$notation-E<gt>as_string> returns an empty string.

C<$notation-E<gt>accept_gi($visitor[, ...])> is implemented as a
synonym for C<accept>.

C<children_accept> and C<children_accept_gi> do nothing.

=head1 AUTHOR

Ken MacLeod, ken@bitsko.slc.ut.us

=head1 SEE ALSO

perl(1), SGML::SPGrove(3), Text::EntityMap(3), SGML::Element(3),
SGML::PI(3).

=cut

sub as_string {
    my $self = shift;
    my $context = shift;

    return ("");
}

sub accept {
    my $self = shift;
    my $visitor = shift;

    $visitor->visit_SGML_Notation ($self, @_);
}

# synonomous to `accept'
sub accept_gi {
    my $self = shift;
    my $visitor = shift;

    $visitor->visit_SGML_Notation ($self, @_);
}

# these are here just for type compatibility
sub children_accept { }
sub children_accept_gi { }
sub contents { return [] }

1;
