package POE::Component::YubiAuth;

use warnings;
use strict;

=head1 NAME

POE::Component::YubiAuth - Use Yubico Web Service API to check One Time Passwords

=cut

our $VERSION = '0.01';


=head1 SYNOPSIS

POE::Component::YubiAuth uses Yubico's public Web Service API to verify One Time Passwords
generated by Yubikey.

    use POE::Component::YubiAuth;

    # Get your API id and key at https://api.yubico.com/get-api-key/
    POE::Component::YubiAuth->spawn('<key id>');

    # subref as callback
    POE::Kernel->post('yubi', 'verify', '<otp data>', sub {
        print Data::Dumper::Dumper($_[0]);
    });

    # session event as callback
    POE::Session->create(
        inline_states => {
            _start => sub {
                my ($kernel, $heap) = @_[KERNEL, HEAP];
                $kernel->post('yubi', 'verify', '<otp data>', 'dump');
                $kernel->alias_set('foo');
            },
            dump => sub {
                my ($kernel, $heap) = @_[KERNEL, HEAP];
                print Data::Dumper::Dumper($_[ARG0]);
                $kernel->alias_remove;
                $kernel->post('yubi', 'shutdown');
            },
        }
    );
    POE::Kernel->run();

=head1 CONSTRUCTOR

=head2 POE::Component::YubiAuth->spawn(<key id>)

spawn() takes Yubico key ID as a parameter and spawns a POE session named
'yubi' that will respond to various events later:

    POE::Kernel->post('yubi', 'verify', ...);

        or

    $kernel->post('yubi', 'shutdown');

=cut

=head1 EVENTS

=head2 verify

verify() takes two parameters - Yubikey One Time Password and a callback. Callback can be a
subroutine reference or a name of a POE event in the current session. Examples on how to use
both are provided in the SYNOPSIS:

    POE::Kernel->post('yubi', 'verify', '<otp data>', sub {
        print Data::Dumper::Dumper($_[0]);
    });

The callback will receive a hash reference with response from Yubico's server. If the
'status' key has the value 'OK', then the verification process was successfull. If callback
receives an undefined value instead of a hash reference, then some strange error has occured
(e.g. no connection to the Yubico's server).

Please visit http://www.yubico.com/developers/api/ for more information.

=head2 shutdown

shutdown() terminates the 'yubi' session.

=cut


=head1 AUTHOR

Kirill Miazine, C<< <km@krot.org> >>

=head1 COPYRIGHT & LICENSE

Copyright 2010 Kirill Miazine.

This software is distributed under an ISC-style license, please see
<http://km.krot.org/code/license.txt> for details.

=cut

use POE::Session;
use POE::Component::Client::HTTP;
use HTTP::Request;

sub spawn {
    my $proto = shift;
    my $class = ref($proto) || $proto;

    my $id = shift or die "Yubico ID is required\n";
    my $key = shift;

    POE::Session->create(
        inline_states => {
            _start => sub {
                my ($kernel, $heap) = @_[KERNEL, HEAP];
                $kernel->alias_set('yubi');
                POE::Component::Client::HTTP->spawn(Alias => '_yubi_ua');
            },
            _stop => sub { },
            shutdown => sub {
                my ($kernel, $heap) = @_[KERNEL, HEAP];
                POE::Kernel->post('_yubi_ua', 'shutdown');
                $kernel->alias_remove();
            },
            verify => sub {
                my ($kernel, $sender, $heap) = @_[KERNEL, SENDER, HEAP];
                my ($otp, $callback) = @_[ARG0, ARG1];
                my $req = HTTP::Request->new(GET =>
                    "http://api.yubico.com/wsapi/verify?id=$heap->{'id'}&timestamp=1&otp=$otp"
                );
                $kernel->post('_yubi_ua', 'request', '_verify_cb', $req, [$sender, $callback]);
            },
            _verify_cb => sub {
                my ($kernel, $heap) = @_[KERNEL, HEAP];
                my ($req, $res) = map { $_->[0] } @_[ARG0, ARG1];
                my ($sender, $callback) = @{$_[ARG0]->[1]};
                my $args;
                $args = {map { split /=/, $_, 2 } grep { /=/ } split /\r?\n/, $res->content}
                    if $res->is_success;
                if (ref $callback eq 'CODE') {
                    $callback->($args);
                } elsif (ref $callback) {
                    $callback->postback->($args);
                } else {
                    $kernel->post($sender, $callback, $args);
                }
            }
        },
        heap => {id => $id, key => $key},
    );
}

1; # End of POE::Component::YubiAuth
