package App::Manoc::Logger;
##ABSTRACT: A tiny wrapper around Log4perl

use strict;
use warnings;

our $VERSION = '2.99.3'; ##TRIAL VERSION

use Carp;

use FindBin;
use Log::Log4perl;
use Log::Log4perl::Level;


sub initialized { Log::Log4perl->initialized(); }


sub logger {
    my $self = shift;
    Log::Log4perl->get_logger(@_);
}

sub _init_screen_logger {
    my ($category) = @_;
    my $logger = Log::Log4perl->get_logger($category);

    my $appender =
        Log::Log4perl::Appender->new( "Log::Log4perl::Appender::Screen", name => 'screenlog' );

    my $layout = Log::Log4perl::Layout::PatternLayout->new("[%d] %p %m%n");
    $appender->layout($layout);
    $logger->add_appender($appender);
    $logger->level($DEBUG);
}


sub init {
    my $self  = shift;
    my %args  = ( scalar(@_) == 1 ) ? %{ $_[0] } : @_;
    my $class = $args{class} || '';

    return if Log::Log4perl->initialized();

    if ( $ENV{MANOC_SUPPRESS_LOG} ) {
        Log::Log4perl->easy_init($OFF);
        return;
    }

    if ( $args{debug} ) {
        _init_screen_logger('');
        return;
    }

    my $config_file = $ENV{APP_MANOC_LOG_CONFIG};
    unless ( defined($config_file) && -f $config_file ) {

        my @config_paths;
        exists $ENV{APP_MANOC_CONFIG} and
            push @config_paths, $ENV{APP_MANOC_CONFIG};
        push @config_paths, File::Spec->catdir( $FindBin::Bin, File::Spec->updir() );
        -d '/etc' and push @config_paths, '/etc';

        foreach my $p (@config_paths) {
            my $file = File::Spec->catfile( $p, 'manoc_log.conf' );
            -f $file or next;
            $config_file = $file;
            last;
        }
    }

    if ( defined($config_file) ) {
        -f $config_file or croak "Cannot open config file $config_file";
        Log::Log4perl->init($config_file);
    }
    else {
        _init_screen_logger('');
    }
}

1;

# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Logger - A tiny wrapper around Log4perl

=head1 VERSION

version 2.99.3

=head1 METHODS

=head2 logger

Return a Log4Perl logger.

=head2 init

Initialize Manoc loggers.

=head1 FUNCTIONS

=head2 initialized

Return true if the logger has been initialized.

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
