package Dancer2::Template::Handlebars 0.1;

# ABSTRACT: Dancer2 wrapper for Handlebars templating engine

use strict;
use warnings;

use Text::Handlebars;
use Module::Runtime qw/ use_module /;

use Moo;
use Try::Tiny;
with 'Dancer2::Core::Role::Template';

has '+default_tmpl_ext' => ( default => sub { 'hbs' }, );

has helpers => (
    is      => 'ro',
    lazy    => 1,
    builder => '_build_helpers',
);

has _engine => (
    is      => 'ro',
    lazy    => 1,
    default => sub {
        my $self = shift;

        return Text::Handlebars->new( helpers => $self->helpers, );
    },
);

has _config => (
    is      => 'ro',
    lazy    => 1,
    default => sub {
        return $_[0]->settings->{engines}->{handlebars} || {};
    }
);

sub _build_helpers {
    my $self = shift;

    my %helpers;

    if ( my $h = $self->_config->{helpers} ) {
        for my $module ( ref $h ? @$h : $h ) {
            my %h = try {
                use_module($module)->HANDLEBARS_HELPERS
            }
            catch {
                die "couldn't import helper functions from $module: $_";
            };

            @helpers{ keys %h } = values %h;
        }
    }

    return \%helpers;
}

sub render {
    my ( $self, $template, $tokens ) = @_;

    my $method = 'render';

    if ( ref $template ) {    # it's a ref to a string
        $template = $$template;
        $method .= '_string';
    }

    return $self->_engine->$method( $template, $tokens );
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Dancer2::Template::Handlebars - Dancer2 wrapper for Handlebars templating engine

=head1 VERSION

version 0.1

=head1 SYNOPSIS

Dancer2 wrapper around Handlebars templating engine

=head1 GRATEFUL THANKS

...to Yanick. His prior work on L<Dancer::Template::Handlebars> and
L<Dancer2::Template::Mustache>. Most all of the code you see in this
module is his work, or very, very close to his original code; I
merely remixed it, and got tests working for my own purposes.

=head1 AUTHOR

Ruth Holloway <ruth@hiruthie.me>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020 by Ruth Holloway.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
