package Prty::ContentProcessor;
use base qw/Prty::Hash/;

use strict;
use warnings;

our $VERSION = 1.096;

use Prty::Option;
use Prty::Perl;
use Prty::Path;
use Prty::Section::Parser;
use Prty::Section::Object;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Prty::ContentProcessor - Prozessiere Abschnitts-Dateien

=head1 BASE CLASS

L<Prty::Hash>

=head1 DESCRIPTION

Ein Objekt der Klasse repräsentiert einen Prozessor für
Abschnitts-Dateien, also Dateien, die von einem Abschnittsparser
(s. Klasse Prty::Section::Parser) geparsed werden.

=head1 METHODS

=head2 Konstruktor

=head3 new() - Instantiiere ContentProcessor

=head4 Synopsis

    $cop = $class->new($storageDir,@opt);

=head4 Description

Instantiiere ein ContentProcessor-Objekt und liefere eine Referenz
auf dieses Objekt zurück.

=head4 Arguments

=over 4

=item $storage

Der Name des Storage-Verzeichnisses ohne Verzeichnisanteil,
z.B. '.storage'.

=back

=head4 Options

=over 4

=item -extensions => \@extensions

Liste der Datei-Erweiterungen der Dateien, die der
ContentProcessor verarbeitet. Hinzu kommen die Erweiterungen, die
die per registerType() hinzugefügten Typen definieren.

=back

=head4 Returns

ContentProcessor-Objekt

=cut

# -----------------------------------------------------------------------------

sub new {
    my $class = shift;
    my $storage = shift;

    # Optionen

    my $extensions = [];

    Prty::Option->extract(\@_,
        -extensions=>\$extensions,
    );

    # Objekt instantiieren

    my $self = $class->SUPER::new(
        storage=>$storage,
        baseExtensionA=>$extensions,
        typeA=>[],
        entityA=>[],
        # memoize
        extensionRegex=>undef,
    );

    return $self;
}

# -----------------------------------------------------------------------------

=head2 Plugins

=head3 registerType() - Registriere Plugin-Klasse für Typ

=head4 Synopsis

    $cop->registerType($pluginClass,$extension,$sectionType,@keyVal);

=head4 Description

Registriere Plugin-Klasse $pluginClass für Abschnitts-Dateien mit
Identifier $sectionType und den Abschnitts-Eigenschaften @keyVal.
Die Plugin-Klasse wird automatisch geladen, falls sie nicht
existiert (sie kann für Tests also auch "inline" definiert werden).

=head4 Arguments

=over 4

=item $pluginClass

Name der Plugin-Klasse, z.B. 'Program::Shell'.

=item $extension

Datei-Erweiterung für Dateien dieses Typs, z.B. '.prg'.

=item $sectionType

Abschnitts-Bezeichner einschl. Klammerung, z.B. '[Program]'.

=item @keyVal

Abschnitts-Attribute, die über den Abschnitts-Bezeichner hinaus
den Dateityp kennzeichnen, z.B. Language=>'Shell'.

=back

=head4 Returns

nichts

=cut

# -----------------------------------------------------------------------------

sub registerType {
    my $self = shift;
    # @_: $pluginClass,$extension,$sectionType,@keyVal

    Prty::Perl->loadClass($_[0]);
    push @{$self->{'typeA'}},\@_;
        
    return;
}

# -----------------------------------------------------------------------------

=head2 Operationen

=head3 process() - Verarbeite Dateien

=head4 Synopsis

    $cop->process(@paths);

=head4 Description

Verarbeite alle ContentProcessor-Dateien der Pfade @paths.

=head4 Arguments

=over 4

=item @paths

Liste der Verzeichnisse und Dateien. Der Pfad '-' bedeutet STDIN.

=back

=head4 Returns

nichts

=cut

# -----------------------------------------------------------------------------

sub process {
    my $self = shift;
    my @paths = @_;

    my @files;
    for my $path (@paths) {
        if (-d $path) {
            # Verzeichnis: nach Dateien durchsuchen

            push @files,Prty::Path->find($path,
                -type=>'f',
                -pattern=>$self->extensionRegex,
            );
        }
        else {
            # Datei: direkt hinzufügen
            push @files,$path;
        }
    }
    
    # Instantiiere Parser

    my $par = Prty::Section::Parser->new(
        encoding=>'utf-8',
    );

    # Parse Dateien zu Entitäten

    my $entityA = $self->get('entityA');
    for my $file (@files) {
        $par->parse($file,sub {
            my $sec = Prty::Section::Object->new(@_);

            my $class = $self->typeClass($sec);
            if ($class) {
                $sec->weaken(root=>$self); # Verweis auf ContentProcessor
                push @$entityA,$class->new($sec);
            }
            elsif (@$entityA) {
                my $ent = $entityA->[-1];
                $ent->addSubSection($sec);
                $ent->transferSource($sec);
            }
            else {
                $sec->error(
                    q{COP-00001: Unexpected Section},
                     Section=>$sec->fullType,
                );
            }

            return;
        });
    }

    # TODO: Abgleich mit Storage
    
    return;
}

# -----------------------------------------------------------------------------

=head2 Intern

=head3 extensionRegex() - Regex zum Auffinden von ContentProcessor-Dateien

=head4 Synopsis

    $regex = $cop->extensionRegex;

=head4 Description

Liefere den regulären Ausdruck, der die Dateinamen matcht, die vom
ContentProcessor verarbeitet werden. Der Regex wird genutzt, wenn
ein I<Verzeichnis> nach ContentProcessor-Dateien durchsucht wird.

=head4 Returns

kopilierter Regex

=cut

# -----------------------------------------------------------------------------

sub extensionRegex {
    my $self = shift;

    return $self->memoize('extensionRegex',sub {
        my ($self,$key) = @_;
        
        # Datei-Erweiterungen ermitteln

        my @extensions = @{$self->{'baseExtensionA'}};
        for my $arr (@{$self->{'typeA'}}) {
            push @extensions,$arr->[1],
        }        

        # Regex erstellen

        my $regex;
        for (@extensions) {
            my $ext = $_;
            if ($regex) {
                $regex .= '|';
            }
            $ext =~ s/^\.//; # führenden Punkt entfernen
            $regex .= $ext;
        }

        return qr/\.($regex)$/;
    });
}

# -----------------------------------------------------------------------------

=head3 typeClass() - Plugin-Klasse zum Abschnitts-Objekt

=head4 Synopsis

    $typeClass = $cop->typeClass($sec);

=head4 Description

Ermittele die Plugin-Klasse zum Abschnitts-Objekt $sec. Existiert
keine Plugin-Klasse zum Abschnitts-Objekt, liefere C<undef>.

=head4 Arguments

=over 4

=item $sec

Abschnitts-Objekt

=back

=head4 Returns

Name der Plugin-Kasse

=cut

# -----------------------------------------------------------------------------

sub typeClass {
    my ($self,$sec) = @_;

    my $fullType = $sec->fullType;
    for my $arr (@{$self->{'typeA'}}) {
        if ($arr->[2] eq $fullType) {
            my $ok = 1;
            for (my $i = 3; $i < @$arr; $i += 2) {
                if ($sec->get($arr->[$i]) ne $arr->[$i+1]) {
                    $ok = 0;
                    last;
                }
            }
            if ($ok) {
                return $arr->[0];
            }
        }
    }

    return undef;
}

# -----------------------------------------------------------------------------

=head1 VERSION

1.096

=head1 AUTHOR

Frank Seitz, L<http://fseitz.de/>

=head1 COPYRIGHT

Copyright (C) 2016 Frank Seitz

=head1 LICENSE

This code is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

# -----------------------------------------------------------------------------

1;

# eof
