package CXC::Optics::Prescription::LVS::Mirror;

# ABSTRACT: Parsing and extracting LVS mirror prescriptions

use strict;
use warnings;

use feature ':5.20';
use experimental qw[ signatures postderef switch];

our $VERSION = '0.01';


use File::ShareDir ();
use YAML ();
use File::Spec::Functions ();

use Data::Dump ();


use Iterator::Flex::Base 0.07;
use Iterator::Flex::Failure;

use Regexp::Common qw/ number /;

sub _geo_to_PS ( $geo ) {

    { 'hyperbola' => 'secondary', parabola => 'primary' }->{ lc $geo };
}

#pod =sub load_mirror_model
#pod
#pod   $model = CXC::Optics::Prescription::LVS::Mirror->load_mirror_model( $model_name );
#pod
#pod Creates a CXC::Optics::Prescription::LVS::Mirror object from
#pod the pre-parsed model with the given name.
#pod
#pod Available models are:
#pod
#pod   ek05lvs
#pod
#pod =cut

sub load_mirror_model ( $class, $model_name ) {

    my $file = eval {
        File::ShareDir::dist_file( 'CXC-Optics-Prescription-LVS',
            File::Spec::Functions::catfile( "mirror", $model_name . '.yaml' ) );
    };
    die( "unable to find mirror model $model_name\n" )
      if $@;

    return bless YAML::LoadFile( $file ), $class;
}


#pod =sub parse_mirror_output
#pod
#pod   $model = parse_mrror_output( $file );
#pod
#pod Creates a CXC::Optics::Prescription::LVS::Mirror object by
#pod parsing the output file written by LVS' B<mirror> program.
#pod
#pod =cut

sub parse_mirror_output ( $class, $file ) {

    my $fh = IO::File->new( $file )
      or die( "error opening mirror output file: $file\n" );


    my $iter = _fh_iterator( $fh );

    return bless [
        map { $_->@* } _parse_axial_distance( $iter ),
        _parse_dimensions( $iter ),
        _parse_prescription( $iter ),
        _parse_element_lengths( $iter ),
        _parse_sagital_depth( $iter ),
        _parse_focal_length_plate_scale( $iter ),
        _parse_OSAC_parameters( $iter ) ], $class;
}

sub _parse_axial_distance ( $iter ) {

    $iter->reset;

    my @re = (
        qr/Axial Distance to\s+(?<station>.*), (?<unit>.*)$/,
        qr/Axial Distance in (?<unit>.*) to station\s+(?<station>.*?)\s+\g{unit} from focus$/
    );

    my @data;

    for my $re ( @re ) {

        for ( 0, 1 ) {

            my $block = _parse_block( $iter, $re );

            my $station = lc $block->{title_match}{station};
            $station =~ s/\s/_/g;

            my $unit = $block->{title_match}{unit};

            foreach ( $block->{data}->@* ) {
                $_->{other} =~ s/\s/_/g;
                $_->{location} = delete $_->{other};
                $_->{param}    = 'axial_distance';
                $_->{unit}     = $unit;
                $_->{station}  = $station;
            }
            push @data, $block->{data};
        }
    }

    return @data;
}

sub _parse_dimensions ( $iter ) {

    my @data;

    $iter->reset;
    $iter->drain( qr/(Radius|Diameter), /i );

    for ( 1 .. 4 ) {
        my $block
          = _parse_block( $iter, qr/\s*(?<dimension>.*),\s+(?<unit>.*)$/ );

        my $dimension = lc $block->{title_match}{dimension};
        my $unit      = $block->{title_match}{unit};


        foreach ( $block->{data}->@* ) {

            $_->{other} =~ s/\s/_/g;
            $_->{location} = delete $_->{other};
            $_->{unit}     = $unit;
            $_->{param}    = $dimension;
        }

        push @data, $block->{data};
    }

    return @data;
}

sub _parse_prescription ( $iter ) {

    $iter->reset;
    $iter->drain( qr/Optical constants for paraboloids and hyperboloids/, 0 );

    my @data;
    my @titles = (
        [ e    => qr/Eccentricity/ ],
        [ d    => qr/Conic constant/ ],
        [ zoff => qr/z offset/ ],
    );

    for my $title ( @titles ) {
        my ( $name, $qr ) = $title->@*;
        my $block = _parse_block( $iter, $qr,
            hdrs => [ [ 0, 1, 1, 2, 2 ], [ 0, 1, 2, 3, 4 ] ] );

        foreach ( $block->{data}->@* ) {
            $_->{system} = 'lvs';
            $_->{param}  = delete $_->{other} // $name;
        }
        push @data, $block->{data};
    }

    $iter->drain( qr/Grazing Angle/ );

    my $block = _parse_block( $iter, undef,
        hdrs => [ [ 0, 1, 1, 2, 2 ], [ 0, 1, 2, 3, 4 ], ] );
    foreach my $key ( $block->{data}->@* ) {

        given ( $key->{other} ) {

            when ( /Grazing Angle,\s+(?<unit>.*) (?<optic>.*)/i ) {
                $key->{optic} = _geo_to_PS( $+{optic} );
                $key->{param} = 'graze_angle';
                $key->{unit}  = $+{unit};
            }

            when ( /a\s+(?<unit>mm|inches)/ ) {
                $key->{param} = 'a';
                $key->{unit}  = $+{unit};
            }

            default {
                die "can't parse @{[ Data::Dump::pp $_ ]}";
            }

        }
        delete $key->{other};
    }
    push @data, $block->{data};

    return @data;
}

sub _parse_element_lengths( $iter ) {

    $iter->reset;
    my $block = _parse_block(
        $iter,
        qr/Element Lengths/,
        hdrs => [ [ 0, 1, 1, 2, 2 ], [ 0, 1, 2, 3, 4 ], ],
    );
    foreach ( $block->{data}->@* ) {
        $_->{param} = 'length';
        $_->{unit}  = delete $_->{unit};
    }
    $block->{data};
}

sub _parse_sagital_depth ( $iter ) {
    my $block = _parse_block(
        $iter,
        qr/Sagital Depths/,
        hdrs => [ [ 0, 1, 1, 2, 2 ], [ 0, 1, 2, 3, 4 ], ],
    );

    foreach ( $block->{data}->@* ) {
        $_->{param} = 'sagital_depth';
        $_->{unit}  = delete $_->{unit};
    }
    $block->{data};
}

sub _parse_focal_length_plate_scale ( $iter ) {

    $iter->reset;
    my @data;

    for ( 0, 1 ) {
        $iter->drain( qr/Foc ln/ );
        my $block = _parse_block( $iter, undef, ncolhdr => 1 );

        for my $key ( $block->{data}->@* ) {

            given ( $key->{other} ) {

                when ( /foc ln\s+(?<unit>mm|in)/i ) {
                    $key->{param} = 'flength';
                    $key->{unit} = $+{unit} eq 'in' ? 'inches' : $+{unit}
                }
                delete $key->{other};
            }

            default {
                $key->{param} = 'plate_scale';
                $key->{unit}  = ( delete $key->{other} ) =~ s/ //gr;
            }
        }

        push @data, $block->{data};
    }

    return @data;
}

sub _parse_OSAC_parameters ( $iter ) {

    $iter->reset;
    return map { _parse_OSAC_parameters_block( $iter ) } 1, 2;
}
sub _parse_OSAC_parameters_block ( $iter ) {

    $iter->drain( qr/P=conic constant/ );
    my $block = _parse_block(
        $iter, undef,
        hdrs => [
            [ 0, 1, 2, [ 3, 4 ], 5, [ 6 .. 8 ], [ 9 .. 11 ] ],
            [ 0, 1, 2, undef, 3, 4, 5 ],
        ],
        colsplit => ' ',
    );

    foreach my $key ( $block->{data}->@* ) {

        $key->{system} = 'OSAC';

        given ( $key->{other} ) {

            when ( /(?<field>.*)\s+(?<unit>mm|inches)$/ ) {
                ( my $field ) = split( /[\s,-]/, $+{field} );
                delete $key->{other};
                $key->{param} = $field;
                $key->{unit}  = $+{unit};
            }

            when ( /p=conic constant/i ) {
                delete $key->{other};
                $key->{param} = 'p';
            }
        }
    }

    return $block->{data};
}

sub _parse_block ( $iter, $title_re = undef, %attr ) {

    $attr{ncolhdr}  //= 2;
    $attr{hdrs}     //= [ [ 0, 1, 1, 2, 3, 3 ], [ 0, 1, 2, 3, 4, 5 ], ];
    $attr{colsplit} //= qr/[\t]|\s{2,}/;

    die( "hdrs attribute arrays do not have the same number of elements\n" )
      unless $attr{hdrs}[0]->$#* == $attr{hdrs}[1]->$#*;

    # title
    # empty line
    # column label line 1
    # column label line 2
    # empty line
    # columns of data

    # loop until we hit the title

    my ( $title, %title );
    if ( defined $title_re ) {
        while ( $iter->next !~ $title_re ) {
            # say "LINE = @{[ $iter->current ]}";
        }

        $title = $iter->current;
        chomp $title;
        $title =~ $title_re;
        %title = %+;
        $title =~ s/(^\s+)|(\s+$)//g;
    }

    my @raw_col_labels;
    for ( 1 .. $attr{ncolhdr} ) {
        my $line = $iter->next;
        redo unless length $line;
        $raw_col_labels[ $_ - 1 ] = [ split( $attr{colsplit}, $line ) ];
    }

    my @columns;

    if ( $attr{ncolhdr} == 2 ) {
        @columns = map {

            my ( $idx1, $idx2 )
              = map { ref $_ || !defined $_ ? $_ : [$_] } $attr{hdrs}[0][$_],
              $attr{hdrs}[1][$_];
            join( ' ',
                ( defined $idx1 ? $raw_col_labels[0]->@[@$idx1] : () ),
                ( defined $idx2 ? $raw_col_labels[1]->@[@$idx2] : () ),
            );
        } 0 .. $attr{hdrs}[0]->$#*;

    }
    else {
        @columns = $raw_col_labels[0]->@*;
    }

    my @meta;
    while ( my ( $idx, $colval ) = each @columns ) {

        my %meta;

        if ( $colval =~ /^(?<optic>hyperbola|parabola)\s+(?<other>.*)$/i ) {
            $meta{optic} = _geo_to_PS( $+{optic} );
            my $other = lc $+{other};
            if ( $other =~ /(?<unit>mm|inches)/i ) {
                $meta{unit} = $other;
            }
            else {
                $meta{other} = $other;
            }
        }
        else {
            $meta{other} = lc $colval;
        }

        $meta[$idx] = \%meta;

    }

    $iter->next;
    my @datum;
    my $line;

    my $real = $RE{num}{real}{ -expno => '[EeDd]' };

    while ( ( $line = $iter->next ) =~ /^\s*\d/ ) {
        $line =~ s/(^\s+)|(\s+$)//g;
        my @data = map {
            $_ =~ $real
              ? do { my $num = s/d/e/ri; $num +0 }
              : $_
        } split( /[\t]|\s+/, $line );

        my $sidx = 0;
        my %lmeta;
        if ( $columns[0] =~ /mirror|pair num/i ) {
            $lmeta{mirror} = $data[ $sidx++ ];
        }
        if ( $columns[1] =~ /surface p=1,h=2/i ) {
            $lmeta{optic}
              = { 1 => 'primary', 2 => 'secondary' }->{ $data[ $sidx++ ] };
        }

        for ( $sidx .. $#meta ) {

            my %meta = ( %lmeta, $meta[$_]->%*, value => $data[$_] );
            push @datum, \%meta;
        }
    }

    $iter->pushback;

    return {
        title       => $title,
        title_match => \%title,
        data        => \@datum,
    };
}


sub _fh_iterator( $fh ) {

    my ( $prev, $current, $repeat );
    $repeat = 0;

    Iterator::Flex::Base->construct(

        name => 'fh_iterator',
        prev => sub {
            return $prev;
        },

        current => sub {
            return $current;
        },

        next => sub ( $self ) {

            if ( $repeat ) {
                $repeat = 0;
            }
            else {
                $prev    = $current;
                $current = $fh->getline;
            }

            if ( !defined $current ) {
                $self->set_exhausted;
                Iterator::Flex::Failure::Exhausted->throw;
            }
            else {
                chomp $current;
            }

            return $current;
        },

        reset => sub ( $self ) {
            $fh->seek( 0, 0 );
            $prev = $current = undef;
            $repeat = 0;
        },

        methods => {
            pushback => sub {
                $repeat = 1;
            },
            drain => sub ( $iter, $qr, $pushback = 1 ) {
                $iter->next
                  until defined $iter->current && $iter->current =~ $qr;
                $iter->pushback if $pushback;
            }
        } );
}

###################################################################

sub _prescriptions ( $self, $system ) {

    my @pars = grep {
             defined $_->{system}
          && $_->{system} eq $system
          && ( defined $_->{unit} ? $_->{unit} eq 'mm' : 1 )
    } $self->@*;

    my @prescription;
    $prescription[ $_->{mirror} ]{ $_->{optic} }{ $_->{param} } = $_->{value}
      for @pars;

    return @prescription;
}


#pod =method prescriptions
#pod
#pod   @prescriptions = $model->prescriptions;
#pod
#pod Return the prescription for the model.  Linear parameters are in mm.
#pod The prescriptions are returned as an array indexed off of the mirror
#pod number (1-6).
#pod
#pod Each array element looks like this:
#pod
#pod  {
#pod      primary => {
#pod 	 d     => 2.49570504674018,
#pod 	 e     => 1,
#pod 	 'e-1' => 0,
#pod 	 zoff  => 10069.2189948357
#pod      },
#pod      secondary => {
#pod 	 d     => 2.49723582188823,
#pod 	 e     => 1.00024809920262,
#pod 	 'e-1' => 0.000248099202621096,
#pod 	 zoff  => 0
#pod      },
#pod  }
#pod
#pod =cut

sub prescriptions ( $self ) {
    $self->_prescriptions( 'lvs' );
}


#pod =method osac_prescriptions
#pod
#pod   @prescriptions = $model->osac_prescriptions;
#pod
#pod Return the OSAC prescription for the model.  Linear parameters are in mm.
#pod The prescriptions are returned as an array indexed off of the mirror number (1-6).
#pod
#pod Each array element looks like this:
#pod
#pod  {
#pod      primary => {
#pod 	 k     => -2.49570504674018,
#pod 	 p     => 0,
#pod 	 rho_0 => 320.569777256348,
#pod 	 z_0   => 419.1
#pod      },
#pod      secondary => {
#pod 	 k     => -7.26202481526188,
#pod 	 p     => -0.000496259958456534,
#pod 	 rho_0 => 306.098516687762,
#pod 	 z_0   => 1338.1
#pod      },
#pod  }
#pod
#pod =cut

sub osac_prescriptions ( $self ) {
    $self->_prescriptions( 'OSAC' );
}


#pod =method axial_distances
#pod
#pod   @distances = $model->axial_distances;
#pod
#pod Return the axial distances in mm from the focal plane to the
#pod C<primary>, C<secondary>, and C<intersect plane> for each shell.
#pod
#pod The distances are returned as an array indexed off of the mirror number (1-6).
#pod Each array element looks like this:
#pod
#pod   {
#pod       intersect_plane => -3.12348,
#pod       primary         => {
#pod           back  => 40.4,
#pod           front => 878.6
#pod       },
#pod       secondary => {
#pod           back  => -878.6,
#pod           front => -40.4
#pod       },
#pod   }
#pod
#pod =cut


sub axial_distances ( $self ) {

    my @pars = grep {
             $_->{param} eq 'axial_distance'
          && $_->{station} eq 'focal_plane'
          && $_->{unit} eq 'mm'
    } $self->@*;

    my @distance;
    $distance[ $_->{mirror} ]{ $_->{optic} }{ $_->{location} } = $_->{value}
      for grep { exists $_->{optic} } @pars;

    $distance[ $_->{mirror} ]{ $_->{location} } = $_->{value}
      for grep { !exists $_->{optic} } @pars;

    return @distance;
}

#pod =method dimensions
#pod
#pod   @distances = $model->dimensions;
#pod
#pod Return the optic and intersect plane dimensions, in mm.
#pod C<primary>, C<secondary>, and C<intersect plane> for each shell.
#pod
#pod The distances are returned as an array indexed off of the mirror number (1-6).
#pod Each array element looks like this:
#pod
#pod  {
#pod      intersect_plane => {
#pod          radius => 316.96956
#pod      },
#pod      primary => {
#pod          length => 838.2,
#pod          radius => {
#pod              back  => 317.29022,
#pod              front => 323.81612
#pod          }
#pod      },
#pod      secondary => {
#pod          length => 838.2,
#pod          radius => {
#pod              back  => 296.13584,
#pod              front => 316.02294
#pod          }
#pod      },
#pod  }
#pod
#pod =cut

sub dimensions ( $self ) {

    my @pars = grep {
        ( $_->{param} eq 'radius' || $_->{param} eq 'length' )
          && $_->{unit} eq 'mm'
    } $self->@*;

    my @dimension;
    for ( grep { exists $_->{optic} } @pars ) {

        if ( exists $_->{location} ) {
            $dimension[ $_->{mirror} ]{ $_->{optic} }{ $_->{param} }
              { $_->{location} } = $_->{value};
        }
        else {
            $dimension[ $_->{mirror} ]{ $_->{optic} }{ $_->{param} }
              = $_->{value};
        }
    }

    $dimension[ $_->{mirror} ]{ $_->{location} }{ $_->{param} } = $_->{value}
      for grep { !exists $_->{optic} } @pars;

    return @dimension;
}

#pod =method grazing_angles
#pod
#pod   @angles = $model->grazing_angles;
#pod
#pod Return the grazing angles, in radians.
#pod
#pod The angles are returned as an array indexed off of the mirror number (1-6).
#pod Each array element looks like this:
#pod
#pod  {
#pod      primary   => 0.0149014052178367,
#pod      secondary => 0.0149201261086361
#pod  }
#pod
#pod =cut

sub grazing_angles ( $self ) {

    my @angle;
    $angle[ $_->{mirror} ]{ $_->{optic} } = $_->{value}
      for grep { $_->{param} eq 'graze_angle' } $self->@*;

    return @angle;
}


1;

__END__

=pod

=head1 NAME

CXC::Optics::Prescription::LVS::Mirror - Parsing and extracting LVS mirror prescriptions

=head1 VERSION

version 0.01

=head1 SUBROUTINES

=head2 load_mirror_model

  $model = CXC::Optics::Prescription::LVS::Mirror->load_mirror_model( $model_name );

Creates a CXC::Optics::Prescription::LVS::Mirror object from
the pre-parsed model with the given name.

Available models are:

  ek05lvs

=head2 parse_mirror_output

  $model = parse_mrror_output( $file );

Creates a CXC::Optics::Prescription::LVS::Mirror object by
parsing the output file written by LVS' B<mirror> program.

=head1 METHODS

=head2 prescriptions

  @prescriptions = $model->prescriptions;

Return the prescription for the model.  Linear parameters are in mm.
The prescriptions are returned as an array indexed off of the mirror
number (1-6).

Each array element looks like this:

 {
     primary => {
	 d     => 2.49570504674018,
	 e     => 1,
	 'e-1' => 0,
	 zoff  => 10069.2189948357
     },
     secondary => {
	 d     => 2.49723582188823,
	 e     => 1.00024809920262,
	 'e-1' => 0.000248099202621096,
	 zoff  => 0
     },
 }

=head2 osac_prescriptions

  @prescriptions = $model->osac_prescriptions;

Return the OSAC prescription for the model.  Linear parameters are in mm.
The prescriptions are returned as an array indexed off of the mirror number (1-6).

Each array element looks like this:

 {
     primary => {
	 k     => -2.49570504674018,
	 p     => 0,
	 rho_0 => 320.569777256348,
	 z_0   => 419.1
     },
     secondary => {
	 k     => -7.26202481526188,
	 p     => -0.000496259958456534,
	 rho_0 => 306.098516687762,
	 z_0   => 1338.1
     },
 }

=head2 axial_distances

  @distances = $model->axial_distances;

Return the axial distances in mm from the focal plane to the
C<primary>, C<secondary>, and C<intersect plane> for each shell.

The distances are returned as an array indexed off of the mirror number (1-6).
Each array element looks like this:

  {
      intersect_plane => -3.12348,
      primary         => {
          back  => 40.4,
          front => 878.6
      },
      secondary => {
          back  => -878.6,
          front => -40.4
      },
  }

=head2 dimensions

  @distances = $model->dimensions;

Return the optic and intersect plane dimensions, in mm.
C<primary>, C<secondary>, and C<intersect plane> for each shell.

The distances are returned as an array indexed off of the mirror number (1-6).
Each array element looks like this:

 {
     intersect_plane => {
         radius => 316.96956
     },
     primary => {
         length => 838.2,
         radius => {
             back  => 317.29022,
             front => 323.81612
         }
     },
     secondary => {
         length => 838.2,
         radius => {
             back  => 296.13584,
             front => 316.02294
         }
     },
 }

=head2 grazing_angles

  @angles = $model->grazing_angles;

Return the grazing angles, in radians.

The angles are returned as an array indexed off of the mirror number (1-6).
Each array element looks like this:

 {
     primary   => 0.0149014052178367,
     secondary => 0.0149201261086361
 }

=head1 SEE ALSO

Please see those modules/websites for more information related to this module.

=over 4

=item *

L<CXC::Optics::Prescription::LVS|CXC::Optics::Prescription::LVS>

=back

=head1 AUTHOR

Diab Jerius <djerius@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
