#!/usr/bin/perl

# $Id: static_page2base_page.pl,v 1.4 2001/12/02 23:32:49 lachoy Exp $

# Transition all your static_page objects to base_page. This assumes
# that you are using all-database storage and want to transition to
# all-filesystem storage.

use strict;
use OpenInteract::Startup;

use constant DEFAULT_EXTENSION => '.html';

my ( $OPT_add_html, $OPT_as_file, $OPT_as_database );

# Add your own transformations here -- this gets called to set the
# location of the new object given the old location. Default is to add
# the DEFAULT_EXTENSION if '--add_html' was specified and the location
# doesn't have an extension, and change 'home.html' to the more
# standard 'index.html'
#
# An example of adding your own would be to lower-case all URLs

sub transform_location {
    my ( $old_location ) = @_;
    my ( $location );
    if ( $OPT_add_html and $old_location !~ /\.\w+$/ ) {
        $location = join( '', $old_location, DEFAULT_EXTENSION );
    }
    else {
        $location = $old_location;
    }
    $location =~ s|^/NTRC|/ntrc|;
    $location =~ s|^\s*http://.*?/|/|;
    $location = "/$location" unless ( $location =~ m|^/| );
    $location =~ s/home\.html$/index.html/;
    return $location;
}


sub skip_page {
    my ( $location ) = @_;
    return 1 if ( $location =~ m|^/oi_docs| );
    return 1 if ( $location =~ m|\.bak| );
    return 1 if ( $location =~ m|~| );
    return 1 if ( $location =~ m|home\d+| );
    return undef;
}


{
    my %opts = ( 'add_html'    => \$OPT_add_html,
                 'as_file'     => \$OPT_as_file,
                 'as_database' => \$OPT_as_database );
    my $R = OpenInteract::Startup->setup_static_environment_options( '', \%opts );

    open( STDERR, "> static_page2base_page.err" )
          || die "Cannot open error log: $!";

    $OPT_as_file++ unless ( $OPT_as_file or $OPT_as_database );

    # Are the old objects indexed? If so, we remove the items from the
    # index after a successful save of the new object.

    my $is_old_indexed = $R->static_page->isa( 'OpenInteract::FullText' );

    my $start_time = time;

    # First retrieve all the static page objects

    my $iter = $R->static_page->fetch_iterator({ skip_security => 1,
                                                 order         => 'location' });

    my $page_class = $R->page;
    my ( $total, $skipped, $success, $error ) = ( 0, 0, 0, 0 );

    while ( my $static_page = $iter->get_next ) {

        $total++;

        # Callback for custom settings to skip certain pages see
        # routine at top to add your own settings

        if ( skip_page( $static_page->{location} ) ) {
            $skipped++;
            printf( "%4d SKIPPED: %s\n", $total, $static_page->{location} );
            next;
        }

        # Modify the location as necessary -- you might want to
        # customize the transform_location() routine at the top of the
        # script if you have modification you want to make.

        my $location = transform_location( $static_page->{location} );

        # Set most page properties

        my $page = $page_class->new({
                         title         => $static_page->{title},
                         author        => $static_page->{author},
                         keywords      => $static_page->{keywords},
                         boxes         => $static_page->{boxes},
                         main_template => $static_page->{main_template},
                         active_on     => $static_page->{active_on},
                         expires_on    => $static_page->{expires_on},
                         notes         => $static_page->{notes},
                         location      => $location,
                         is_active     => 'yes',
        });

        # Set file status

        $page->{storage} = 'database' if ( $OPT_as_database );
        $page->{storage} = 'file'     if ( $OPT_as_file );

        # Set the content

        if ( $static_page->{script} ) {
            $page->{content} = join( "\n", $static_page->{pagetext},
                                           qq(<script language="JavaScript">\n<!--\n),
                                           $static_page->{script},
                                           qq(\n// -->\n</script>) );
        }
        else {
            $page->{content} = $static_page->{pagetext};
        }
        $page->{content} =~ s/\r//g;

        # Set the MIME type and size of the content

        $page->{mime_type} = $page->mime_type_content( $page->{content} ) ||
                             'text/html';
        $page->{size}      = length $page->{content};

        # Save the page object (automatically saves the content) and
        # display a status

        eval { $page->save({ skip_security => 1 }) };
        if ( $@ ) {
            printf "%4d ERROR: %37s --> %s\n", $total, $static_page->{location}, $page->{location};
            print STDERR "----------------------------------------\n",
                         "Error converting ($static_page->{location}) to ($page->{location})\n",
                         "$@\n";
            $error++;
        }
        else {
            $static_page->remove_object_from_index if ( $is_old_indexed );
            printf "%4d OK: %40s --> %s\n", $total, $static_page->{location}, $page->{location};
            $success++;
        }
    }

    my $end_time = time;
    my $elapsed  = $end_time - $start_time;
    print "\nDone\n\n";
    print "Total:     $total pages\n",
          "Skipped:   $skipped pages\n",
          "Converted: $success pages\n",
          "Error:     $error pages (see static_page2base_page.err)\n",
          "Started:   ", scalar localtime( $start_time ), "\n",
          "Ended:     ", scalar localtime( $end_time ), "\n",
          "Elapsed: ", sprintf( "%5.1f minutes\n", $elapsed / 60 ),
          "          ", sprintf( "%5.2f seconds/page\n", $elapsed / ( $total - $skipped ) );
}

__END__

=pod

=head1 NAME

static_page2base_page.pl - Migrate pages from the 'static_page' to the 'base_page' package

=head1 SYNOPSIS

 $ export OIWEBSITE=/home/httpd/mysite
 $ perl static_page2base_page.pl --add_html --as_file

=head1 DESCRIPTION

As of OpenInteract 1.35, all functionality for serving 'static' pages
and files is now in the 'base_page' package. (Previously it was in the
'static_page' package.)

The new package has been entirely rewritten and uses a different table
structure, so you will need to migrate the pages from the structure of
the earlier package to this one. This script should do the job.

=head1 OPTIONS

B<--add_html>

If pass we will add C<.html> to any file that does not have an
extension.

B<--as_file>

B<--as_database>

These two options are mutually exclusive. They determine where the
page content will be stored. If not specified, we use B<--as_file>.

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 SEE ALSO

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
