package OpenInteract::SiteTemplate;

# $Id: SiteTemplate.pm,v 1.7 2002/01/02 02:43:55 lachoy Exp $

use strict;
use Data::Dumper qw( Dumper );

$OpenInteract::SiteTemplate::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

my $NAME_SEPARATOR = '::';

########################################
# CLASS METHODS
########################################

# Parse a combined 'package::name' label into the package and
# name. Yes, it's a simple format but we might need to change it in
# the future, so why not create a method

sub parse_name {
    my ( $class, $full_name ) = @_;
    return split /$NAME_SEPARATOR/, $full_name, 2;
}


# Create a combined 'package::name' label

sub create_name {
    my ( $item, $package, $name ) = @_;
    if ( ref $item ) {
        $package = $item->{package};
        $name    = $item->{name};
    }
    return join $NAME_SEPARATOR, $package, $name;
}

# Retrieve a template by name. Note that $name can be a simple name or
# the combined 'package::name'. We might also *require* that the name
# is combined or that $package is passed in.

sub fetch_by_name {
    my ( $class, $name, $package, $p ) = @_;
    $name    = lc $name;
    $package = lc $package;
    $p     ||= {};
    
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "Trying to retrieve template for name ($name) and ",
                               "package ($package)" );

    my ( $cmb_package, $cmb_name ) = $class->parse_name( $name );
    if ( $cmb_package and $cmb_name ) {
        $package = $cmb_package;
        $name    = $cmb_name;
    }

    my $where = 'name = ?';
    my @value = ( $name );
    if ( $package ) {
        $where   .= ' AND package = ?';
        push @value, $package;
    }
    my $obj = eval { $class->fetch_group({ %{ $p }, where => $where, value => \@value }) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $OpenInteract::Error::user_msg = "Cannot retrieve site template for name " .
                                         "<<$name>> and package <<$package>>";
        die $OpenInteract::Error::user_msg;
    }
    if ( scalar @{ $obj } > 1 ) {
        $R->scrib( 0, "Too many items found in response to name <<$name>> ",
                      "and package <<$package>>! Returning ",
                      "$obj->[0]->{name} - $obj->[0]->{package}" );
    }
    $R->DEBUG && $R->scrib( 2, "Template object(s) found:\n", Dumper( $obj ) );
    return $obj->[0];
}


########################################
# RULESET METHODS
########################################

sub ruleset_add {
    my ( $class, $rs_table ) = @_;
    push @{ $rs_table->{pre_save_action} }, \&calculate_update_time;
    return __PACKAGE__;
}


sub calculate_update_time {
    my ( $self, $p ) = @_;
    $self->{last_update} = time;
    return 1;
}

1;

__END__

=pod

=head1 NAME

OpenInteract::SiteTemplate - Object to represent templates and accompanying info

=head1 SYNOPSIS

 my $R = OpenInteract::Request->instance;

 # Retreive a single template based on name

 my $tmpl = eval { $R->site_template->fetch_by_name( 
                            'user_info_box', 
                            'base_box' 
                   ) };
 die "Cannot retrieve box: $@" if ( $@ );
 print "Template contents: $tmpl->{template}\n";

 # Retrieve multiple templates from a package

 my $tmpl_list = eval { $R->site_template->fetch_group({
                            package => 'base_box',
                        }) };
 die "Cannot retrieve templates from base_box: $@" if ( $@ );
 foreach my $tmpl ( @{ $tmpl_list } ) {
    print "Template contents: $tmpl->{template}\n";
 }

 # Parse the common 'package::name' format

 my $full_name = 'base_box::main_box_shell';
 my ( $tmpl_package, $tmpl_name ) = $R->site_template->parse_name( $full_name );

=head1 DESCRIPTION

SiteTemplate objects are used throughout OpenInteract -- in fact, on
every single request multiple objects will be retrieved from the
database.

Each object represents a template which can be interpreted by the
template processing engine (normally L<Template Toolkit>) and replaced
with information from the OpenInteract environment along with data
that you decide to display on a page.

However, most of the time you will not deal directly with template
objects. The core OpenInteract modules L<OpenInteract::Template> and
L<OpenInteract::Template::Toolkit> will retrieve templates for you
based on the name and package specified.

=head1 METHODS

B<parse_name( $full_template_name )>

Parse a full template name (in 'package::name' format) into the
package and name comprising it.

Returns a two-item list: C<( $package, $name )>.

B<fetch_by_name( template_name, [ package_name ])>

Retrieves template(s) by name and/or package. (Just from the database
for now, not from the filesystem.)

Parameters:

=over 4

=item *

B<template_name> ($)

Name of the template to retrieve. Note that this can be the combined
'package::name' format as well.

B<package_name> ($) (optional, see note)

Name of the package to use for retrieving the template -- both the
'template_name' and 'package_name' are required to uniquely identify a
template. If you use the 'package::name' format for 'template_name'
then this is optional, otherwise it is required.

=head1 TO DO

B<Use Template::Toolkit-specifics>

Whatever is in here should be replaced by a L<Template::Provider>
class that will eventually be written.

=head1 BUGS

None known.

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
