package OpenInteract::Handler::Module;

# $Id: Module.pm,v 1.3 2001/05/30 03:13:03 lachoy Exp $

use strict;
use SPOPS::Secure  qw( :all );

@OpenInteract::Handler::Module::ISA     = qw( OpenInteract::Handler::GenericDispatcher SPOPS::Secure );
$OpenInteract::Handler::Module::VERSION = sprintf("%d.%02d", q$Revision: 1.3 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Module::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Module::default_method    = 'listing';
@OpenInteract::Handler::Module::forbidden_methods = ();
%OpenInteract::Handler::Module::security          = ( 
 listing       => SEC_LEVEL_WRITE,  
 security_show => SEC_LEVEL_WRITE, 
 security_edit => SEC_LEVEL_WRITE,
);

use constant DEBUG => 0;

my %PERM = (
  SEC_LEVEL_NONE()  => SEC_LEVEL_NONE_VERBOSE,
  SEC_LEVEL_READ()  => SEC_LEVEL_READ_VERBOSE,
  SEC_LEVEL_WRITE() => SEC_LEVEL_WRITE_VERBOSE,
);

my %SCOPE = (
  SEC_SCOPE_WORLD() => 'World',
  SEC_SCOPE_GROUP() => 'Group',
  SEC_SCOPE_USER()  => 'User',
);

##########
# NOTE
#
#  We will be getting rid of this module, so don't make any
#  improvements to it. The other handler (Security.pm) will be able to
#  do everything this does, and do it in a consistent manner.
#
##########

sub global_security_object_class { return OpenInteract::Request->security; }

# Just list all the current modules
sub listing {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;

  my $params = { main_script => '/Module', error_msg => $p->{error_msg} };

  # Get all the modules and put the relevant into into a list
  my $module_info = $R->CONFIG->{action};
  my %use = ();
  foreach my $module ( keys %{ $module_info } ) {
    next if ( ! $module );                   # skip the module used for no path
    next if ( $module eq 'default' );        # skip the default information
    next if ( $module =~ /^_/ );             # skip 'meta' modules
    next if ( $module_info->{ $module }->{security} eq 'no' ); # skip modules not doing security 
   
    $use{ $module } = $module_info->{ $module }->{class};
  }
  $params->{module_list} = [ map { { module => $_, class => $use{ $_ } } } sort keys %use ];
  
  $R->{page}->{title} = 'Listing of Modules';
  return $R->template->handler( {}, $params, { db => 'module_list',
                                               package => 'base_security' } );
}

# Display security settings for a particular module
sub security_show {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;

  my $module       = $R->apache->param( 'module' );
  unless ( $module ) {
    my $user_msg = 'Required parameter missing, cannot continue with specific security display.';
    $R->throw( { code => 313, type => 'module',
                 user_msg => $user_msg,
                 system_msg => 'Cannot display security for a module when no module given.' } );
    return $class->listing( { error_msg => $user_msg } );
  }

  # Scope defaults to GROUP if not specified

  my $scope = $R->apache->param( 'scope' ) || SEC_SCOPE_GROUP;

  # Set initial parameters

  my $params = { 
    main_script => '/Module', error_msg => $p->{error_msg},
    module_name => $module, scope => $scope, 
    module_class => $R->CONFIG->{action}->{ $module }->{class},
    scope_desc => \%SCOPE, level_desc => \%PERM,
    sec_scope => { user => SEC_SCOPE_USER, group => SEC_SCOPE_GROUP,
                   world => SEC_SCOPE_WORLD } 
  };
  $R->DEBUG && $R->scrib( 1, "Finding security information for $params->{module_class} in scope $scope" );

  my $sec_obj_class = $R->security;
  my ( @assigned_list, @unassigned_list );

  # For WORLD, just display the world setting 

  if ( $scope eq SEC_SCOPE_WORLD ) {
    my $sec = eval { $sec_obj_class->fetch_match( $params->{module_class}, 
                                                  { scope => SEC_SCOPE_WORLD } ) };
    if ( $@ ) {
      OpenInteract::Error->set( SPOPS::Error->get );
      $R->throw( { code => 308 } );
    }
    $sec ||= { level => SEC_LEVEL_NONE };
    push @assigned_list, $sec->{level};
  }

  # For both USER and GROUP, we develop a list of
  # the items that are not yet assigned a security
  # level and a list of those that are currently assigned
  # a level. 

  else { 
    my $where = 'class = ? AND object_id = ? AND scope = ?';
    my $value = [ $params->{module_class}, '0', $scope ];
    my $sec_list  = eval { $sec_obj_class->fetch_group({ where => $where,
                                                         value => $value }) };
    if ( $@ ) {
      OpenInteract::Error->set( SPOPS::Error->get );
      $R->throw( { code => 403 } );
    }
    my $obj_class  = ( $scope eq SEC_SCOPE_USER ) ? $R->user : $R->group;
    my $name_field = ( $scope eq SEC_SCOPE_USER ) ? 'login_name' : 'name';
    foreach my $sec ( @{ $sec_list } ) {
      my $obj = eval { $obj_class->fetch( $sec->{scope_id} ); };
      if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 404 } );
        warn "Found error on fetching $obj_class $sec->{scope_id}: $@\n";
      }
      next if ( ! $obj );
     $R->DEBUG && $R->scrib( 1, "Assigned: $obj->{name_field} (", $obj->id, ") with $sec->{level} from security object $sec->{sid}" );
      push @assigned_list, { id => $obj->id, name => $obj->{ $name_field },
                             perm => $sec->{level} };
    }
    my $unassigned_items = eval { $obj_class->fetch_group( { order => $name_field } ) } || [];
    my %all_unassigned = map { $_->id => $_->{ $name_field } }
                             @{ $unassigned_items };
    foreach ( @assigned_list ) { delete $all_unassigned{ $_->{id} } }
    delete $all_unassigned{1}; # remove superuser/admin from the list
    @unassigned_list = map { { id => $_, name => $all_unassigned{ $_ } } }
                           sort { $all_unassigned{ $a } cmp $all_unassigned{ $b } } 
                           keys %all_unassigned;
  }

  $params->{level} = { map { $PERM{ $_ } => $_ } keys %PERM };
  $params->{scope_list} = [ map { { scope => $_, name => $SCOPE{ $_ } } } keys %SCOPE ];
  $params->{unassigned_list} = \@unassigned_list;
  $params->{assigned_list}   = \@assigned_list;
  $R->{page}->{title} = 'Editing Module Security';
  return $R->template->handler( {}, $params, { db => 'module_security',
                                               package => 'base_security' } );
}

sub security_edit {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  $R->{page}->{return_url} = '/Module/';

  # Find the module and the scope of security we are changing;
  # if there is no module or scope, bail

  my $scope       = $R->apache->param( 'scope' );
  unless ( $scope ) {
    my $user_msg = 'Cannot modify security for a module with no scope specified!';
    $R->throw( { code => 312, type => 'module',
                 user_msg => $user_msg,
                 system_msg => 'No scope passed into the Module/security_edit handler' } );
    return $class->listing( { error_msg => $user_msg . ' No changes made.' } );
  }

  my $module  = $R->apache->param( 'module' );
  unless ( $module ) {
    my $user_msg = 'Cannot modify security for a module when no module specified!';
    $R->throw( { code => 312, type => 'module',
                 user_msg => $user_msg,
                 system_msg => 'No module name passed into the Module/security_edit handler' } );
    return $class->listing( { error_msg => $user_msg . ' No changes made.' } );
  }
  my $module_info   = $R->CONFIG->{action}->{ $module };
  my $module_class  = $module_info->{class};
  my $sec_obj_class = $R->security;
  $R->DEBUG && $R->scrib( 1, "Setting scope <$scope> on class $module_class" );

  # If this is a WORLD scope, just get the level and modify 

  if ( $scope eq SEC_SCOPE_WORLD ) {
    my $level = $R->apache->param( 'level' );
    eval { $class->set_item_security({ class     => $module_class, 
                                       object_id => '0',
                                       scope     => SEC_SCOPE_WORLD, 
                                       level     => $level }) };
    if ( $@ ) {
      warn ">> Error trying to set WORLD security: $@";
      OpenInteract::Error->set( SPOPS::Error->get );
      $OpenInteract::Error::user_msg = "Cannot set security for module in scope WORLD.";
      $R->throw( { code => 406 } );
    }
  }

  # If this is a USER or GROUP scope, scroll through the items that
  # were assigned and setup a hashref of id => permissions mapping;
  # send those items to be added, and if those were done ok
  # remove the difference

  else {

    # First retrieve the existing items.

    my $where = 'class = ? AND object_id = ? AND scope = ?';
    my $value = [ $module_class, '0', $scope ];
    my $existing_list = eval { $sec_obj_class->fetch_group({ where => $where,
                                                             value => $value }) };
    if ( $@ ) {
      OpenInteract::Error->set( SPOPS::Error->get );
      $R->throw( { code => 403 } );
    }
    my %existing_by_id = map { $_->{scope_id} => $_->{level} } @{ $existing_list };
    delete $existing_by_id{1}; # get rid of superuser/admin

   # Now create the hashref of new items

    my ( %new_perm );
    my @perm = split /\|/, $R->apache->param( 'assigned_tally' );
    $R->DEBUG && $R->scrib( 1, scalar @perm, " items in assigned." );
    foreach my $info ( @perm ) {
      next if ( ! $info );
      $R->DEBUG && $R->scrib( 1, "Processing <<$info>> from form." );
      my ( $id, $perm, $name ) = split ';', $info;
      $new_perm{ $id } = $perm;
    }
    eval { $class->set_security({ class     => $module_class, 
                                  object_id => '0',
                                  scope     => $scope, 
                                  level     => \%new_perm } ); };
    if ( $@ ) {
      OpenInteract::Error->set( SPOPS::Error->get );
      $OpenInteract::Error::user_msg = "Cannot set security for module in scope $SCOPE{ $scope }.";
      $R->throw( { code => 406 } );
   }
    foreach ( keys %new_perm ) { delete $existing_by_id{ $_ }; }
    
    # everything left in %existing_by_id needs to be removed
    foreach my $id ( keys %existing_by_id ) {
      $R->DEBUG && $R->scrib( 1, "Try to remove $module_class scope $scope ($id)" );
      eval { $class->remove_item_security({ class     => $module_class, 
                                            object_id => '0', 
                                            scope     => $scope, 
                                            scope_id  => $id } ); };
      if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $OpenInteract::Error::user_msg = "Cannot remove security for module in scope $SCOPE{ $scope }.";
        $R->throw( { code => 406 } );
      }
    }
    
  }
  return $class->listing( $p );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Module - Edit module information

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head1 NOTES

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
