//
// (C) Copyright 2011-2012 Sergey A. Babkin.
// This file is a part of Triceps.
// See the file COPYRIGHT for the copyright notice and license information
//
// Helpers to call Perl code back from C++.

#include <typeinfo>
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "ppport.h"

#include "TricepsPerl.h"
#include "PerlCallback.h"

// ###################################################################################

using namespace TRICEPS_NS;

namespace TRICEPS_NS
{
namespace TricepsPerl 
{

///////////////////////// PerlCallback ///////////////////////////////////////////////

PerlCallback::PerlCallback() :
	code_(NULL)
{ }

PerlCallback::~PerlCallback()
{
	clear();
}

void PerlCallback::clear()
{
	if (code_) {
		SvREFCNT_dec(code_);
		code_ = NULL;
	}
	if (!args_.empty()) {
		for (size_t i = 0; i < args_.size(); i++) {
			SvREFCNT_dec(args_[i]);
		}
		args_.clear();
	}
}

bool PerlCallback::setCode(SV *code, const char *fname)
{
	clear();

	if (!SvROK(code) || SvTYPE(SvRV(code)) != SVt_PVCV) {
		setErrMsg( string(fname) + ": code must be a reference to Perl function" );
		return false;
	}

	code_ = newSV(0);
	sv_setsv(code_, code);
	return true;
}

// Append another argument to args_.
// @param arg - argument value to append; will make a copy of it.
void PerlCallback::appendArg(SV *arg)
{
	SV *argcp = newSV(0);
	sv_setsv(argcp, arg);
	args_.push_back(argcp);
}

bool PerlCallback::equals(const PerlCallback *other) const
{
	if (args_.size() != other->args_.size())
		return false;
	if ((code_ == NULL) ^ (other->code_ == NULL))
		return false;

	if (code_ != NULL && SvIV(code_) != SvIV(other->code_)) // same reference
		return false;

	dSP;

	for (size_t i = 0; i < args_.size(); ++i) {
		int nv;
		int result;
		bool error = false;
		SV *a1 = args_[i];
		SV *a2 = other->args_[i];

		ENTER; SAVETMPS; 

		PUSHMARK(SP);
		XPUSHs(a1);
		XPUSHs(a2);
		PUTBACK; 

		const char *func = ((SvIOK(a1) || SvNOK(a1)) && (SvIOK(a2) || SvNOK(a2))) ? "Triceps::_compareNumber" :  "Triceps::_compareText" ;
		nv = call_pv(func, G_SCALAR|G_EVAL);

		if (SvTRUE(ERRSV)) {
			warn("Internal error in function %s: %s", func, SvPV_nolen(ERRSV));
			error = true;
		}

		SPAGAIN;
		if (nv < 1) { 
			result = 1; // doesn't match
		} else {
			for (; nv > 1; nv--)
				POPs;
			SV *perlres = POPs;
			result = SvTRUE(perlres);
		}
		PUTBACK; 

		FREETMPS; LEAVE;

		if (error || result) // if equal, the comparison will be 0
			return false;
	}
	
	return true;
}

bool callbackEquals(const PerlCallback *p1, const PerlCallback *p2)
{
	if (p1 == NULL || p2 == NULL) {
		return p1 == p2;
	} else {
		return p1->equals(p2);
	}
}

void callbackSuccessOrThrow(const char *fmt, ...)
{
	if (SvTRUE(ERRSV)) {
		clearErrMsg(); // in case if it was thrown by Triceps, clean up
		// propagate to the caller
		Erref err = new Errors(SvPV_nolen(ERRSV));

		va_list ap;
		va_start(ap, fmt);
		string s = vstrprintf(fmt, ap);
		va_end(ap);
		err->appendMsg(true, s);

		throw TRICEPS_NS::Exception(err, false);
	}
}

///////////////////////// PerlLabel ///////////////////////////////////////////////

PerlLabel::PerlLabel(Unit *unit, const_Onceref<RowType> rtype, const string &name, 
		Onceref<PerlCallback> clr, Onceref<PerlCallback> cb) :
	Label(unit, rtype, name),
	clear_(clr),
	cb_(cb)
{ }

PerlLabel::~PerlLabel()
{ }

Onceref<PerlLabel> PerlLabel::makeSimple(Unit *unit, const_Onceref<RowType> rtype,
	const string &name, SV *code, const char *fmt, ...)
{
	Onceref<PerlCallback> clr = new PerlCallback();
	if (!clr->setCode(get_sv("Triceps::_DEFAULT_CLEAR_LABEL", 0), "")) {
		// should really never fail, but just in case
		va_list ap;
		va_start(ap, fmt);
		string s = vstrprintf(fmt, ap);
		va_end(ap);
		throw Exception(strprintf("%s: internal error, bad value in $Triceps::_DEFAULT_CLEAR_LABEL", s.c_str()), false);
	}

	Onceref<PerlCallback> cb = new PerlCallback();
	if (!cb->setCode(code, "")) {
		// should really never fail, but just in case
		va_list ap;
		va_start(ap, fmt);
		string s = vstrprintf(fmt, ap);
		va_end(ap);
		const char *errtxt = ": unknown error in creating a Perl label";
		SV *errmsg = get_sv("!", 0);
		if (SvPOK(errmsg))
			errtxt = SvPV_nolen(errmsg);
		throw Exception(strprintf("%s%s", s.c_str(), errtxt), false);
	}
	return new PerlLabel(unit, rtype, name, clr, cb);
}

void PerlLabel::execute(Rowop *arg) const
{
	dSP;

	if (cb_.isNull()) {
		warn("Error in label %s handler: attempted to call the label that has been cleared", getName().c_str());
		return;
	}

	WrapRowop *wrop = new WrapRowop(arg);
	SV *svrop = newSV(0);
	sv_setref_pv(svrop, "Triceps::Rowop", (void *)wrop);

	WrapLabel *wlab = new WrapLabel(const_cast<PerlLabel *>(this));
	SV *svlab = newSV(0);
	sv_setref_pv(svlab, "Triceps::Label", (void *)wlab);

	PerlCallbackStartCall(cb_);

	XPUSHs(svlab);
	XPUSHs(svrop);

	PerlCallbackDoCall(cb_);

	// this calls the DELETE methods on wrappers
	SvREFCNT_dec(svrop);
	SvREFCNT_dec(svlab);

	callbackSuccessOrThrow("Detected in the unit '%s' label '%s' execution handler.", getUnitName().c_str(), getName().c_str());
}

void PerlLabel::clearSubclass()
{
	dSP;

	cb_ = NULL; // drop the execution callback

	if (clear_.isNull()) 
		return; // nothing to do
	
	WrapLabel *wlab = new WrapLabel(const_cast<PerlLabel *>(this));
	SV *svlab = newSV(0);
	sv_setref_pv(svlab, "Triceps::Label", (void *)wlab);

	PerlCallbackStartCall(clear_);

	XPUSHs(svlab);

	PerlCallbackDoCall(clear_);

	// this calls the DELETE methods on wrappers
	SvREFCNT_dec(svlab);

	clear_ = NULL; // eventually drop the callback, before any chance of throwing!

	callbackSuccessOrThrow("Detected in the unit '%s' label '%s' clearing handler.", getUnitName().c_str(), getName().c_str());
}

///////////////////////// UnitTracerPerl ///////////////////////////////////////////////

UnitTracerPerl::UnitTracerPerl(Onceref<PerlCallback> cb) :
	cb_(cb)
{ }

void UnitTracerPerl::execute(Unit *unit, const Label *label, const Label *fromLabel, Rowop *rop, Unit::TracerWhen when)
{
	dSP;

	if (cb_.isNull()) {
		warn("Error in unit %s tracer: attempted to call the tracer that has been cleared", 
			unit->getName().c_str());
		return;
	}

	SV *svunit = newSV(0);
	sv_setref_pv(svunit, "Triceps::Unit", (void *)(new WrapUnit(unit)));

	SV *svlab = newSV(0);
	sv_setref_pv(svlab, "Triceps::Label", (void *)(new WrapLabel(const_cast<Label *>(label))));

	SV *svfrlab = newSV(0);
	if (fromLabel != NULL)
		sv_setref_pv(svfrlab, "Triceps::Label", (void *)(new WrapLabel(const_cast<Label *>(fromLabel))));

	SV *svrop = newSV(0);
	sv_setref_pv(svrop, "Triceps::Rowop", (void *)(new WrapRowop(rop)));

	SV *svwhen = newSViv(when);

	PerlCallbackStartCall(cb_);

	XPUSHs(svunit);
	XPUSHs(svlab);
	XPUSHs(svfrlab);
	XPUSHs(svrop);
	XPUSHs(svwhen);

	PerlCallbackDoCall(cb_);

	// this calls the DELETE methods on wrappers
	SvREFCNT_dec(svunit);
	SvREFCNT_dec(svlab);
	SvREFCNT_dec(svfrlab);
	SvREFCNT_dec(svrop);
	SvREFCNT_dec(svwhen);

	if (SvTRUE(ERRSV)) {
		// If in eval, croak may cause issues by doing longjmp(), so better just warn.
		// Would exit(1) be better?
		warn("Error in unit %s tracer: %s", 
			unit->getName().c_str(), SvPV_nolen(ERRSV));

	}
}

Onceref<PerlCallback> GetSvCall(SV *svptr, const char *fmt, ...)
{
	Autoref<PerlCallback> cb = new PerlCallback();

	if (SvROK(svptr)) {
		if (SvTYPE(SvRV(svptr)) == SVt_PVAV) {
			AV *array = (AV*)SvRV(svptr);
			int len = av_len(array)+1; // av_len returns the index of last element
			if (len > 0) {
				SV *code = *av_fetch(array, 0, 0);
				if (SvROK(code) && SvTYPE(SvRV(code)) == SVt_PVCV) {
					cb->setCode(code, ""); // can't fail
					for (int i = 1; i < len; i++) { // pick up the args
						cb->appendArg(*av_fetch(array, i, 0));
					}
					return cb;
				}
			}
		} else if (SvTYPE(SvRV(svptr)) == SVt_PVCV) {
			cb->setCode(svptr, ""); // can't fail
			return cb;
		}
	}

	va_list ap;
	va_start(ap, fmt);
	string s = vstrprintf(fmt, ap);
	va_end(ap);
	throw TRICEPS_NS::Exception::f("%s value must be a reference to a function or an array starting with a reference to function", s.c_str());
}

}; // Triceps::TricepsPerl
}; // Triceps


