package Evo::Class;
use Evo;
use Module::Load;
use Module::Loaded;
use Evo::Util 'monkey_patch';

sub base      {'Evo::Class::Base'}
sub init_base { load shift->base }

sub attr_simple {
  my ($class, $target, $attrs) = @_;
  foreach my $attr (@$attrs) {
    monkey_patch($target, $attr,
      sub { return $_[0]{$attr} if @_ == 1; $_[0]{$attr} = $_[1]; $_[0] });
  }
}

sub attr_default {
  my ($class, $target, $attrs, $value) = @_;
  foreach my $attr (@$attrs) {

    monkey_patch $target, $attr, sub {
      return exists $_[0]{$attr} ? $_[0]{$attr} : ($_[0]{$attr} = $value)
        if @_ == 1;
      $_[0]{$attr} = $_[1];
      $_[0];
    };

  }
}

sub attr_lazy {
  my ($class, $target, $attrs, $fn) = @_;
  Carp::croak('Default has to be a code reference or constant value')
    if ref $fn ne 'CODE';

  for my $attr (@$attrs) {

    monkey_patch $target, $attr, sub {
      return exists $_[0]{$attr} ? $_[0]{$attr} : ($_[0]{$attr} = $fn->($_[0]))
        if @_ == 1;
      $_[0]{$attr} = $_[1];
      $_[0];
    };

  }
}

sub attr {
  my ($class, $target, $attrs, $value) = @_;
  Carp::croak "attribute should have a name at least" unless $attrs;
  $attrs = ref $attrs eq 'ARRAY' ? $attrs : [$attrs];
  do { Carp::croak(qq{Attribute "$_" invalid}) unless $_ =~ /^[a-zA-Z_]\w*$/ }
    for @$attrs;

  return $class->attr_simple($target, $attrs) unless defined $value;
  ref $value
    ? $class->attr_lazy($target, $attrs, $value)
    : $class->attr_default($target, $attrs, $value);
}

sub init_class {
  my ($class, $target, @bases) = @_;
  $class->init_base;

  mark_as_loaded($target) unless is_loaded($target);
  monkey_patch $target, 'has', sub(*@) { $class->attr($target, @_) };
  monkey_patch $target, 'extends', sub { $class->isa($target, @_) };

  return $class->isa($target, @bases) if @bases;

  no strict 'refs';    ## no critic
  my $isa = \@{"${target}::ISA"};

  # has bases, nothing to do
  return if @$isa;
  push @$isa, $class->base;
}

sub isa {
  my ($class, $target, @bases) = @_;
  no strict 'refs';    ## no critic
  my $isa = \@{"${target}::ISA"};

  # delete last :: and push it to isa
  foreach my $base (map { /(.+)::+$/ ? $1 : $_ } @bases) {
    load $base unless is_loaded($base);
    push @$isa, $base;
  }
}

1;

# ABSTRACT: Evo object system

__END__

=pod

=encoding UTF-8

=head1 NAME

Evo::Class - Evo object system

=head1 VERSION

version 0.0171

=head1 SYNOPSIS

  package main;
  use Evo;
  use Test::More;

  {

    package My::Obj;
    use Evo -class;

    # simple attr, quotes can be omitted;
    # has 'foo';
    has foo;

    # attr with default value
    has bar => 'bar';

    # couple of attrs
    has ['baz', 'name'] => 'baz';

  }

  my $obj = My::Obj->new;
  say $obj->bar;

  $obj = My::Obj->new(foo => 'ok', bar => 'ok');
  $obj = My::Obj->new({foo => 'ok', bar => 'ok'});
  say $obj->bar;

  # chaining
  $obj->foo('1')->bar('2');
  say $obj->bar;

  {

    # inherit in compile-time
    package My::Obj::Son;
    use Evo 'My::Obj';
    has name => 'Alex';

    # inherit in run-time, like Moose extends does
    package My::Obj::Daughter;
    use Evo -class;
    extends 'My::Obj';
    has name => 'Alisa';
  }

  my $son      = My::Obj::Son->new;
  my $daughter = My::Obj::Daughter->new;

  say $son->bar;
  say $son->name;
  say $daughter->bar;
  say $daughter->name;

=head1 DESCRIPTION

This class provides OO functionality for you modules. This module shouldn't be
used directly

=head1 USAGE

To transform package to the class, use on of the following code

  # first compile-time form
  use Evo 'Parent::Class';
  use Evo 'Parrent::';

  # second run-time form
  use Evo -class;
  extends 'Parent';

That will install in you package neccessary methods. If you inherit from the
top level package in the first compile-time form, you should end it with '::'
to it to tell L<Evo> that you want enable OO features instead of trying import
a probably missing function C<Parent>

Also you package will be marked with L<Evo/"-loaded"> to make it possible
using inline classes, and parent packages will be loaded (if aren't already)
and also marked as L<Evo/"-loaded">

Multiple inheritance is also supported

=head2 extends

Extends you class with parents (fill C<@ISA>)

=head2 new

Creates a new object

=head2 has

Setup attribute

=head2 METHODS

=head3 attr

Install attribute to the specified package. Probably shouln't be used directly, use generated L</"has"> instead

  require Evo::Class;
  Evo::Class->attr('My::Obj', domain => 'alexbyk.com');

  my $foo = My::Obj->new;
  say $foo->domain;

=head2 Internal methods

These methods shouldn't be used directly and can be changed in the future

=head3 base
=head3 init_base
=head3 init_class
=head3 attr_simple
=head3 attr_default
=head3 attr_lazy
=head3 isa

=head1 AUTHOR

alexbyk.com

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
