package Evo::Mdn;
use Evo -class;
use Hash::Util::FieldHash qw(register id id_2obj fieldhashes fieldhash);
use Evo::Guard;

has senders => sub { +{} };
has guards  => sub { +{} };
has stores  => sub { +{} };

sub new {
  my $self = shift->SUPER::new(@_);
  fieldhashes($self->senders, $self->stores, $self->guards);
  $self;
}

sub broadcast {
  my ($self, $sender, $msg) = @_;
  my $hash = $self->senders->{$sender};
  $hash->{$_}->($self, id_2obj($_), $msg, $sender) for keys %$hash;
}

# don't try to register subscriptions.
# hashes works fast enought and in most
# keyses shoud work much faster because you don't need
# to register a second entity

sub _when_message_cb {
  my ($self, $me) = (shift, shift);
  my $guard = Evo::dep_new('Evo::Guard',
    sub { $_[0] and $self->unsubscribe_from_all($me) });
  $me->when_message(@_);
}

sub subscribe {
  my ($self, $me, $sndr) = (shift, shift, shift);
  my ($store_me, $cb) = @_ ? @_ : (1, undef);

  unless ($cb) {
    Carp::croak "implement when_message in $me"
      unless $me->can('when_message');
    $cb = \&_when_message_cb;
  }
  my ($stores, $senders, $guards)
    = ($self->stores, $self->senders, $self->guards);

  my $hash = $senders->{$sndr} //= {};
  Hash::Util::FieldHash::fieldhash(%$hash);
  Carp::croak "$me has been subscribed to $sndr before" if $hash->{$me};

  # protect from closures cicle refs
  $guards->{$sndr} = Evo::dep_new('Evo::Guard', sub { %$hash = () })
    unless ($guards->{$sndr});

  $hash->{$me} = $cb;
  $stores->{$sndr}->{id $me} = $me if $store_me;
}

sub unsubscribe {
  my ($self, $me, $sndr) = @_;
  delete $self->senders->{$sndr}->{$me};
  delete $self->stores->{$sndr}->{id $me};
}

sub unsubscribe_from_all {
  my ($self, $me) = @_;
  $self->unsubscribe($me, $_) for keys %{$self->senders};
}

sub delete_all_subscribers {
  my ($self, $sender) = @_;
  %{$self->senders->{$sender}} = ();
}

sub _clear_all {
  my $self = shift;
  %{$self->stores} = ();
  $self->delete_all_subscribers($_) for keys %{$self->senders};
  $self;
}

sub DESTROY { shift->_clear_all->SUPER::DESTROY; }

1;

# ABSTRACT: A message delivery network

__END__

=pod

=encoding UTF-8

=head1 NAME

Evo::Mdn - A message delivery network

=head1 VERSION

version 0.0170

=head1 SYNOPSIS

  use Evo::Base -strict;
  use Evo::MDN;
  my $mdn = Evo::MDN->new;

  my $sender = Evo::Base->new;
  do {
    my $foo  = Evo::Base->new;
    my $file = IO::File->new;
    $mdn->subscribe($foo,  $sender, 1, sub($me, $msg, @) { say "$me got $msg" });
    $mdn->subscribe($file, $sender, 0, sub($me, $msg, @) { say "$me got $msg" });
    $mdn->broadcast($sender, "hello");
  };

  # only Foo is alive
  $mdn->broadcast($sender, "alive");

=head1 DESCRIPTION

Message delivery network. Allows to send messages from one object to another.
It do the right things in most cases.

The benefit to use it that almost any object can send and any object can
receive messages without modification. So you can use your existing code
or write new modules and build your app using independent components that
communicate.

You can expect about 500K-1M messages/s perfomance per 1 process/processor core

=head1 METHODS

=head2 broadcast

Sends a message to all subscribers. when_message will be invoked with
the sender and message for the subscribers

=head2 subscribe

Subscribe one object to another
If the third passed arbument is true, stores an object and prevent
it from destruction while sender exists. Default values are

  $mdn->subscribe($foo, $sender, 1, sub { shift->when_message(@_) });

But if callback is not provided, an object will be checked that method
C<when_message> exists in object's class, or an exception will be thrown

=head2 unsubscribe

Unsubscribe one object from another and deletet itself, if it was stored

=head2 unsubscribe_from_all

Unsubscribe me from all senders

=head1 AUTHOR

alexbyk.com

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
