package EixoUtils::Duplicity;
use strict;
use Eixo::Base::Clase;
use EixoUtils::Runner;
use IPC::Cmd qw(can_run);

my $DEBUG = 1;

sub debugOn{$DEBUG=1};
sub debugOff{$DEBUG=undef};

#### properties
my $DEFAULT_ARCHIVE_DIR = '/var/cache/duplicity';
my $DEFAULT_BACKEND = 'file';


has(
    encrypt_key =>undef,
    sign_key => undef,
    encrypt_secret_keyring => undef,
    backup_type => undef,
    action => undef,
    archive_dir => $DEFAULT_ARCHIVE_DIR,
    source_path => '/',
    target_url => undef,
    source_url => undef,
    target_folder=> undef,
    volsize => undef,

    backend_config => {

        backend => $DEFAULT_BACKEND,
        user => undef,
        secret => undef,
        container => undef,
    },

    passphrase => undef,
    sign_passphrase => undef,
    tmpdir => undef,

    force => 1 , #default to remove files intead of list
    log_fd => 1, #default to stdout,
    include => undef,
    exclude => undef,
    no_encryption => undef,
    no_compression => undef,
    allow_source_mismatch => undef,
    duplicity_path => can_run('duplicity'),
    gpg_path => can_run('gpg'),


    command_executed => undef,
    status => undef,
    output => undef,
    error => undef,
    die_on_error => 0

);

sub initialize{
    my ($self, %args) = @_;

    $self->SUPER::initialize(%args);
    
    $self->verifyRequirements();

}

sub build_backend_url{
    my ($self, $folder) = @_;

    $folder ||= '';
    
    my $bconfig = $_[0]->backend_config;

    my $url;

    if($bconfig->{backend} =~ /azure|s3|file/){
        
        $url = $bconfig->{backend}.'://'.$bconfig->{container};
        $url .= '/'.$folder if($folder);
    }
    elsif($bconfig->{backend} eq 'rsync') {
        
    }

    return $url;

}



### verifications
# chequear que duplicity esta instalado
# se lle pasan keys 
# - chequear que gpg este instalado
# - chequear que a key exista no gpg keyring
# - chequear que este seteada no entorno a passphrase
# - que se consiga descifrar a key correctamente
# ao facer backup que exista o source

sub verifyRequirements{

    $_[0]->__checkDuplicity();
    $_[0]->__checkGpg();
    $_[0]->__checkGpgKeys();
}

sub __checkDuplicity{
    die('duplicity not found') 
        unless($_[0]->duplicity_path);

}

sub __checkGpg{

    die("GPG not found")
        unless($_[0]->gpg_path);
}

sub __checkGpgKeys{
    ## testear que as keys estan dadas de alta no gpg
}

# 
# chequea si existen os parametros especificados, e estan definidos
# dentro do target
#
sub existsParameters{

    my ($self, $parameters, $target) = @_;
    
    # if no target, search for parameter in $self
    $target ||= $self;

    foreach my $p (@$parameters){

        my @opcionales = split(/\|/, $p);

        unless(grep {exists($target->{$_}) && defined($target->{$_})} @opcionales){
            die("Required parameter '$p' not specified")
        }

    }

}


#### methods

# backup
#
sub fullBackup{
    my ($self, %args) = @_;

    $self->backup(%args, backup_type => 'full')
}

sub incrementalBackup{
    my ($self, %args) = @_;

    $self->backup(%args, backup_type => 'incr')
}

sub backup{
    my ($self,%args) = @_;

    # merge args with object properties
    my %merged_args = (%$self, %args);

    ## validations
    $self->existsParameters(
        [qw(source_path target_url|target_folder)], 
        \%merged_args
    );

    die("Source path '$merged_args{source_path}' not exists in filesystem") unless(-e $merged_args{source_path});

    # 
    # si so nos pasan o target_folder, construimos a url cos parametros do backend
    #
    my $target_url = $merged_args{target_url} ||
        $self->build_backend_url($merged_args{target_folder});


    my @args;

    if($merged_args{backup_type}){
        push @args, $merged_args{backup_type}
    }

    foreach my $inc (@{$merged_args{include}}){
        push @args, '--include', $inc;
    };

    foreach my $exc (@{$merged_args{exclude}}){
        push @args, '--exclude', $exc;
    };

    # 
    # other duplicity options
    #
    foreach my $opt (qw(
        exclude_filelist 
        exclude_if_present
        exclude_older_than
        exclude_regexp
        volsize
    )){
        if(my $val = $merged_args{$opt}){
            my $duplicity_opt = ($opt =~ tr/_/-/r);
            push @args,"--$duplicity_opt", $val; 
        }
    }

    #
    # other duplicity flag options
    #
    foreach my $opt (qw(
        allow_source_mismatch
        copy_links
        exclude_device_files
        exclude_other_filesystems
        
    )){

        if(my $val = $merged_args{$opt}){
            my $duplicity_opt = ($opt =~ tr/_/-/r);
            push @args,"--$duplicity_opt"; 
        }
    }

    # source and destination
    push @args, $merged_args{source_path}, $target_url;

    $self->__execDuplicity(@args);
}


#
# verify compares the latest backup with the current files
#
sub verify{
    my ($self, %args) = @_;

    my %merged_args = (%$self, %args);

    ## validations
    $self->existsParameters(
        [qw(source_url|source_folder target_folder)], 
        \%merged_args
    );

    my $source_url = $merged_args{source_url} || 
        $self->build_backend_url($merged_args{source_folder});

    my @cmd = (
        "verify",
        $source_url,
        $merged_args{target_folder}
    );

    if($merged_args{'compare_data'}){
        push @cmd, '--compare-data';
    }

    if(my $time = $merged_args{time}){
        push @cmd, '--time', $time;
    }

    $self->__execDuplicity(@cmd)

}


# 
# restore a backup
#
sub restore{
    my ($self, %args) = @_;

    my %merged_args = (%$self, %args);

    ## validations
    $self->existsParameters(
        [qw(source_url|source_folder target_folder)], 
        \%merged_args
    );

    my $source_url = $merged_args{source_url} ||
        $self->build_backend_url($merged_args{source_folder});

    my @cmd = (
        "restore",
        $source_url,
        $merged_args{target_folder}
    );

    if(my $file_to_restore = $merged_args{file_to_restore} ){
        push @cmd, '--file-to-restore', $file_to_restore;
    }

    if(my $time = $merged_args{time}){
        push @cmd, '--time', $time;
    }

    $self->__execDuplicity(
        @cmd
    );
}

#
# check collection status
#
sub collectionStatus{
    my ($self, %args) = @_;
    my %merged_args = (%$self, %args);

    ## validations
    $self->existsParameters(
        [qw(source_url|source_folder)], 
        \%merged_args
    );

    my $source_url = $merged_args{source_url} ||
        $self->build_backend_url($merged_args{source_folder});

    my @cmd = (
        "collection-status",
        $source_url
    );

    if(my $file_changed = $merged_args{file_changed}){
        push @cmd, '--file-changed', $file_changed;
    }

    $self->__execDuplicity(@cmd);
}

# 
# list files in a backup
#
sub listCurrentFiles{
    my ($self, %args) = @_;
    my %merged_args = (%$self, %args);

    ## validations
    $self->existsParameters(
        [qw(source_url|source_folder)], 
        \%merged_args
    );

    my $source_url = $merged_args{source_url} ||
        $self->build_backend_url($merged_args{source_folder});

    my @cmd = (
        "list-current-files",
        $source_url
    );

    if(my $time = $merged_args{time}){
        push @cmd, '--time', $time;
    }

    $self->__execDuplicity(@cmd);

}

#
# cleanup archive_dir 
#
sub cleanup{
    
    my ($self,%args) = @_;

    my %merged_args = (%$self, %args);
    
    $self->existsParameters(
        [qw(target_url|target_folder)], 
        \%merged_args
    );

    my $target_url = $merged_args{target_url} ||
        $self->build_backend_url($merged_args{target_folder});


    $self->__execDuplicity(
        'cleanup',
        '--force',
        $target_url
    )
}

# 
# purgue older backups
#
sub removeOlderThan{
    my ($self, %args) = @_;

    my %merged_args = (%$self, %args);
    
    
    ## validations
    $self->existsParameters(
        [qw(target_url|target_folder time)], 
        \%merged_args
    );

    my $target_url = $merged_args{target_url} ||
        $self->build_backend_url($merged_args{target_folder});

    my @cmd = (
        'remove-older-than',
        $merged_args{time},
        $target_url,
    );

    if($merged_args{'force'}){
        push @cmd, '--force';
    }

    $self->__execDuplicity(@cmd);

}


sub __execDuplicity{
    my ($self, @args) = @_;

    my @cmd = ($self->duplicity_path);
    $self->{action} = $args[0];

    # 
    # general duplicity options
    #
    foreach my $opt (qw(
        volsize
        archive_dir
        encrypt_key
        sign_key
        log_fd
    )){
        if(my $val = $self->{$opt}){
            my $duplicity_opt = ($opt =~ tr/_/-/r);
            push @args,"--$duplicity_opt", $val; 
        }
    }


    # 
    # general duplicity flag options
    #
    foreach my $opt (qw(
        dry_run
        no_compression
        no_encryption
        
    )){

        if(my $val = $self->{$opt}){
            my $duplicity_opt = ($opt =~ tr/_/-/r);
            push @cmd,"--$duplicity_opt"; 
        }
    }

    if($DEBUG){
        print "running: ", join(' ', @cmd,@args), "\n";
    }


    my $env = $self->prepareEnv();

    my $r = EixoUtils::Runner->new(
        die_on_error => $self->die_on_error,
        env => $env
    )->run(
        @cmd, @args, '--log-file', '/tmp/aaa'
    );

    $self->command_executed($r->command);

    $self->__processOutput($r);
}


#
# environment vars
#
#FTP_PASSWORD
#PASSPHRASE
#SIGN_PASSPHRASE
#
#Azure Backend
#AZURE_ACCOUNT_NAME
#AZURE_ACCOUNT_KEY
#
#S3 Backend
#------------
#AWS_ACCESS_KEY_ID
#AWS_SECRET_ACCESS_KEY

sub prepareEnv{
    my ($self) = @_;

    my $bconfig = $self->backend_config;
    my %backend_credentials = ();
    
    if($bconfig->{backend} eq 'azure'){

        %backend_credentials = (
            AZURE_ACCOUNT_KEY => $bconfig->{key},
            AZURE_ACCOUNT_NAME => $bconfig->{user}
        )
    
    }
    elsif($bconfig->{backend} eq 's3'){
        %backend_credentials = (
            AWS_SECRET_ACCESS_KEY => $bconfig->{key} ,
            AWS_ACCESS_KEY_ID => $bconfig->{user}
        )
    }
    elsif($bconfig->{backend} eq 'ftp'){
        %backend_credentials = (
            FTP_PASSWORD => $bconfig->{key}
        )
    }

    # 
    # construimos o hash do env
    #
    my $env = {%backend_credentials};

    $env->{PASSPHRASE} = $self->passphrase 
        if($self->passphrase);

    $env->{SIGN_PASSPHRASE} = $self->sign_passphrase
        if($self->sign_passphrase);

    $env->{TMPDIR} = $self->tmpdir
        if($self->tmpdir);

    return $env;

}


sub __processOutput{
    my ($self, $runner) = @_;

    $self->{__runner} = $runner;

    if($DEBUG){
        print 'status:', $runner->status,"\n", "salida:", $runner->stdout,"\n", 'stderr:',$runner->stderr, "\n";
    }

    $self->status($runner->status);
    $self->output($runner->stdout.$runner->stderr);

    if($runner->status != 0){
        $self->error("Error in duplicity ".$self->action. ". Details: ".$self->output);
    }

    return $self;
}

1;

#duplicity cleanup --force --no-print-statistics --encrypt-key $encryptkey --sign-key $encryptkey --full-if-older-than $increments --archive-dir /var/cache/backupninja/duplicity $desturl
#
#duplicity remove-older-than $keep --force  --no-print-statistics --encrypt-key $encryptkey --sign-key $encryptkey --full-if-older-than $increments --extra-clean --archive-dir /var/cache/backupninja/duplicity $desturl
#
#duplicity --no-print-statistics --encrypt-key $encryptkey --sign-key $encryptkey --full-if-older-than $increments --extra-clean --archive-dir /var/cache/backupninja/duplicity  "
#for i in "${include[@]}"
#do
#        instr="$instr --include '$i'"
#done
#instr="$instr --exclude '**' / $desturl"
#
### Lista los ficheiros disponibles: 
#export PASSPHRASE=xxxxx
#duplicity list-current-files rsync://backups@backups.situm.es//media/backup-data/data/pre-old/sql --encrypt-key 41CBC7A8
#
### restaurar ficheiro dun backup
#duplicity -t 12D --file-to-restore var/lib/pgsql/backup-new/alarms.sql.gz  rsync://backups@backups.situm.es//media/backup-data/data/pro-old/sql --encrypt-key 41CBC7A8 /tmp/alarms.sql.gz
#
