package main;

use strict;
use warnings;

use lib qw{ inc };

use Astro::Coord::ECI::Test qw{ :tolerance };
use Astro::Coord::ECI::TLE;
use Astro::Coord::ECI::Utils qw{max};
use Test::More 0.88;

my %body;
my $data;
my $handler = sub {};
my $model;
my $tle;
my @tolerance;

my %action = (
    data	=> sub {
	$data = undef;
	return \&data;
    },
    model	=> sub {
	my ( $new_model ) = @_;
	defined $new_model
	    and $model = lc $new_model;
	return \&model;
    },
    tolerance	=> sub {
	my @args = @_;
	@args < 6
	    and die "Need six tolerances";
	@tolerance = @args;
	return sub {};
    },
);

while ( <DATA> ) {
    m/ \A \s* (?: [#] | \z ) /smx
	and next;
    if ( m/ \A \s* - [[:lower:]]+ \b /smx ) {
	s/ \A \s* - //smx;
	s/ \s+ \z //smx;
	my ( $act, @args ) = split qr{ \s+ }smx;
	$action{$act}
	    or die "Unknown action -$act\n";
	$handler->();
	$handler = $action{$act}->( @args );
    } else {
	$handler->( $_ );
    }
}

$handler->();

done_testing;

sub data {
    my ( $line ) = @_;
    if ( defined $line ) {
	$data .= $line;
    } else {
	my @bodies = Astro::Coord::ECI::TLE->parse( $data );
	foreach my $sat ( @bodies ) {
	    my $oid = format_oid( $sat->get( 'id' ) );
	    $sat->set( id => $oid );
	    $body{$oid} = $sat;
	}
	$tle = $bodies[-1];
    }
    return;
}

{
    my @name;
    my $note;

    BEGIN {
	@name = qw{ X Y Z Xdot Ydot Zdot };
	$note = '';
    }

    sub model {
	my ( $line ) = @_;
	defined $line
	    or return;
	$line =~ s/ \s+ \z //smx;
	$line =~ s/ \A \s+ //smx;

	my ( $deltat, @want ) = split qr{ \s+ }smx, $line;

	if ( @want == 1 && $want[0] eq 'xx' ) {
	    $tle = $body{ format_oid( $deltat ) }
		or die "OID $deltat not seen";
	} else {

	    my $new_note = sprintf 'OID %s model %s',
		$tle->get( 'id' ), $model;
	    if ( $note ne $new_note ) {
		$note = $new_note;
		note $note;
	    }

	    $deltat *= 60;

	    my @got = $tle->$model(
		$tle->get( 'epoch' ) + $deltat
	    )->eci();

	    foreach my $inx ( 0 .. 5 ) {
		my $title = sprintf '%s %s ECI %s at %s',
		    $tle->get( 'id' ), $model, $name[$inx],
		    format_deltat( $deltat );
		tolerance $got[$inx], $want[$inx], $tolerance[$inx], $title;
	    }
	}
	return;
    }
}

sub format_deltat {
    my ( $deltat ) = @_;

    $deltat
	or return 'epoch';

    my $sign = '+';
    if ( $deltat < 0 ) {
	$sign = '-';
	$deltat = - $deltat;
    }

    my $sec = $deltat % 60;
    $deltat = ( $deltat - $sec ) / 60;
    my $min = $deltat % 60;
    $deltat = ( $deltat - $min ) / 60;
    my $hr = $deltat;

    my @args = ( $sign, $hr, pluralize( hour => $hr ) );
    my $tplt = 'epoch %s %i %s';
    if ( $sec ) {
	push @args, $min, pluralize( minute => $min ), $sec, pluralize(
	    second => $sec );
	$tplt .= ' %i %s %i %s';
    } elsif ( $min ) {
	push @args, $min, pluralize( minute => $min );
	$tplt .= ' %i %s';
    }

    return sprintf $tplt, @args;
}

sub format_oid {
    my ( $oid ) = @_;
    return sprintf '%05i', $oid;
}

sub pluralize {
    my ( $word, $number ) = @_;
    $number == 1
	and return $word;
    return "${word}s";
}

1;
__DATA__

-tolerance .02 .02 .02 .00002 .00002 .00002

-data
1 88888U          80275.98708465  .00073094  13844-3  66816-4 0    8
2 88888  72.8435 115.9689 0086731  52.6988 110.5714 16.05824518  105

-model sgp
# TSINCE X Y Z XDOT YDOT ZDOT
0. 2328.96594238 -5995.21600342 1719.97894287 2.91110113 -0.98164053 -7.09049922
360.00000000 2456.00610352 -6071.94232177 1222.95977784 2.67852119 -0.44705850 -7.22800565
720.00000000 2567.39477539 -6112.49725342 713.97710419 2.43952477 0.09884824 -7.31889641
1080.00000000 2663.03179932 -6115.37414551 195.73919105 2.19531813 0.65333930 -7.36169147
1440.00000000 2742.85470581 -6079.13580322 -328.86091614 1.94707947 1.21346101 -7.35499924

-model sgp4
# TSINCE X Y Z XDOT YDOT ZDOT
0. 2328.97048951 -5995.22076416 1719.97067261 2.91207230 -0.98341546 -7.09081703
360.00000000 2456.10705566 -6071.93853760 1222.89727783 2.67938992 -0.44829041 -7.22879231
720.00000000 2567.56195068 -6112.50384522 713.96397400 2.44024599 0.09810869 -7.31995916
1080.00000000 2663.09078980 -6115.48229980 196.39640427 2.19611958 0.65241995 -7.36282432
1440.00000000 2742.55133057 -6079.67144775 -326.38095856 1.94850229 1.21106251 -7.35619372

-model sgp8
# TSINCE X Y Z XDOT YDOT ZDOT
0. 2328.87265015 -5995.21289063 1720.04884338 2.91210661 -0.98353850 -7.09081554
360.00000000 2456.04577637 -6071.90490722 1222.84086609 2.67936245 -0.44820847 -7.22888553
720.00000000 2567.68383789 -6112.40881348 713.29282379 2.43992555 0.09893919 -7.32018769
1080.00000000 2663.49508667 -6115.18182373 194.62816810 2.19525236 0.65453661 -7.36308974
1440.00000000 2743.29238892 -6078.90783691 -329.73434067 1.94680957 1.21500109 -7.35625595

-data
1 11801U          80230.29629788  .01431103  00000-0  14311-1
2 11801  46.7916 230.4354 7318036  47.4722  10.4117  2.28537848

-model sdp4
# TSINCE X Y Z XDOT YDOT ZDOT
0. 7473.37066650 428.95261765 5828.74786377 5.10715413 6.44468284 -0.18613096
360.00000000 -3305.22537232 32410.86328125 -24697.17675781 -1.30113538 -1.15131518 -0.28333528
720.00000000 14271.28759766 24110.46411133 -4725.76837158 -0.32050445 2.67984074 -2.08405289
1080.00000000 -9990.05883789 22717.35522461 -23616.89062501 -1.01667246 -2.29026759 0.72892364
1440.00000000 9787.86975097 33753.34667969 -15030.81176758 -1.09425066 0.92358845 -1.52230928

-model sdp8
# TSINCE X Y Z XDOT YDOT ZDOT
0. 7469.47631836 415.99390792 5829.64318848 5.11402285 6.44403201 -0.18296110
360.00000000 -3337.38992310 32351.39086914 -24658.63037109 -1.30200730 -1.15603013 -0.28164955
720.00000000 14226.54333496 24236.08740234 -4856.19744873 -0.33951668 2.65315416 -2.08114153
1080.00000000 -10151.59838867 22223.69848633 -23392.39770508 -1.00112480 -2.33532837 0.76987664
1440.00000000 9420.08203125 33847.21875000 -15391.06469727 -1.11986055 0.85410149 -1.49506933

-tolerance .00001 .00001 .00001 .00000001 .00000001 .00000001

-data

# The following TLE data are from sgp4-ver.tle, and ultimately from
# "Revisiting Spacetrack Report #3" by David A. Vallado, Paul Crawford,
# Richard Hujsak, and T. S. Kelso, presented at the 2006 AIAA/AAS
# Astrodynamics Specialist Conference.
# This report was obtained from the Celestrak web site, specifically
# http://celestrak.com/publications/AIAA/2006-6753/

#   ------------------ Verification test cases ----------------------
#                       # TEME example
1 00005U 58002B   00179.78495062  .00000023  00000-0  28098-4 0  4753
2 00005  34.2682 348.7242 1859667 331.7664  19.3264 10.82419157413667     0.00      4320.0        360.00
#                       ## fig show lyddane fix error with gsfc ver
1 04632U 70093B   04031.91070959 -.00000084  00000-0  10000-3 0  9955
2 04632  11.4628 273.1101 1450506 207.6000 143.9350  1.20231981 44145  -5184.0     -4896.0        120.00
#   DELTA 1 DEB         # near earth normal drag equation
#                       # perigee = 377.26km, so moderate drag case
1 06251U 62025E   06176.82412014  .00008885  00000-0  12808-3 0  3985
2 06251  58.0579  54.0425 0030035 139.1568 221.1854 15.56387291  6774      0.0      2880.0        120.00
#   MOLNIYA 2-14              # 12h resonant ecc in 0.65 to 0.7 range
1 08195U 75081A   06176.33215444  .00000099  00000-0  11873-3 0   813
2 08195  64.1586 279.0717 6877146 264.7651  20.2257  2.00491383225656      0.0      2880.0        120.00
#   MOLNIYA 1-36              ## fig 12h resonant ecc in 0.7 to 0.715 range
1 09880U 77021A   06176.56157475  .00000421  00000-0  10000-3 0  9814
2 09880  64.5968 349.3786 7069051 270.0229  16.3320  2.00813614112380      0.0      2880.0        120.00
#   SMS 1 AKM           # show the integrator problem with gsfc ver
1 09998U 74033F   05148.79417928 -.00000112  00000-0  00000+0 0  4480
2 09998   9.4958 313.1750 0270971 327.5225  30.8097  1.16186785 45878  -1440.0      -720.00         60.0
#                       # Original STR#3 SDP4 test
1 11801U          80230.29629788  .01431103  00000-0  14311-1      13
2 11801  46.7916 230.4354 7318036  47.4722  10.4117  2.28537848    13      0.0      1440.0        360.00
#   EUTELSAT 1-F1 (ECS1)## fig lyddane choice in GSFC at 2080 min
1 14128U 83058A   06176.02844893 -.00000158  00000-0  10000-3 0  9627
2 14128  11.4384  35.2134 0011562  26.4582 333.5652  0.98870114 46093      0.0      2880.0        120.00
#   SL-6 R/B(2)         # Deep space, perigee = 82.48 (<98) for
#                       # s4 > 20 mod
1 16925U 86065D   06151.67415771  .02550794 -30915-6  18784-3 0  4486
2 16925  62.0906 295.0239 5596327 245.1593  47.9690  4.88511875148616      0.0      1440.0        120.00
#   SL-12 R/B           # Shows Lyddane choice at 1860 and 4700 min
1 20413U 83020D   05363.79166667  .00000000  00000-0  00000+0 0  7041
2 20413  12.3514 187.4253 7864447 196.3027 356.5478  0.24690082  7978   1440.0      4320.0        120.00
#   MOLNIYA 1-83                # 12h resonant, ecc > 0.715 (negative BSTAR)
1 21897U 92011A   06176.02341244 -.00001273  00000-0 -13525-3 0  3044
2 21897  62.1749 198.0096 7421690 253.0462  20.1561  2.01269994104880      0.0      2880.0        120.00
#   SL-6 R/B(2)         # last tle given, decayed 2006-04-04, day 94
1 22312U 93002D   06094.46235912  .99999999  81888-5  49949-3 0  3953
2 22312  62.1486  77.4698 0308723 267.9229  88.7392 15.95744531 98783  54.2028672   1440.0         20.00
#   SL-6 R/B(2)         # 12h resonant ecc in the > 0.715 range
1 22674U 93035D   06176.55909107  .00002121  00000-0  29868-3 0  6569
2 22674  63.5035 354.4452 7541712 253.3264  18.7754  1.96679808 93877      0.0      2880.0        120.00
#   ARIANE 44L+ R/B     # Lyddane bug at <= 70 min for atan2(),
#                       # no quadrant fix
1 23177U 94040C   06175.45752052  .00000386  00000-0  76590-3 0    95
2 23177   7.0496 179.8238 7258491 296.0482   8.3061  2.25906668 97438      0.0      1440.0        120.00
#   WIND                        # STR#3 Kepler failes past about 200 min
1 23333U 94071A   94305.49999999 -.00172956  26967-3  10000-3 0    15
2 23333  28.7490   2.3720 9728298  30.4360   1.3500  0.07309491    70      0.0      1600.0        120.00
#   ARIANE 42P+3 R/B    ## fig Lyddane bug at > 280.5 min for AcTan()
1 23599U 95029B   06171.76535463  .00085586  12891-6  12956-2 0  2905
2 23599   6.9327   0.2849 5782022 274.4436  25.2425  4.47796565123555      0.0       720.0         20.00
#   ITALSAT 2           # 24h resonant GEO, inclination > 3 deg
1 24208U 96044A   06177.04061740 -.00000094  00000-0  10000-3 0  1600
2 24208   3.8536  80.0121 0026640 311.0977  48.3000  1.00778054 36119      0.0      1440.0        120.00
#   AMC-4               ## fig low incl, show incl shift with
#                       ## gsfc version from 240 to 1440 min
1 25954U 99060A   04039.68057285 -.00000108  00000-0  00000-0 0  6847
2 25954   0.0004 243.8136 0001765  15.5294  22.7134  1.00271289 15615  -1440.0      1440.0        120.00
#   INTELSAT 902                # negative incl at 9313 min then
#                       # 270 deg Lyddane bug at 37606 min
1 26900U 01039A   06106.74503247  .00000045  00000-0  10000-3 0  8290
2 26900   0.0164 266.5378 0003319  86.1794 182.2590  1.00273847 16981   9300.00     9400.00        60.00
#   COSMOS 1024 DEB     # 12h resonant ecc in 0.5 to 0.65 range
1 26975U 78066F   06174.85818871  .00000620  00000-0  10000-3 0  6809
2 26975  68.4714 236.1303 5602877 123.7484 302.5767  2.05657553 67521      0.0      2880.0        120.00
#   CBERS 2                     # Near Earth, ecc = 8.84E-5 (< 1.0e-4)
#                       # drop certain normal drag terms
1 28057U 03049A   06177.78615833  .00000060  00000-0  35940-4 0  1836
2 28057  98.4283 247.6961 0000884  88.1964 271.9322 14.35478080140550      0.0      2880.0        120.00
#   NAVSTAR 53 (USA 175)# 12h non-resonant GPS (ecc < 0.5 ecc)
1 28129U 03058A   06175.57071136 -.00000104  00000-0  10000-3 0   459
2 28129  54.7298 324.8098 0048506 266.2640  93.1663  2.00562768 18443      0.0      1440.0        120.00
#   COSMOS 2405         # Near Earth, perigee = 127.20 (< 156) s4 mod
1 28350U 04020A   06167.21788666  .16154492  76267-5  18678-3 0  8894
2 28350  64.9977 345.6130 0024870 260.7578  99.9590 16.47856722116490      0.0      2880.0        120.00
#   H-2 R/B                   # Deep space, perigee = 135.75 (<156) s4 mod
1 28623U 05006B   06177.81079184  .00637644  69054-6  96390-3 0  6000
2 28623  28.5200 114.9834 6249053 170.2550 212.8965  3.79477162 12753      0.0      1440.0        120.00
#   XM-3                      # 24h resonant geo, incl < 3 deg goes
#                       # negative around 1130 min
1 28626U 05008A   06176.46683397 -.00000205  00000-0  10000-3 0  2190
2 28626   0.0019 286.9433 0000335  13.7918  55.6504  1.00270176  4891      0.0      1440.0        120.00
#   MINOTAUR R/B        # Sub-orbital case - Decayed 2005-11-29
#                       #(perigee = -51km), lost in 50 minutes
1 28872U 05037B   05333.02012661  .25992681  00000-0  24476-3 0  1534
2 28872  96.4736 157.9986 0303955 244.0492 110.6523 16.46015938 10708      0.0        50.0          5.00
#   SL-14 DEB           # Last stage of decay - lost in under 420 min
1 29141U 85108AA  06170.26783845  .99999999  00000-0  13519-0 0   718
2 29141  82.4288 273.4882 0015848 277.2124  83.9133 15.93343074  6828      0.0       440.0         20.00
#   SL-12 DEB           # Near Earth, perigee = 212.24 < 220
#                       # simplified drag eq
1 29238U 06022G   06177.28732010  .00766286  10823-4  13334-2 0   101
2 29238  51.5595 213.7903 0202579  95.2503 267.9010 15.73823839  1061      0.0      1440.0        120.00
#                       # Original STR#3 SGP4 test
1 88888U          80275.98708465  .00073094  13844-3  66816-4 0    87
2 88888  72.8435 115.9689 0086731  52.6988 110.5714 16.05824518  1058      0.0      1440.0        120.00
#

-model sgp4r

# The following solution data are from sgp4r.out, and ultimately from
# "Revisiting Spacetrack Report #3" by David A. Vallado, Paul Crawford,
# Richard Hujsak, and T. S. Kelso, presented at the 2006 AIAA/AAS
# Astrodynamics Specialist Conference.
# This report was obtained from the Celestrak web site, specifically
# http://celestrak.com/publications/AIAA/2006-6753/

  5  xx
       0.00000000    7022.46529266   -1400.08296755       0.03995155    1.893841015    6.405893759    4.534807250
     360.00000000   -7154.03120202   -3783.17682504   -3536.19412294    4.741887409   -4.151817765   -2.093935425
     720.00000000   -7134.59340119    6531.68641334    3260.27186483   -4.113793027   -2.911922039   -2.557327851
    1080.00000000    5568.53901181    4492.06992591    3863.87641983   -4.209106476    5.159719888    2.744852980
    1440.00000000    -938.55923943   -6268.18748831   -4294.02924751    7.536105209   -0.427127707    0.989878080
    1800.00000000   -9680.56121728    2802.47771354     124.10688038   -0.905874102   -4.659467970   -3.227347517
    2160.00000000     190.19796988    7746.96653614    5110.00675412   -6.112325142    1.527008184   -0.139152358
    2520.00000000    5579.55640116   -3995.61396789   -1518.82108966    4.767927483    5.123185301    4.276837355
    2880.00000000   -8650.73082219   -1914.93811525   -3007.03603443    3.067165127   -4.828384068   -2.515322836
    3240.00000000   -5429.79204164    7574.36493792    3747.39305236   -4.999442110   -1.800561422   -2.229392830
    3600.00000000    6759.04583722    2001.58198220    2783.55192533   -2.180993947    6.402085603    3.644723952
    3960.00000000   -3791.44531559   -5712.95617894   -4533.48630714    6.668817493   -2.516382327   -0.082384354
    4320.00000000   -9060.47373569    4658.70952502     813.68673153   -2.232832783   -4.110453490   -3.157345433
  4632  xx
       0.00000000    2334.11450085  -41920.44035349      -0.03867437    2.826321032   -0.065091664    0.570936053
   -5184.00000000  -29020.02587128   13819.84419063   -5713.33679183   -1.768068390   -3.235371192   -0.395206135
   -5064.00000000  -32982.56870101  -11125.54996609   -6803.28472771    0.617446996   -3.379240041    0.085954707
   -4944.00000000  -22097.68730513  -31583.13829284   -4836.34329328    2.230597499   -2.166594667    0.426443070
   -4896.00000000  -15129.94694545  -36907.74526221   -3487.56256701    2.581167187   -1.524204737    0.504805763
  6251  xx
       0.00000000    3988.31022699    5498.96657235       0.90055879   -3.290032738    2.357652820    6.496623475
     120.00000000   -3935.69800083     409.10980837    5471.33577327   -3.374784183   -6.635211043   -1.942056221
     240.00000000   -1675.12766915   -5683.30432352   -3286.21510937    5.282496925    1.508674259   -5.354872978
     360.00000000    4993.62642836    2890.54969900   -3600.40145627    0.347333429    5.707031557    5.070699638
     480.00000000   -1115.07959514    4015.11691491    5326.99727718   -5.524279443   -4.765738774    2.402255961
     600.00000000   -4329.10008198   -5176.70287935     409.65313857    2.858408303   -2.933091792   -6.509690397
     720.00000000    3692.60030028    -976.24265255   -5623.36447493    3.897257243    6.415554948    1.429112190
     840.00000000    2301.83510037    5723.92394553    2814.61514580   -5.110924966   -0.764510559    5.662120145
     960.00000000   -4990.91637950   -2303.42547880    3920.86335598   -0.993439372   -5.967458360   -4.759110856
    1080.00000000     642.27769977   -4332.89821901   -5183.31523910    5.720542579    4.216573838   -2.846576139
    1200.00000000    4719.78335752    4798.06938996    -943.58851062   -2.294860662    3.492499389    6.408334723
    1320.00000000   -3299.16993602    1576.83168320    5678.67840638   -4.460347074   -6.202025196   -0.885874586
    1440.00000000   -2777.14682335   -5663.16031708   -2462.54889123    4.915493146    0.123328992   -5.896495091
    1560.00000000    4992.31573893    1716.62356770   -4287.86065581    1.640717189    6.071570434    4.338797931
    1680.00000000      -8.22384755    4662.21521668    4905.66411857   -5.891011274   -3.593173872    3.365100460
    1800.00000000   -4966.20137963   -4379.59155037    1349.33347502    1.763172581   -3.981456387   -6.343279443
    1920.00000000    2954.49390331   -2080.65984650   -5754.75038057    4.895893306    5.858184322    0.375474825
    2040.00000000    3363.28794321    5559.55841180    1956.05542266   -4.587378863    0.591943403    6.107838605
    2160.00000000   -4856.66780070   -1107.03450192    4557.21258241   -2.304158557   -6.186437070   -3.956549542
    2280.00000000    -497.84480071   -4863.46005312   -4700.81211217    5.960065407    2.996683369   -3.767123329
    2400.00000000    5241.61936096    3910.75960683   -1857.93473952   -1.124834806    4.406213160    6.148161299
    2520.00000000   -2451.38045953    2610.60463261    5729.79022069   -5.366560525   -5.500855666    0.187958716
    2640.00000000   -3791.87520638   -5378.82851382   -1575.82737930    4.266273592   -1.199162551   -6.276154080
    2760.00000000    4730.53958356     524.05006433   -4857.29369725    2.918056288    6.135412849    3.495115636
    2880.00000000    1159.27802897    5056.60175495    4353.49418579   -5.968060341   -2.314790406    4.230722669
  8195  xx
       0.00000000    2349.89483350  -14785.93811562       0.02119378    2.721488096   -3.256811655    4.498416672
     120.00000000   15223.91713658  -17852.95881713   25280.39558224    1.079041732    0.875187372    2.485682813
     240.00000000   19752.78050009   -8600.07130962   37522.72921090    0.238105279    1.546110924    0.986410447
     360.00000000   19089.29762968    3107.89495018   39958.14661370   -0.410308034    1.640332277   -0.306873818
     480.00000000   13829.66070574   13977.39999817   32736.32082508   -1.065096849    1.279983299   -1.760166075
     600.00000000    3333.05838525   18395.31728674   12738.25031238   -1.882432221   -0.611623333   -4.039586549
     720.00000000    2622.13222207  -15125.15464924     474.51048398    2.688287199   -3.078426664    4.494979530
     840.00000000   15320.56770017  -17777.32564586   25539.53198382    1.064346229    0.892184771    2.459822414
     960.00000000   19769.70267785   -8458.65104454   37624.20130236    0.229304396    1.550363884    0.966993056
    1080.00000000   19048.56201523    3260.43223119   39923.39143967   -0.418015536    1.639346953   -0.326094840
    1200.00000000   13729.19205837   14097.70014810   32547.52799890   -1.074511043    1.270505211   -1.785099927
    1320.00000000    3148.86165643   18323.19841703   12305.75195578   -1.895271701   -0.678343847   -4.086577951
    1440.00000000    2890.80638268  -15446.43952300     948.77010176    2.654407490   -2.909344895    4.486437362
    1560.00000000   15415.98410712  -17699.90714437   25796.19644689    1.049818334    0.908822332    2.434107329
    1680.00000000   19786.00618538   -8316.74570581   37723.74539119    0.220539813    1.554518900    0.947601047
    1800.00000000   19007.28688729    3412.85948715   39886.66579255   -0.425733568    1.638276809   -0.345353807
    1920.00000000   13627.93015254   14216.95401307   32356.13706868   -1.083991976    1.260802347   -1.810193903
    2040.00000000    2963.26486560   18243.85063641   11868.25797486   -1.908015447   -0.747870342   -4.134004492
    2160.00000000    3155.85126036  -15750.70393364    1422.32496953    2.620085624   -2.748990396    4.473527039
    2280.00000000   15510.15191770  -17620.71002219   26050.43525345    1.035454678    0.925111006    2.408534465
    2400.00000000   19801.67198812   -8174.33337167   37821.38577439    0.211812700    1.558576937    0.928231880
    2520.00000000   18965.46529379    3565.19666242   39847.97510998   -0.433459945    1.637120585   -0.364653213
    2640.00000000   13525.88227400   14335.15978787   32162.13236536   -1.093537945    1.250868256   -1.835451681
    2760.00000000    2776.30574260   18156.98538451   11425.73046481   -1.920632199   -0.820370733   -4.181839232
    2880.00000000    3417.20931586  -16038.79510665    1894.74934058    2.585515864   -2.596818146    4.456882556
  9880  xx
       0.00000000   13020.06750784   -2449.07193500       1.15896030    4.247363935    1.597178501    4.956708611
     120.00000000   19190.32482476    9249.01266902   26596.71345328   -0.624960193    1.324550562    2.495697637
     240.00000000   11332.67806218   16517.99124008   38569.78482991   -1.400974747    0.710947006    0.923935636
     360.00000000     328.74217398   19554.92047380   40558.26246145   -1.593281066    0.126772913   -0.359627307
     480.00000000  -10684.90590680   18057.15728839   33158.75253886   -1.383205997   -0.582328999   -1.744412556
     600.00000000  -17069.78000550    9944.86797897   13885.91649059    0.044133354   -1.853448464   -3.815303117
     720.00000000   13725.09398980   -2180.70877090     863.29684523    3.878478111    1.656846496    4.944867241
     840.00000000   19089.63879226    9456.29670247   27026.79562883   -0.656614299    1.309112636    2.449371941
     960.00000000   11106.41248373   16627.60874079   38727.35140296   -1.409722680    0.698582526    0.891383535
    1080.00000000      72.40958621   19575.08054144   40492.12544001   -1.593394604    0.113655142   -0.390556063
    1200.00000000  -10905.89252576   17965.41205111   32850.07298244   -1.371396120   -0.601706604   -1.782817058
    1320.00000000  -17044.61207568    9635.48491849   13212.59462953    0.129244030   -1.903551430   -3.884569098
    1440.00000000   14369.90303735   -1903.85601062    1722.15319852    3.543393116    1.701687176    4.913881358
    1560.00000000   18983.96210441    9661.12233804   27448.99557732   -0.687189304    1.293808870    2.403630759
    1680.00000000   10878.79336704   16735.31433954   38879.23434264   -1.418239666    0.686235750    0.858951848
    1800.00000000    -184.03743100   19593.09371709   40420.40606889   -1.593348925    0.100448697   -0.421571993
    1920.00000000  -11125.12138631   17870.19488928   32534.21521208   -1.359116236   -0.621413776   -1.821629856
    2040.00000000  -17004.43272827    9316.53926351   12526.11883812    0.220330736   -1.955594322   -3.955058575
    2160.00000000   14960.06492693   -1620.68430805    2574.96359381    3.238634028    1.734723385    4.868880331
    2280.00000000   18873.46347257    9863.57004586   27863.46574735   -0.716736981    1.278632817    2.358448535
    2400.00000000   10649.86857581   16841.14172669   39025.48035006   -1.426527152    0.673901057    0.826632332
    2520.00000000    -440.53459323   19608.95524423   40343.10675451   -1.593138597    0.087147884   -0.452680559
    2640.00000000  -11342.45028909   17771.44223942   32211.12535721   -1.346344015   -0.641464291   -1.860864234
    2760.00000000  -16948.06005711    8987.64254880   11826.28284367    0.318007297   -2.009693492   -4.026726648
    2880.00000000   15500.53445068   -1332.90981042    3419.72315308    2.960917974    1.758331634    4.813698638
  9998  xx
       0.00000000   25532.98947267  -27244.26327953      -1.11572421    2.410283885    2.194175683    0.545888526
   -1440.00000000  -11362.18265118  -35117.55867813   -5413.62537994    3.137861261   -1.011678260    0.267510059
   -1380.00000000     309.25349929  -36960.43090143   -4198.48007670    3.292429375   -0.002166046    0.402111628
   -1320.00000000   11949.04009077  -35127.37816804   -2565.89806468    3.119942784    1.012096444    0.497284100
   -1260.00000000   22400.45329336  -29798.63236321    -677.91515122    2.638533344    1.922477736    0.542792913
   -1200.00000000   30640.84752458  -21525.02340201    1277.34808722    1.903464941    2.634294312    0.534540934
   -1140.00000000   35899.56788035  -11152.71158138    3108.72535238    0.997393045    3.079858548    0.474873291
   -1080.00000000   37732.45438600     288.18821054    4643.87587495    0.016652226    3.225184410    0.371669746
   -1020.00000000   36045.92961699   11706.61816230    5746.32646574   -0.942409065    3.069888941    0.236662980
    -960.00000000   31076.77273609   22063.44379776    6325.93403705   -1.794027976    2.642072476    0.083556127
    -900.00000000   23341.26015320   30460.88002531    6342.91707895   -2.469409743    1.990861658   -0.073612096
    -840.00000000   13568.39733054   36204.45930900    5806.79548733   -2.919354203    1.178920217   -0.221646814
    -780.00000000    2628.58762420   38840.10855897    4771.91979854   -3.114400514    0.276239109   -0.348926401
    -720.00000000   -8535.81598158   38171.79073851    3331.00311285   -3.043839958   -0.644462527   -0.445808894
  11801  xx
       0.00000000    7473.37102491     428.94748312    5828.74846783    5.107155391    6.444680305   -0.186133297
     360.00000000   -3305.22148694   32410.84323331  -24697.16974954   -1.301137319   -1.151315600   -0.283335823
     720.00000000   14271.29083858   24110.44309009   -4725.76320143   -0.320504528    2.679841539   -2.084054355
    1080.00000000   -9990.05800009   22717.34212448  -23616.88515553   -1.016674392   -2.290267981    0.728923337
    1440.00000000    9787.87836256   33753.32249667  -15030.79874625   -1.094251553    0.923589906   -1.522311008
  14128  xx
       0.00000000   34747.57932696   24502.37114079      -1.32832986   -1.731642662    2.452772615    0.608510081
     120.00000000   18263.33439094   38159.96004751    4186.18304085   -2.744396611    1.255583260    0.528558932
     240.00000000   -3023.38840703   41783.13186459    7273.03412906   -3.035574793   -0.271656544    0.309645251
     360.00000000  -23516.34391907   34424.42065671    8448.49867693   -2.529120477   -1.726186020    0.009582303
     480.00000000  -37837.46699511   18028.39727170    7406.25540271   -1.360069525   -2.725794686   -0.292555349
     600.00000000  -42243.58460661   -3093.72887774    4422.91711801    0.163110919   -3.009980598   -0.517584362
     720.00000000  -35597.57919549  -23407.91145393     282.09554383    1.641405246   -2.506773678   -0.606963478
     840.00000000  -19649.19834455  -37606.11623860   -3932.71525948    2.689647056   -1.349150016   -0.537710698
     960.00000000    1431.30912160  -41982.04949668   -7120.45467057    3.035263353    0.160882945   -0.327993994
    1080.00000000   22136.97605384  -35388.19823762   -8447.62393401    2.587624889    1.630097136   -0.032349004
    1200.00000000   37050.15790219  -19537.23321425   -7564.83463543    1.461844494    2.674654256    0.272202191
    1320.00000000   42253.81760945    1431.81867593   -4699.87621174   -0.049247334    3.019518960    0.505890058
    1440.00000000   36366.59147396   22023.54245720    -601.47121821   -1.549681546    2.571788981    0.607057418
    1560.00000000   20922.12287985   36826.33975981    3654.91125886   -2.644070068    1.447521216    0.548722983
    1680.00000000     -23.77224182   41945.51688402    6950.29891751   -3.043358385   -0.057417440    0.346112094
    1800.00000000  -20964.17821076   36039.06206172    8418.91984963   -2.642795221   -1.546099886    0.052725852
    1920.00000000  -36401.63863057   20669.75286162    7677.19769359   -1.549488154   -2.627052310   -0.254079652
    2040.00000000  -42298.30327543    -119.03351118    4922.96388841   -0.052232768   -3.018152669   -0.493827331
    2160.00000000  -37125.62383511  -20879.63058368     879.86971348    1.456499841   -2.619358421   -0.604081694
    2280.00000000  -22250.12320553  -36182.74736487   -3393.15365183    2.583161226   -1.536647628   -0.556404555
    2400.00000000   -1563.06258654  -42035.43179159   -6780.02161760    3.034917506   -0.052702046   -0.363395654
    2520.00000000   19531.64069587  -36905.65470956   -8395.46892032    2.693682199    1.446079999   -0.075256054
    2640.00000000   35516.53506142  -22123.71916638   -7815.04516935    1.646882125    2.568416058    0.232985912
    2760.00000000   42196.03535976   -1547.32646751   -5187.39401981    0.166491841    3.019211549    0.480665780
    2880.00000000   37802.25393045   19433.57330019   -1198.66634226   -1.359930580    2.677830903    0.602507466
  16925  xx
       0.00000000    5559.11686836  -11941.04090781     -19.41235206    3.392116762   -1.946985124    4.250755852
     120.00000000   12339.83273749   -2771.14447871   18904.57603433   -0.871247614    2.600917693    0.581560002
     240.00000000   -3385.00215658    7538.13955729     200.59008616   -2.023512865   -4.261808344   -6.856385787
     360.00000000   12805.22442200  -10258.94667177   13780.16486738    0.619279224    1.821510542    2.507365975
     480.00000000    5682.46556318    7199.30270473   15437.67134070   -2.474365406    2.087897336   -2.583767460
     600.00000000    7628.94243982  -12852.72097492    2902.87208981    2.748131081   -0.740084579    4.125307943
     720.00000000   11531.64866625    -858.27542736   19086.85993771   -1.170071901    2.660311986    0.096005705
     840.00000000   -3866.98069515    2603.73442786   -4577.36484577    1.157257298   -8.453281164   -4.683959407
     960.00000000   13054.77732721   -8707.92757730   15537.63259903    0.229846748    2.119467054    2.063396852
    1080.00000000    3496.91064652    8712.83919778   12845.81838327   -2.782184997    1.552950644   -3.554436131
    1200.00000000    9593.07424729  -13023.75963608    6250.46484931    2.072666376    0.278735334    3.778111073
    1320.00000000   10284.79205084    1487.89914169   18824.37381327   -1.530335053    2.663107730   -0.542205966
    1440.00000000    -984.62035146   -5187.03480813   -5745.59594144    4.340271916   -7.266811354    1.777668888
  20413  xx
       0.00000000   25123.29290741  -13225.49966287    3249.40351869    0.488683419    4.797897593   -0.961119693
    1440.00000000 -151669.05280515   -5645.20454550   -2198.51592118   -0.869182889   -0.870759872    0.156508219
    1560.00000000 -157497.71657495  -11884.99595074   -1061.44439402   -0.749657961   -0.864016715    0.157766101
    1680.00000000 -162498.32255577  -18062.99733167      81.00915253   -0.638980378   -0.853687105    0.158098992
    1800.00000000 -166728.76010920  -24155.99648299    1222.84128677   -0.535600687   -0.840455444    0.157680857
    1920.00000000 -169935.81924592  -31767.29787964    2749.01540345   -0.430050431   -0.828904183    0.157812340
    2040.00000000 -172703.07831815  -37662.95639336    3883.60052579   -0.338004891   -0.810277487    0.156020035
    2160.00000000 -174823.19337404  -43417.55605219    5003.26312809   -0.250258622   -0.789828672    0.153764903
    2280.00000000 -176324.63925775  -49018.51958648    6104.85025002   -0.166136613   -0.767706262    0.151092242
    2400.00000000 -177231.42142458  -54454.12699497    7185.48661607   -0.085067854   -0.744001567    0.148033403
    2520.00000000 -177563.73583232  -59713.14859144    8242.48472591   -0.006561730   -0.718760309    0.144608676
    2640.00000000 -177338.48026483  -64784.54644698    9273.27220003    0.069809946   -0.691990238    0.140829236
    2760.00000000 -176569.65151461  -69657.21976255   10275.33063459    0.144426878   -0.663665876    0.136698419
    2880.00000000 -175268.65299073  -74319.77625463   11246.14177160    0.217631370   -0.633731091    0.132212491
    3000.00000000 -173444.53039609  -78760.31560396   12183.13775212    0.289737325   -0.602099929    0.127361017
    3120.00000000 -171104.14813653  -82966.21323591   13083.65278381    0.361037779   -0.568655903    0.122126889
    3240.00000000 -168252.31543803  -86923.89363433   13944.87382716    0.431811396   -0.533249797    0.116486022
    3360.00000000 -164891.86832887  -90618.58225954   14763.78794247    0.502328269   -0.495695896    0.110406725
    3480.00000000 -161023.71139825  -94034.02398835   15537.12375729    0.572855321   -0.455766412    0.103848688
    3600.00000000 -156646.82136726  -97152.15370791   16261.28409305    0.643661538   -0.413183688    0.096761524
    3720.00000000 -151758.21285737  -99952.70098346   16932.26607548    0.715023254   -0.367609561    0.089082727
    3840.00000000 -146352.86521283 -102412.70506284   17545.56394158    0.787229695   -0.318630913    0.080734873
    3960.00000000 -140423.60777444 -104505.90799734   18096.04807097    0.860588979   -0.265739987    0.071621768
    4080.00000000 -133960.95961851 -106201.98091318   18577.81121953    0.935434758   -0.208307307    0.061623110
    4200.00000000 -126952.91860010 -107465.51906186   18983.96903112    1.012133628   -0.145543878    0.050587007
    4320.00000000 -119384.69396454 -108254.71115372   19306.39581892    1.091093313   -0.076447479    0.038319282
  21897  xx
       0.00000000  -14464.72135182   -4699.19517587       0.06681686   -3.249312013   -3.281032707    4.007046940
     120.00000000  -19410.46286123  -19143.03318969   23114.05522619    0.508602237   -1.156882269    2.379923455
     240.00000000  -12686.06129708  -23853.75335645   35529.81733588    1.231633829   -0.221718202    1.118440291
     360.00000000   -2775.46649359  -22839.64574119   39494.64689967    1.468963405    0.489481769   -0.023972788
     480.00000000    7679.87883570  -16780.50760106   34686.21815555    1.364171080    1.211183897   -1.385151371
     600.00000000   14552.40023028   -4819.50121461   17154.70672449    0.109201591    2.176124494   -3.854856805
     720.00000000  -15302.38845375   -5556.43440300    1095.95088753   -2.838224312   -3.134231137    3.992596326
     840.00000000  -19289.20066748  -19427.04851118   23759.45685636    0.552495087   -1.112499437    2.325112654
     960.00000000  -12376.21976437  -23893.38020018   35831.33691892    1.246701529   -0.194294048    1.074867282
    1080.00000000   -2400.55677665  -22698.62264640   39482.75964390    1.472582922    0.513555654   -0.069306561
    1200.00000000    8031.66819252  -16455.77592085   34298.94391742    1.351357426    1.239633234   -1.448195324
    1320.00000000   14559.48780372   -4238.43773813   16079.23154704   -0.026409655    2.218938770   -4.012628896
    1440.00000000  -16036.04980660   -6372.51406468    2183.44834232   -2.485113443   -2.994994355    3.955891272
    1560.00000000  -19156.71583814  -19698.89059957   24389.29473934    0.594278133   -1.069418599    2.271152044
    1680.00000000  -12062.72925552  -23925.82362911   36120.66680667    1.261238798   -0.167201856    1.031478939
    1800.00000000   -2024.96136966  -22551.56626703   39458.50085787    1.475816889    0.537615764   -0.114887472
    1920.00000000    8379.80916204  -16123.95878459   33894.75123231    1.337468254    1.268432783   -1.512473301
    2040.00000000   14527.86748873   -3646.33817120   14960.74306518   -0.180035839    2.261273515   -4.179355590
    2160.00000000  -16680.12147335   -7149.80800425    3257.64227208   -2.178897351   -2.863927095    3.904876943
    2280.00000000  -19013.58793448  -19958.93766022   25003.81778666    0.634100431   -1.027559823    2.218002685
    2400.00000000  -11745.76155818  -23951.19438627   36397.87676581    1.275261813   -0.140425132    0.988259441
    2520.00000000   -1648.81945070  -22398.50594576   39421.83273890    1.478660174    0.561671519   -0.160733093
    2640.00000000    8723.97652795  -15784.99406275   33473.35215527    1.322433593    1.297602497   -1.578055493
    2760.00000000   14452.25571587   -3043.42332645   13796.84870805   -0.355190169    2.302485443   -4.355767077
    2880.00000000  -17246.31075678   -7890.72601508    4315.39410307   -1.910968458   -2.740945672    3.844722726
  22312  xx
       0.00000000    1442.10132912    6510.23625449       8.83145885   -3.475714837    0.997262768    6.835860345
      54.20286720     306.10478453   -5816.45655525   -2979.55846068    3.950663855    3.415332543   -5.879974329
      74.20286720    3282.82085464    2077.46972905   -5189.17988770    0.097342701    7.375135692    2.900196702
      94.20286720     530.82729176    6426.20790003    1712.37076793   -3.837120395   -1.252430637    6.561602577
     114.20286720   -3191.69170212     170.27219912    5956.29807775   -1.394956872   -7.438073471   -0.557553115
     134.20286720   -1818.99222465   -6322.45146616     681.95247154    3.349795173   -1.530140265   -6.831522765
     154.20286720    2515.66448634   -2158.83091224   -5552.13320544    2.571979660    7.311930509   -1.639865620
     174.20286720    2414.52833210    5749.10150922   -1998.59693165   -2.681032960    3.527589301    6.452951429
     194.20286720   -1877.98944331    3862.27848302    5112.48435863   -3.261489804   -6.026859137    3.433254768
     214.20286720   -3117.36584395   -4419.74773864    3840.85960912    1.545479182   -5.475416581   -5.207913748
     234.20286720     815.32034678   -5231.67692249   -3760.04690354    3.870864200    4.455588552   -5.211082191
     254.20286720    3269.54341810    3029.00081083   -4704.67969713   -0.526711345    6.812157950    3.929825087
     274.20286720     -10.18099756    6026.23341453    2643.50518407   -3.953623254   -2.616070012    6.145637500
     294.20286720   -3320.58819584   -1248.42679945    5563.06017927   -0.637046974   -7.417786044   -2.076120187
     314.20286720   -1025.48974616   -6366.98945782    -911.23559153    3.811771909    0.438071490   -6.829260617
     334.20286720    3003.75996128    -413.85708003   -5706.15591435    1.674350083    7.694169068    0.316915204
     354.20286720    1731.42816980    6258.27676925    -409.32527982   -3.400497806    1.447945424    6.904010052
     374.20286720   -2582.52111460    2024.19020680    5647.55650268   -2.530348121   -7.221719393    1.438141553
     394.20286720   -2440.56848578   -5702.77311877    1934.81094689    2.731792947   -3.350576075   -6.527773339
     414.20286720    1951.22934391   -3423.59443045   -5121.67808201    3.249039133    6.465974362   -3.069806659
     434.20286720    2886.50939356    4888.68626216   -3096.29885989   -1.973162139    4.877039020    5.832414910
     454.20286720   -1276.55532182    4553.26898463    4406.19787375   -3.715146421   -5.320176914    4.418210777
     474.20286720   -3181.54698042   -3831.29976506    4096.80242787    1.114159970   -6.104773578   -4.829967400
  22674  xx
       0.00000000   14712.22023280   -1443.81061850       0.83497888    4.418965470    1.629592098    4.115531802
     120.00000000   25418.88807860    9342.60307989   23611.46690798    0.051284086    1.213127306    2.429004159
     240.00000000   21619.59550749   16125.24978864   36396.79365831   -0.963604380    0.685454965    1.177181937
     360.00000000   12721.50543331   19258.96193362   40898.47648359   -1.457448565    0.179955469    0.071502601
     480.00000000    1272.80760054   18458.41971897   37044.74742696   -1.674863386   -0.436454983   -1.201040990
     600.00000000  -10058.43188619   11906.60764454   21739.62097733   -1.245829683   -1.543789125   -3.324449221
     720.00000000   10924.40116466   -2571.92414170   -2956.34856294    6.071727751    1.349579102    3.898430260
     840.00000000   25332.14851525    8398.91099924   21783.90654357    0.222320754    1.272214306    2.580527192
     960.00000000   22317.71926039   15574.82086129   35495.77144092   -0.892750056    0.737383381    1.291738834
    1080.00000000   13795.68675885   19088.83051008   40803.69584385   -1.420277669    0.235599456    0.185517056
    1200.00000000    2515.17145049   18746.63776282   37864.58088636   -1.668016053   -0.360431458   -1.052854596
    1320.00000000   -9084.48602106   12982.62608646   24045.63900249   -1.378032363   -1.373184736   -3.013963835
    1440.00000000    5647.00909495   -3293.90518693   -5425.85235063    8.507977176    0.414560797    2.543322806
    1560.00000000   25111.63372210    7412.55109488   19844.25781729    0.416496290    1.332106006    2.739301737
    1680.00000000   22961.47461641   14985.74459578   34511.09257381   -0.816711048    0.789391108    1.407901804
    1800.00000000   14841.15301459   18876.91439870   40626.25901619   -1.380403341    0.290228810    0.298258120
    1920.00000000    3750.70174081   18978.57939698   38578.11783220   -1.656939412   -0.287930881   -0.910825599
    2040.00000000   -8027.30219489   13939.54436955   26136.49045637   -1.474476061   -1.222693624   -2.737178731
    2160.00000000   -1296.95657093   -2813.69369768   -5871.09587258    9.881929371   -1.978467207   -1.922261005
    2280.00000000   24738.60364819    6383.41644019   17787.27631900    0.639556952    1.392554379    2.906206324
    2400.00000000   23546.85388669   14358.15602832   33441.67679479   -0.734895006    0.841564851    1.526009909
    2520.00000000   15855.87696303   18624.05633582   40367.13420574   -1.337753546    0.343969522    0.410018472
    2640.00000000    4976.44933591   19156.75504042   39189.68603184   -1.642084365   -0.218525096   -0.774148204
    2760.00000000   -6909.20746210   14790.44707042   28034.46732222   -1.545152610   -1.088119523   -2.487447214
    2880.00000000   -7331.65006707    -604.17323419   -2723.51014575    6.168997265   -3.634011554   -5.963531682
  23177  xx
       0.00000000   -8801.60046706      -0.03357557      -0.44522743   -3.835279101   -7.662552175    0.944561323
     120.00000000   -1684.34352858  -31555.95196340    3888.99944319    2.023055719   -2.151306405    0.265065778
     240.00000000   12325.51410155  -38982.15046244    4802.88832275    1.763224157   -0.102514446    0.012397139
     360.00000000   22773.66831936  -34348.02176606    4228.77407391    1.067616787    1.352427865   -0.166956367
     480.00000000   26194.40441089  -19482.94203672    2393.84774063   -0.313732186    2.808771328   -0.346204118
     600.00000000    8893.50573448    5763.38890561    -713.69884164   -7.037399220    3.022613131   -0.370272416
     720.00000000   -6028.75686537  -25648.99913786    3164.37107274    1.883159288   -3.177051976    0.390793162
     840.00000000    8313.57299056  -38146.45710922    4697.80777535    1.905002133   -0.625883074    0.076098187
     960.00000000   20181.29108622  -36842.60674073    4529.12568218    1.326244476    0.921916487   -0.114527455
    1080.00000000   26302.61794569  -25173.39539436    3084.65309986    0.245398835    2.329974347   -0.287495880
    1200.00000000   19365.07045602   -2700.00490122     317.42727417   -3.009733018    3.902496058   -0.478928582
    1320.00000000   -9667.81878780  -16930.19112642    2095.87469034    1.279288285   -4.736005905    0.582878255
    1440.00000000    4021.31438583  -36066.09209609    4442.91587411    2.007322354   -1.227461376    0.149383897
  23333  xx
       0.00000000   -9301.24542291    3326.10200382    2318.36441127   -8.729303005   -0.828225037   -0.122314827
     120.00000000  -44672.91239677   -6213.11996580   -1738.80131727   -3.719475070   -1.336673022   -0.621888261
     240.00000000  -67053.08885387  -14994.69685946   -5897.99072793   -2.860576613   -1.183771565   -0.568473909
     360.00000000  -85227.84253161  -22897.08484467   -9722.59184563   -2.426469823   -1.078592475   -0.525341431
     480.00000000 -100986.00419137  -30171.19698696  -13283.77044766   -2.147108978   -1.000530827   -0.491587582
     600.00000000 -115093.00686386  -36962.56316477  -16634.15682929   -1.945446188   -0.938947736   -0.464199202
     720.00000000 -127965.80064891  -43363.32967165  -19809.90480432   -1.789652016   -0.888278463   -0.441254468
     840.00000000 -139863.28332203  -49436.45704152  -22836.80438138   -1.663762568   -0.845315913   -0.421548627
     960.00000000 -150960.22978258  -55227.45413896  -25734.01408879   -1.558730986   -0.808061065   -0.404293846
    1080.00000000 -161381.71414630  -60770.64040903  -28516.26290017   -1.468977174   -0.775190459   -0.388951810
    1200.00000000 -171221.18736949  -66092.76474443  -31195.19847387   -1.390837596   -0.745785633   -0.375140398
    1320.00000000 -180550.82888745  -71215.23290630  -33780.24938270   -1.321788672   -0.719184752   -0.362579495
    1440.00000000 -189427.87533073  -76155.54943343  -36279.19882816   -1.260024473   -0.694896053   -0.351058133
    1560.00000000 -197898.69401407  -80928.29015180  -38698.57972447   -1.204211888   -0.672544709   -0.340413731
    1600.00000000 -200638.82986235  -82484.14969881  -39488.34331447   -1.186748462   -0.665472422   -0.337037582
  23599  xx
       0.00000000    9892.63794341      35.76144969      -1.08228838    3.556643237    6.456009375    0.783610890
      20.00000000   11931.95642997    7340.74973750     886.46365987    0.308329116    5.532328972    0.672887281
      40.00000000   11321.71039205   13222.84749156    1602.40119049   -1.151973982    4.285810871    0.521919425
      60.00000000    9438.29395675   17688.05450261    2146.59293402   -1.907904054    3.179955046    0.387692479
      80.00000000    6872.08634639   20910.11016811    2539.79945034   -2.323995367    2.207398462    0.269506121
     100.00000000    3933.37509798   23024.07662542    2798.25966746   -2.542860616    1.327134966    0.162450076
     120.00000000     816.64091546   24118.98675475    2932.69459428   -2.626838010    0.504502763    0.062344306
     140.00000000   -2334.41705804   24246.86096326    2949.36448841   -2.602259646   -0.288058266   -0.034145135
     160.00000000   -5394.31798039   23429.42716149    2850.86832586   -2.474434068   -1.074055982   -0.129868366
     180.00000000   -8233.35130237   21661.24480883    2636.51456118   -2.230845533   -1.875742344   -0.227528603
     200.00000000  -10693.96497348   18909.88168891    2302.33707548   -1.835912433   -2.716169865   -0.329931880
     220.00000000  -12553.89669904   15114.63990716    1840.93573231   -1.212478879   -3.619036996   -0.439970633
     240.00000000  -13450.20591864   10190.57904289    1241.95958736   -0.189082511   -4.596701971   -0.559173899
     260.00000000  -12686.60437121    4079.31106161     498.27078614    1.664498211   -5.559889865   -0.676747779
     280.00000000   -8672.55867753   -2827.56823315    -342.59644716    5.515079852   -5.551222962   -0.676360044
     300.00000000    1153.31498060   -6411.98692060    -779.87288941    9.689818102    1.388598425    0.167868798
     320.00000000    9542.79201056    -533.71253081     -65.73165428    3.926947087    6.459583539    0.785686755
     340.00000000   11868.80960100    6861.59590848     833.72780602    0.452957852    5.632811328    0.685262323
     360.00000000   11376.23941678   12858.97121366    1563.40660172   -1.087665695    4.374693347    0.532207051
     380.00000000    9547.70300782   17421.48570758    2118.56907515   -1.876540262    3.253891728    0.395810243
     400.00000000    7008.51470263   20725.47471227    2520.56064289   -2.308703599    2.270724438    0.276138613
     420.00000000    4082.28135104   22911.04184601    2786.37568309   -2.536665546    1.383670232    0.168153407
     440.00000000     969.17978149   24071.23673676    2927.31326579   -2.626695115    0.557172428    0.067536854
     460.00000000   -2184.71515444   24261.21671601    2950.08142825   -2.607072866   -0.236887607   -0.029125215
     480.00000000   -5253.42223370   23505.37595671    2857.66120738   -2.484424544   -1.022255436   -0.124714444
     500.00000000   -8108.27961017   21800.81688388    2649.72981961   -2.247597251   -1.821159176   -0.221925624
     520.00000000  -10594.77795556   19117.80779221    2322.72136979   -1.863118484   -2.656426668   -0.323521502
     540.00000000  -12497.32045995   15398.64085906    1869.69983897   -1.258130763   -3.551583368   -0.432338888
     560.00000000  -13467.92475245   10560.90147785    1280.78399181   -0.271870523   -4.520514224   -0.550016092
     580.00000000  -12848.18843590    4541.21901842     548.53826427    1.494157156   -5.489585384   -0.667472039
     600.00000000   -9152.70552728   -2344.24950144    -287.98121970    5.127921095   -5.650383025   -0.685989008
     620.00000000     280.38490909   -6500.10264018    -790.36092984    9.779619614    0.581815811    0.074171345
     640.00000000    9166.25784315   -1093.12552651    -129.49428887    4.316668714    6.438636494    0.785116609
     660.00000000   11794.48942915    6382.21138354     780.88439015    0.604412453    5.731729369    0.697574333
     680.00000000   11424.30138324   12494.26088864    1524.33165488   -1.021328075    4.463448968    0.542532698
     700.00000000    9652.09867350   17153.84762075    2090.48038336   -1.844516637    3.327522235    0.403915232
     720.00000000    7140.41945884   20539.25485336    2501.21469368   -2.293173684    2.333507912    0.282716311
  24208  xx
       0.00000000    7534.10987189   41266.39266843      -0.10801028   -3.027168008    0.558848996    0.207982755
     120.00000000  -14289.19940414   39469.05530051    1428.62838591   -2.893205245   -1.045447840    0.179634249
     240.00000000  -32222.92014955   26916.25425799    2468.59996594   -1.973007929   -2.359335071    0.102539376
     360.00000000  -41413.95109398    7055.51656639    2838.90906671   -0.521665080   -3.029172207   -0.002066843
     480.00000000  -39402.72251896  -14716.42475223    2441.32678358    1.066928187   -2.878714619   -0.105865729
     600.00000000  -26751.08889828  -32515.13982431    1384.38865570    2.366228869   -1.951032799   -0.181018498
     720.00000000   -6874.77975542  -41530.38329422     -46.60245459    3.027415087   -0.494671177   -0.207337260
     840.00000000   14859.52039042  -39302.58907247   -1465.02482524    2.869609883    1.100123969   -0.177514425
     960.00000000   32553.14863770  -26398.88401807   -2485.45866002    1.930064459    2.401574539   -0.099250520
    1080.00000000   41365.67576837   -6298.09965811   -2828.05254033    0.459741276    3.051680214    0.006431872
    1200.00000000   38858.83295070   15523.39314924   -2396.86850752   -1.140211488    2.867567143    0.110637217
    1320.00000000   25701.46068162   33089.42617648   -1308.68556638   -2.428713821    1.897381431    0.184605907
    1440.00000000    5501.08137100   41590.27784405     138.32522930   -3.050691874    0.409203052    0.207958133
  25954  xx
       0.00000000    8827.15660472  -41223.00971237       3.63482963    3.007087319    0.643701323    0.000941663
   -1440.00000000    8118.18519221  -41368.40537378       4.11046687    3.017696741    0.591994297    0.000933016
   -1320.00000000   27766.34015328  -31724.97000557       9.93297846    2.314236153    2.024903193    0.000660861
   -1200.00000000   39932.57237973  -13532.60040454      13.12958252    0.987382819    2.911942843    0.000213298
   -1080.00000000   41341.01365441    8305.71681955      12.84988501   -0.605098224    3.014378268   -0.000291034
    -960.00000000   31614.99210558   27907.29155353       9.16618797   -2.034243523    2.305014102   -0.000718418
    -840.00000000   13375.75227587   39994.27017651       3.05416854   -2.915424366    0.975119874   -0.000955576
    -720.00000000   -8464.89963309   41312.93549892      -3.86622919   -3.011600615   -0.617275050   -0.000939664
    -600.00000000  -28026.23406158   31507.89995661      -9.76047869   -2.296840160   -2.043607595   -0.000674889
    -480.00000000  -40040.01314363   13218.00579413     -13.06594832   -0.963328772   -2.919827983   -0.000231414
    -360.00000000  -41268.43291976   -8632.06859693     -12.90661266    0.630042315   -3.009677376    0.000273163
    -240.00000000  -31377.85317015  -28156.13970334      -9.32605530    2.054021717   -2.288554158    0.000704959
    -120.00000000  -13031.41552688  -40092.33381029      -3.27636660    2.924657466   -0.950541167    0.000949381
       0.00000000    8827.15660472  -41223.00971237       3.63482963    3.007087319    0.643701323    0.000941663
     120.00000000   28306.85426674  -31243.80147394       9.57216891    2.279137743    2.064316875    0.000684127
     240.00000000   40159.05128805  -12845.39151157      12.96086316    0.937265422    2.928448287    0.000245505
     360.00000000   41192.55903455    9013.79606759      12.90495666   -0.656727442    3.003543458   -0.000257479
     480.00000000   31131.69755798   28445.55681731       9.42419238   -2.073484842    2.269770851   -0.000691233
     600.00000000   12687.81846530   40217.83324639       3.44726249   -2.931721827    0.924962230   -0.000940766
     720.00000000   -9172.23500245   41161.63475527      -3.43575757   -3.000571486   -0.668847508   -0.000940101
     840.00000000  -28562.51093192   31022.45987587      -9.39562161   -2.261449202   -2.082713897   -0.000689669
     960.00000000  -40260.77504549   12529.11484344     -12.84915105   -0.913097031   -2.935933528   -0.000256181
    1080.00000000  -41114.14376538   -9338.87194483     -12.87952404    0.681588815   -2.998432565    0.000245006
    1200.00000000  -30890.01512240  -28690.40750792      -9.48037212    2.092989805   -2.252978152    0.000680459
    1320.00000000  -12341.46194020  -40310.06316386      -3.55833201    2.940537098   -0.900219523    0.000934170
    1440.00000000    9533.27750818  -41065.52390214       3.30756482    2.995596171    0.695200236    0.000938525
  26900  xx
       0.00000000  -42014.83795787    3702.34357772     -26.67500257   -0.269775247   -3.061854393    0.000336726
    9300.00000000   40968.68133298   -9905.99156086      11.84946837    0.722756848    2.989645389   -0.000161261
    9360.00000000   42135.66858481    1072.99195618      10.83481752   -0.078150602    3.074772455   -0.000380063
    9400.00000000   41304.75156132    8398.27742944       9.74006214   -0.612515135    3.014117469   -0.000511575
  26975  xx
       0.00000000  -14506.92313768  -21613.56043281      10.05018894    2.212943308    1.159970892    3.020600202
     120.00000000    7309.62197950    6076.00713664    6800.08705263    1.300543383    5.322579615   -4.788746312
     240.00000000   -3882.62933791   11960.00543452  -25088.14383845   -2.146773699   -1.372461491   -2.579382089
     360.00000000  -16785.45507465    -734.79159704  -34300.57085853   -1.386528125   -1.907762641   -0.220949641
     480.00000000  -23524.16689356  -13629.45124622  -30246.27899200   -0.462846784   -1.586139830    1.269293624
     600.00000000  -22890.23597092  -22209.35900155  -16769.91946116    0.704351342   -0.671112594    2.432433851
     720.00000000  -11646.39698980  -19855.44222106    3574.00109607    2.626712727    1.815887329    2.960883901
     840.00000000    7665.76124241   11159.78946577     345.93813117   -0.584818007    3.193514161   -5.750338922
     960.00000000   -6369.35388112   10204.80073022  -27844.52150384   -2.050573276   -1.582940542   -2.076075232
    1080.00000000  -18345.64763145   -2977.76684430  -34394.90760612   -1.243589864   -1.892050757    0.060372061
    1200.00000000  -23979.74839255  -15436.44139571  -28616.50540218   -0.294973425   -1.482987916    1.478255628
    1320.00000000  -21921.97167880  -22852.45147658  -13784.85308485    0.945455629   -0.428940995    2.596964378
    1440.00000000   -8266.43821031  -17210.74590112    6967.95546070    3.082244069    2.665881872    2.712555075
    1560.00000000    6286.85464535   13809.56328971   -6321.60663781   -1.615964016    1.383135377   -5.358719132
    1680.00000000   -8730.87526788    8244.63344365  -30039.92372791   -1.935622871   -1.724162072   -1.631224738
    1800.00000000  -19735.81883249   -5191.76593007  -34166.14974143   -1.097835530   -1.860148418    0.324401050
    1920.00000000  -24232.73847703  -17112.08243255  -26742.88893252   -0.119786184   -1.364365317    1.680220468
    2040.00000000  -20654.45640708  -23184.54386047  -10611.55144716    1.209238113   -0.144169639    2.748054938
    2160.00000000   -4337.15988957  -13410.46817244    9870.45949215    3.532753095    3.772236461    2.088424247
    2280.00000000    4074.62263523   14698.07548285  -12248.65327973   -2.053824693    0.203325817   -4.607867718
    2400.00000000  -10950.23438984    6148.66879447  -31736.65532865   -1.809875605   -1.816179062   -1.233364913
    2520.00000000  -20952.40702045   -7358.71507895  -33633.06643074   -0.948973031   -1.813594137    0.573893078
    2640.00000000  -24273.48944134  -18637.15546906  -24633.27702390    0.064161440   -1.228537560    1.875728935
    2760.00000000  -19057.55468077  -23148.29322082   -7269.38614178    1.500802809    0.195383037    2.879031237
    2880.00000000      43.69305308   -8145.90299207   11634.57079913    3.780661682    5.105315423    0.714401345
  28057  xx
       0.00000000   -2715.28237486   -6619.26436889      -0.01341443   -1.008587273    0.422782003    7.385272942
     120.00000000   -1816.87920942   -1835.78762132    6661.07926465    2.325140071    6.655669329    2.463394512
     240.00000000    1483.17364291    5395.21248786    4448.65907172    2.560540387    4.039025766   -5.736648561
     360.00000000    2801.25607157    5455.03931333   -3692.12865694   -0.595095864   -3.951923117   -6.298799125
     480.00000000     411.09332812   -1728.99769152   -6935.45548810   -2.935970964   -6.684085058    1.492800886
     600.00000000   -2506.52558454   -6628.98655094    -988.07784497   -1.390577189   -0.556164143    7.312736468
     720.00000000   -2090.79884266   -2723.22832193    6266.13356576    1.992640665    6.337529519    3.411803080
     840.00000000    1091.80560222    4809.88229503    5172.42897894    2.717483546    4.805518977   -5.030019896
     960.00000000    2811.14062300    5950.65707171   -2813.23705389   -0.159662742   -3.121215491   -6.775341949
    1080.00000000     805.72698304    -812.16627907   -7067.58483968   -2.798936020   -6.889265977    0.472770873
    1200.00000000   -2249.59837532   -6505.84890714   -1956.72365062   -1.731234729   -1.528750230    7.096660885
    1320.00000000   -2311.57375797   -3560.99112891    5748.16749600    1.626569751    5.890482233    4.293545048
    1440.00000000     688.16056594    4124.87618964    5794.55994449    2.810973665    5.479585563   -4.224866316
    1560.00000000    2759.94088230    6329.87271798   -1879.19518331    0.266930672   -2.222670878   -7.119390567
    1680.00000000    1171.50677137     125.82053748   -7061.96626202   -2.605687852   -6.958489749   -0.556333225
    1800.00000000   -1951.43708472   -6251.71945820   -2886.95472355   -2.024131483   -2.475214272    6.741537478
    1920.00000000   -2475.70722288   -4331.90569958    5117.31234924    1.235823539    5.322743371    5.091281211
    2040.00000000     281.46097847    3353.51057102    6302.87900650    2.840647273    6.047222485   -3.337085992
    2160.00000000    2650.33118860    6584.33434851    -908.29027134    0.675457235   -1.274044972   -7.323921567
    2280.00000000    1501.17226597    1066.31132756   -6918.71472952   -2.361891904   -6.889669974   -1.574718619
    2400.00000000   -1619.73468334   -5871.14051991   -3760.56587071   -2.264093975   -3.376316601    6.254622256
    2520.00000000   -2581.04202505   -5020.05572531    4385.92329047    0.829668458    4.645048038    5.789262667
    2640.00000000    -119.22080628    2510.90620488    6687.45615459    2.807575712    6.496549689   -2.384136661
    2760.00000000    2486.23806726    6708.18210028      80.43349581    1.057274905   -0.294294027   -7.384689123
    2880.00000000    1788.42334580    1990.50530957   -6640.59337725   -2.074169091   -6.683381288   -2.562777776
  28129  xx
       0.00000000   21707.46412351  -15318.61752390       0.13551152    1.304029214    1.816904974    3.161919976
     120.00000000   18616.75971861    3166.15177043   18833.41523210   -2.076122016    2.838457575    1.586210535
     240.00000000   -3006.50596328   18522.20742011   18941.84078154   -3.375452789    1.032680773   -1.559324534
     360.00000000  -21607.02086957   15432.59962630     206.62470309   -1.306049851   -1.817011568   -3.163725018
     480.00000000  -18453.06134549   -3150.83256134  -18685.83030936    2.106017925   -2.860236337   -1.586151870
     600.00000000    3425.11742384  -18514.73232706  -18588.67200557    3.394666340   -1.003072030    1.610061295
     720.00000000   21858.23838149  -15101.51661554     387.34517048    1.247973967    1.856017403    3.161439948
     840.00000000   18360.69935796    3506.55256762   19024.81678979   -2.122684184    2.830618605    1.537510677
     960.00000000   -3412.84765409   18646.85269710   18748.00359987   -3.366815728    0.986039922   -1.607874972
    1080.00000000  -21758.08331586   15215.44829478    -180.82181406   -1.250144680   -1.856490448   -3.163774870
    1200.00000000  -18193.41290284   -3493.85876912  -18877.14757717    2.153326942   -2.852221264   -1.536617760
    1320.00000000    3833.57386848  -18635.77026711  -18388.68722885    3.384748179   -0.955363841    1.658785020
    1440.00000000   22002.20074562  -14879.72595593     774.32827099    1.191573619    1.894561165    3.159953047
  28350  xx
       0.00000000    6333.08123128   -1580.82852326      90.69355720    0.714634423    3.224246550    7.083128132
     120.00000000   -3990.93845855    3052.98341907    4155.32700629   -5.909006188   -0.876307966   -5.039131404
     240.00000000    -603.55232010   -2685.13474569   -5891.70274282    7.572519907   -1.975656726    0.121722605
     360.00000000    4788.22345627     782.56169214    4335.14284621   -4.954509026    3.683346464    4.804645839
     480.00000000   -6291.84601644    1547.82790772    -453.67116498   -0.308625588   -3.341538574   -7.082659115
     600.00000000    4480.74573428   -3028.55200374   -3586.94343641    5.320920857    1.199736275    5.626350481
     720.00000000    -446.42460916    2932.28872588    5759.19389757   -7.561000245    1.550975493   -1.374970885
     840.00000000   -3713.79581831   -1382.66125130   -5122.45131136    6.090931626   -3.512629733   -3.467571746
     960.00000000    6058.32017522    -827.47406722    2104.04678651   -1.798403024    3.787067272    6.641439744
    1080.00000000   -5631.73659006    2623.70953644    1766.49125084   -3.216401578   -2.309140959   -6.788609120
    1200.00000000    2776.84991560   -3255.36941953   -4837.19667790    6.748135564   -0.193044825    4.005718698
    1320.00000000    1148.04430837    2486.07343386    5826.34075913   -7.420162295    2.589456382    0.356350006
    1440.00000000   -4527.90871828    -723.29199041   -4527.44608319    5.121674217   -3.909895427   -4.500218556
  28623  xx
       0.00000000  -11665.70902324   24943.61433357      25.80543633   -1.596228621   -1.476127961    1.126059754
     120.00000000  -11645.35454950     979.37668356    5517.89500058    3.407743502   -5.183094988   -0.492983277
     240.00000000    5619.19252274   19651.44862280   -7261.38496765   -2.013634213    3.106842861    0.284235517
     360.00000000   -9708.68629714   26306.14553149   -1204.29478856   -1.824164290   -0.931909596    1.113419052
     480.00000000  -14394.03162892    6659.30765074    5593.38345858    1.556522911   -4.681657614    0.296912248
     600.00000000    7712.09476270   15565.72627434   -7342.40465571   -1.646800364    4.070313571   -0.109483081
     720.00000000   -7558.36739603   27035.11367962   -2385.12054184   -1.999583791   -0.393409283    1.078093515
     840.00000000  -15495.61862220   11550.15897828    5053.83178121    0.469277336   -4.029761073    0.679054742
     960.00000000    9167.02568222   10363.65204210   -6871.52576042   -0.881621027    5.223361510   -0.740696297
    1080.00000000   -5275.80272094   27151.78486008   -3494.50687216   -2.129609388    0.150196480    1.021038089
    1200.00000000  -15601.37656145   15641.29379850    4217.03266850   -0.249183123   -3.405238557    0.888214503
    1320.00000000    9301.05872300    3883.15265574   -5477.86477017    0.871447821    6.493677331   -1.885545282
    1440.00000000   -2914.31065828   26665.20392758   -4511.09814335   -2.216261909    0.710067769    0.940691824
  28626  xx
       0.00000000   42080.71852213   -2646.86387436       0.81851294    0.193105177    3.068688251    0.000438449
     120.00000000   37740.00085593   18802.76872802       3.45512584   -1.371035206    2.752105932    0.000336883
     240.00000000   23232.82515008   35187.33981802       4.98927428   -2.565776620    1.694193132    0.000163365
     360.00000000    2467.44290178   42093.60909959       5.15062987   -3.069341800    0.179976276   -0.000031739
     480.00000000  -18962.59052991   37661.66243819       4.04433258   -2.746151982   -1.382675777   -0.000197633
     600.00000000  -35285.00095313   23085.44402778       2.08711880   -1.683277908   -2.572893625   -0.000296282
     720.00000000  -42103.20138132    2291.06228893      -0.13274964   -0.166974816   -3.070104560   -0.000311007
     840.00000000  -37580.31858370  -19120.40485693      -2.02755702    1.394367848   -2.740341612   -0.000248591
     960.00000000  -22934.20761876  -35381.23870806      -3.16495932    2.580167539   -1.672360951   -0.000134907
    1080.00000000   -2109.90332389  -42110.71508198      -3.36507889    3.070935369   -0.153808390   -0.000005855
    1200.00000000   19282.77774728  -37495.59250598      -2.71861462    2.734400524    1.406220933    0.000103486
    1320.00000000   35480.60990600  -22779.03375285      -1.52841859    1.661210676    2.587414593    0.000168300
    1440.00000000   42119.96263499   -1925.77567263      -0.19827433    0.140521206    3.071541613    0.000179561
  28872  xx
       0.00000000   -6131.82730456    2446.52815528    -253.64211033   -0.144920228    0.995100963    7.658645067
       5.00000000   -5799.24256134    2589.14811119    2011.54515100    2.325207364   -0.047125672    7.296234071
      10.00000000   -4769.05061967    2420.46580562    4035.30855837    4.464585796   -1.060923209    6.070907874
      15.00000000   -3175.45157340    1965.98738086    5582.12569607    6.049639376   -1.935777558    4.148607019
      20.00000000   -1210.19024802    1281.54541294    6474.68172772    6.920746273   -2.580517337    1.748783868
      25.00000000     896.73799533     447.12357305    6607.22400507    6.983396282   -2.925846168   -0.872655207
      30.00000000    2896.99663534    -440.04738594    5954.92675486    6.211488246   -2.926949815   -3.433959806
      35.00000000    4545.78970167   -1273.55952872    4580.16512984    4.656984233   -2.568711513   -5.638510954
      40.00000000    5627.43299371   -1947.94282469    2634.16714930    2.464141047   -1.873985161   -7.195743032
      45.00000000    5984.72318534   -2371.37691609     349.87996209   -0.121276950   -0.911981546   -7.859613894
      50.00000000    5548.43325922   -2480.16469245   -1979.24314527   -2.763269534    0.199691915   -7.482796996
  29141  xx
       0.00000000     423.99295524   -6658.12256149     136.13040356    1.006373613    0.217309983    7.662587892
      20.00000000     931.80883587   -1017.17852239    6529.19244527   -0.298847918    7.613891977    1.226399480
      40.00000000     -83.44906141    6286.20208453    2223.49837161   -1.113515974    2.530970283   -7.219445568
      60.00000000    -958.57681221    3259.26005348   -5722.63732467   -0.101225813   -6.735338321   -3.804851872
      80.00000000    -255.25619985   -5132.59762974   -4221.27233118    1.077709303   -4.905938824    5.892521264
     100.00000000     867.44295097   -5038.40402933    4256.73810533    0.479447535    5.032326446    5.857126248
     120.00000000     559.16882013    3376.30587937    5699.22017391   -0.906749328    6.646149867   -3.852331832
     140.00000000    -669.85184205    6196.00229484   -2281.95741770   -0.795804092   -2.752114827   -7.202478520
     160.00000000    -784.20708019   -1278.53125553   -6449.19892596    0.636702380   -7.595425203    1.431090802
     180.00000000     406.15811659   -6607.03115799     148.33021477    1.009818575    0.231843765    7.692047844
     200.00000000     916.34911813    -884.08649248    6491.09810362   -0.302163049    7.669887109    1.084336909
     220.00000000    -104.02490970    6304.31821405    1960.08739882   -1.108873823    2.259522809   -7.351147710
     240.00000000    -944.61642849    2872.17248379   -5846.94103362   -0.051117686   -6.989747076   -3.413102600
     260.00000000    -187.16569888   -5404.86163467   -3731.97057618    1.094696706   -4.412110995    6.326060952
     280.00000000     884.59720467   -4465.74516163    4725.83632696    0.380656028    5.691554046    5.303910983
     300.00000000     446.40767236    4086.66839620    5093.05596650   -0.982424447    6.072965199   -4.791630682
     320.00000000    -752.24467495    5588.35473301   -3275.04092573   -0.661161370   -4.016290740   -6.676898026
     340.00000000    -643.72872525   -2585.02528560   -5923.01306608    0.807922142   -7.171597814    3.041115058
     360.00000000     584.40295819   -6202.35605817    1781.00536019    0.869250450    2.226927514    7.471676765
     380.00000000     779.59211765    1100.73728301    6311.59529480   -0.599552305    7.721032522   -1.275153027
     400.00000000    -403.03155588    6399.18000837    -364.12735875   -1.008861924   -0.516636615   -7.799812287
     420.00000000    -852.93910071     192.65232023   -6322.47054784    0.396006194   -7.882964919   -0.289331517
  29238  xx
       0.00000000   -5566.59512819   -3789.75991159      67.60382245    2.873759367   -3.825340523    6.023253926
     120.00000000    4474.27915495   -1447.72286142    4619.83927235    4.712595822    5.668306153   -2.701606741
     240.00000000    1922.17712474    5113.01138342   -4087.08470203   -6.490769651   -0.522350158   -3.896001154
     360.00000000   -6157.93546882   -2094.70798790   -1941.63730960    0.149900661   -5.175192523    5.604262034
     480.00000000    2482.64052411   -3268.45944555    5146.38006190    6.501814698    4.402848754   -0.350943511
     600.00000000    4036.26455287    4827.43347201   -2507.99063955   -5.184409515    1.772280695   -5.331390168
     720.00000000   -5776.81371622    -118.64155319   -3641.22052418   -2.539917207   -5.622701582    4.403125405
     840.00000000      67.98699487   -4456.49213473    4863.71794283    7.183809420    2.418917791    2.015642495
     960.00000000    5520.62207038    3782.38203554    -596.73193161   -3.027966069    3.754152525   -6.013506363
    1080.00000000   -4528.05104455    1808.46273329   -4816.99727762   -4.808419763   -5.185789345    2.642104494
    1200.00000000   -2356.61468078   -4852.51202272    3856.53816184    6.688446735    0.118520958    4.021854210
    1320.00000000    6149.65800134    2173.59423261    1369.29488732   -0.345832777    5.109857861   -5.842951828
    1440.00000000   -2629.55011449    3400.98040158   -5344.38217129   -6.368548448   -3.998963509    0.577253064
  88888  xx
       0.00000000    2328.96975262   -5995.22051338    1719.97297192    2.912073281   -0.983417956   -7.090816210
     120.00000000    1020.69234558    2286.56260634   -6191.55565927   -3.746543902    6.467532721    1.827985678
     240.00000000   -3226.54349155    3503.70977525    4532.80979343    1.000992116   -5.788042888    5.162585826
     360.00000000    2456.10706533   -6071.93855503    1222.89768554    2.679390040   -0.448290811   -7.228792155
     480.00000000     787.16457349    2719.91800946   -6043.86662024   -3.759883839    6.277439314    2.397897864
     600.00000000   -3110.97648029    3121.73026235    4878.15217035    1.244916056   -6.124880425    4.700576353
     720.00000000    2567.56229695   -6112.50383922     713.96374435    2.440245751    0.098109002   -7.319959258
     840.00000000     556.05661780    3144.52288201   -5855.34636178   -3.754660143    6.044752775    2.957941672
     960.00000000   -2982.47940539    2712.61663711    5192.32330472    1.475566773   -6.427737014    4.202420227
    1080.00000000    2663.08964352   -6115.48290885     196.40072866    2.196121564    0.652415093   -7.362824152
    1200.00000000     328.54999674    3557.09490552   -5626.21427211   -3.731193288    5.769341172    3.504058731
    1320.00000000   -2842.06876757    2278.42343492    5472.33437150    1.691852635   -6.693216335    3.671022712
    1440.00000000    2742.55398832   -6079.67009123    -326.39012649    1.948497651    1.211072678   -7.356193131

# ex: set textwidth=72 :
